/**
  *********************************************************************************
  *
  * @file    bsp_uart_lin.c
  * @brief   uart_lin driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_uart_lin.h"

/** @addtogroup ES32F03xx_BSP
  * @{
  */

/**
  * @defgroup UART_LIN uart lin
  * @{
  */

/**
  * @defgroup UART_LIN_Public_Params uart lin Public Variables
  * @{
  */

uart_handle_t h_uart;

/**
  * @}
  */

/**
  * @defgroup UART_LIN_Private_Functions uart lin Private Functions
  * @{
  */

/**
  * @brief  Initializate pin of uart_lin module.
  * @retval None
  */
static void uart_lin_pin_init(void)
{
	gpio_init_t x;

	/* Initialize lin_tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize lin_rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	/* Initialize aj1020_sel pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(GPIOD, GPIO_PIN_3, &x);

	return;
}
/**
  * @}
  */

/**
  * @defgroup UART_LIN_Public_Functions uart lin Public Functions
  * @brief    BSP UART_LIN Functions
  * @verbatim
=========================================================================================
              ##### BSP UART_LIN Functions #####
=========================================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp uart_lin module.
      (+) Send data by uart_lin module.
      (+) Recv data by uart_lin module.
    @endverbatim
  * @{
  */


/**
  * @brief  Initializate uart_lin module using specified baud_rate.
  * @param  baud: specified uart baud_rate.
  * @retval None
  */
void uart_lin_module_init(uint32_t baud)
{
	/* Initialize pin */
	uart_lin_pin_init();

	memset(&h_uart, 0, sizeof(h_uart));

	/* Initialize uart */
	h_uart.perh             = UART1;
	h_uart.init.baud        = baud;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_LIN;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	ald_uart_init(&h_uart);

	/* enable break signal detection interrupt */
	CLEAR_BIT(h_uart.perh->MCR, UART_MCR_LINBDL_MSK);
	/* set break signal length */
	SET_BIT(h_uart.perh->IER, UART_IER_LINBKIE_MSK);
	/* Enable AJ1020 */
	ald_gpio_write_pin(GPIOD, GPIO_PIN_3, 1);

	return ;
}

/**
  * @brief  Send break signal.
  * @retval NULL
  */
void send_break_signal(void)
{
	/* Send break signal */
	SET_BIT(h_uart.perh->MCR, UART_MCR_BKREQ_MSK);

	return ;
}

/**
  * @brief  Sends an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref lin_status_t.
  */
lin_status_t bsp_uart_lin_send(uint8_t *buf, uint16_t size)
{
	if ((buf == NULL) || (size == 0))
		return  LIN_ERROR;

	UART_RX_DISABLE(&h_uart);

	if (ald_uart_send(&h_uart, buf, size, 500) != OK)
		return  LIN_ERROR;

	UART_RX_ENABLE(&h_uart);

	return LIN_OK;
}

/**
  * @brief  Recvs an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref lin_status_t.
  */
lin_status_t bsp_uart_lin_recv(uint8_t *buf, uint16_t size)
{
	if ((buf == NULL) || (size == 0))
		return  LIN_ERROR;

	if (ald_uart_recv(&h_uart, buf, size, 500) != OK)
		return  LIN_ERROR;

	return LIN_OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
