/**
  *********************************************************************************
  *
  * @file    bsp_uart_irda.c
  * @brief   uart_irda driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_uart_irda.h"

/** @addtogroup ES32F03xx_BSP
  * @{
  */

/**
  * @defgroup UART_IRDA uart irda
  * @{
  */

/**
  * @defgroup UART_IRDA_Public_Variable uart irda Public Variables
  * @{
  */
static uart_handle_t h_uart_irda;
/**
  * @}
  */

/**
  * @defgroup UART_IRDA_Private_Functions uart irda Private Functions
  * @{
  */

 /**
  * @brief  initialize pin of uart_irda module  
  * @retval None
  */ 
static void uart_irda_pin_init(void)
{
	gpio_init_t x;

	/* Initialize irda_tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOE, GPIO_PIN_10, &x);

	/* Initialize lin_rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOE, GPIO_PIN_11, &x);

	return;
}
/**
  * @}
  */

/**
  * @defgroup UART_IRDA_Public_Functions uart irda Public Functions
  * @brief    BSP UART_IRDA Functions
  * @verbatim
=========================================================================================
              ##### BSP UART_IRDA Functions #####
=========================================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp uart_irda module.
      (+) Send data by uart_irda module.
      (+) Recv data by uart_irda module.
    @endverbatim
  * @{
  */
 
/**
  * @brief  uart_irda module initialize function
  * @retval None
  */
void uart_irda_module_init(uint32_t baud)
{
	/* Initialize pin */
	uart_irda_pin_init();

	memset(&h_uart_irda, 0, sizeof(h_uart_irda));

	/* Initialize uart */
	h_uart_irda.perh             = UART3;
	h_uart_irda.init.baud        = baud;
	h_uart_irda.init.word_length = UART_WORD_LENGTH_8B;
	h_uart_irda.init.stop_bits   = UART_STOP_BITS_1;
	h_uart_irda.init.parity      = UART_PARITY_NONE;
	h_uart_irda.init.mode        = UART_MODE_IrDA;
	h_uart_irda.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	ald_uart_init(&h_uart_irda);

	/* set RX line reverse recv */
	SET_BIT(h_uart_irda.perh->LCR, UART_LCR_RXINV_MSK);

	return ;
}

/**
  * @brief  Sends an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_irda_status_t.
  */
uart_irda_status_t bsp_uart_irda_send(uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return UART_IRDA_ERROR;

	if (ald_uart_send(&h_uart_irda, buf, size, 500) != OK)
		return UART_IRDA_ERROR;

	return UART_IRDA_OK;
}

/**
  * @brief  Recvs an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_irda_status_t.
  */
uart_irda_status_t bsp_uart_irda_recv(uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return UART_IRDA_ERROR;

	if (ald_uart_recv(&h_uart_irda, buf, size, 2000) != OK)
		return UART_IRDA_ERROR;

	return UART_IRDA_OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
