/**
  *********************************************************************************
  *
  * @file    bsp_uart_485.c
  * @brief   uart_485 driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_uart_485.h"

/** @addtogroup ES32F03xx_BSP
  * @{
  */

/**
  * @defgroup UART_485 uart 485
  * @{
  */

/**
  * @defgroup UART_485_Private_Params uart 485 Private Variables
  * @{
  */
static uart_handle_t h_uart;
/**
  * @}
  */

/**
  * @defgroup UART_485_Private_Functions uart 485 Private Functions
  * @{
  */

 /**
  * @brief  initialize pin of uart 485 module  
  * @retval None
  */ 
static void uart_485_pin_init(void)
{
	gpio_init_t x;

	/* Initialize 485_tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

	/* Initialize 485_rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_11, &x);

	/* Initialize 485 de pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(GPIOB, GPIO_PIN_1, &x);

	return;
}

/**
  * @brief  Enable uart 485 module send function.
  * @retval None.
  */
static void uart_485_send_enable(void)
{
	ald_gpio_write_pin(GPIOB, GPIO_PIN_1, 1);

	return ;
}

/**
  * @brief  Enable uart_485 module recv function.
  * @retval None.
  */
static void uart_485_recv_enable(void)
{
	ald_gpio_write_pin(GPIOB, GPIO_PIN_1, 0);

	return ;
}
/**
  * @}
  */

/**
  * @defgroup UART_485_Public_Functions uart 485 Public Functions
  * @brief    BSP UART_485 Functions
  * @verbatim
=========================================================================================
              ##### BSP UART_485 Functions #####
=========================================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp uart_485 module.
      (+) Send data by uart_485 module.
      (+) Recv data by uart_485 module.
    @endverbatim
  * @{
  */

/**
  * @brief  uart_485 module initialize function
  * @brief  baud: communication baud rate.
  * @brief  addr: communication device addr.
  * @retval None
  */
void uart_485_module_init(uint32_t baud, uint8_t addr)
{
	uart_485_pin_init();
	memset(&h_uart, 0x0, sizeof(h_uart));

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = baud;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_RS485;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;

	ald_uart_init(&h_uart);

	uart_485_recv_enable();
	MODIFY_REG(h_uart.perh->CR, UART_CR_ADDR_MSK, addr << UART_CR_ADDR_POSS);
}

/**
  * @brief  Sends an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_485_status_t.
  */
uart_485_status_t bsp_uart_485_send(uint8_t *buf, uint16_t size)
{
	ald_status_t status;

	if (buf == NULL)
		return UART_485_ERROR;

	uart_485_send_enable();
	status = ald_uart_send(&h_uart, buf, size, 500);
	uart_485_recv_enable();

	if (status != OK)
		return UART_485_ERROR;

	return UART_485_OK;
}

/**
  * @brief  Sends address in blocking mode.
  * @param  addr: address of slave device.
  * @retval Status, see @ref uart_485_status_t.
  */
uart_485_status_t bsp_uart_485_send_addr(uint16_t addr)
{
	ald_status_t status;

	uart_485_send_enable();
	status = ald_uart_rs485_send_addr(&h_uart, (addr | 0x100), 500);
	uart_485_recv_enable();

	if (status != OK)
		return UART_485_ERROR;

	return UART_485_OK;
}

/**
  * @brief  Recvs an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_485_status_t.
  */
uart_485_status_t bsp_uart_485_recv(uint8_t *buf, uint16_t size)
{
	ald_status_t status;

	if (buf == NULL)
		return UART_485_ERROR;

	uart_485_recv_enable();
	status = ald_uart_recv(&h_uart, buf, size, 2000);

	if (status != OK)
		return UART_485_ERROR;

	return UART_485_OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
