/**
  *********************************************************************************
  *
  * @file    bsp_uart_38k.c
  * @brief   uart_38k driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_uart_38k.h"

/** @addtogroup ES32F03xx_BSP
  * @{
  */

/**
  * @defgroup UART_38K uart 38k
  * @{
  */

/**
  * @defgroup UART_38K_Parivate_Params uart 38k Public Variables 
  * @{
  */
static uart_handle_t h_uart;
static timer_handle_t h_tim;
static timer_oc_init_t tim_ocinit;
/**
  * @}
  */

/**
  * @defgroup UART_38K_Private_Functions uart 38k Private Functions
  * @{
  */

 /**
  * @brief  initialize pin of uart_38k module  
  * @retval None
  */ 
static void bsp_uart_38k_pin_init(void)
{
	gpio_init_t x;

	/* Initialize 38k_tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOE, GPIO_PIN_10, &x);

	/* Initialize lin_rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOE, GPIO_PIN_11, &x);

	return;
}
/**
  * @}
  */

/**
  * @defgroup UART_38K_Public_Functions uart 38k Public Functions
  * @brief    BSP UART_38K Functions
  * @verbatim
=========================================================================================
              ##### BSP UART_38K Functions #####
=========================================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp uart_38k module.
      (+) Send data by uart_38k module.
      (+) Recv data by uart_38k module.
    @endverbatim
  * @{
  */

/**
  * @brief  Initializate uart_lin module using specified baud-rate.
  * @param  baud: specified uart baud-rate.
  * @param  mode: specified modulate singal mode.
                  0--High Modulation
		  1--Low Modulation
  * @retval None
  */
void uart_38k_module_init(uint32_t baud, uint8_t mode)
{
	/* clear uart_handle_t structure */
	memset(&h_uart, 0x0, sizeof(h_uart));
	/* clear tim_handle_t structure */
	memset(&h_tim, 0x0, sizeof(h_tim));
	/* clear tim_oc_init_t structure */
	memset(&tim_ocinit, 0x0, sizeof(tim_ocinit));

	/* Initialize pin */
	bsp_uart_38k_pin_init();

	/* Initialize uart */
	h_uart.perh             = UART3;
	h_uart.init.baud        = baud;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	ald_uart_init(&h_uart);

	/* Initialize tim0 */
#ifdef ES32F065x	
	h_tim.perh              = AD16C4T0;
#endif
#ifdef ES32F033x
	h_tim.perh              = GP16C4T0;
#endif
	h_tim.init.prescaler    = 0;
	h_tim.init.mode         = TIMER_CNT_MODE_UP;
	h_tim.init.period       = 699;
	h_tim.init.clk_div      = TIMER_CLOCK_DIV1;
	h_tim.init.re_cnt       = 0;
	ald_timer_pwm_init(&h_tim);

	/* set tim0 channel */
	tim_ocinit.oc_mode        = TIMER_OC_MODE_PWM1;
        tim_ocinit.oc_polarity    = TIMER_OC_POLARITY_HIGH;
        tim_ocinit.oc_fast_en     = DISABLE;
        tim_ocinit.ocn_polarity   = TIMER_OCN_POLARITY_HIGH;
        tim_ocinit.ocn_idle       = TIMER_OCN_IDLE_RESET;
        tim_ocinit.oc_idle        = TIMER_OC_IDLE_RESET;

        /* Set the pulse value for channel 1 */
        tim_ocinit.pulse = 350;
	ald_timer_oc_config_channel(&h_tim, &tim_ocinit, TIMER_CHANNEL_1);

	ald_timer_pwm_set_freq(&h_tim, 38000);
	ald_timer_pwm_set_duty(&h_tim, TIMER_CHANNEL_1, 50);

	ald_timer_pwm_start(&h_tim, TIMER_CHANNEL_1);

	/* select modulate singal source TIM0 */
	MODIFY_REG(PIS->UART3_TXMCR, PIS_TXMCR_TXMSS_MSK, 1 << PIS_TXMCR_TXMSS_POSS);
	/* select modulate singal source channel */
	MODIFY_REG(PIS->UART3_TXMCR, PIS_TXMCR_TXSIGS_MSK, 0 << PIS_TXMCR_TXSIGS_POSS);
	/* select modulate singal mode -- low level*/
	MODIFY_REG(PIS->UART3_TXMCR, PIS_TXMCR_TXMLVLS_MSK, mode << PIS_TXMCR_TXMLVLS_POS);

	return ;
}

/**
  * @brief  Sends an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_38k_status_t.
  */
uart_38k_status_t bsp_uart_38k_send(uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return UART_38K_ERROR;

	if (ald_uart_send(&h_uart, buf, size, 500) != OK)
		return UART_38K_ERROR;

	return UART_38K_OK;
}

/**
  * @brief  Recvs an amount of data in blocking mode.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref uart_38k_status_t.
  */
uart_38k_status_t bsp_uart_38k_recv(uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return UART_38K_ERROR;

	if (ald_uart_recv(&h_uart, buf, size, 2000) != OK)
		return UART_38K_ERROR;

	return UART_38K_OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
