/**
  *********************************************************************************
  *
  * @file    bsp_rgb.c
  * @brief   RBG driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
 #include <stdio.h>
 #include "bsp_rgb.h"

 /** @addtogroup ES32F03xx_BSP
  * @{
  */

 /**
  * @defgroup RGB rgb
  * @{
  */

/**
  * @defgroup RGB_Private_Function rgb Private Functions
  * @{
  */

/**
  * @brief  Initializate pin of tim3 module.
  * @retval None
  */
static void tim3_pin_init(void)
{
	gpio_init_t GPIO_InitStructure;

	/* Initialize tim0 ch1 pin */
	GPIO_InitStructure.mode = GPIO_MODE_OUTPUT;
	GPIO_InitStructure.odos = GPIO_PUSH_PULL;
	GPIO_InitStructure.pupd = GPIO_PUSH_UP;
	GPIO_InitStructure.odrv = GPIO_OUT_DRIVE_NORMAL;
	GPIO_InitStructure.flt  = GPIO_FILTER_DISABLE;
	GPIO_InitStructure.type = GPIO_TYPE_TTL;
	GPIO_InitStructure.func = GPIO_FUNC_3;

        ald_gpio_init(GPIOC, GPIO_PIN_6, &GPIO_InitStructure);
        ald_gpio_init(GPIOC, GPIO_PIN_7, &GPIO_InitStructure);

}

#ifdef ES32F0654
/**
  * @brief  Initializate pin of tim0 module.
  * @retval None
  */
static void tim0_pin_init(void)
{
	gpio_init_t a;

	/* Initialize lptim0 output pin */
	a.mode = GPIO_MODE_OUTPUT;
	a.odos = GPIO_PUSH_PULL;
	a.pupd = GPIO_PUSH_UP;
	a.odrv = GPIO_OUT_DRIVE_NORMAL;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_TTL;
	a.func = GPIO_FUNC_6;

	ald_gpio_init(GPIOB, GPIO_PIN_0, &a);
}
#else
/**
  * @brief  Initializate pin of lptim0 module.
  * @retval None
  */
static void lptim0_pin_init(void)
{
	gpio_init_t a;

	/* Initialize lptim0 output pin */
	a.mode = GPIO_MODE_OUTPUT;
	a.odos = GPIO_PUSH_PULL;
	a.pupd = GPIO_PUSH_UP;
	a.odrv = GPIO_OUT_DRIVE_NORMAL;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_TTL;
	a.func = GPIO_FUNC_2;

	ald_gpio_init(GPIOC, GPIO_PIN_5, &a);
}
#endif

/**
  * @brief  tim3 chan1 chan2 PWM out initialize
  * @retval None
  */
static void tim3_module_init(timer_handle_t *tim_h, timer_oc_init_t *tim_ocinit, timer_clock_config_t *tim_clock)
{
	/* Initialize pin */
	tim3_pin_init();

	/* Initialize TIM3 */
	tim_h->perh              = GP16C2T1;
	tim_h->init.prescaler    = 0;
	tim_h->init.mode         = TIMER_CNT_MODE_UP;
	tim_h->init.period       = (uint32_t)(255 - 1);
	tim_h->init.clk_div      = TIMER_CLOCK_DIV1;
	tim_h->init.re_cnt       = 0;
	ald_timer_pwm_init(tim_h);

	/* Initialize clock source */
	tim_clock->source = TIMER_SRC_INTER;
	ald_timer_config_clock_source(tim_h, tim_clock);

	/* Common configuration for all channels */
	tim_ocinit->oc_mode        = TIMER_OC_MODE_PWM1;
        tim_ocinit->oc_polarity    = TIMER_OC_POLARITY_HIGH;
        tim_ocinit->oc_fast_en     = DISABLE;
        tim_ocinit->ocn_polarity   = TIMER_OCN_POLARITY_HIGH;
        tim_ocinit->ocn_idle       = TIMER_OCN_IDLE_RESET;
        tim_ocinit->oc_idle        = TIMER_OC_IDLE_RESET;

        /* Set the pulse value for channel 1 */
        tim_ocinit->pulse = (uint32_t)(255 * 20 / 100);
	ald_timer_oc_config_channel(tim_h, tim_ocinit, TIMER_CHANNEL_1);

        /* Set the pulse value for channel 2 */
        tim_ocinit->pulse = (uint32_t)(255 * 50 / 100);
	ald_timer_oc_config_channel(tim_h, tim_ocinit, TIMER_CHANNEL_2);

	/* Start input pwm from tim2/3 channel 1 */
	ald_timer_oc_start(tim_h, TIMER_CHANNEL_1);

        /* Start input pwm from tim2/3 channel 2 */
	ald_timer_oc_start(tim_h, TIMER_CHANNEL_2);
}
#ifdef ES32F0654
/**
  * @brief  tim0 chan2 PWM out initialize
  * @retval None
  */
static void tim0_module_init(timer_handle_t *h_tim, timer_oc_init_t *timer_ocinit, timer_clock_config_t *tim_clock)
{
	tim0_pin_init();

	/* Initialize TIM0 */
	h_tim->perh              = AD16C4T0;
	h_tim->init.prescaler    = 0;
	h_tim->init.mode         = TIMER_CNT_MODE_UP;
	h_tim->init.period       = (uint32_t)(255 - 1);
	h_tim->init.clk_div      = TIMER_CLOCK_DIV1;
	h_tim->init.re_cnt       = 0;
	ald_timer_pwm_init(h_tim);

	/* Initialize clock source */
	tim_clock->source = TIMER_SRC_INTER;
	ald_timer_config_clock_source(h_tim, tim_clock);

	/* Common configuration for all channels */
	timer_ocinit->oc_mode        = TIMER_OC_MODE_PWM1;
        timer_ocinit->oc_polarity    = TIMER_OC_POLARITY_HIGH;
        timer_ocinit->oc_fast_en     = DISABLE;
        timer_ocinit->ocn_polarity   = TIMER_OCN_POLARITY_HIGH;
        timer_ocinit->ocn_idle       = TIMER_OCN_IDLE_RESET;
        timer_ocinit->oc_idle        = TIMER_OC_IDLE_RESET;

        /* Set the pulse value for channel 2 */
        timer_ocinit->pulse = (uint32_t)(255 * 50 / 100);
	ald_timer_oc_config_channel(h_tim, timer_ocinit, TIMER_CHANNEL_2);

        /* Start input pwm from tim0 channel 2 */
	ald_timer_ocn_start(h_tim, TIMER_CHANNEL_2);
}
#else
/**
  * @brief  lptim0 PWM out initialize
  * @retval None
  */
static void lptim0_module_init(lptim_handle_t *lptim_h, lptim_trigger_init_t *lptim_trig, lptim_clock_source_init_t *lptim_clk)
{
        ald_cmu_lptim0_clock_select(CMU_LP_PERH_CLOCK_SEL_PLL1);

	/* Initialize LPTIM0 */
	lptim0_pin_init();

	/* Initialize LPTIM0 */
	lptim_h->perh      = LPTIM0;
	lptim_h->init.psc  = LPTIM_PRESC_1;
	lptim_h->init.arr  = 255;
	lptim_h->init.cmp  = 100;
	lptim_h->init.mode = LPTIM_MODE_CONTINUOUS;

	ald_lptim_pwm_init(lptim_h);

	/* Initialize clock source */
	lptim_clk->sel      = LPTIM_CKSEL_INTERNAL;
	lptim_clk->polarity = LPTIM_CKPOL_RISING;
	ald_lptim_clock_source_config(lptim_h, lptim_clk);

	/* Initialize toggle */
	lptim_trig->mode = LPTIM_TRIGEN_SW;
	ald_lptim_trigger_config(lptim_h, lptim_trig);

	ald_lptim_pwm_start(lptim_h);
}
#endif

/**
  * @}
  */

/**
  * @defgroup RGB_Public_Function rgb Public Functions
  * @brief    BSP RGB Functions
  * @verbatim
 ===============================================================================
              ##### BSP RGB Functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp rgb pwm module;
      (+) set lptim pwm duty;

    @endverbatim
  * @{
  */

/**
  * @brief  rgb module initialize
  * @retval None
  */
void rgb_module_init(void *rgb_tim3, void *rgb_lptim0)
{
	rgb_tim3_env_t *p_tim = (rgb_tim3_env_t *)rgb_tim3;

#ifdef ES32F0654
	rgb_tim3_env_t *p_lptim = (rgb_tim3_env_t *)rgb_lptim0;
#else
	rgb_lptim0_env_t *p_lptim = (rgb_lptim0_env_t *)rgb_lptim0;
#endif


	if ((rgb_tim3 == NULL) || (rgb_lptim0 == NULL))
		return;

	tim3_module_init(&p_tim->tim_h, &p_tim->tim_ocinit, &p_tim->tim_clock);

#ifdef ES32F0654
	tim0_module_init(&p_lptim->tim_h, &p_lptim->tim_ocinit, &p_lptim->tim_clock);
#else
	lptim0_module_init(&p_lptim->lptim_h, &p_lptim->lptim_trig, &p_lptim->lptim_clk);
#endif
}

/**
  * @brief  rgb set lptim duty
  * @retval None
  */
void lptim_pwm_set_duty(void *rgb_lptim0, uint8_t duty)
{
#ifdef ES32F0654
	rgb_tim3_env_t *p_lptim = (rgb_tim3_env_t *)rgb_lptim0;
#else
	rgb_lptim0_env_t *p_lptim = (rgb_lptim0_env_t *)rgb_lptim0;
#endif

	if ((rgb_lptim0 == NULL) || (duty > 15))
		return;

#ifdef ES32F0654
	ald_timer_pwm_set_duty(&p_lptim->tim_h, TIMER_CHANNEL_2, 100 - duty * 6);
#else
	ald_lptim_pwm_stop(&p_lptim->lptim_h);
	MODIFY_REG(p_lptim->lptim_h.perh->CMP, LP16T_CMP_CMP_MSK, (duty * 15) << LP16T_CMP_CMP_POSS);
	ald_lptim_pwm_start(&p_lptim->lptim_h);
#endif
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
