/**
  *********************************************************************************
  *
  * @file    bsp_rf.c
  * @brief   rf driver.
  *
  * @version V1.0
  * @date    04 Jul 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          04 Jul 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 
#include "bsp_rf.h"

/** @addtogroup ES32F03xx_BSP
  * @{
  */

/** @defgroup RF rf 
  * @{
  */

/** @defgroup RF_Public_Variable rf Public Variables
  * @{
  */
volatile uint8_t RF_Irq_Flag = 0;
/**
  * @}
  */

/** @defgroup RF_Private_Variable rf Private Variables
  * @{
  */
static spi_handle_t h_spi;
/**
  * @}
  */

/** @defgroup RF_Private_Functions rf Private Functions
  * @{
  */
/**
  * @brief  Initialize RF's pin.
  * @retval None
  */
static void hw3000_ConfigGPIO(void)
{
	/* PD8  --> IRQ  PD9  --> CS PD10 --> GP1 PD11 --> GP2 */
	/* PB3  --> SCLK PB4  --> MISO PB5  --> MOSI PC8  --> PDN */

	gpio_init_t x;
	exti_init_t exti;

	/* MISO */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_4;
	ald_gpio_init(PORT_MISO, PIN_MISO, &x);

	/* GP1 */
	x.func = GPIO_FUNC_1;
	ald_gpio_init(PORT_GP1, PIN_GP1, &x);

	/* GP2 */
	x.func = GPIO_FUNC_1;
	ald_gpio_init(PORT_GP2, PIN_GP2, &x);

	/* IRQ */
	x.func = GPIO_FUNC_1;
	ald_gpio_init(PORT_IRQ, PIN_IRQ, &x);

	/* CS */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(PORT_CS, PIN_CS, &x);

	/* PDN */
	ald_gpio_init(PORT_PDN, PIN_PDN, &x);

	/* SCK */
	x.func = GPIO_FUNC_4;
	ald_gpio_init(PORT_SCK, PIN_SCK, &x);

	/* MOSI */
	x.func = GPIO_FUNC_4;
	ald_gpio_init(PORT_MOSI, PIN_MOSI, &x);

	/* Initialize external interrupt */
	exti.filter      = ENABLE;
	exti.cks         = EXTI_FILTER_CLOCK_32K;
	exti.filter_time = 1;

	ald_gpio_exti_init(PORT_IRQ, PIN_IRQ, &exti);
	ald_gpio_exti_clear_flag_status(PIN_IRQ);
	ald_gpio_exti_interrupt_config(PIN_IRQ, EXTI_TRIGGER_TRAILING_EDGE, ENABLE);
	/* enable exti 8 */
	ald_mcu_irq_config(EXTI8_11_IRQn, 2, ENABLE);

	h_spi.perh               = SPI0;
	h_spi.init.mode          = SPI_MODE_MASTER;
	h_spi.init.dir           = SPI_DIRECTION_2LINES;
	h_spi.init.data_size     = SPI_DATA_SIZE_8;
	h_spi.init.baud          = SPI_BAUD_8;
	h_spi.init.phase         = SPI_CPHA_SECOND;
	h_spi.init.polarity      = SPI_CPOL_LOW;
	h_spi.init.first_bit     = SPI_FIRSTBIT_MSB;
	h_spi.init.ss_en         = DISABLE;
	h_spi.init.crc_calc    	 = DISABLE;
	ald_spi_init(&h_spi);
}

/**
  * @brief  Initialize HW3000 Module.
  * @retval None
  */
static void hw3000_init(void)
{
	uint8_t i;
	uint16_t agc_table[16] = {0x1371, 0x1351, 0x0B51, 0x2F71, 0x2F51, 0x2E51,
				  0x2E31, 0x4B31, 0x4731, 0x4631, 0x4531, 0x4431,
				  0x6131, 0x6031, 0x6011, 0x6009};

	hw3000_write_reg(0x4C, 0x5555);		//Bank 0 open by default
	while (!(hw3000_read_reg(INTFLAG) & 0x4000)); 	//wait for chip ready

	hw3000_write_reg(0x4C, 0x55AA);		//Bank 1 open

	for (i = 0; i < 16; i++)
		hw3000_write_reg(0x1B + i, agc_table[i]);

	hw3000_write_reg(BK03, 0x0508);
	hw3000_write_reg(BK11, 0xC630);

	hw3000_write_reg(BK14, 0x1935);		//10kbps
	hw3000_write_reg(BK40, 0x0008);
	hw3000_write_reg(BK41, 0x0010);
	hw3000_write_reg(BK42, 0x82D8);
	hw3000_write_reg(BK43, 0x3D38);

	hw3000_write_reg(BK17, 0xF6C2);		//470MHz / 433MHzƵ

	hw3000_write_reg(BK51, 0x001B);
	hw3000_write_reg(BK55, 0x8003);
	hw3000_write_reg(BK56, 0x4155);
	hw3000_write_reg(BK62, 0x70ED);

	hw3000_write_reg(0x4C, 0x5555);		//Bank 0 open

	hw3000_write_reg(INTIC, 0x8000);	//clear por int

	hw3000_write_reg(MIXFW, 0x2E35);	//must modify this value
	hw3000_write_reg(MODECTRL, 0x100F); 	//must close GPIO clock for noise reason

	hw3000_write_reg(MODEMCTRL, 0x5201);	//20MHz osc select

	hw3000_write_reg(RFCFG, 0x3312);	//470.1MHz ֱƵģʽ
	hw3000_write_reg(FREQCFG1, 0x8000);	//433Mhz-->0x8013; 470.1Mhz-->0x8000;
	hw3000_write_reg(FREQCFG2, 0xA3D7);
	hw3000_write_reg(FREQCFG0, 0x005E);

	hw3000_write_reg(SYMRATE0, 0x0051);	//10kbps
	hw3000_write_reg(SYMRATE1, 0x00EC);
	hw3000_write_reg(FILTERBAND, 0x0025);
	hw3000_write_reg(DEVIATION, 0x0052);

	hw3000_write_reg(PACFG, 0xFF3F);	//20dBm

	hw3000_write_reg(PKTCTRL, 0xC000);	//֡ʽǿ֡ṹ
	hw3000_write_reg(PKTCFG0, 0x4010);	//SFD 2bytes, preamble 16 bytes
	hw3000_write_reg(PREACFG, 0x0206);
	hw3000_write_reg(PIPECTRL, 0x0001); 	//PIPE 0 enable
}

/**
  * @brief  Write Specific Value To Specific Register.
  * @param  addr: Destination Regsiter address.
  * @param  data: The Point of data which want to write.
  * @param  length: The Length of data which want to write.
  * @retval None.
  */
static void hw3000_write_fifo(uint8_t addr, uint8_t *data, uint16_t length)
{
	uint16_t i;

	addr |= 0x80;

    	RF_CS_0();

	ald_spi_send_byte_fast(&h_spi, addr);

	for (i = 0; i < length; i++)
		ald_spi_send_byte_fast(&h_spi, data[i]);

	while (ald_spi_get_flag_status(&h_spi, SPI_IF_BUSY) != RESET) {};

    	RF_CS_1();
}
/**
  * @brief  Read specific length data from destination Register.
  * @param  addr: Destination Regsiter address.
  * @param  data: Pointer address for storing data.
  * @param  length: The Length of data which want to read.
  * @retval None.
  */
static void hw3000_read_fifo(uint8_t addr, uint8_t *data, uint16_t length)
{
	uint16_t i;

	RF_CS_0();

	ald_spi_send_byte_fast(&h_spi, addr);
	data[0] = ald_spi_recv_byte_fast(&h_spi);	//read the first data as frame len

	for (i = 0; i < length; i++)
		data[i + 1] = ald_spi_recv_byte_fast(&h_spi);

	RF_CS_1();
}
/**
  * @}
  */
/**
  * @defgroup RF_Public_Functions rf Public Functions
  * @brief    BSP RF Public Functions.
  * @verbatim    
  ==============================================================================
              ##### RF Module Functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Initialize RF module.
      (+) Write Specific Value To Specific Register.
      (+) Read Specific Value From Specific Register.
    @endverbatim
  * @{
  */
/**
  * @brief  Initialize BSP HW3000 Module.
  * @retval None
  */
void bsp_InitHW3000(void)
{
	hw3000_ConfigGPIO();		/* GPIO */
	hw3000_init();
}

/**
  * @brief  Write Specific Value To Specific Register.
  * @param  addr: Destination Regsiter address.
  * @param  data: The Value want to write.
  * @retval None.
  */
void hw3000_write_reg(uint8_t addr, uint16_t data)
{
	addr |= 0x80;
    	RF_CS_0();

	ald_spi_send_byte_fast(&h_spi, addr);
	ald_spi_send_byte_fast(&h_spi, (uint8_t)(data >> 8));
	ald_spi_send_byte_fast(&h_spi, (uint8_t)data);

	while (ald_spi_get_flag_status(&h_spi, SPI_IF_BUSY) != RESET) {};

    	RF_CS_1();
}

/**
  * @brief  Read Value from Specific Register.
  * @param  addr: Destination Regsiter address.
  * @retval temp: The Value which read from specific address.
  */
uint16_t hw3000_read_reg(uint8_t addr)
{
	uint16_t temp = 0;

	RF_CS_0();

	ald_spi_send_byte_fast(&h_spi, addr);
	temp = ald_spi_recv_byte_fast(&h_spi);
	temp <<= 8;
	temp |= ald_spi_recv_byte_fast(&h_spi);

	RF_CS_1();

	return temp;
}
/**
  * @brief  Set HW3000 Transmit Power.
  * @param  powerdbm: Transmit Power see @ref POWER_e.
  * @retval None.
  */
void hw3000_power_set(POWER_e powerdbm)
{
	switch (powerdbm) {
	case POWER_N_40dBm:	//-40dBm
		hw3000_write_reg(PACFG, 0x003F);
		break;
	case POWER_N_16dBm:	//-16dBm
		hw3000_write_reg(PACFG, 0x013F);
		break;
	case POWER_N_10dBm:	//-10dBm
		hw3000_write_reg(PACFG, 0x023F);
		break;
	case POWER_N_5dBm:	//-5dBm
		hw3000_write_reg(PACFG, 0x043F);
		break;
	case POWER_0dBm:	//0dBm
		hw3000_write_reg(PACFG, 0x073F);
		break;
	case POWER_5dBm:	//5dBm
		hw3000_write_reg(PACFG, 0x0C3F);
		break;
	case POWER_10dBm:	//10dBm
		hw3000_write_reg(PACFG, 0x183F);
		break;
	case POWER_15dBm:	//15dBm
		hw3000_write_reg(PACFG, 0x2A3F);
		break;
	case POWER_18dBm:	//18dBm
		hw3000_write_reg(PACFG, 0x643F);
		break;
	case POWER_20dBm:	//20dBm
		hw3000_write_reg(PACFG, 0xFF3F);
		break;
	default:
		break;
	}
}
/**
  * @brief  Transmit Specific length of data by hw3000 module.
  * @param  txBuf: Pointer address for transmit data.
  * @param  len: The length of data which want to transmit.
  * @retval None.
  */
void hw3000_tx_data(uint8_t *txBuf, uint8_t len)
{

    	hw3000_write_reg(TRCTRL, 0x0100);	//tx_enable
    	hw3000_write_reg(FIFOSTA, 0x0100);	//flush fifo
    	hw3000_write_fifo(FIFODATA, txBuf, len);//write fifo
    	hw3000_write_reg(FIFOCTRL, 0x0001);	//ocpy = 1

    	while (!RF_Irq_Flag) {}; 			//wait for send finish
    	RF_Irq_Flag = 0;

    	hw3000_write_reg(FIFOCTRL, 0x0000);	//ocpy = 0
    	hw3000_write_reg(INTIC, 0x0001);	//clr_int
    	hw3000_write_reg(TRCTRL, 0x0000);	//send disable

    	return ;
}

/**
  * @brief  Enable HW3000 module recv.
  * @retval None.
  */
void hw3000_rx_enable(void)
{
   	hw3000_write_reg(TRCTRL, 0x0080); 	//enable rx
}

/**
  * @brief  Enable HW3000 module recv.
  * @param  buf: Pointer address for storing data.
  * @retval length: The length of data which recv from HW3000.
  */
uint8_t hw3000_ReadFIFO(uint8_t * buf)
{
	uint16_t reg = 0;
	uint8_t length = 0;

	reg = hw3000_read_reg(FIFOCTRL);

	if (!(reg & 0xC000)) {	                              	//Check if CRC error

		reg = hw3000_read_reg(RXPHR0);

		if (reg & 0xff00)
			length = (reg >> 8) - 3;	// data length

		hw3000_read_fifo(FIFODATA, buf, length);        //read FIFO
        	hw3000_write_reg(FIFOCTRL, 0x0000);	        //ocpy = 0
		hw3000_write_reg(INTIC, 0x0001);	        //FIFOFIFOжϱ־

		return length;
	}

	hw3000_write_reg(FIFOCTRL, 0x0000);		        //ocpy = 0
	hw3000_write_reg(INTIC, 0x0001);		        //FIFOFIFOжϱ־

	return 0;
}


/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
