/**
  *********************************************************************************
  *
  * @file    bsp_mpu9250.c
  * @brief   Mpu9250 driver
  *
  * @version V1.0
  * @date    06 Jun 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          06 Jun 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"
#include "stdlib.h"
#include "bsp_mpu9250.h"
#include "ald_i2c.h"

 /** @addtogroup ES32F03xx_BSP
  * @{
  */

 /**
  * @defgroup MPU9250 mpu9250 
  * @{
  */

/**
  * @defgroup MPU9250_Private_Functions mpu9250 Private Functions
  * @{
  */

/**
  * @brief  Initializate pin of i2c1 module.
  * @retval None
  */
void mpu_i2c_pin_init(void)
{
	gpio_init_t a;

	/* Initialize scl pin */
	a.mode = GPIO_MODE_OUTPUT;
	a.odos = GPIO_PUSH_PULL;
	a.pupd = GPIO_PUSH_UP;
	a.odrv = GPIO_OUT_DRIVE_NORMAL;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_CMOS;
	a.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOA, GPIO_PIN_5, &a);

	/* Initialize sda pin */
	a.mode = GPIO_MODE_CLOSE;
	a.odos = GPIO_PUSH_PULL;
	a.pupd = GPIO_PUSH_UP;
	a.odrv = GPIO_OUT_DRIVE_NORMAL;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_CMOS;
	a.func = GPIO_FUNC_5;
	ald_gpio_init(GPIOA, GPIO_PIN_6, &a);
}

/**
  * @brief  mpu9250 set gyro frequence
  * @retval ald_status_t
  */
ald_status_t mpu9250_set_gyro_freq(bsp_mpu9250_env_t *h_mpu, uint8_t index)
{
	uint8_t data;
	data = index << 3;

	return ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_GYRO_CFG_REG, I2C_MEMADD_SIZE_8BIT, &data, 1, 500);
}

/**
  * @brief  mpu9250 set accel frequence
  * @retval ald_status_t
  */
ald_status_t mpu9250_set_accel_freq(bsp_mpu9250_env_t *h_mpu, uint8_t index)
{
	uint8_t data;
	data = index << 3;

	return ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_ACCEL_CFG_REG, I2C_MEMADD_SIZE_8BIT, &data, 1, 500);
}

/**
  * @brief  mpu9250 set dlpf
  * @retval ald_status_t
  */
ald_status_t mpu9250_set_lpf(bsp_mpu9250_env_t *h_mpu, uint16_t lpf)
{
	uint8_t data;

	if (lpf >= 188)
		data = 1;
	else if (lpf >= 98)
		data = 2;
	else if (lpf >= 42)
		data = 3;
	else if (lpf >= 20)
		data = 4;
	else if (lpf >= 10)
		data = 5;
	else
		data = 6;

	return ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_CFG_REG, I2C_MEMADD_SIZE_8BIT, &data, 1, 500);
}

/**
  * @brief  mpu9250 set sample rate
  * @retval ald_status_t
  */
ald_status_t mpu9250_set_sample_rate(bsp_mpu9250_env_t *h_mpu, uint16_t rate)
{
	uint8_t data;

	if (rate > 1000)
		rate = 1000;
	if (rate < 4)
		rate = 4;

	data = 1000 / rate - 1;

	if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_SAMPLE_RATE_REG, I2C_MEMADD_SIZE_8BIT, &data, 1, 500) != OK)
		return ERROR;

	return mpu9250_set_lpf(h_mpu, rate / 2);
}

/**
  * @}
  */

/**
  * @defgroup MPU9250_Public_Functions mpu9250 Public Functions
  * @brief    BSP MPU9250 Functions
  * @verbatim
 ===============================================================================
              ##### MPU9250 Public Functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp mpu9250 module;
      (+) get mpu9250 temperature;
      (+) get mpu9250 gyro data;
      (+) get mpu9250 acce data;
      (+) mpu9250 data measure;

    @endverbatim
  * @{
  */

/**
  * @brief  get mpu9250 temperature 
  * @retval temperature data
  */
uint32_t bsp_mpu_get_temperature(bsp_mpu9250_env_t *h_mpu)
{
	uint8_t buf[2] = {0};
	uint8_t raw;
	uint32_t temp;

	if (ald_i2c_mem_read(&h_mpu->h_i2c, MPU9250_ADDR, MPU_TEMP_OUTH_REG, I2C_MEMADD_SIZE_8BIT, buf, 2, 500) != OK)
		return 0;

	raw = ((uint16_t)buf[0] << 8) | buf[1];
	temp = 2100 + raw * 10000 / 33387;
	return temp;
}

/**
  * @brief  get mpu9250 gyro data 
  * @retval ald_status_t
  */
ald_status_t bsp_mpu_get_gyro(bsp_mpu9250_env_t *h_mpu, uint16_t *gx, uint16_t *gy, uint16_t *gz)
{
	uint8_t buf[6] = {0};

	if (ald_i2c_mem_read(&h_mpu->h_i2c, MPU9250_ADDR, MPU_GYRO_XOUTH_REG, I2C_MEMADD_SIZE_8BIT, buf, 6, 500) != OK)
		return ERROR;

	*gx = ((uint16_t)buf[0] << 8) | buf[1];
	*gy = ((uint16_t)buf[2] << 8) | buf[3];
	*gz = ((uint16_t)buf[4] << 8) | buf[5];

	return OK;
}

/**
  * @brief  get mpu9250 accel data 
  * @retval ald_status_t
  */
ald_status_t bsp_mpu_get_acce(bsp_mpu9250_env_t *h_mpu, uint16_t *ax, uint16_t *ay, uint16_t *az)
{
	uint8_t buf[6] = {0};

	if (ald_i2c_mem_read(&h_mpu->h_i2c, MPU9250_ADDR, MPU_ACCEL_XOUTH_REG, I2C_MEMADD_SIZE_8BIT, buf, 6, 500) != OK)
		return ERROR;

	*ax = ((uint16_t)buf[0] << 8) | buf[1];
	*ay = ((uint16_t)buf[2] << 8) | buf[3];
	*az = ((uint16_t)buf[4] << 8) | buf[5];

	return OK;
}

/**
  * @brief  init mpu9250 module
  * @retval ald_status_t
  */
ald_status_t bsp_mpu_init(bsp_mpu9250_env_t *h_mpu)
{
	uint8_t chip_id;
	uint8_t data[6];

	mpu_i2c_pin_init();

	h_mpu->h_i2c.perh              = I2C1;
	h_mpu->h_i2c.init.clk_speed    = 100000;
	h_mpu->h_i2c.init.duty         = I2C_DUTYCYCLE_2;
	h_mpu->h_i2c.init.own_addr1    = 0x0A;
	h_mpu->h_i2c.init.addr_mode    = I2C_ADDR_7BIT;
	h_mpu->h_i2c.init.general_call = I2C_GENERALCALL_DISABLE;
	h_mpu->h_i2c.init.no_stretch   = I2C_NOSTRETCH_ENABLE;
	ald_i2c_reset(&h_mpu->h_i2c);
	ald_i2c_init(&h_mpu->h_i2c);

	data[0] = BIT_RESET;
	if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_PWR_MGMT1_REG, I2C_MEMADD_SIZE_8BIT, data, 1, 500) != OK)
		return ERROR;

	/* Wake up chip */
	ald_delay_ms(100);
	data[0] = 0;
	if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_PWR_MGMT1_REG, I2C_MEMADD_SIZE_8BIT, data, 1, 500) != OK)
		return ERROR;

	if (mpu9250_set_gyro_freq(h_mpu, 3) != OK)
		return ERROR;

	if (mpu9250_set_accel_freq(h_mpu, 0) != OK)
		return ERROR;

	if (mpu9250_set_sample_rate(h_mpu, 50) != OK)
		return ERROR;

	if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_INT_EN_REG, I2C_MEMADD_SIZE_8BIT, data, 1, 500) != OK)
		return ERROR;

	if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_USER_CTRL_REG, I2C_MEMADD_SIZE_8BIT, data, 1, 500) != OK)
		return ERROR;

	if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_FIFO_EN_REG, I2C_MEMADD_SIZE_8BIT, data, 1, 500) != OK)
		return ERROR;

	if (ald_i2c_mem_read(&h_mpu->h_i2c, MPU9250_ADDR, MPU_DEVICE_ID_REG, I2C_MEMADD_SIZE_8BIT, &chip_id, 1, 500) != OK)
		return ERROR;

	if (chip_id == MPU_ID) {
		data[0] = 0x01;
		if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_PWR_MGMT1_REG, I2C_MEMADD_SIZE_8BIT, data, 1, 500) != OK)
			return ERROR;

		data[0] = 0;
		if (ald_i2c_mem_write(&h_mpu->h_i2c, MPU9250_ADDR, MPU_PWR_MGMT2_REG, I2C_MEMADD_SIZE_8BIT, data, 1, 500) != OK)
			return ERROR;

		if (mpu9250_set_sample_rate(h_mpu, 50) != OK)
			return ERROR;
	}
	else
		return ERROR;

	return OK;
}

/**
  * @brief  get mpu9250 data 
  * @retval ald_status_t
  */
ald_status_t bsp_mpu_data_measure(bsp_mpu9250_env_t *h_mpu, bsp_mpu9250_data_t *mpu_data)
{
	if (mpu_data == NULL)
		return ERROR;

	if (bsp_mpu_get_gyro(h_mpu, &mpu_data->gyro[0], &mpu_data->gyro[1], &mpu_data->gyro[2]) == ERROR)
		return ERROR;

	if (bsp_mpu_get_acce(h_mpu, &mpu_data->acce[0], &mpu_data->acce[1], &mpu_data->acce[2]) == ERROR)
		return ERROR;

	mpu_data->temperature = bsp_mpu_get_temperature(h_mpu);

	return OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
