/**
  *********************************************************************************
  *
  * @file    bsp_key.c
  * @brief   Key driver.
  *
  * @version V1.0
  * @date    04 Jul 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          04 Jul 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "bsp_key.h"

/** @defgroup ES32F03xx_BSP ES32F03xx BSP
  * @{
  */

/** @defgroup Key key
  * @{
  */

/** @defgroup Key_Private_Variable Key Private Variables
  * @{
  */
bsp_key_env_t bsp_key_env;
/**
  * @}
  */

/** @defgroup Key_Private_Functions Key Private Functions
  * @{
  */
/**
  * @brief  Initialize key's pin.
  * @retval None
  */
void bsp_key_pin_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_DOWN;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;

	ald_gpio_init(BSP_KEY1_PORT, BSP_KEY1_PIN, &x);
	ald_gpio_init(BSP_KEY2_PORT, BSP_KEY2_PIN, &x);
	ald_gpio_init(BSP_KEY3_PORT, BSP_KEY3_PIN, &x);
	ald_gpio_init(BSP_KEY4_PORT, BSP_KEY4_PIN, &x);

	return;
}

/**
  * @brief  Read key1's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key1(void)
{
	return ald_gpio_read_pin(BSP_KEY1_PORT, BSP_KEY1_PIN);
}

/**
  * @brief  Read key2's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key2(void)
{
	return ald_gpio_read_pin(BSP_KEY2_PORT, BSP_KEY2_PIN);
}

/**
  * @brief  Read key3's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key3(void)
{
	return ald_gpio_read_pin(BSP_KEY3_PORT, BSP_KEY3_PIN);
}

/**
  * @brief  Read key4's value
  * @retval The status:
  *           0 - No press
  *           1 - Press
  */
uint8_t bsp_key_read_key4(void)
{
	return ald_gpio_read_pin(BSP_KEY4_PORT, BSP_KEY4_PIN);
}
/**
  * @}
  */

/** @defgroup Key_Public_Functions Key Public Functions
  * @{
  */

/** @defgroup Key_Public_Functions_Group1 Key Initialization Function
  * @brief    Initialize Key module.
  *
  * @verbatim    
  ==============================================================================
              ##### Key Initialization Functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Initialize key module.

    @endverbatim
  * @{
  */
/**
  * @brief  Initialization key module.
  * @retval None
  */
void bsp_key_init(void)
{
	memset(&bsp_key_env, 0x0, sizeof(bsp_key_env_t));
	bsp_key_pin_init();

	bsp_key_env.read_cbk[0] = bsp_key_read_key1;
	bsp_key_env.read_cbk[1] = bsp_key_read_key2;
	bsp_key_env.read_cbk[2] = bsp_key_read_key3;
	bsp_key_env.read_cbk[3] = bsp_key_read_key4;

	return;
}
/**
  * @}
  */

/** @defgroup Key_Public_Functions_Group2 Key Functions
  * @brief    Key functions
  *
  * @verbatim    
  ==============================================================================
              ##### Key Functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Scan key.
      (+) bond callback function.
      (+) unbond callback function.

    @endverbatim
  * @{
  */

/**
  * @brief  Scan key.
  * @retval None
  */
void bsp_key_scan(void)
{
	int32_t i;

	for (i = 0; i < BSP_KEY_MAX; ++i) {
		if (!(bsp_key_env.valid & (1U << i)))
			continue;

		if ((bsp_key_env.read_cbk[i]() == 1) && (bsp_key_env.state[i].press == 0)) {
			if (++bsp_key_env.state[i].hold >= 3) {
				bsp_key_env.state[i].hold  = 0;
				bsp_key_env.state[i].press = 1;

				if (bsp_key_env.cbk[i << 1])
					bsp_key_env.cbk[i << 1]((bsp_key_event_t)(i << 1));
			}
		}
		else if ((bsp_key_env.read_cbk[i]() == 0) && (bsp_key_env.state[i].press == 1)) {
			if (++bsp_key_env.state[i].hold >= 3) {
				bsp_key_env.state[i].hold  = 0;
				bsp_key_env.state[i].press = 0;

				if (bsp_key_env.cbk[(i << 1) + 1])
					bsp_key_env.cbk[(i << 1) + 1]((bsp_key_event_t)((i << 1) + 1));
			}
		}
	}
}

/**
  * @brief  Bond callback function.
  * @param  event: Type of event.
  * @param  cbk: Callback function.
  * @retval None
  */
void bsp_key_bond_func(bsp_key_event_t event, bsp_key_cbk_t cbk)
{
	int32_t i;

	if (event == BSP_KEY_ALL) {
		for (i = 0; i < BSP_KEY_MAX; ++i) {
			bsp_key_env.valid            |= (1U << i);
			bsp_key_env.cbk[i << 1]       = cbk;
			bsp_key_env.cbk[(i << 1) + 1] = cbk;
		}
		return;
	}

	bsp_key_env.valid     |= 1 << (event >> 1);
	bsp_key_env.cbk[event] = cbk;
	return;
}

/**
  * @brief  Unbond callback function.
  * @param  event: Type of event.
  * @retval None
  */
void bsp_key_unbond_func(bsp_key_event_t event)
{
	int32_t i;

	if (event == BSP_KEY_ALL) {
		for (i = 0; i < BSP_KEY_MAX; ++i) {
			bsp_key_env.valid            &= ~(1U << i);
			bsp_key_env.cbk[i << 1]       = NULL;
			bsp_key_env.cbk[(i << 1) + 1] = NULL;
		}
		return;
	}

	bsp_key_env.cbk[event] = NULL;

	if ((bsp_key_env.cbk[(event >> 1) << 1] == NULL)
			&& (bsp_key_env.cbk[((event >> 1) << 1) + 1] == NULL))
		bsp_key_env.valid  &= ~(1 << (event >> 1));

	return;
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
