/**
  *********************************************************************************
  *
  * @file    bsp_eeprom.c
  * @brief   Eeprom driver
  *
  * @version V1.0
  * @date    16 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

 #include <stdio.h>
 #include "bsp_eeprom.h"
 #include "ald_gpio.h"

 /**
   * @addtogroup ES32F03xx_BSP
   * @{
   */

/**
  * @defgroup EEPROM eeprom
  * @{
  */

/**
  * @defgroup EEPROM_Private_Params eeprom Private Variables
  * @{
  */
i2c_handle_t h_i2c;
/**
  * @}
  */

/**
  * @defgroup EEPROM_Private_Functions eeprom Private Functions
  * @{
  */

/**
  * @brief  Initializate eeprom pin
  * @retval None.
  */
static void eeprom_pin_init(void)
{
	gpio_init_t a;

	/* Initialize scl pin */
	a.mode = GPIO_MODE_OUTPUT;
	a.odos = GPIO_OPEN_DRAIN;
	a.pupd = GPIO_PUSH_UP;
	a.odrv = GPIO_OUT_DRIVE_NORMAL;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_TTL;
	a.func = EEPROM_SCL_FUN;
	ald_gpio_init(EEPROM_SCL_PORT, EEPROM_SCL_PIN, &a);

	/* Initialize sda pin */
	a.mode = GPIO_MODE_OUTPUT;
	a.odos = GPIO_OPEN_DRAIN;
	a.pupd = GPIO_PUSH_UP;
	a.odrv = GPIO_OUT_DRIVE_NORMAL;
	a.flt  = GPIO_FILTER_DISABLE;
	a.type = GPIO_TYPE_TTL;
	a.func = EEPROM_SDA_FUN;
	ald_gpio_init(EEPROM_SDA_PORT, EEPROM_SDA_PIN, &a);
}

/**
  * @}
  */

/**
  * @defgroup EEPROM_Public_Functions eeprom Public Functions
  * @brief    BSP EEPROM Functions
  * @verbatim
 ===============================================================================
              ##### BSP EEPROM Functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp eeprom module;
      (+) Write an amount of data to a specific memory address
      (+) Read an amount of data to a specific memory address

    @endverbatim
  * @{
  */

/**
  * @brief  Initializate eeprom moudle
  * @retval None.
  */
void eeprom_module_init(void)
{
	eeprom_pin_init();

	/* Initialize I2C struct */
	memset(&h_i2c, 0, sizeof(h_i2c));

	/* Initialize I2C */
	h_i2c.perh = I2C1;
	h_i2c.init.clk_speed = 100000;
	h_i2c.init.duty = I2C_DUTYCYCLE_2;
	h_i2c.init.own_addr1 = 0x0A;
	h_i2c.init.addr_mode = I2C_ADDR_7BIT;
	h_i2c.init.general_call = I2C_GENERALCALL_DISABLE;
	h_i2c.init.no_stretch = I2C_NOSTRETCH_ENABLE;

	ald_i2c_reset(&h_i2c);
	ald_i2c_init(&h_i2c);
}

/**
  * @brief  Write an amount of data in non-blocking mode with Interrupt to a specific memory address
  * @param  dev_addr: Target device address
  * @param  mem_addr: Internal memory address
  * @param  add_size: size of internal memory address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref eep_operat_t.
  */
eep_operat_t write_eeprom(uint16_t dev_addr, uint16_t mem_addr,
                              i2c_addr_size_t add_size, uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return EEP_ERROR;

	if (ald_i2c_mem_write_by_it(&h_i2c, dev_addr, mem_addr, add_size, buf, size) != OK)
		return EEP_ERROR;

	return EEP_OK;
}

/**
  * @brief  Read an amount of data in non-blocking mode with Interrupt to a specific memory address
  * @param  dev_addr: Target device address
  * @param  mem_addr: Internal memory address
  * @param  add_size: size of internal memory address
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be read
  * @retval Status, see @ref eep_operat_t.
  */
eep_operat_t read_eeprom(uint16_t dev_addr, uint16_t mem_addr,
                              i2c_addr_size_t add_size, uint8_t *buf, uint16_t size)
{
	if (buf == NULL)
		return EEP_ERROR;

	if (ald_i2c_mem_read_by_it(&h_i2c, dev_addr, mem_addr, add_size, buf, size) != OK)
		return EEP_ERROR;

	return EEP_OK;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
