/**
  *********************************************************************************
  *
  * @file    bsp_display.c
  * @brief   Display driver
  *
  * @version V1.0
  * @date    16 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "bsp_display.h"
#include "ald_calc.h"


 /** @addtogroup ES32F03xx_BSP
  * @{
  */

 /**
  * @defgroup Display_Module display
  * @{
  */

/** @defgroup Display_Private_Variables display Private Variables
  * @{
  */  

/**
  * @brief Touch Structure Init.
  */
static _tp_dev tp_dev =
{
	bsp_tp_scan,
	bsp_xtp2046_adjust,
	 {0} ,
	 {0} ,
	0,
	0.14,
	0.20,
	0xFFE8,
	0xFFE3,
};


/**
  * @brief Touch adjust table.
  */
const uint8_t TP_ADJDIS_TBL[3][4] = {{0, 1, 2, 3} , {0, 2, 1, 3} , {1, 2, 0, 3} };

/**
  * @brief ASCII 16*8
  */
const uint8_t asc2_1608[1520] = {
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
0x00, 0x48, 0x6C, 0x24, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x24, 0x24, 0x24, 0x7F, 0x12, 0x12, 0x12, 0x7F, 0x12, 0x12, 0x12, 0x00, 0x00,
0x00, 0x00, 0x08, 0x1C, 0x2A, 0x2A, 0x0A, 0x0C, 0x18, 0x28, 0x28, 0x2A, 0x2A, 0x1C, 0x08, 0x08,
0x00, 0x00, 0x00, 0x22, 0x25, 0x15, 0x15, 0x15, 0x2A, 0x58, 0x54, 0x54, 0x54, 0x22, 0x00, 0x00,
0x00, 0x00, 0x00, 0x0C, 0x12, 0x12, 0x12, 0x0A, 0x76, 0x25, 0x29, 0x11, 0x91, 0x6E, 0x00, 0x00,
0x00, 0x06, 0x06, 0x04, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x40, 0x20, 0x10, 0x10, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x10, 0x20, 0x40, 0x00,
0x00, 0x02, 0x04, 0x08, 0x08, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x08, 0x08, 0x04, 0x02, 0x00,
0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x6B, 0x1C, 0x1C, 0x6B, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x08, 0x08, 0x7F, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06, 0x04, 0x03,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x00,
0x00, 0x00, 0x80, 0x40, 0x40, 0x20, 0x20, 0x10, 0x10, 0x08, 0x08, 0x04, 0x04, 0x02, 0x02, 0x00,
0x00, 0x00, 0x00, 0x18, 0x24, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x24, 0x18, 0x00, 0x00,
0x00, 0x00, 0x00, 0x08, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x3E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x20, 0x20, 0x10, 0x08, 0x04, 0x42, 0x7E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x20, 0x18, 0x20, 0x40, 0x40, 0x42, 0x22, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x20, 0x30, 0x28, 0x24, 0x24, 0x22, 0x22, 0x7E, 0x20, 0x20, 0x78, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7E, 0x02, 0x02, 0x02, 0x1A, 0x26, 0x40, 0x40, 0x42, 0x22, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x38, 0x24, 0x02, 0x02, 0x1A, 0x26, 0x42, 0x42, 0x42, 0x24, 0x18, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7E, 0x22, 0x22, 0x10, 0x10, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x24, 0x18, 0x24, 0x42, 0x42, 0x42, 0x3C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x18, 0x24, 0x42, 0x42, 0x42, 0x64, 0x58, 0x40, 0x40, 0x24, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x04,
0x00, 0x00, 0x00, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x00, 0x00, 0x00, 0x7F, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x46, 0x40, 0x20, 0x10, 0x10, 0x00, 0x18, 0x18, 0x00, 0x00,
0x00, 0x00, 0x00, 0x1C, 0x22, 0x5A, 0x55, 0x55, 0x55, 0x55, 0x2D, 0x42, 0x22, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x08, 0x08, 0x18, 0x14, 0x14, 0x24, 0x3C, 0x22, 0x42, 0x42, 0xE7, 0x00, 0x00,
0x00, 0x00, 0x00, 0x1F, 0x22, 0x22, 0x22, 0x1E, 0x22, 0x42, 0x42, 0x42, 0x22, 0x1F, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7C, 0x42, 0x42, 0x01, 0x01, 0x01, 0x01, 0x01, 0x42, 0x22, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x1F, 0x22, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x22, 0x1F, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3F, 0x42, 0x12, 0x12, 0x1E, 0x12, 0x12, 0x02, 0x42, 0x42, 0x3F, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3F, 0x42, 0x12, 0x12, 0x1E, 0x12, 0x12, 0x02, 0x02, 0x02, 0x07, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3C, 0x22, 0x22, 0x01, 0x01, 0x01, 0x71, 0x21, 0x22, 0x22, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0xE7, 0x42, 0x42, 0x42, 0x42, 0x7E, 0x42, 0x42, 0x42, 0x42, 0xE7, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x3E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7C, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x11, 0x0F,
0x00, 0x00, 0x00, 0x77, 0x22, 0x12, 0x0A, 0x0E, 0x0A, 0x12, 0x12, 0x22, 0x22, 0x77, 0x00, 0x00,
0x00, 0x00, 0x00, 0x07, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x42, 0x7F, 0x00, 0x00,
0x00, 0x00, 0x00, 0x77, 0x36, 0x36, 0x36, 0x36, 0x2A, 0x2A, 0x2A, 0x2A, 0x2A, 0x6B, 0x00, 0x00,
0x00, 0x00, 0x00, 0xE3, 0x46, 0x46, 0x4A, 0x4A, 0x52, 0x52, 0x52, 0x62, 0x62, 0x47, 0x00, 0x00,
0x00, 0x00, 0x00, 0x1C, 0x22, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x22, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x3F, 0x42, 0x42, 0x42, 0x42, 0x3E, 0x02, 0x02, 0x02, 0x02, 0x07, 0x00, 0x00,
0x00, 0x00, 0x00, 0x1C, 0x22, 0x41, 0x41, 0x41, 0x41, 0x41, 0x4D, 0x53, 0x32, 0x1C, 0x60, 0x00,
0x00, 0x00, 0x00, 0x3F, 0x42, 0x42, 0x42, 0x3E, 0x12, 0x12, 0x22, 0x22, 0x42, 0xC7, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7C, 0x42, 0x42, 0x02, 0x04, 0x18, 0x20, 0x40, 0x42, 0x42, 0x3E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7F, 0x49, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0xE7, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x3C, 0x00, 0x00,
0x00, 0x00, 0x00, 0xE7, 0x42, 0x42, 0x22, 0x24, 0x24, 0x14, 0x14, 0x18, 0x08, 0x08, 0x00, 0x00,
0x00, 0x00, 0x00, 0x6B, 0x49, 0x49, 0x49, 0x49, 0x55, 0x55, 0x36, 0x22, 0x22, 0x22, 0x00, 0x00,
0x00, 0x00, 0x00, 0xE7, 0x42, 0x24, 0x24, 0x18, 0x18, 0x18, 0x24, 0x24, 0x42, 0xE7, 0x00, 0x00,
0x00, 0x00, 0x00, 0x77, 0x22, 0x22, 0x14, 0x14, 0x08, 0x08, 0x08, 0x08, 0x08, 0x1C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x7E, 0x21, 0x20, 0x10, 0x10, 0x08, 0x04, 0x04, 0x42, 0x42, 0x3F, 0x00, 0x00,
0x00, 0x78, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x78, 0x00,
0x00, 0x00, 0x02, 0x02, 0x04, 0x04, 0x08, 0x08, 0x08, 0x10, 0x10, 0x20, 0x20, 0x20, 0x40, 0x40,
0x00, 0x1E, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x1E, 0x00,
0x00, 0x38, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
0x00, 0x06, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x42, 0x78, 0x44, 0x42, 0x42, 0xFC, 0x00, 0x00,
0x00, 0x00, 0x00, 0x03, 0x02, 0x02, 0x02, 0x1A, 0x26, 0x42, 0x42, 0x42, 0x26, 0x1A, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x44, 0x02, 0x02, 0x02, 0x44, 0x38, 0x00, 0x00,
0x00, 0x00, 0x00, 0x60, 0x40, 0x40, 0x40, 0x78, 0x44, 0x42, 0x42, 0x42, 0x64, 0xD8, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x42, 0x7E, 0x02, 0x02, 0x42, 0x3C, 0x00, 0x00,
0x00, 0x00, 0x00, 0xF0, 0x88, 0x08, 0x08, 0x7E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x3E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x22, 0x22, 0x1C, 0x02, 0x3C, 0x42, 0x42, 0x3C,
0x00, 0x00, 0x00, 0x03, 0x02, 0x02, 0x02, 0x3A, 0x46, 0x42, 0x42, 0x42, 0x42, 0xE7, 0x00, 0x00,
0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00, 0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x3E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x30, 0x30, 0x00, 0x00, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x22, 0x1E,
0x00, 0x00, 0x00, 0x03, 0x02, 0x02, 0x02, 0x72, 0x12, 0x0A, 0x16, 0x12, 0x22, 0x77, 0x00, 0x00,
0x00, 0x00, 0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x3E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F, 0x92, 0x92, 0x92, 0x92, 0x92, 0xB7, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3B, 0x46, 0x42, 0x42, 0x42, 0x42, 0xE7, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x42, 0x42, 0x42, 0x42, 0x42, 0x3C, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1B, 0x26, 0x42, 0x42, 0x42, 0x22, 0x1E, 0x02, 0x07,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x44, 0x42, 0x42, 0x42, 0x44, 0x78, 0x40, 0xE0,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x77, 0x4C, 0x04, 0x04, 0x04, 0x04, 0x1F, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x42, 0x02, 0x3C, 0x40, 0x42, 0x3E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x3E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x30, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x42, 0x42, 0x42, 0x42, 0x62, 0xDC, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE7, 0x42, 0x24, 0x24, 0x14, 0x08, 0x08, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xEB, 0x49, 0x49, 0x55, 0x55, 0x22, 0x22, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0x24, 0x18, 0x18, 0x18, 0x24, 0x6E, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE7, 0x42, 0x24, 0x24, 0x14, 0x18, 0x08, 0x08, 0x07,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0x22, 0x10, 0x08, 0x08, 0x44, 0x7E, 0x00, 0x00,
0x00, 0xC0, 0x20, 0x20, 0x20, 0x20, 0x20, 0x10, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xC0, 0x00,
0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10,
0x00, 0x06, 0x08, 0x08, 0x08, 0x08, 0x08, 0x10, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x06, 0x00,
0x0C, 0x32, 0xC2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

/**
  * @}
  */

/**
  * @defgroup Display_Private_Functions display Private Functions
  * @{
  */

/**
  * @brief Spi write function.
  * @param display:display structure.
  * @param data:data start address.
  * @param size:data length.
  * @retval None
  */
static void lcd_write_bus(bsp_display_env_t *display, uint8_t *data, uint32_t size)
{
        while (size--)
        	ald_spi_send_byte_fast(&display->h_spi, *data++);
}

/**
  * @brief  lcd write 8-bit data.
  * @param  display: display structure.
  * @param  data: written data.
  * @retval None
  */
static void lcd_wr_data8(bsp_display_env_t *display, uint8_t data)
{
	OLED_CS_CLR();
	OLED_DC_SET();
	lcd_write_bus(display, &data, 1);
	OLED_CS_SET();
}

/**
  * @brief  lcd write 16-bit data.
  * @param  display: display structure.
  * @param  data: written data.
  * @retval None
  */
static void lcd_wr_data16(bsp_display_env_t * display, uint16_t data)
{
        uint8_t temp = 0;
        OLED_CS_CLR();
	OLED_DC_SET();
        temp = data >> 8;
	lcd_write_bus(display, &temp, 1);
        temp = (uint8_t)data;
        lcd_write_bus(display, &temp, 1);
	OLED_CS_SET();
}

/**
  * @brief  lcd write data.
  * @param  display: display structure.
  * @param  data: data start address.
  * @param  size: data length.
  * @retval None
  */
static void lcd_wr_data(bsp_display_env_t *display, uint8_t *data, uint32_t size)
{
	OLED_CS_CLR();
	OLED_DC_SET();
	lcd_write_bus(display, data, size);
}

/**
  * @brief  lcd write register.
  * @param  display: display structure.
  * @param  data: written data.
  * @retval None
  */
static void lcd_wr_reg(bsp_display_env_t *display, uint8_t data)
{
	OLED_CS_CLR();
	OLED_DC_CLR();
	lcd_write_bus(display, &data, 1);
	OLED_CS_SET();
}

/**
  * @brief  lcd target point address set.
  * @param  display:display structure.
  * @param  x1: target x1 axis value.
  * @param  y1: target y1 axis value.
  * @param  x2: target x2 axis value.
  * @param  y2: target y2 axis value.
  * @retval None
  */
static void address_set(bsp_display_env_t *display, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2)
{
	lcd_wr_reg(display, 0x2a);
	lcd_wr_data8(display, (x1 >> 8));
	lcd_wr_data8(display, x1);
	lcd_wr_data8(display, (x2 >> 8));
	lcd_wr_data8(display, x2);

	lcd_wr_reg(display, 0x2b);
	lcd_wr_data8(display, (y1 >> 8));
	lcd_wr_data8(display, y1);
	lcd_wr_data8(display, (y2 >> 8));
	lcd_wr_data8(display, y2);

	lcd_wr_reg(display, 0x2c);
}

/**
  * @brief  spi tx callback function.
  * @param  hperh: spi instance.
  * @retval None
  */
static void spi_tx_cbk(spi_handle_t *hperh)
{
	OLED_CS_SET();
}

/**
  * @brief  Read touch ADC value.
  * @param  display: display structure.
  * @param  cmd: operation command.
  * @retval None
  */
static uint16_t bsp_read_xtp_adc(bsp_display_env_t *display, unsigned char cmd)
{
        uint16_t temp = 0;
        TOUCH_CS_Clr();
        for (temp = 100; temp > 0; temp--);
        ald_spi_send_byte_fast(&display->h_spi, cmd);
        for (temp = 400; temp > 0; temp--);
        temp = 0;
        temp = ald_spi_recv_byte_fast(&display->h_spi);
        temp <<= 8;
        temp = temp | ald_spi_recv_byte_fast(&display->h_spi);
        temp >>= 4;
        TOUCH_CS_Set();
        return temp;
}

/**
 * @brief  Read touch ADC register.
 * @param  display: display structure.
 * @param  cmd: read register command.
 * @retval temp: register value.
 */
static uint16_t bsp_read_xtp_adcs(bsp_display_env_t *display, uint16_t cmd)
{
	uint16_t i, j;
	uint16_t buf[READ_TIMES];
	uint16_t sum = 0;
	uint16_t temp;
	for (i = 0; i < READ_TIMES; i++) buf[i] = bsp_read_xtp_adc(display, cmd);
	for (i = 0; i < READ_TIMES - 1; i++)
	 {
		for (j = i + 1; j < READ_TIMES; j++)
		 {
			if (buf[i] > buf[j])
			 {
				temp   = buf[i];
				buf[i] = buf[j];
				buf[j] = temp;
			}
		}
	}
	sum = 0;
	for (i = LOST_VAL; i < READ_TIMES - LOST_VAL; i++)sum += buf[i];
	temp = sum / (READ_TIMES - 2 * LOST_VAL);
	return temp;
}

/**
 * @brief  Read touch ADC one times.
 * @param  display: display structure.
 * @param  x: store x point.
 * @param  y: store y point.
 * @retval 1,success;0,failed.
 */
static uint8_t bsp_get_xtp_point(bsp_display_env_t *display, uint16_t *x, uint16_t *y)
{
	uint16_t xtemp, ytemp;
	xtemp = bsp_read_xtp_adcs(display, CMD_RDX);
	ytemp = bsp_read_xtp_adcs(display, CMD_RDY);
	*x = xtemp;
	*y = ytemp;
        return 1;
}

/**
  * @brief  Touch Initializate.
  * @param  display: display structure.
  * @retval None
  */
static void bsp_tp_init(bsp_display_env_t *display)
{
        bsp_get_xtp_point(display, &tp_dev.x[0], &tp_dev.y[0]);//һζȡʼ
}

/**
  * @brief  lcd draw point.
  * @param  display: display structure.
  * @param  x: store x point.
  * @param  y: store y point.
  * @param  color: draw point color.
  * @retval None
  */
static void TP_Draw_Touch_Point(bsp_display_env_t *display, uint16_t x, uint16_t y, uint16_t color)
{
	bsp_draw_line(display, x - 12, y, x + 13, y, color); //
        bsp_draw_line(display, x, y - 12, x, y + 13, color); //

	bsp_draw_point(display, x + 1, y + 1, color);
	bsp_draw_point(display, x - 1, y + 1, color);
	bsp_draw_point(display, x + 1, y - 1, color);
	bsp_draw_point(display, x - 1, y - 1, color);
	bsp_draw_circle(display, x, y, 6, color);//Ȧ
}

/**
  * @brief  lcd initialize.
  * @param  display: display structure.
  * @retval None
  */
static void oled_init(bsp_display_env_t *display)
{
	gpio_init_t x;
        exti_init_t exti;

	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
        x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;

	/* Enable spi flash cs set */
	ald_gpio_init(GPIOE, GPIO_PIN_0, &x);
	ald_gpio_write_pin(GPIOE, GPIO_PIN_0, 1);

	/* PE14 -> CS_LCD */
	ald_gpio_init(GPIOE, GPIO_PIN_15, &x);
	ald_gpio_write_pin(GPIOE, GPIO_PIN_15, 1);

	/* PC13 -> RST */
	ald_gpio_init(GPIOC, GPIO_PIN_13, &x);
	ald_gpio_write_pin(GPIOC, GPIO_PIN_13, 1);

	/* PE3 -> DC */
	ald_gpio_init(GPIOE, GPIO_PIN_3, &x);
	ald_gpio_write_pin(GPIOE, GPIO_PIN_3, 1);

        /* PE14 -> CS_PEN */
        ald_gpio_init(GPIOE, GPIO_PIN_14, &x);
        ald_gpio_write_pin(GPIOE, GPIO_PIN_14, 1);

	/* PB15 MOSI INIT */
	x.func = GPIO_FUNC_4;
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x);

	/* PB13 SCK INIT */
	x.func = GPIO_FUNC_4;
	ald_gpio_init(GPIOB, GPIO_PIN_13, &x);

        /* PB14 MISO INIT */
        /* MISO Initializes */	
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
        x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_4;
        ald_gpio_init(GPIOB, GPIO_PIN_14, &x);

        /* PE7--> PEN */
        x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
        ald_gpio_init(GPIOE, GPIO_PIN_7, &x);

        /* Initialize external interrupt */
	exti.filter      = ENABLE;
	exti.cks         = EXTI_FILTER_CLOCK_32K;
	exti.filter_time = 100;
        ald_gpio_exti_init(GPIOE, GPIO_PIN_7, &exti);
        ald_gpio_exti_clear_flag_status(GPIO_PIN_7);
        ald_gpio_exti_interrupt_config(GPIO_PIN_7, EXTI_TRIGGER_TRAILING_EDGE, ENABLE);
        ald_mcu_irq_config(EXTI4_7_IRQn, 3, DISABLE);

	display->h_spi.perh           = SPI1;
	display->h_spi.init.mode      = SPI_MODE_MASTER;
	display->h_spi.init.dir       = SPI_DIRECTION_2LINES;
	display->h_spi.init.data_size = SPI_DATA_SIZE_8;
	display->h_spi.init.baud      = SPI_BAUD_8;
	display->h_spi.init.phase     = SPI_CPHA_FIRST;
	display->h_spi.init.polarity  = SPI_CPOL_LOW;
	display->h_spi.init.first_bit = SPI_FIRSTBIT_MSB;
	display->h_spi.init.ss_en     = DISABLE;
	display->h_spi.init.crc_calc  = DISABLE;
	display->h_spi.tx_cplt_cbk    = spi_tx_cbk;
	ald_spi_init(&display->h_spi);

        //lcdʼ
	OLED_CS_CLR();
	OLED_RST_CLR();
	ald_delay_ms(20);
	OLED_RST_SET();
	ald_delay_ms(20);

	//************* Start Initial Sequence **********// 
	lcd_wr_reg(display, 0xCF);
	lcd_wr_data8(display, 0x00);
	lcd_wr_data8(display, 0xD9);
	lcd_wr_data8(display, 0x30);

	lcd_wr_reg(display, 0xED);
	lcd_wr_data8(display, 0x64);
	lcd_wr_data8(display, 0x03);
	lcd_wr_data8(display, 0X12);
	lcd_wr_data8(display, 0X81);

	lcd_wr_reg(display, 0xE8);
	lcd_wr_data8(display, 0x85);
	lcd_wr_data8(display, 0x10);
	lcd_wr_data8(display, 0x78);

	lcd_wr_reg(display, 0xCB);
	lcd_wr_data8(display, 0x39);
	lcd_wr_data8(display, 0x2C);
	lcd_wr_data8(display, 0x00);
	lcd_wr_data8(display, 0x34);
	lcd_wr_data8(display, 0x02);

	lcd_wr_reg(display, 0xF7);
	lcd_wr_data8(display, 0x20);

	lcd_wr_reg(display, 0xEA);
	lcd_wr_data8(display, 0x00);
	lcd_wr_data8(display, 0x00);

	lcd_wr_reg(display, 0xC0);
	lcd_wr_data8(display, 0x21);

	lcd_wr_reg(display, 0xC1);
	lcd_wr_data8(display, 0x12);

	lcd_wr_reg(display, 0xC5);
	lcd_wr_data8(display, 0x32);
	lcd_wr_data8(display, 0x3C);

	lcd_wr_reg(display, 0xC7);
	lcd_wr_data8(display, 0xC1);

	lcd_wr_reg(display, 0x36);
	lcd_wr_data8(display, 0x08);

	lcd_wr_reg(display, 0x3A);
	lcd_wr_data8(display, 0x55);

	lcd_wr_reg(display, 0xB1);
	lcd_wr_data8(display, 0x00);
	lcd_wr_data8(display, 0x18);

	lcd_wr_reg(display, 0xB6);
	lcd_wr_data8(display, 0x0A);
	lcd_wr_data8(display, 0xA2);

	lcd_wr_reg(display, 0xF2);
	lcd_wr_data8(display, 0x00);

	lcd_wr_reg(display, 0x26);
	lcd_wr_data8(display, 0x01);

	lcd_wr_reg(display, 0xE0);
	lcd_wr_data8(display, 0x0F);
	lcd_wr_data8(display, 0x20);
	lcd_wr_data8(display, 0x1E);
	lcd_wr_data8(display, 0x09);
	lcd_wr_data8(display, 0x12);
	lcd_wr_data8(display, 0x0B);
	lcd_wr_data8(display, 0x50);
	lcd_wr_data8(display, 0XBA);
	lcd_wr_data8(display, 0x44);
	lcd_wr_data8(display, 0x09);
	lcd_wr_data8(display, 0x14);
	lcd_wr_data8(display, 0x05);
	lcd_wr_data8(display, 0x23);
	lcd_wr_data8(display, 0x21);
	lcd_wr_data8(display, 0x00);

	lcd_wr_reg(display, 0xE1);
	lcd_wr_data8(display, 0x00);
	lcd_wr_data8(display, 0x19);
	lcd_wr_data8(display, 0x19);
	lcd_wr_data8(display, 0x00);
	lcd_wr_data8(display, 0x12);
	lcd_wr_data8(display, 0x07);
	lcd_wr_data8(display, 0x2D);
	lcd_wr_data8(display, 0x28);
	lcd_wr_data8(display, 0x3F);
	lcd_wr_data8(display, 0x02);
	lcd_wr_data8(display, 0x0A);
	lcd_wr_data8(display, 0x08);
	lcd_wr_data8(display, 0x25);
	lcd_wr_data8(display, 0x2D);
	lcd_wr_data8(display, 0x0F);

	lcd_wr_reg(display, 0x11);
	ald_delay_ms(120);
	lcd_wr_reg(display, 0x29);

        bsp_tp_init(display);
}

/**
  * @}
  */

/**
  * @defgroup Display_Public_Functions display Public Functions
  * @brief    BSP Display Functions
  * @verbatim
=====================================================================================
                ##### BSP Display Functions #####
=====================================================================================
    [..]   This section provides functions allowing to:
       (+) Initialize bsp display.
       (+) display screen by specific color.
       (+) display string.
       (+) display chinese.
       (+) display image.
       (+) display chinese.
       (+) display point.
       (+) display line.
       (+) display rectangle.
       (+) display circle.

  * @endverbatim
  * @{
  */

/**
  * @brief  Initialize display module
  * @retval None
  */
void bsp_display_init(bsp_display_env_t *display, touch_callback_t cbk)
{
	oled_init(display);
        display->cbk = cbk;
	return;
}

/**
 * @brief reflush the screen is the specified color.
 * @param display: display structure.
 * @param color: specified color.
 * @retval None.
 */
void bsp_display_clear(bsp_display_env_t *display, uint16_t *color)
{
	uint16_t i;

	address_set(display, 0, 0, 239, 319);

	for (i = 0; i < 512; ++i)
		display->lcd_buf[i] = *color;
	for (i = 0; i < 153; ++i)
		lcd_wr_data(display, (uint8_t *)display->lcd_buf, 1024);
}

/**
 * @brief Display char.
 * @param display: display structure.
 * @param x: display start x axis.
 * @param y: display start y axis.
 * @param data: char.
 * @retval None.
 */
void bsp_display_char(bsp_display_env_t *display, uint16_t x, uint16_t y, char data)
{
	uint8_t temp, i, j, k = 0;

	if (x > LCD_W - 16 || y > LCD_H - 16)
		return;

	/* clear */
	if (data == 0) {
		for (i = 0; i < 128; ++i)
			display->lcd_buf[i] = display->back_color;

		lcd_wr_data(display, (uint8_t *)display->lcd_buf, 256);
		return;
	}

	data -= ' ';
	address_set(display, x, y, x + 7, y + 15);

	for (i = 0; i < 16; ++i) {
		temp = asc2_1608[((uint16_t)data << 4) + i];

		for (j = 0; j < 8; ++j) {
			if (temp & 0x01)
				display->lcd_buf[k] = display->point_color;
			else
				display->lcd_buf[k] = display->back_color;

			temp >>= 1;
			++k;
		}
	}
	lcd_wr_data(display, (uint8_t *)display->lcd_buf, 256);
}

/**
 * @brief Display string.
 * @param display: display structure.
 * @param x: display start x axis.
 * @param y: display start y axis.
 * @param str: string start address.
 * @param clear: 1, clear; 0, display.
 * @retval None.
 */
void bsp_display_string(bsp_display_env_t *display, uint16_t x, uint16_t y, char *str, uint8_t clear)
{
	uint32_t len = strlen(str);

	if (x == 0xFFFF && y == 0xFFFF) {
		x = display->xpos;
		y = display->ypos;
	}

	while (len--) {
		if (x > LCD_W - 8) {
			x  = 0;
			y += 16;
		}

		if (y > LCD_H - 16) {
			display->xpos = 0;
			display->ypos = 0;
			return;
		}

		bsp_display_char(display, x, y, clear ? 0 : *str++);
		x += 8;
	}

	display->xpos = x;
	display->ypos = y;
	return;
}

/**
 * @brief Display Chinese.
 * @param display: display structure.
 * @param x: display start x axis.
 * @param y: display start y axis.
 * @param data: data start address.
 * @param size: length.
 * @retval None.
 */
void bsp_display_gb(bsp_display_env_t *display, uint8_t x, uint8_t y, uint8_t *data, uint16_t size)
{
	uint8_t times;
	uint8_t i;
	uint8_t j;
	uint16_t x_temp;
	uint16_t y_temp;
	uint16_t loop = 0;
	uint8_t loop2;
	uint8_t word_num;
	uint8_t *temp;

	if (x + 32 > 239)
		x = 0;

	if (y + 32 > 320)
		y = x = 0;

	ald_delay_ms(2);
	x_temp = x;
	y_temp = y;
	temp = data;
	word_num = (size / (WRITE_SIZE - 1)) + 1;

	for (loop2 = 0; loop < word_num; loop2++) {
		if (x_temp + 31 > 239) {
			x_temp = x;
			y_temp += y_temp + 32;
		}

		if (y_temp + 31 > 319) {
			x_temp = x;
			y_temp = y;
		}

		address_set(display, x_temp, y_temp, x_temp + 31, y_temp + 31);
		for (times = 0; times < 2; times++) {
			loop = 0;
			for (j = 0; j < 64; j++) {
				for (i = 0; i < 8; i++) {
					if ((*temp & (1 << i)) != 0)
						display->lcd_buf[loop] = display->point_color;
					else
						display->lcd_buf[loop] = display->back_color;
					loop++;
				}
				temp++;
		 	}
			lcd_wr_data(display, (uint8_t *)display->lcd_buf, 1024);
		}
	}
}

/**
 * @brief Display the image.
 * @param display: display structure.
 * @param data: image start address.
 * @param size: image size.
 * @param index: image index.
 * @retval None.
 */
void bsp_display_image(bsp_display_env_t *display, uint8_t *data, uint16_t size, uint8_t index)
{
	uint16_t loop;
	uint16_t write_time;
	uint16_t delay;
	uint16_t ypos_inc;
	uint16_t xremaind;

	if (index == 1) {
		display->image_xpos = 0;
		display->image_ypos = 0;
	}

	write_time = (size / (WRITE_SIZE - 1)) + 1;
	xremaind = size % LCD_W;
	ypos_inc = size / LCD_W;

	ald_delay_ms(2);
 	address_set(display, display->image_xpos, display->image_ypos, 239, 319);
	for (loop = 0; loop < write_time - 1; loop++) {
		for (delay = 0; delay < 2000; delay++ );
		lcd_wr_data(display, data + loop * WRITE_SIZE, 1024);
 	}

	if ((size % 1024) != 0) {
		lcd_wr_data(display, data + (write_time - 1) * WRITE_SIZE, size - (write_time - 1) * WRITE_SIZE);
	}
	else {
		lcd_wr_data(display, data + (write_time - 1) * WRITE_SIZE, WRITE_SIZE);

	}

	if (display->image_xpos + xremaind > 239) {
		display->image_xpos = display->image_xpos + xremaind - 239;
		display->image_ypos = display->image_ypos + ypos_inc + 1;
	}
	else {
		display->image_xpos = display->image_xpos + xremaind;
		display->image_ypos = display->image_ypos + ypos_inc;
	}

	if (display->image_ypos > 319) {
		display->image_xpos = 0;
		display->image_ypos = 0;
	}
}


/**
 * @brief   Display the zip image.
 * @param  display: display structure.
 * @param  cbk: callback function.
 * @retval None.
 */
void bsp_display_image_zip(bsp_display_env_t *display, get_image_frame cbk)
{
	uint16_t len;

	ald_delay_ms(10);
 	address_set(display, 0, 0, 239, 319);

	while ((len = cbk(display->data_buf)) != 0) {
		lcd_wr_data(display, display->data_buf, len);
	}

	return;
}

/**
  * @brief  Print function
  * @param  display: display structure.
  * @param x:display start x axis.
  * @param y:display start y axis.
  * @param  fmt: format parameters.
  * @retval None
  */
void bsp_printf_d(bsp_display_env_t *display, uint16_t x, uint16_t y, const char *fmt, ...)
{
	int i, k, len = 0;
	va_list args;

	memset(display->out_buf, 0x0, 64);

	va_start(args, fmt);
	vsnprintf(display->out_buf, 64, fmt, args);
	va_end(args);

	for (i = 0; i < 64; ++i) {
		if (display->out_buf[i] == 0x0A) {
			for (k = 63; k > i + 1; --k) {
				if (display->out_buf[k - 1] == 0)
					continue;

				display->out_buf[k] = display->out_buf[k - 1];
			}

			display->out_buf[k] = 0x0D;
		}
	}

	while (display->out_buf[len] && len < 64)
		++len;

	bsp_display_string(display, x, y, (char *)display->out_buf, 0);
	return;
}

/**
 * @brief LCD display a string in the specified row.
 * @param display: display structure.
 * @param p: string start address.
 * @param row: specified row.
 * @retval None.
 */
void bsp_display_by_row(bsp_display_env_t *display, char *p, uint16_t row)
{
	uint8_t len = strlen(p);

	if (row > 20 || len > 30)
		return;

	display->xpos = 0;
	display->ypos = (row - 1) << 4;

	while (len--) {
		bsp_display_char(display, display->xpos, display->ypos, *p++);
		display->xpos += 8;
	}
}


/**
 * @brief  LCD draw a circle centered on (x0,y0),r is the radius and the color color. 
 * @param  display: display structure.
 * @param  x0: x0 axis value.
 * @param  y0: y0 axis value.
 * @param  r: the radius.
 * @param  color: circle color.
 * @retval None.
 */
void bsp_draw_circle(bsp_display_env_t *display, uint16_t x0, uint16_t y0, uint8_t r, uint16_t color)
{
	int a, b;
	int di;
	a = 0;
	b = r;
	di = 3 - ( r << 1);
	while (a <= b) {
		bsp_draw_point(display, x0 + a, y0 - b, color);             //3
		bsp_draw_point(display, x0 + b, y0 - a, color);             //0
		bsp_draw_point(display, x0 + b, y0 + a, color);             //1
		bsp_draw_point(display, x0 + a, y0 + b, color);             //7
		bsp_draw_point(display, x0 - a, y0 + b, color);             //2
		bsp_draw_point(display, x0 - b, y0 + a, color);             //4
		bsp_draw_point(display, x0 - a, y0 - b, color);             //5
		bsp_draw_point(display, x0 - b, y0 - a, color);             //6
		a++;
		if (di < 0)
                        di += 4 * a + 6;
		else {
			di += 10 + 4 * (a - b);
			b--;
		}
	}
}

/**
 * @brief LCD draw a big point with specified color.
 * @param display: display structure.
 * @param x: x axis value.
 * @param y: y axis value.
 * @param color: point color.
 * @retval None.
 */
void bsp_draw_big_point(bsp_display_env_t *display, uint16_t x, uint16_t y, uint16_t color)
{
        bsp_draw_point(display, x, y, color);
        bsp_draw_point(display, x + 1, y, color);
        bsp_draw_point(display, x, y + 1, color);
        bsp_draw_point(display, x + 1, y + 1, color);
}


/**
 * @brief  Draw a big point and then clear a big point.
 * @param  display: display structure.
 * @param  x1: start axis value.
 * @param  y1: start axis value.
 * @param  color: point color.
 * @param  x2: end axis value.
 * @param  y2: end axis value.
 * @retval None.
 */
void bsp_draw_clear_big_point(bsp_display_env_t *display, uint16_t x1, uint16_t y1,
					uint16_t color, uint16_t x2, uint16_t y2)
{
	uint8_t k;

	for (k = 0; k < SQ_SIZE; ++k)
		display->lcd_buf[k] = display->back_color;

	address_set(display, x2, y2, x2 + (POINT_SIZE - 1), y2 + (POINT_SIZE - 1));
	lcd_wr_data(display, (uint8_t *)display->lcd_buf, (SQ_SIZE << 1));

	for (k = 0; k < SQ_SIZE; ++k)
		display->lcd_buf[k] = color;

	address_set(display, x1, y1, x1 + (POINT_SIZE - 1), y1 + (POINT_SIZE - 1));
	lcd_wr_data(display, (uint8_t *)display->lcd_buf, (SQ_SIZE << 1));

	return;

}

/**
 * @brief LCD draw a point with specified color.
 * @param display: display structure.
 * @param x: x axis value.
 * @param y: y axis value.
 * @param color: point color.
 * @retval None.
 */
void bsp_draw_point(bsp_display_env_t *display, uint16_t x, uint16_t y, uint16_t color)
{
	address_set(display, x, y, x, y);
	lcd_wr_data16(display, color);
}

/**
 * @brief LCD draw a cross.
 * @param display: display structure.
 * @param x: center x point.
 * @param y: center y point.
 * @retval None.
 */
void bsp_draw_cross(bsp_display_env_t *display, uint16_t x, uint16_t y)
{
        bsp_draw_line(display, x - 10, y, x + 10, y, RED);
        bsp_draw_line(display, x, y - 10, x, y + 10, RED);
}

/**
 * @brief LCD draw a line.
 * @param display:display structure.
 * @param x1: start x1 point.
 * @param y1: start y1 point.
 * @param x2 :end x2 point.
 * @param y2: end y2 point.
 * @param color: draw line color.
 * @retval None.
 */
void bsp_draw_line(bsp_display_env_t *display, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color)
{
	uint16_t t;
	int xerr = 0, yerr = 0, delta_x, delta_y, distance;
	int incx, incy, uRow, uCol;

	delta_x = x2 - x1;
	delta_y = y2 - y1;
	uRow = x1;
	uCol = y1;
	if (delta_x > 0)
		incx = 1;
	else if (delta_x == 0)
		incx = 0;
	else {
		incx = - 1;
		delta_x = - delta_x;
	}

	if (delta_y > 0)
		incy = 1;
	else if (delta_y == 0)
		incy = 0;
	else {
		incy = - 1;
		delta_y = - delta_y;
	}

	if ( delta_x > delta_y)
		distance = delta_x;
	else distance = delta_y;

	for (t = 0; t <= distance + 1; t++ )
	 {
		bsp_draw_point(display, uRow, uCol, color);
		xerr += delta_x ;
		yerr += delta_y ;
		if (xerr > distance) {
			xerr -= distance;
			uRow += incx;
		}
		if (yerr > distance) {
			yerr -= distance;
			uCol += incy;
		}
	}
}

/**
 * @brief  LCD draw a rectangle.
 * @param  display:display structure.
 * @param  x1: start x1 point.
 * @param  y1: start y1 point.
 * @param  x2: end x2 point.
 * @param  y2: end y2 point.
 * @param  color: draw line color.
 * @retval None.
 */
void bsp_draw_rectangle(bsp_display_env_t *display, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color)
{
	bsp_draw_line(display, x1, y1, x2, y1, color);
	bsp_draw_line(display, x1, y1, x1, y2, color);
	bsp_draw_line(display, x1, y2, x2, y2, color);
	bsp_draw_line(display, x2, y1, x2, y2, color);
}

/**
  * @brief  Read touch ADC two times.
  * @param  display:display structure.
  * @param  x: store x point.
  * @param  y: store y point.
  * @retval 1,success; 0,failed.
  */
uint8_t bsp_get_xtp_point2(bsp_display_env_t * display, uint16_t*x, uint16_t *y)
{
        uint16_t x1, y1;
        uint16_t x2, y2;
        uint8_t res;

	res = bsp_get_xtp_point(display, &x1, &y1);
	if (res == 0)
		return(0);
	res = bsp_get_xtp_point(display, &x2, &y2);
	if (res == 0)
		return(0);
	if (((x2 <= x1 && x1 < x2 + ERR_RANGE) || (x1 <= x2 && x2 < x1 + ERR_RANGE))
	&& ((y2 <= y1 && y1 < y2 + ERR_RANGE) || (y1 <= y2 && y2 < y1 + ERR_RANGE))) {
		*x = (x1 + x2) / 2;
		*y = (y1 + y2) / 2;
		return 1;
	} else return 0;
}

/**
  * @brief  Touch Scan function.
  * @param  display: display structure.
  * @param  tp: if 1,read touch point; if 0,read display point.
  * @retval touch state.
  */
uint8_t bsp_tp_scan(bsp_display_env_t *display, uint8_t tp)
{
	if (XPT2046_EXTI_Read() == XPT2046_EXTI_ActiveLevel) {
		if (tp)
                        bsp_get_xtp_point2(display, &tp_dev.x[0], &tp_dev.y[0]);
		else if (bsp_get_xtp_point2(display, &tp_dev.x[0], &tp_dev.y[0])) {
	 		tp_dev.x[0] = (uint32_t)(tp_dev.xfac * tp_dev.x[0] + tp_dev.xoff);
			tp_dev.y[0] = (uint32_t)(tp_dev.yfac * tp_dev.y[0] + tp_dev.yoff);
	 	}

		if ((tp_dev.sta & TP_PRES_DOWN) == 0) {
			tp_dev.sta  = TP_PRES_DOWN | TP_CATH_PRES;
			tp_dev.x[4] = tp_dev.x[0];
			tp_dev.y[4] = tp_dev.y[0];
		}
	}
        else {
		if (tp_dev.sta & TP_PRES_DOWN)
			tp_dev.sta &= ~(1 << 7);
		else {
			tp_dev.x[4] = 0;
			tp_dev.y[4] = 0;
			tp_dev.x[0] = 0xffff;
			tp_dev.y[0] = 0xffff;
		}
	}
	return tp_dev.sta & TP_PRES_DOWN;
}

/**
  * @brief  Touch adjust function.
  * @param  display:display structure.
  * @retval None
  */
void bsp_xtp2046_adjust(bsp_display_env_t *display)
{
	uint16_t pos_temp[4][2] = {0};
	uint8_t cnt = 0;
	uint16_t d1, d2;
	uint32_t tem1, tem2;
	float fac = 0;
        display->back_color = WHITE;
	bsp_display_clear(display, &display->back_color);
	TP_Draw_Touch_Point(display, 20, 20, RED);
	tp_dev.sta = 0;
	tp_dev.xfac = 0;
	while (1)
	 {
READJ:
                tp_dev.scan(display, 1);
                if ((tp_dev.sta & 0xc0) == TP_CATH_PRES) {
			tp_dev.sta &= ~(1 << 6);
			pos_temp[cnt][0] = tp_dev.x[0];
			pos_temp[cnt][1] = tp_dev.y[0];
			cnt++;
                        switch (cnt)
			 {
				case 1:
					TP_Draw_Touch_Point(display, 20, 20, (WHITE));
					TP_Draw_Touch_Point(display, LCD_W - 20, 20, RED);
					break;
				case 2:
 					TP_Draw_Touch_Point(display, LCD_W - 20, 20,  (WHITE));
					TP_Draw_Touch_Point(display, 20, LCD_H - 20, (RED));
					break;
				case 3:
 					TP_Draw_Touch_Point(display, 20, LCD_H - 20, (WHITE));
 					TP_Draw_Touch_Point(display, LCD_W - 20, LCD_H - 20, (RED));
					break;
				case 4:
					for (cnt = 0; cnt < 3; cnt++)
					 {
						tem1 = abs(pos_temp[TP_ADJDIS_TBL[cnt][0]][0] - pos_temp[TP_ADJDIS_TBL[cnt][1]][0]);//x1 - x2 / x1 - x3 / x2 - x3
						tem2 = abs(pos_temp[TP_ADJDIS_TBL[cnt][0]][1] - pos_temp[TP_ADJDIS_TBL[cnt][1]][1]);//y1 - y2 / y1 - y3 / y2 - y3
						tem1 *= tem1;
						tem2 *= tem2;
						d1 = ald_calc_sqrt(tem1 + tem2);
						tem1 = abs(pos_temp[TP_ADJDIS_TBL[cnt][2]][0] - pos_temp[TP_ADJDIS_TBL[cnt][3]][0]);//x3 - x4 / x2 - x4 / x1 - x4
						tem2 = abs(pos_temp[TP_ADJDIS_TBL[cnt][2]][1] - pos_temp[TP_ADJDIS_TBL[cnt][3]][1]);//y3 - y4 / y2 - y4 / y1 - y4
						tem1 *= tem1;
						tem2 *= tem2;
						d2 = ald_calc_sqrt(tem1 + tem2);
						fac = (float)d1 / d2;
						if (fac < 0.95 || fac > 1.05 || d1 == 0 || d2 == 0) {
							cnt = 0;
							TP_Draw_Touch_Point(display, LCD_W - 20, LCD_H - 20, (WHITE));
							TP_Draw_Touch_Point(display, 20, 20, (RED));
                                                        bsp_display_string(display, 35, 110, "Error, Readjust!", 0);
                                                        goto READJ;
						}
					}
					tp_dev.xfac = (float)(LCD_W - 40) / (pos_temp[1][0] - pos_temp[0][0]);
					tp_dev.xoff = (uint32_t)((LCD_W - tp_dev.xfac * (pos_temp[1][0] + pos_temp[0][0])) / 2);
 					tp_dev.yfac = (float)(LCD_H - 40) / (pos_temp[2][1] - pos_temp[0][1]);
					tp_dev.yoff = (uint32_t)((LCD_H - tp_dev.yfac * (pos_temp[2][1] + pos_temp[0][1])) / 2);

					bsp_display_string(display, 35, 110, "Touch Screen Adjust OK!", 0);
					ald_delay_ms(1000);

 					bsp_display_clear(display, &display->back_color);
					return;
			}
                }
	}

}

/**
  * @brief Touch Callback function.
  * @param display:display structure.
  * @retval None
  */
void touch_call_back(bsp_display_env_t *display)
{
        uint16_t count = 300;
        uint16_t cnt[2] = {0};
        uint16_t buf[2] = {0};
        uint16_t point[2] = {0};

        tp_dev.scan(display, 0);
        point[0] = tp_dev.x[0];
        point[1] = tp_dev.y[0];
        while (XPT2046_EXTI_Read() == XPT2046_EXTI_ActiveLevel) {
                if (--count != 0) {
                        tp_dev.scan(display, 1);
                        if (tp_dev.x[0] > buf[0]) {               //right count
                                cnt[0]++;
                                buf[0] = tp_dev.x[0];
                        }
                        else if (tp_dev.x[0] < buf[0]) {          //left count
                                cnt[1]++;
                                buf[1] = tp_dev.x[0];
                        }
                }
                else break;
        }

        if ((cnt[0] > 60) && ((cnt[0] - cnt[1]) > 30))
               (display->cbk)(0, 0, 1);                            //right slide
        else if ((cnt[1] > 60) && ((cnt[1] - cnt[0]) > 30))
                (display->cbk)(0, 0, 2);                          //left slide
        else {
                (display->cbk)(point[0], point[1], 3);
        }
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
