/**
  *********************************************************************************
  *
  * @file    bsp_dht11.c
  * @brief   DHT11  driver
  *
  * @version V1.0
  * @date    06 Jun 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          06 Jun 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include "string.h"
#include "ald_gpio.h"
#include "ald_cmu.h"
#include "ald_timer.h"
#include "bsp_dht11.h"


 /** @addtogroup ES32F03xx_BSP
  * @{
  */

 /**
  * @defgroup DHT11 dht11 
  * @{
  */
/**
  * @defgroup DHT11_Private_Functions dht11 Private Functions
  * @{
  */

/**
  * @brief  delay us.
  * @retval None
  */
static void delay_us(uint32_t delay)
{
	uint32_t ticks;
	uint32_t told, tnow, tcnt = 0;
	uint32_t reload = SysTick->LOAD;
	ticks = delay * 48;
	told = SysTick->VAL;

	while (1)
	 {
		tnow = SysTick->VAL;
		if (tnow != told)
		 {
			if (tnow < told)tcnt += told - tnow;
			else tcnt += reload - tnow + told;
			told = tnow;
			if (tcnt >= ticks)break;
		}
	};
}

/**
  * @brief  Initializate pin of pf7.
  * @retval None
  */
static void bsp_dht111_pin_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_OUTPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odos = GPIO_PUSH_PULL;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(GPIOF, GPIO_PIN_7, &x);
	ald_gpio_write_pin(GPIOF, GPIO_PIN_7, 1);
}

/**
  * @brief  reset dht11 .
  * @retval None
  */
static void bsp_dht11_reset(void)
{
	DHT11_PIN_OUTPUT();
	ald_gpio_write_pin(GPIOF, GPIO_PIN_7, 0);
	/* 拉低至少18MS */
	ald_delay_ms(20);
	ald_gpio_write_pin(GPIOF, GPIO_PIN_7, 1);
	/* 主机拉高20-40US */
	delay_us(30);
}

/**
  * @brief  check dht11 return 1 or 0.
  * @retval return bit
  */
static uint8_t bsp_dht11_check(void)
{
	uint8_t retry = 0;
	DHT11_PIN_INPUT();
	/* DHT11会拉低40-80us */
	while (!ald_gpio_read_pin(GPIOF, GPIO_PIN_7) && retry < 100) {
		retry++;
		delay_us(1);
	}
	if (retry >= 100)
		return 1;
	else
		retry = 0;

	/* dht11会再拉高40-80us */
	while (!ald_gpio_read_pin(GPIOF, GPIO_PIN_7) && retry < 100) {
		retry++;
		delay_us(1);
	}
	if (retry >= 100)
		return 1;
	return 0;
}

/**
  * @brief  read dht11 bit
  * @retval bit data
  */
static uint8_t bsp_dht11_read_bit(void)
{
	uint8_t retry = 0;
	while (ald_gpio_read_pin(GPIOF, GPIO_PIN_7) && retry < 100) {
		retry++;
		delay_us(1);
	}
	retry = 0;
	while (!ald_gpio_read_pin(GPIOF, GPIO_PIN_7) && retry < 100) {
		retry++;
		delay_us(1);
	}
	/* delay 40 us */
	delay_us(35);
	if (ald_gpio_read_pin(GPIOF, GPIO_PIN_7))
		return 1;
	else
		return 0;
}

/**
  * @brief  read dht11 byte
  * @retval byte data
  */
static uint8_t bsp_dht11_read_byte(void)
{
	uint8_t i, dat = 0;
	for (i = 0; i < 8; i++) {
		dat <<= 1;
		dat |= bsp_dht11_read_bit();
	}
	return dat;
}
/**
  * @}
  */

/**
  * @defgroup DHT11_Public_Functions dht11 Public Functions
  * @brief    BSP DHT11 Functions
  * @verbatim
 ===============================================================================
              ##### BSP DHT11 Functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp dht11.
      (+) get dht11 data.

    @endverbatim
  * @{
  */

/**
  * @brief  get dht11 data 
  * @retval result 1 or 0
  */
uint8_t bsp_dht11_read_data(uint8_t *temp, uint8_t *humi)
{
	uint8_t buf[5] = {0};
	uint8_t i;

	bsp_dht11_reset();
	if (bsp_dht11_check() == 0) {
		for (i = 0; i < 5; i++)
			buf[i] = bsp_dht11_read_byte();

		if ((buf[0] + buf[1] + buf[2] + buf[3]) == buf[4]) {
			*humi = buf[0];
			*temp = buf[2];
		}
	}
	else
		return 1;

	return 0;
}

/**
  * @brief  init dht11 
  * @retval result 1 or 0
  */
uint8_t bsp_dht11_init(void)
{
	bsp_dht111_pin_init();
	ald_delay_ms(100);
	bsp_dht11_reset();

	return bsp_dht11_check();
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
