/**
  *********************************************************************************
  *
  * @file    bsp_can.c
  * @brief   can driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <stdio.h>
#include "bsp_can.h"

/** @addtogroup ES32F0xx_BSP
  * @{
  */

/**
  * @defgroup CAN can
  * @{
  */

/**
  * @defgroup Can_Public_Params can Public Variables
  * @{
  */

can_handle_t h_can;

/**
  * @}
  */

/**
  * @defgroup Can_Private_Functions can Private Functions
  * @{
  */

 /**
  * @brief  initialize pin of can module  
  * @retval None
  */ 
static void can_pin_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = CAN_TX_FUNC;
	ald_gpio_init(CAN_TX_PORT, CAN_TX_PIN, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = CAN_RX_FUNC;
	ald_gpio_init(CAN_RX_PORT, CAN_RX_PIN, &x);

	return;
}
/**
  * @}
  */

/**
  * @defgroup Can_Public_Functions can Public Functions
  * @brief    BSP can Functions
  * @verbatim
=========================================================================================
              ##### BSP can Functions #####
=========================================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp can module.
      (+) Send data by can module.
      (+) Recv data by can module.
    @endverbatim
  * @{
  */

/**
  * @brief  can module initialize function
  * @retval None
  */
void can_module_init(void)
{
	/* clear can_handle_t structure */
	memset(&h_can, 0x0, sizeof(h_can));
	/* Initialize pin */
	can_pin_init();

	/* Initialize CAN */
	h_can.perh        = CAN0;
	h_can.init.psc    = 8;
	h_can.init.mode   = CAN_MODE_NORMAL;
	h_can.init.sjw    = CAN_SJW_1;
	h_can.init.seg1   = CAN_SEG1_7;
	h_can.init.seg2   = CAN_SEG2_4;
	h_can.init.ttcm   = DISABLE;
	h_can.init.abom   = DISABLE;
	h_can.init.awk    = DISABLE;
	h_can.init.artx   = ENABLE;
	h_can.init.rfom   = DISABLE;
	h_can.init.txmp   = DISABLE;

	ald_can_init(&h_can);
}

/**
  * @brief  Sends an amount of data in blocking mode.
  * @param  tx_msg: struct of data to be sent.
  * @retval Status, see @ref can_status_t.
  */
can_status_t can_send_msg(can_tx_msg_t tx_msg)
{
#ifdef CAN_ID_EXTEND
	tx_msg.type = CAN_ID_EXT;
	tx_msg.ext  = 0x486489;
#else
	tx_msg.type = CAN_ID_STD;
	tx_msg.std  = 0x486;
#endif
	tx_msg.rtr  = CAN_RTR_DATA;
	tx_msg.len  = 8;

	return (can_status_t)ald_can_send(&h_can, &tx_msg, 100);
}

/**
  * @brief  Recvs an amount of data in blocking mode.
  * @param  rx_msg: Pointer to data buffer
  * @retval Status, see @ref can_status_t.
  */
can_status_t can_recv_msg(can_rx_msg_t* rx_msg)
{
	h_can.rx0_msg = rx_msg;

	return (can_status_t)ald_can_recv(&h_can, CAN_RX_FIFO0, rx_msg, 10000);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
