/**
  *********************************************************************************
  *
  * @file    bsp_buzzer.c
  * @brief   BUZZER driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include "bsp_buzzer.h"
#include "ald_gpio.h"
#include "ald_timer.h"
#include "ald_cmu.h"
#include "string.h"

 /**
   * @addtogroup ES32F03xx_BSP
   * @{
   */

/**
  * @defgroup BUZZER buzzer 
  * @{
  */
/**
  * @defgroup BUZZER_Public_Variable buzzer Public Variables
  * @{
  */
const uint8_t Music[] = {
21, 40, 22, 40, 23, 40, 21, 40,
21, 40, 22, 40, 23, 40, 21, 40,
23, 40, 24, 40, 25, 40, 00, 40,
23, 40, 24, 40, 25, 40, 00, 40,
25, 20, 26, 20, 25, 20, 24, 20,
23, 40, 21, 40, 25, 20, 26, 20,
25, 20, 24, 20, 23, 40, 21, 40,
21, 40, 15, 40, 21, 40, 00, 40,
21, 40, 15, 40, 21, 40, 00, 40,
0xFF
};

/**
  * @}
  */

/**
  * @defgroup BUZZER_Private_Params buzzer Private Variables
  * @{
  */
const uint16_t BuzzerTab[] =
{
00, 3816, 3400, 3032, 2866, 2552, 2272, 2024, 00, 00,
00, 1912, 1730, 1517, 1432, 1276, 1136, 1012, 00, 00,
00, 956,  851,  758,  715,  637,  568,  506,  00, 00,
00, 769,  684,  615,  577,  513,  461,  410,  384,
};

static timer_handle_t h_tim;
static timer_clock_config_t tim_clock;
static timer_oc_init_t tim_ocinit;

/**
  * @}
  */

/**
  * @defgroup BUZZER_Private_Functions buzzer Private Functions
  * @{
  */

/**
  * @brief  Initializate buzzer pin
  * @retval None.
  */
static void buzzer_pin_init(void)
{
	gpio_init_t x;

	/* Initialize cs pin PE0 */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
        x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_2;

        ald_gpio_init(GPIOA, GPIO_PIN_15, &x);
}

/**
  * @brief  Configure pwm freq and duty.
  * @param  h_tim: Pointer to a tim_handle_t structure.
  * @param  freq: pwm's freq.
  * @param  duty: pwm's duty.
  * @retval None
  */
static void bsp_buzzer_freq_duty_set(timer_handle_t *h_tim, uint16_t freq, uint8_t duty)
{
	ald_timer_pwm_set_freq(h_tim, freq);
	ald_timer_pwm_set_duty(h_tim, TIMER_CHANNEL_1, duty);

	return;
}

/**
  * @brief  Change buzzer frequency
  * @param  h_tim: Pointer to a tim_handle_t structure.
  * @param  frequency: the frequency want change
  * @retval None
  */
static void bsp_buzzer_change_freq(timer_handle_t *h_tim, uint16_t frequency)
{
	uint32_t temp;

	if (frequency == 0) {
		bsp_buzzer_freq_duty_set(h_tim, 0, 50);
		return;
	}

	temp = ald_cmu_get_pclk1_clock();
	temp = temp / 4;
	temp = temp * 2 / frequency;

	//Cur_Frequency = (ald_cmu_get_pclk2_clock() / 4) / frequency;
	bsp_buzzer_freq_duty_set(h_tim, temp, 50);
}

/**
  * @brief  Starts pwm.
  * @param  h_tim: Pointer to a tim_handle_t structure.
  * @retval None
  */
static void bsp_buzzer_start(timer_handle_t *h_tim)
{
	ald_timer_pwm_start(h_tim, TIMER_CHANNEL_1);
}

/**
  * @brief  Stops pwm.
  * @param  h_tim: Pointer to a tim_handle_t structure.
  * @retval None
  */
static void bsp_buzzer_stop(timer_handle_t *h_tim)
{
	ald_timer_pwm_stop(h_tim, TIMER_CHANNEL_1);
}

/**
  * @}
  */

/**
  * @defgroup BUZZER_Public_Functions buzzer Public Functions
  * @brief    BSP BUZZER Functions
  * @verbatim
 ===============================================================================
              ##### BSP BUZZER Functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp buzzer module
      (+) play music by buzzer whit specific data.

    @endverbatim
  * @{
  */

/**
  * @brief  Initializate buzzer moudle
  * @retval None.
  */
void bsp_buzzer_init(void)
{
	buzzer_pin_init();

	/* Initialize TIM2 */
	h_tim.perh            = GP16C2T0;
	h_tim.init.prescaler  = 0;
	h_tim.init.mode       = TIMER_CNT_MODE_UP;
	h_tim.init.period     = 0xffff;
	h_tim.init.clk_div    = TIMER_CLOCK_DIV4;
	h_tim.init.re_cnt     = 0;
	ald_timer_pwm_init(&h_tim);

	/* Initialize clock source */
	tim_clock.source = TIMER_SRC_INTER;
	ald_timer_config_clock_source(&h_tim, &tim_clock);

	/* Common configuration for all channels */
	tim_ocinit.oc_mode        = TIMER_OC_MODE_PWM1;
        tim_ocinit.oc_polarity    = TIMER_OC_POLARITY_HIGH;
        tim_ocinit.oc_fast_en     = DISABLE;
        tim_ocinit.ocn_polarity   = TIMER_OCN_POLARITY_HIGH;
        tim_ocinit.ocn_idle       = TIMER_OCN_IDLE_RESET;
        tim_ocinit.oc_idle        = TIMER_OC_IDLE_RESET;

        /* Set the pulse value for channel 1 */
        tim_ocinit.pulse = (uint32_t)(0x0088);
	ald_timer_oc_config_channel(&h_tim, &tim_ocinit, TIMER_CHANNEL_1);
}

/**
  * @brief  Play music by buzzer.
  * @param  sheet: the array name of music 
  * @retval None.
  */
void PlayMusic(const uint8_t *sheet)
{
	uint16_t tone, length, i;

	while (1) {
		if (*sheet == 0xFF) {
			return;
		}
		else if (*sheet == 0) {
			bsp_buzzer_stop(&h_tim);

			sheet++;
			length = *sheet;

			for (i = 0; i < length; i++)
				ald_delay_ms(10);

			sheet++;
		}
		else {
			tone = *sheet;
			bsp_buzzer_change_freq(&h_tim, BuzzerTab[tone]);
			bsp_buzzer_start(&h_tim);
			sheet++;
			length = *sheet;

			for (i = 0; i < length; i++)
				ald_delay_ms(10);

			sheet++;
		}
	}
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
