/**
  *********************************************************************************
  *
  * @file    bsp_adc.c
  * @brief   ADC driver
  *
  * @version V1.0
  * @date    24 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          24 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
 #include <stdio.h>
 #include "string.h"
 #include "bsp_adc.h"

/** @addtogroup ES32F03xx_BSP
  * @{
  */

/**
  * @defgroup ADC adc 
  * @{
  */

/**
  * @defgroup ADC_Private_Param adc Private Variables
  * @{
  */

uint32_t reg_value = 0;
uint32_t inj_value = 0;
adc_handle_t h_adc;
adc_nch_conf_t reg_config;

/**
  * @}
  */

/**
  * @defgroup ADC_Private_Functions adc Private Functions
  * @{
  */

/**
  * @brief  Initializate pin of adc
  * @retval None.
  */
static void adc_pin_init(void)
{
	/* Initialize ADC pin */
	gpio_init_t x;

	/* Initialize PC2 pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIOC, GPIO_PIN_2, &x);

	return;
}
/**
  * @}
  */

/**
  * @addtogroup ADC_Public_Functions adc Public Functions
  * @brief      BSP ADC Functions
  * @verbatim
=======================================================================================
                 ##### BSP ADC Functions #####
=======================================================================================
   [..]   This section provides functions allowing to:
      (+) Initialize bsp adc module.
      (+) Return ADC Sample data average value.

    @endverbatim
  * @{
  */

/**
  * @brief  ADC Module initialization.
  * @retval None
  */
void adc_module_init(void)
{
	/* clear lpuart_handle_t structure */
	memset(&h_adc, 0x0, sizeof(adc_handle_t));
	/* clear adc_nch_conf_t structure */
	memset(&reg_config, 0x0, sizeof(adc_nch_conf_t));

	adc_pin_init();

	/* Initialize adc */
	h_adc.perh          = ADC0;
	h_adc.init.align    = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan     = DISABLE;
	h_adc.init.cont     = DISABLE;
	h_adc.init.nch_nr   = ADC_NCH_NR_1;
	h_adc.init.disc     = ADC_ALL_DISABLE;
	h_adc.init.disc_nr  = ADC_DISC_NR_1;
	h_adc.init.data_bit = ADC_CONV_BIT_12;
	h_adc.init.div      = ADC_CKDIV_128;
	h_adc.init.n_ref    = ADC_NEG_REF_VSS;
	h_adc.init.p_ref    = ADC_POS_REF_VDD;

	ald_adc_init(&h_adc);

	/* Initialize regular convert channel */
	reg_config.ch   = ADC_CHANNEL_2;
	reg_config.idx  = ADC_NCH_IDX_1;
	reg_config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &reg_config);

	return;
}

/**
  * @brief  Get adc's 20 times average value.
  * @retval The average value of adc output.
  */
uint32_t adc_average(void)
{
	int32_t i;
	uint32_t adc_max, adc_min, adc_sum, adc_value;
	uint32_t reg_value;

	/* Start rejular convert */
	adc_max = 0,
	adc_min = 0xffffffff;
	adc_sum = 0;

	for (i = 0; i < 20; i++) {
		ald_adc_normal_start(&h_adc);

		/* Polling convert finish and read result */
		if (ald_adc_normal_poll_for_conversion(&h_adc, 5000) == OK)
			reg_value = ald_adc_normal_get_value(&h_adc);

		if (reg_value > adc_max)
			adc_max = reg_value;

		if (reg_value < adc_min)
			adc_min = reg_value;

		adc_sum = adc_sum + reg_value;
	}

	adc_sum = adc_sum - adc_max - adc_min;
	adc_sum = adc_sum / 18;
	adc_value = 3300 * 10 / 4096 * adc_sum / 10;

	return adc_value;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
