/**
  *********************************************************************************
  *
  * @file    bsp_display.c
  * @brief   Header of display driver
  *
  * @version V1.0
  * @date    16 Apr 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Apr 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __BSP_DISPLAY_H__
#define __BSP_DISPLAY_H__

#ifdef _cplusplus
extern "C" {
#endif

#include "utils.h"
#include <stdlib.h>
#include "ald_gpio.h"
#include "ald_spi.h"


/**
  * @addtogroup ES32F03xx_BSP
  * @{
  */

 /**
  * @addtogroup Display_Module
  * @{
  */  

/** @defgroup Display_Private_Macro display Private Macros
  * @{
  */ 

/**
  * @brief Touch operation command.
  */
#define CMD_RDX    0xD0
#define CMD_RDY    0x90
#define READ_TIMES 5
#define LOST_VAL   1
#define POINT_SIZE 5
#define SQ_SIZE    (POINT_SIZE * POINT_SIZE)
#define ERR_RANGE  50

/**
  *@}
  */

/** @defgroup Display_Public_Macro display Public Macros
  * @{
  */ 

/**
 * @brief LCD Color Value
   */ 	
#define LCD_W 		240
#define LCD_H 		320
#define OLED_MODE	0
#define SIZE		16
#define XLevelL		0x00
#define XLevelH		0x10
#define Max_Column	128
#define Max_Row		64
#define	Brightness	0xFF
#define X_WIDTH 	128
#define Y_WIDTH 	64
#define ROW_SIZE	29
#define WRITE_SIZE	1024
#define WHITE		0xFFFF
#define BLACK		0x0000
#define BLUE		0x001F
#define BRED		0XF81F
#define GRED		0XFFE0
#define GBLUE		0X07FF
#define RED		0xF800
#define MAGENTA		0xF81F
#define GREEN		0x07E0
#define CYAN		0x7FFF
#define YELLOW		0xFFE0
#define BROWN		0XBC40
#define BRRED		0XFC07
#define GRAY		0X8430

/**
 * @brief LCD Color Value Transfer
   */
#define COLOR_TRAN(x)	(uint16_t)(((x) << 8) | ((x) >> 8))


/**
 * @brief Touch Interrupt IO 
   */
#define XPT2046_EXTI_ActiveLevel        0
#define XPT2046_EXTI_Read()             ald_gpio_read_pin(GPIOE, GPIO_PIN_7)

/**
 * @brief LCD Control Command
   */
#define OLED_CMD  0
#define OLED_DATA 1


/**
 * @brief Touch State
   */
#define TP_PRES_DOWN 0x80
#define TP_CATH_PRES 0x40

/**
 * @brief LCD and Touch SPI Chip Select Control
   */
#define OLED_CS_CLR()	ald_gpio_write_pin(GPIOE, GPIO_PIN_15, 0);\
			ald_gpio_write_pin(GPIOE, GPIO_PIN_14, 1)
#define OLED_CS_SET()	ald_gpio_write_pin(GPIOE, GPIO_PIN_15, 1);\
			ald_gpio_write_pin(GPIOE, GPIO_PIN_14, 0)
#define TOUCH_CS_Clr()	ald_gpio_write_pin(GPIOE, GPIO_PIN_14, 0);\
                        ald_gpio_write_pin(GPIOE, GPIO_PIN_15, 1)
#define TOUCH_CS_Set()	ald_gpio_write_pin(GPIOE, GPIO_PIN_14, 1);\
                        ald_gpio_write_pin(GPIOE, GPIO_PIN_15, 0)
#define OLED_RST_CLR() 	ald_gpio_write_pin(GPIOC, GPIO_PIN_13, 0)
#define OLED_RST_SET() 	ald_gpio_write_pin(GPIOC, GPIO_PIN_13, 1)
#define OLED_DC_CLR() 	ald_gpio_write_pin(GPIOE, GPIO_PIN_3, 0)
#define OLED_DC_SET() 	ald_gpio_write_pin(GPIOE, GPIO_PIN_3, 1)

/**
  * @}
  */

/**
  * @defgroup Display_Module_Public_Type display Public Types
  * @{ 
  */

/**
 * @brief LCD and Touch Callback Function Prototype
 */
typedef int (*get_image_frame)(uint8_t *buf);
typedef void (* touch_callback_t)(uint16_t x, uint16_t y, uint8_t slide);

/**
  * @brief Display environment structure
  */
typedef struct bsp_display_env_s {
	spi_handle_t h_spi;
	char out_buf[64];
	uint16_t back_color;
	uint16_t point_color;
	uint16_t xpos;
	uint16_t ypos;
        uint16_t ad_xpos;
        uint16_t ad_ypos;
	uint16_t image_xpos;
	uint16_t image_ypos;
	uint16_t time_index;
        touch_callback_t cbk;
	uint8_t data_buf[580];
	uint16_t lcd_buf[512];
} bsp_display_env_t;

/**
  * @brief touch module timer structure,refer to zhengdian yuanzi
  */
typedef struct
{
	uint8_t (*scan)(bsp_display_env_t *display, uint8_t);	        //ɨ败.0, Ļɨ;1, ;
	void (*adjust)(bsp_display_env_t *display);		        //У׼
	uint16_t x[5]; 		                                        //ǰ
	uint16_t y[5];		                                        //x[0], y[0]:˴ɨʱ, , 
	uint8_t  sta;				                        //ʵ״̬
	float xfac;
	float yfac;
	short xoff;
	short yoff;
} _tp_dev;

/**
  * @}
  */

/** @addtogroup Display_Public_Functions
  * @{
  */ 
extern void bsp_display_init(bsp_display_env_t *display, touch_callback_t cbk);
extern void bsp_display_clear(bsp_display_env_t *display, uint16_t *color);
extern void bsp_display_string(bsp_display_env_t *display, uint16_t x, uint16_t y, char *str, uint8_t clear);
extern void bsp_display_gb(bsp_display_env_t *display, uint8_t x, uint8_t y, uint8_t *data, uint16_t size);
extern void bsp_display_image(bsp_display_env_t *display, uint8_t *data, uint16_t size, uint8_t index);
extern void bsp_display_image_zip(bsp_display_env_t *display, get_image_frame cbk);
extern void bsp_printf_d(bsp_display_env_t *display, uint16_t x, uint16_t y, const char *fmt, ...);
extern void bsp_display_by_row(bsp_display_env_t *display, char *p, uint16_t row);
extern void bsp_draw_big_point(bsp_display_env_t *display, uint16_t x, uint16_t y, uint16_t color);
extern void bsp_draw_clear_big_point(bsp_display_env_t *display, uint16_t x1, uint16_t y1,
					uint16_t color, uint16_t x2, uint16_t y2);
extern void bsp_draw_point(bsp_display_env_t *display, uint16_t x, uint16_t y, uint16_t color);
extern void bsp_draw_line(bsp_display_env_t *display, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color);
extern void bsp_draw_rectangle(bsp_display_env_t *display, uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint16_t color);
extern void bsp_draw_circle(bsp_display_env_t *display, uint16_t x0, uint16_t y0, uint8_t r, uint16_t color);
extern uint8_t bsp_tp_scan(bsp_display_env_t *display, uint8_t tp);
extern void bsp_xtp2046_adjust(bsp_display_env_t *display);
extern void touch_call_back(bsp_display_env_t *display);

/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef _cplusplus
}
#endif

#endif
