/**
  *********************************************************************************
  *
  * @file    ald_rmu.h
  * @brief   Header file of RMU module driver.
  *
  * @version V1.0
  * @date    30 Jun 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jun 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_RMU_H__
#define __ALD_RMU_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"

/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup RMU
  * @{
  */

/** @defgroup RMU_Public_Types RMU Public Types
  * @{
  */
/**
  * @brief RMU BOR fliter
  */
typedef enum
{
    RMU_BORFLT_1 = 0x1, /**< 1 cycle  */
    RMU_BORFLT_2 = 0x2, /**< 2 cycles */
    RMU_BORFLT_3 = 0x3, /**< 3 cycles */
    RMU_BORFLT_4 = 0x4, /**< 4 cycles */
    RMU_BORFLT_5 = 0x5, /**< 5 cycles */
    RMU_BORFLT_6 = 0x6, /**< 6 cycles */
    RMU_BORFLT_7 = 0x7, /**< 7 cycles */
} rmu_bor_filter_t;

/**
  * @brief RMU BOR voltage
  */
typedef enum
{
    RMU_VOL_1_8 = 0x0,  /**< 1.8V */
    RMU_VOL_2_0 = 0x1,  /**< 2.0V */
    RMU_VOL_2_2 = 0x2,  /**< 2.2V */
    RMU_VOL_2_4 = 0x3,  /**< 2.4V */
    RMU_VOL_2_6 = 0x4,  /**< 2.6V */
    RMU_VOL_2_8 = 0x5,  /**< 2.8V */
    RMU_VOL_3_0 = 0x6,  /**< 3.0V */
    RMU_VOL_3_2 = 0x7,  /**< 3.2V */
    RMU_VOL_3_4 = 0x8,  /**< 3.4V */
    RMU_VOL_3_6 = 0x9,  /**< 3.6V */
    RMU_VOL_3_8 = 0xA,  /**< 3.8V */
    RMU_VOL_4_0 = 0xB,  /**< 4.0V */
    RMU_VOL_4_2 = 0xC,  /**< 4.2V */
    RMU_VOL_4_4 = 0xD,  /**< 4.4V */
    RMU_VOL_4_6 = 0xE,  /**< 4.6V */
    RMU_VOL_4_8 = 0xF,  /**< 4.8V */
} rmu_bor_vol_t;

/**
  * @brief RMU reset status
  */
typedef enum
{
    RMU_RST_POR    = (1U << 0), /**< POR */
    RMU_RST_WAKEUP = (1U << 1), /**< WAKEUP */
    RMU_RST_BOR    = (1U << 2), /**< BOR */
    RMU_RST_NMRST  = (1U << 3), /**< NMRST */
    RMU_RST_IWDT   = (1U << 4), /**< IWDT */
    RMU_RST_WWDT   = (1U << 5), /**< WWDT */
    RMU_RST_LOCKUP = (1U << 6), /**< LOCKUP */
    RMU_RST_CHIP   = (1U << 7), /**< CHIP */
    RMU_RST_MCU    = (1U << 8), /**< MCU */
    RMU_RST_CPU    = (1U << 9), /**< CPU */
    RMU_RST_CFG    = (1U << 10),    /**< CFG */
    RMU_RST_CFGERR = (1U << 16),    /**< CFG Error */
	RMU_RST_ALL    = (0xFFFFFU),    /**< ALL */
} rmu_state_t;

/**
  * @brief RMU periperal select bit
  */
typedef enum
{
    RMU_PERH_GPIO     = (1U << 0),          /**< AHB1: GPIO */
    RMU_PERH_CRC      = (1U << 1),          /**< AHB1: CRC */
    RMU_PERH_CALC     = (1U << 2),          /**< AHB1: CALC */
    RMU_PERH_PIS      = (1U << 5),          /**< AHB1: PIS */
    RMU_PERH_CHIP     = (1U << 0)  | (1U << 27),    /**< AHB2: CHIP */
    RMU_PERH_CPU      = (1U << 1)  | (1U << 27),    /**< AHB2: CPU */
    RMU_PERH_EMI      = (1U << 8)  | (1U << 27),    /**< AHB2: EMI */
    RMU_PERH_BS16T0   = (1U << 1)  | (1U << 28),    /**< APB1: BS16T0 */
    RMU_PERH_GP16C2T0 = (1U << 2)  | (1U << 28),    /**< APB1: GP16C2T0 */
    RMU_PERH_GP16C2T1 = (1U << 3)  | (1U << 28),    /**< APB1: GP16C2T1 */
    RMU_PERH_BS16T1   = (1U << 4)  | (1U << 28),    /**< APB1: BS16T1 */
    RMU_PERH_GP16C4T0 = (1U << 6)  | (1U << 28),    /**< APB1: GP16C4T0 */
    RMU_PERH_UART0    = (1U << 8)  | (1U << 28),    /**< APB1: UART0 */
    RMU_PERH_USART0   = (1U << 12) | (1U << 28),    /**< APB1: EUART0 */
    RMU_PERH_USART1   = (1U << 13) | (1U << 28),    /**< APB1: EUART1 */
    RMU_PERH_SPI0     = (1U << 16) | (1U << 28),    /**< APB1: SPI0 */
    RMU_PERH_SPI1     = (1U << 17) | (1U << 28),    /**< APB1: SPI1 */
    RMU_PERH_I2C0     = (1U << 20) | (1U << 28),    /**< APB1: I2C0 */
    RMU_PERH_I2C1     = (1U << 21) | (1U << 28),    /**< APB1: I2C1 */
    RMU_PERH_LPTIM0   = (1U << 0)  | (1U << 29),    /**< APB2: LPTIM0 */
    RMU_PERH_LPUART0  = (1U << 2)  | (1U << 29),    /**< APB2: LPUART */
    RMU_PERH_ADC0     = (1U << 4)  | (1U << 29),    /**< APB2: ADC0 */
    RMU_PERH_ACMP0    = (1U << 6)  | (1U << 29),    /**< APB2: ACMP0 */
    RMU_PERH_ACMP1    = (1U << 7)  | (1U << 29),    /**< APB2: ACMP1 */
    RMU_PERH_WWDT     = (1U << 12) | (1U << 29),    /**< APB2: WWDT */
    RMU_PERH_LCD      = (1U << 13) | (1U << 29),    /**< APB2: LCD */
    RMU_PERH_IWDT     = (1U << 14) | (1U << 29),    /**< APB2: IWDT */
    RMU_PERH_RTC      = (1U << 15) | (1U << 29),    /**< APB2: RTC */
    RMU_PERH_DBGC     = (1U << 19) | (1U << 29),    /**< APB2: DBGC */
    RMU_PERH_TKS      = (1U << 20) | (1U << 29),    /**< APB2: TKS */
} rmu_peripheral_t;
/**
  * @}
  */

/**
  * @defgroup RMU_Private_Macros RMU Private Macros
  * @{
  */
#define IS_RMU_BORFLT(x)    (((x) == RMU_BORFLT_1) || \
                             ((x) == RMU_BORFLT_2) || \
                             ((x) == RMU_BORFLT_3) || \
                             ((x) == RMU_BORFLT_4) || \
                             ((x) == RMU_BORFLT_5) || \
                             ((x) == RMU_BORFLT_6) || \
                             ((x) == RMU_BORFLT_7))
#define IS_RMU_BORVOL(x)    (((x) == RMU_VOL_1_8) || \
                             ((x) == RMU_VOL_2_0) || \
                             ((x) == RMU_VOL_2_2) || \
                             ((x) == RMU_VOL_2_4) || \
                             ((x) == RMU_VOL_2_6) || \
                             ((x) == RMU_VOL_2_8) || \
                             ((x) == RMU_VOL_3_0) || \
                             ((x) == RMU_VOL_3_2) || \
                             ((x) == RMU_VOL_3_4) || \
                             ((x) == RMU_VOL_3_6) || \
                             ((x) == RMU_VOL_3_8) || \
                             ((x) == RMU_VOL_4_0) || \
                             ((x) == RMU_VOL_4_2) || \
                             ((x) == RMU_VOL_4_4) || \
                             ((x) == RMU_VOL_4_6) || \
                             ((x) == RMU_VOL_4_8))
#define IS_RMU_STATE(x)     (((x) == RMU_RST_POR)    || \
                             ((x) == RMU_RST_WAKEUP) || \
                             ((x) == RMU_RST_BOR)    || \
                             ((x) == RMU_RST_NMRST)  || \
                             ((x) == RMU_RST_IWDT)   || \
                             ((x) == RMU_RST_WWDT)   || \
                             ((x) == RMU_RST_LOCKUP) || \
                             ((x) == RMU_RST_CHIP)   || \
                             ((x) == RMU_RST_MCU)    || \
                             ((x) == RMU_RST_CPU)    || \
                             ((x) == RMU_RST_CFG)    || \
                             ((x) == RMU_RST_CFGERR) || \
							 ((x) == RMU_RST_ALL))
#define IS_RMU_STATE_CLEAR(x)   (((x) == RMU_RST_POR)    || \
                                 ((x) == RMU_RST_WAKEUP) || \
                                 ((x) == RMU_RST_BOR)    || \
                                 ((x) == RMU_RST_NMRST)  || \
                                 ((x) == RMU_RST_IWDT)   || \
                                 ((x) == RMU_RST_WWDT)   || \
                                 ((x) == RMU_RST_LOCKUP) || \
                                 ((x) == RMU_RST_CHIP)   || \
                                 ((x) == RMU_RST_MCU)    || \
                                 ((x) == RMU_RST_CPU)    || \
                                 ((x) == RMU_RST_CFG)    || \
								 ((x) == RMU_RST_CFGERR) || \
								 ((x) == RMU_RST_ALL))
#define IS_RMU_PERH(x)      (((x) == RMU_PERH_GPIO)     || \
                             ((x) == RMU_PERH_CRC)      || \
                             ((x) == RMU_PERH_CALC)     || \
                             ((x) == RMU_PERH_PIS)      || \
                             ((x) == RMU_PERH_CHIP)     || \
                             ((x) == RMU_PERH_CPU)      || \
                             ((x) == RMU_PERH_EMI)      || \
                             ((x) == RMU_PERH_BS16T0)   || \
                             ((x) == RMU_PERH_GP16C2T0) || \
                             ((x) == RMU_PERH_GP16C2T1) || \
                             ((x) == RMU_PERH_BS16T1)   || \
                             ((x) == RMU_PERH_GP16C4T0) || \
                             ((x) == RMU_PERH_UART0)    || \
                             ((x) == RMU_PERH_USART0)   || \
                             ((x) == RMU_PERH_USART1)   || \
                             ((x) == RMU_PERH_SPI0)     || \
                             ((x) == RMU_PERH_SPI1)     || \
                             ((x) == RMU_PERH_I2C0)     || \
                             ((x) == RMU_PERH_I2C1)     || \
                             ((x) == RMU_PERH_LPTIM0)   || \
                             ((x) == RMU_PERH_LPUART0)  || \
                             ((x) == RMU_PERH_ADC0)     || \
                             ((x) == RMU_PERH_ACMP0)    || \
                             ((x) == RMU_PERH_ACMP1)    || \
                             ((x) == RMU_PERH_WWDT)     || \
                             ((x) == RMU_PERH_LCD)      || \
                             ((x) == RMU_PERH_IWDT)     || \
                             ((x) == RMU_PERH_RTC)      || \
                             ((x) == RMU_PERH_DBGC)     || \
                             ((x) == RMU_PERH_TKS))
/**
  * @}
  */

/** @addtogroup RMU_Public_Functions
  * @{
  */
void ald_rmu_bor_config(rmu_bor_filter_t flt, rmu_bor_vol_t vol, type_func_t state);
flag_status_t ald_rmu_get_reset_status(rmu_state_t state);
void ald_rmu_clear_reset_status(rmu_state_t state);
void ald_rmu_reset_periperal(rmu_peripheral_t perh);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_RMU_H__ */
