/**
  *********************************************************************************
  *
  * @file    ald_pis.h
  * @brief   Header file of PIS driver.
  *
  * @version V1.0
  * @date    30 Jun 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jun 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_PIS_H__
#define __ALD_PIS_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup PIS
  * @{
  */

/** @defgroup PIS_Public_Types PIS Public Types
  * @{
  */
/**
  * @brief Producer entry
  */
typedef enum
{
    PIS_NON                 = 0x0U,      /**< No async */
    PIS_GPIO_PIN0           = 0x10U,     /**< PIN0, level,support async */
    PIS_GPIO_PIN1           = 0x11U,     /**< PIN1, level,support async */
    PIS_GPIO_PIN2           = 0x12U,     /**< PIN2, level,support async */
    PIS_GPIO_PIN3           = 0x13U,     /**< PIN3, level,support async */
    PIS_GPIO_PIN4           = 0x14U,     /**< PIN4, level,support async */
    PIS_GPIO_PIN5           = 0x15U,     /**< PIN5, level,support async */
    PIS_GPIO_PIN6           = 0x16U,     /**< PIN6, level,support async */
    PIS_GPIO_PIN7           = 0x17U,     /**< PIN7, level,support async */
    PIS_GPIO_PIN8           = 0x18U,     /**< PIN8, level,support async */
    PIS_GPIO_PIN9           = 0x19U,     /**< PIN9, level,support async */
    PIS_GPIO_PIN10          = 0x1aU,     /**< PIN10, level,support async */
    PIS_GPIO_PIN11          = 0x1bU,     /**< PIN11, level,support async */
    PIS_GPIO_PIN12          = 0x1cU,     /**< PIN12, level,support async */
    PIS_GPIO_PIN13          = 0x1dU,     /**< PIN13, level,support async */
    PIS_GPIO_PIN14          = 0x1eU,     /**< PIN14, level,support async */
    PIS_GPIO_PIN15          = 0x1fU,     /**< PIN15, level,support async */
    PIS_FDET                = 0x23U,     /**< FDET, level,support async */
    PIS_ACMP_OUT0           = 0x30U,     /**< ACMP0 output, level,support async */
    PIS_ACMP_OUT1           = 0x31U,     /**< ACMP1 output, level,support async */
    PIS_ADC0_INJECT         = 0x60U,     /**< ADC0 inject, pclk2 pulse,support async */
    PIS_ADC0_REGULAT        = 0x61U,     /**< ADC0 regulat, pclk2 pulse,support async */
    PIS_ADC0_WINDOW         = 0x62U,     /**< ADC0 window, no have */
    PIS_LVD                 = 0x70U,     /**< LVD, level,support async */
    PIS_UART0_IRDAOUT       = 0x82U,     /**< UART0 irdaout, level,support async */
    PIS_UART0_RTSOUT        = 0x83U,     /**< UART0 rtsout, level,support async */
    PIS_UART0_TXOUT         = 0x84U,     /**< UART0 txout, level,support async */
    PIS_UART0_SEND          = 0x85U,     /**< UART0 syn send, pulse,support async */
    PIS_UART0_RECV          = 0x86U,     /**< UART0 syn recv, pulse,support async */
    PIS_USART0_RECV         = 0xc0U,     /**< USART0 recv, plck1 pulse */
    PIS_USART0_SEND         = 0xc1U,     /**< USART0 send, plck1 pulse */
    PIS_USART0_TXOUT        = 0xc2U,     /**< USART0 txout, plck1 level */
    PIS_USART1_RECV         = 0xd0U,     /**< USART1 recv, plck1 pulse */
    PIS_USART1_SEND         = 0xd1U,     /**< USART1 send, plck1 pulse */
    PIS_USART1_TXOUT        = 0xd2U,     /**< USART1 txout, plck1 level */
    PIS_SPI0_RECV           = 0xe0U,     /**< SPI0 recv, plck1 pulse */
    PIS_SPI0_SEND           = 0xe1U,     /**< SPI0 send, plck1 pulse */
    PIS_SPI0_NE             = 0xe2U,     /**< SPI0 ne, plck1 level */
    PIS_SPI1_RECV           = 0xf0U,     /**< SPI1 recv, plck1 pulse */
    PIS_SPI1_SEND           = 0xf1U,     /**< SPI1 send, plck1 pulse */
    PIS_SPI1_NE             = 0xf2U,     /**< SPI1 ne, plck1 level */
    PIS_I2C0_RECV           = 0x100U,    /**< I2C0 recv, plck1 level */
    PIS_I2C0_SEND           = 0x101U,    /**< I2C0 send, plck1 level */
    PIS_I2C1_RECV           = 0x110U,    /**< I2C1 recv, plck1 level */
    PIS_I2C1_SEND           = 0x111U,    /**< I2C1 send, plck1 level */
    PIS_BS16T0_UPDATA       = 0x130U,    /**< BS16T0 updata, plck1 pulse */
    PIS_GP16C2T0_UPDATA     = 0x140U,    /**< GP16C2T0 updata, plck1 pulse */
    PIS_GP16C2T0_TRIG       = 0x141U,    /**< GP16C2T0 trig, plck1 pulse */
    PIS_GP16C2T0_CH1_INPUT  = 0x142U,    /**< GP16C2T0 channel1 input, plck1 pulse */
    PIS_GP16C2T0_CH1_OUTPUT = 0x143U,    /**< GP16C2T0 channel1 output, plck1 pulse */
    PIS_GP16C2T0_CH2_INPUT  = 0x144U,    /**< GP16C2T0 channel2 input, plck1 pulse */
    PIS_GP16C2T0_CH2_OUTPUT = 0x145U,    /**< GP16C2T0 channel2 output, plck1 pulse */
    PIS_GP16C2T1_UPDATA     = 0x150U,    /**< GP16C2T1 updata, plck1 pulse */
    PIS_GP16C2T1_TRIG       = 0x151U,    /**< GP16C2T1 trig, plck1 pulse */
    PIS_GP16C2T1_CH1_INPUT  = 0x152U,    /**< GP16C2T1 channel1 input, plck1 pulse */
    PIS_GP16C2T1_CH1_OUTPUT = 0x153U,    /**< GP16C2T1 channel1 output, plck1 pulse */
    PIS_GP16C2T1_CH2_INPUT  = 0x154U,    /**< GP16C2T1 channel2 input, plck1 pulse */
    PIS_GP16C2T1_CH2_OUTPUT = 0x155U,    /**< GP16C2T1 channel2 output, plck1 pulse */
    PIS_RTC_CLOCK           = 0x160U,    /**< RTC clock, pulse,support async */
    PIS_RTC_ALARM           = 0x161U,    /**< RTC alarm, pulse,support async */
    PIS_LP16T0_SYN_UPDATA   = 0x170U,    /**< LP16T0 syn updata, pulse,support async */
    PIS_LP16T0_ASY_UPDATA   = 0x171U,    /**< LP16T0 asy updata, pulse,support async */
    PIS_LPUART0_ASY_RECV    = 0x180U,    /**< Lpuart0 asy recv, pulse,support async */
    PIS_LPUART0_ASY_SEND    = 0x181U,    /**< Lpuart0 asy send, pulse,support async */
    PIS_LPUART0_SYN_RECV    = 0x182U,    /**< Lpuart0 syn recv, pulse,support async */
    PIS_LPUART0_SYN_SEND    = 0x183U,    /**< Lpuart0 syn recv, pulse,support async */
    PIS_DMA_CH0             = 0x190U,    /**< DMA channel 0, pulse,support async */
    PIS_DMA_CH1             = 0x191U,    /**< DMA channel 1, pulse,support async */
    PIS_DMA_CH2             = 0x192U,    /**< DMA channel 2, pulse,support async */
    PIS_DMA_CH3             = 0x193U,    /**< DMA channel 3, pulse,support async */
    PIS_DMA_CH4             = 0x194U,    /**< DMA channel 4, pulse,support async */
    PIS_DMA_CH5             = 0x195U,    /**< DMA channel 5, pulse,support async */
    PIS_DMA_CH_ALL          = 0x19fU,    /**< DMA all channel, pulse,support async */
    PIS_BS16T1_UPDATA       = 0x1b0U,    /**< BS16T1 updata, plck1 pulse */
    PIS_GP16C4T0_UPDATA     = 0x1d0U,    /**< GP16C4T0 updata, plck1 pulse */
    PIS_GP16C4T0_TRIG       = 0x1d1U,    /**< GP16C4T0 trig, plck1 pulse */
    PIS_GP16C4T0_CH1_INPUT  = 0x1d2U,    /**< GP16C4T0 channel1 input, plck1 pulse */
    PIS_GP16C4T0_CH1_OUTPUT = 0x1d3U,    /**< GP16C4T0 channel1 output, plck1 pulse */
    PIS_GP16C4T0_CH2_INPUT  = 0x1d4U,    /**< GP16C4T0 channel2 input, plck1 pulse */
    PIS_GP16C4T0_CH2_OUTPUT = 0x1d5U,    /**< GP16C4T0 channel2 output, plck1 pulse */
    PIS_GP16C4T0_CH3_INPUT  = 0x1d6U,    /**< GP16C4T0 channel3 input, plck1 pulse */
    PIS_GP16C4T0_CH3_OUTPUT = 0x1d7U,    /**< GP16C4T0 channel3 output, plck1 pulse */
    PIS_GP16C4T0_CH4_INPUT  = 0x1d8U,    /**< GP16C4T0 channel4 input, plck1 pulse */
    PIS_GP16C4T0_CH4_OUTPUT = 0x1d9U,    /**< GP16C4T0 channel4 output, plck1 pulse */
} pis_src_t;

/**
  * @brief Consumer entry
  */
typedef enum
{
    PIS_CH0_SPI1_CLK       = 0x0F10U,   /**< SPI1 CLK */
    PIS_CH0_L16T0_EXT0     = 0x0030U,   /**< L16T0 EXT0 */
    PIS_CH1_UART0_RX_IRDA  = 0x0011U,   /**< UART0 rx IRDA */
    PIS_CH1_GP16C4T0_CH1IN = 0x0801U,   /**< GP16C4T0 CH1IN  */
    PIS_CH1_L16T0_EXT1     = 0x0031U,   /**< L16T0 EXT1 */
    PIS_CH2_GP16C4T0_CH2IN = 0x0902U,   /**< GP16C4T0 CH2IN */
    PIS_CH2_L16T0_EXT2     = 0x0032U,   /**< L16T0 EXT2 */
    PIS_CH2_TKS_TRIG       = 0x0032U,   /**< TKS Trigger */
    PIS_CH3_GP16C4T0_CH3IN = 0x0A03U,   /**< GP16C4T0 CH3IN */
    PIS_CH3_L16T0_EXT3     = 0x0033U,   /**< L16T0 EXT3 */
    PIS_CH4_GP16C4T0_ITR0  = 0x0034U,   /**< GP16C4T0 ITR0 */
    PIS_CH4_GP16C4T0_CH4IN = 0x0B04U,   /**< GP16C4T0 CH4IN */
    PIS_CH4_GP16C2T0_ITR0  = 0x0034U,   /**< GP16C2T0 ITR0 */
    PIS_CH4_GP16C2T1_ITR0  = 0x0034U,   /**< GP16C2T1 ITR0 */
    PIS_CH4_L16T0_EXT4     = 0x0034U,   /**< L16T0 EXT4 */
    PIS_CH5_USART0_RX      = 0x0615U,   /**< USART0 RX */
    PIS_CH5_SPI0_RX        = 0x0C15U,   /**< SPI0 RX */
    PIS_CH5_GP16C4T0_ITR1  = 0x0035U,   /**< GP16C4T0 ITR1 */
    PIS_CH5_GP16C2T0_ITR1  = 0x0035U,   /**< GP16C2T0 ITR1 */
    PIS_CH5_GP16C2T0_CH1IN = 0x1005U,   /**< GP16C2T0 CH1IN */
    PIS_CH5_GP16C2T1_ITR1  = 0x0035U,   /**< GP16C2T1 ITR1 */
    PIS_CH5_GP16C2T1_CH1IN = 0x1805U,   /**< GP16C2T1 CH1IN */
    PIS_CH5_L16T0_EXT5     = 0x0035U,   /**< L16T0 EXT5 */
    PIS_CH6_USART1_RX      = 0x0716U,   /**< USART1 RX */
    PIS_CH6_SPI0_CLK       = 0x0D16U,   /**< SPI0 CLK */
    PIS_CH6_GP16C4T0_ITR2  = 0x0036U,   /**< GP16C4T0 ITR2 */
    PIS_CH6_GP16C2T0_ITR2  = 0x0036U,   /**< GP16C2T0 ITR2 */
    PIS_CH6_GP16C2T0_CH2IN = 0x1106U,   /**< GP16C2T0 CH2IN */
    PIS_CH6_GP16C2T1_ITR2  = 0x0036U,   /**< GP16C2T1 ITR2 */
    PIS_CH6_GP16C2T1_CH2IN = 0x1906U,   /**< GP16C2T1 CH2IN */
    PIS_CH6_ADC0_NORMAL    = 0x0036U,   /**< ADC0 NORMAL */
    PIS_CH6_L16T0_EXT6     = 0x0036U,   /**< L16T0 EXT6 */
    PIS_CH7_SPI1_RX        = 0x0E17U,   /**< SPI1 RX */
    PIS_CH7_GP16C4T0_ITR3  = 0x0037U,   /**< GP16C4T0 ITR3 */
    PIS_CH7_GP16C2T0_ITR3  = 0x0037U,   /**< GP16C2T0 ITR3 */
    PIS_CH7_GP16C2T1_ITR3  = 0x0037U,   /**< GP16C2T1 ITR3 */
    PIS_CH7_ADC0_INSERT    = 0x0037U,   /**< ADC0 INSERT */
    PIS_CH7_L16T0_EXT7     = 0x0037U,   /**< L16T0 EXT7 */
    PIS_CH7_LPUART_RX      = 0x0817U,   /**< LPUART RX */
    PIS_CH7_DMA            = 0x0037U,   /**< DMA */
} pis_trig_t;

/**
  * @brief Clock select
  */
typedef enum
{
    PIS_CLK_PCLK1 = 0U, /**< Pclock1 */
    PIS_CLK_PCLK2 = 1U, /**< Pclock2 */
    PIS_CLK_SYS   = 2U, /**< Sys clock */
    PIS_CLK_LP    = 3U, /**< Low power clock */
} pis_clock_t;

/**
  * @brief Level select
  */
typedef enum
{
    PIS_EDGE_NONE    = 0U,  /**< None edge */
    PIS_EDGE_UP      = 1U,  /**< Up edge */
    PIS_EDGE_DOWN    = 2U,  /**< Down edge */
    PIS_EDGE_UP_DOWN = 3U,  /**< Up and down edge */
} pis_edge_t;

/**
  * @brief Output style
  */
typedef enum
{
    PIS_OUT_LEVEL = 0U, /**< Level */
    PIS_OUT_PULSE = 1U, /**< Pulse */
} pis_output_t;
/**
  * @brief Sync select
  */
typedef enum
{
    PIS_SYN_DIRECT      = 0U,   /**< Direct */
    PIS_SYN_ASY_PCLK1   = 1U,   /**< Asy pclk1 */
    PIS_SYN_ASY_PCLK2   = 2U,   /**< Asy pclk2 */
    PIS_SYN_ASY_PCLK    = 3U,   /**< Asy pclk */
    PIS_SYN_PCLK2_PCLK1 = 4U,   /**< Pclk2 to pclk1 */
    PIS_SYN_PCLK1_PCLK2 = 5U,   /**< Pclk1 to pclk2 */
    PIS_SYN_PCLK12_SYS  = 6U,   /**< Pclk1 or pclk2 to sysclk */
} pis_syncsel_t;

/**
  * @brief Pis channel
  */
typedef enum
{
    PIS_CH_0 = 0U,  /**< Channel 0 */
    PIS_CH_1 = 1U,  /**< Channel 1 */
    PIS_CH_2 = 2U,  /**< Channel 2 */
    PIS_CH_3 = 3U,  /**< Channel 3 */
    PIS_CH_4 = 4U,  /**< Channel 4 */
    PIS_CH_5 = 5U,  /**< Channel 5 */
    PIS_CH_6 = 6U,  /**< Channel 6 */
    PIS_CH_7 = 7U,  /**< Channel 7 */
} pis_ch_t;

/**
  * @brief Pis output channel
  */
typedef enum
{
    PIS_OUT_CH_0 = 0U,  /**< Channel 0 */
    PIS_OUT_CH_1 = 1U,  /**< Channel 1 */
    PIS_OUT_CH_2 = 2U,  /**< Channel 2 */
    PIS_OUT_CH_3 = 3U,  /**< Channel 3 */
} pis_out_ch_t;

/**
  * @brief Indirect value,no care of it.
  */
typedef enum
{
    PIS_CON_0    = 0U,  /**< Con 0 */
    PIS_CON_1    = 1U,  /**< Con 1 */
    PIS_CON_NONE = 2U,  /**< None */
} pis_con_t;

/**
  * @brief PIS state structures definition
  */
typedef enum
{
    PIS_STATE_RESET   = 0x00U,  /**< Peripheral is not initialized */
    PIS_STATE_READY   = 0x01U,  /**< Peripheral Initialized and ready for use */
    PIS_STATE_BUSY    = 0x02U,  /**< An internal process is ongoing */
    PIS_STATE_TIMEOUT = 0x03U,  /**< Timeout state */
    PIS_STATE_ERROR   = 0x04U,  /**< Error */
} pis_state_t;

/**
  * @brief PIS modulate target
  */
typedef enum
{
    PIS_UART0_TX   = 0x0U,  /**< Modulate uart0 tx */
    PIS_USART0_TX  = 0x1U,  /**< Modulate usart0 tx */
    PIS_USART1_TX  = 0x2U,  /**< Modulate usart1 tx */
    PIS_LPUART0_TX = 0x4U,  /**< Modulate lpuart0 tx */
} pis_modu_targ_t;

/**
  * @brief PIS modulate level
  */
typedef enum
{
    PIS_LOW_LEVEL  = 0U,    /**< Modulate low level */
    PIS_HIGH_LEVEL = 1U,    /**< Modulate high level */
} pis_modu_level_t;

/**
  * @brief PIS modulate source
  */
typedef enum
{
    PIS_SRC_NONE     = 0x0U,    /**< Stop modulate */
    PIS_SRC_GP16C2T0 = 0x3U,    /**< Modulate source is GP16C2T0 */
    PIS_SRC_GP16C2T1 = 0x4U,    /**< Modulate source is GP16C2T1 */
    PIS_SRC_GP16C4T0 = 0x5U,    /**< Modulate source is GP16C4T0 */
    PIS_SRC_LP16T0   = 0x7U,    /**< Modulate source is LP16T0 */
    PIS_SRC_BUZ      = 0x8U,    /**< Modulate source is BUZZ */
} pis_modu_src_t;

/**
  * @brief PIS modulate channel
  */
typedef enum
{
    PIS_TIMER_CH1 = 0U, /**< Src is TIMERx and choose channel 1 */
    PIS_TIMER_CH2 = 1U, /**< Src is TIMERx and choose channel 2 */
    PIS_TIMER_CH3 = 2U, /**< Src is TIMERx and choose channel 3 */
    PIS_TIMER_CH4 = 3U, /**< Src is TIMERx and choose channel 4 */
} pis_modu_channel_t;

/**
  * @brief PIS init structure definition
  */
typedef struct
{
    pis_src_t producer_src;     /**< Producer entry */
    pis_clock_t producer_clk;   /**< Producer module clock */
    pis_edge_t producer_edge;   /**< Producer module pin output edge */
    pis_trig_t consumer_trig;   /**< Consumer entry */
    pis_clock_t consumer_clk;   /**< Consumer clock */
} pis_init_t;

/**
  * @brief PIS modulate config structure definition
  */
typedef struct
{
    pis_modu_targ_t target;     /**< Modulate target */
    pis_modu_level_t level;     /**< Modulate level */
    pis_modu_src_t src;     /**< Modulate src */
    pis_modu_channel_t channel; /**< Modulate channel */
} pis_modulate_config_t;

/**
  * @brief  PIS Handle Structure definition
  */
typedef struct pis_handle_s
{
    PIS_TypeDef *perh;      /**< Register base address */
    pis_init_t init;        /**< PIS required parameters */
    pis_ch_t consumer_ch;       /**< Indirect value, no care of it */
    pis_con_t consumer_con;     /**< Indirect value, no care of it */
    uint32_t consumer_pos;          /**< Indirect value, no care of it */
    uint32_t check_info;        /**< When destroy a handle ,user need check whether is right that ready to destroy */
    lock_state_t lock;      /**< Locking object */
    pis_state_t state;      /**< PIS operation state */
} pis_handle_t;
/**
  * @}
  */


/** @defgroup PIS_Private_Macros PIS Private Macros
  * @{
  */
#define IS_PIS(x)       (((x) == PIS))
#define IS_PIS_SRC(x)   (((x) == PIS_NON)                 || \
                         ((x) == PIS_GPIO_PIN0)           || \
                         ((x) == PIS_GPIO_PIN1)           || \
                         ((x) == PIS_GPIO_PIN2)           || \
                         ((x) == PIS_GPIO_PIN3)           || \
                         ((x) == PIS_GPIO_PIN4)           || \
                         ((x) == PIS_GPIO_PIN5)           || \
                         ((x) == PIS_GPIO_PIN6)           || \
                         ((x) == PIS_GPIO_PIN7)           || \
                         ((x) == PIS_GPIO_PIN8)           || \
                         ((x) == PIS_GPIO_PIN9)           || \
                         ((x) == PIS_GPIO_PIN10)          || \
                         ((x) == PIS_GPIO_PIN11)          || \
                         ((x) == PIS_GPIO_PIN12)          || \
                         ((x) == PIS_GPIO_PIN13)          || \
                         ((x) == PIS_GPIO_PIN14)          || \
                         ((x) == PIS_GPIO_PIN15)          || \
                         ((x) == PIS_FDET)                || \
                         ((x) == PIS_ACMP_OUT0)           || \
                         ((x) == PIS_ACMP_OUT1)           || \
                         ((x) == PIS_ACMP_OUT1)           || \
                         ((x) == PIS_ADC0_INJECT)         || \
                         ((x) == PIS_ADC0_REGULAT)        || \
                         ((x) == PIS_ADC0_WINDOW)         || \
                         ((x) == PIS_LVD)                 || \
                         ((x) == PIS_UART0_IRDAOUT)       || \
                         ((x) == PIS_UART0_RTSOUT)        || \
                         ((x) == PIS_UART0_TXOUT)         || \
                         ((x) == PIS_UART0_SEND)          || \
                         ((x) == PIS_UART0_RECV)          || \
                         ((x) == PIS_USART0_RECV)         || \
                         ((x) == PIS_USART0_SEND)         || \
                         ((x) == PIS_USART0_TXOUT)        || \
                         ((x) == PIS_USART1_RECV)         || \
                         ((x) == PIS_USART1_SEND)         || \
                         ((x) == PIS_USART1_TXOUT)        || \
                         ((x) == PIS_SPI0_RECV)           || \
                         ((x) == PIS_SPI0_SEND)           || \
                         ((x) == PIS_SPI0_NE)             || \
                         ((x) == PIS_SPI1_RECV)           || \
                         ((x) == PIS_SPI1_SEND)           || \
                         ((x) == PIS_SPI1_NE)             || \
                         ((x) == PIS_I2C0_RECV)           || \
                         ((x) == PIS_I2C0_SEND)           || \
                         ((x) == PIS_I2C1_RECV)           || \
                         ((x) == PIS_I2C1_SEND)           || \
                         ((x) == PIS_BS16T0_UPDATA)       || \
                         ((x) == PIS_GP16C2T0_UPDATA)     || \
                         ((x) == PIS_GP16C2T0_TRIG)       || \
                         ((x) == PIS_GP16C2T0_CH1_INPUT)  || \
                         ((x) == PIS_GP16C2T0_CH1_OUTPUT) || \
                         ((x) == PIS_GP16C2T0_CH2_INPUT)  || \
                         ((x) == PIS_GP16C2T0_CH2_OUTPUT) || \
                         ((x) == PIS_GP16C2T1_UPDATA)     || \
                         ((x) == PIS_GP16C2T1_TRIG)       || \
                         ((x) == PIS_GP16C2T1_CH1_INPUT)  || \
                         ((x) == PIS_GP16C2T1_CH1_OUTPUT) || \
                         ((x) == PIS_GP16C2T1_CH2_INPUT)  || \
                         ((x) == PIS_GP16C2T1_CH2_OUTPUT) || \
                         ((x) == PIS_RTC_CLOCK)           || \
                         ((x) == PIS_RTC_ALARM)           || \
                         ((x) == PIS_LP16T0_SYN_UPDATA)   || \
                         ((x) == PIS_LP16T0_ASY_UPDATA)   || \
                         ((x) == PIS_LPUART0_ASY_RECV)    || \
                         ((x) == PIS_LPUART0_ASY_SEND)    || \
                         ((x) == PIS_LPUART0_SYN_RECV)    || \
                         ((x) == PIS_LPUART0_SYN_SEND)    || \
                         ((x) == PIS_DMA_CH0)             || \
                         ((x) == PIS_DMA_CH1)             || \
                         ((x) == PIS_DMA_CH2)             || \
                         ((x) == PIS_DMA_CH3)             || \
                         ((x) == PIS_DMA_CH4)             || \
                         ((x) == PIS_DMA_CH5)             || \
                         ((x) == PIS_DMA_CH_ALL)          || \
                         ((x) == PIS_BS16T1_UPDATA)       || \
                         ((x) == PIS_GP16C4T0_UPDATA)     || \
                         ((x) == PIS_GP16C4T0_TRIG)       || \
                         ((x) == PIS_GP16C4T0_CH1_INPUT)  || \
                         ((x) == PIS_GP16C4T0_CH1_OUTPUT) || \
                         ((x) == PIS_GP16C4T0_CH2_INPUT)  || \
                         ((x) == PIS_GP16C4T0_CH2_OUTPUT) || \
                         ((x) == PIS_GP16C4T0_CH3_INPUT)  || \
                         ((x) == PIS_GP16C4T0_CH3_OUTPUT) || \
                         ((x) == PIS_GP16C4T0_CH4_INPUT)  || \
                         ((x) == PIS_GP16C4T0_CH4_OUTPUT))
#define IS_PIS_TRIG(x)  (((x) == PIS_CH0_SPI1_CLK)      || \
                         ((x) == PIS_CH0_L16T0_EXT0)     || \
                         ((x) == PIS_CH1_UART0_RX_IRDA)  || \
                         ((x) == PIS_CH1_GP16C4T0_CH1IN) || \
                         ((x) == PIS_CH1_L16T0_EXT1)     || \
                         ((x) == PIS_CH2_GP16C4T0_CH2IN) || \
                         ((x) == PIS_CH2_L16T0_EXT2)     || \
                         ((x) == PIS_CH2_TKS_TRIG)       || \
                         ((x) == PIS_CH3_GP16C4T0_CH3IN) || \
                         ((x) == PIS_CH3_L16T0_EXT3)     || \
                         ((x) == PIS_CH4_GP16C4T0_ITR0)  || \
                         ((x) == PIS_CH4_GP16C4T0_CH4IN) || \
                         ((x) == PIS_CH4_GP16C2T0_ITR0)  || \
                         ((x) == PIS_CH4_GP16C2T1_ITR0)  || \
                         ((x) == PIS_CH4_L16T0_EXT4)     || \
                         ((x) == PIS_CH5_USART0_RX)      || \
                         ((x) == PIS_CH5_SPI0_RX)        || \
                         ((x) == PIS_CH5_GP16C4T0_ITR1)  || \
                         ((x) == PIS_CH5_GP16C2T0_ITR1)  || \
                         ((x) == PIS_CH5_GP16C2T0_CH1IN) || \
                         ((x) == PIS_CH5_GP16C2T1_ITR1)  || \
                         ((x) == PIS_CH5_GP16C2T1_CH1IN) || \
                         ((x) == PIS_CH5_L16T0_EXT5)     || \
                         ((x) == PIS_CH6_USART1_RX)      || \
                         ((x) == PIS_CH6_SPI0_CLK)       || \
                         ((x) == PIS_CH6_GP16C4T0_ITR2)  || \
                         ((x) == PIS_CH6_GP16C2T0_ITR2)  || \
                         ((x) == PIS_CH6_GP16C2T0_CH2IN) || \
                         ((x) == PIS_CH6_GP16C2T1_ITR2)  || \
                         ((x) == PIS_CH6_GP16C2T1_CH2IN) || \
                         ((x) == PIS_CH6_ADC0_NORMAL)    || \
                         ((x) == PIS_CH6_L16T0_EXT6)     || \
                         ((x) == PIS_CH7_SPI1_RX)        || \
                         ((x) == PIS_CH7_GP16C4T0_ITR3)  || \
                         ((x) == PIS_CH7_GP16C2T0_ITR3)  || \
                         ((x) == PIS_CH7_GP16C2T1_ITR3)  || \
                         ((x) == PIS_CH7_ADC0_INSERT)    || \
                         ((x) == PIS_CH7_L16T0_EXT7)     || \
                         ((x) == PIS_CH7_LPUART_RX)      || \
                         ((x) == PIS_CH7_DMA))
#define IS_PIS_CLOCK(x)     (((x) == PIS_CLK_PCLK1)  || \
                             ((x) == PIS_CLK_PCLK2)  || \
                             ((x) == PIS_CLK_SYS) || \
                             ((x) == PIS_CLK_LP))
#define IS_PIS_EDGE(x)      (((x) == PIS_EDGE_NONE) || \
                             ((x) == PIS_EDGE_UP)   || \
                             ((x) == PIS_EDGE_DOWN) || \
                             ((x) == PIS_EDGE_UP_DOWN))
#define IS_PIS_OUTPUT(x)    (((x) == PIS_OUT_LEVEL) || \
                             ((x) == PIS_OUT_PULSE))
#define IS_PIS_OUPUT_CH(x)  (((x) == PIS_OUT_CH_0) || \
                             ((x) == PIS_OUT_CH_1) || \
                             ((x) == PIS_OUT_CH_2) || \
                             ((x) == PIS_OUT_CH_3))
#define IS_PIS_MODU_TARGET(x)   (((x) == PIS_UART0_TX)  || \
                                 ((x) == PIS_USART0_TX) || \
                                 ((x) == PIS_USART1_TX) || \
                                 ((x) == PIS_LPUART0_TX))
#define IS_PIS_MODU_LEVEL(x)    (((x) == PIS_LOW_LEVEL) || \
                                 ((x) == PIS_HIGH_LEVEL))
#define IS_PIS_MODU_SRC(x)  (((x) == PIS_SRC_NONE)     || \
                             ((x) == PIS_SRC_GP16C2T0) || \
                             ((x) == PIS_SRC_GP16C2T1) || \
                             ((x) == PIS_SRC_GP16C4T0) || \
                             ((x) == PIS_SRC_LP16T0)   || \
                             ((x) == PIS_SRC_BUZ))
#define IS_PIS_MODU_CHANNEL(x)  (((x) == PIS_TIMER_CH1) || \
                                 ((x) == PIS_TIMER_CH2) || \
                                 ((x) == PIS_TIMER_CH3) || \
                                 ((x) == PIS_TIMER_CH4))
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions
  * @{
  */

/** @addtogroup PIS_Public_Functions_Group1
  * @{
  */
ald_status_t ald_pis_create(pis_handle_t *hperh);
ald_status_t ald_pis_destroy(pis_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions_Group2
  * @{
  */
ald_status_t ald_pis_output_start(pis_handle_t *hperh, pis_out_ch_t ch);
ald_status_t ald_pis_output_stop(pis_handle_t *hperh, pis_out_ch_t ch);
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions_Group3
  * @{
  */
pis_state_t ald_pis_get_state(pis_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions_Group4
  * @{
  */
ald_status_t ald_pis_modu_config(pis_handle_t *hperh, pis_modulate_config_t *config);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __ALD_PIS_H__ */
