/**
  *********************************************************************************
  *
  * @file    ald_tks.c
  * @brief   TKS module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of the Touch Key Sense (TKS) peripheral:
  *           + Initialization and Configuration functions
  *           + Configure functions
  *           + Peripheral Control functions
  *
  * @version V1.0
  * @date    19 Mar 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  ******************************************************************************
  */

#include "ald_tks.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup TKS TKS
  * @brief TKS module driver
  * @{
  */
#ifdef ALD_TKS
/** @defgroup TKS_Public_Functions TKS Public Functions
  * @{
  */

/** @defgroup TKS_Public_Functions_Group1 Initialization functions
  * @brief    Initialization functions
  * @{
  */

/**
  * @brief  Reset TKS peripheral
  * @param  hperh: Pointer to a tks_handle_t structure that contains
  *         the configuration information for the TKS module.
  * @retval None
  */
void ald_tks_reset(tks_handle_t *hperh)
{
    assert_param(IS_TKS(hperh->perh));

    WRITE_REG(hperh->perh->CON0, 0x0);
    WRITE_REG(hperh->perh->CON1, 0x0);
    WRITE_REG(hperh->perh->CHEN, 0x0);
    WRITE_REG(hperh->perh->GAIN, 0x0);
    WRITE_REG(hperh->perh->IE, 0x0);
    WRITE_REG(hperh->perh->IFC, 0xFF);
    WRITE_REG(hperh->perh->SFJTR, 0x0);

    return;
}

/**
  * @brief  Initializes the TKS according to the specified
  *         parameters in the tks_handle_t.
  * @param  hperh: Pointer to a tks_handle_t structure that contains
  *         the configuration information for the TKS module.
  * @retval None
  */
void ald_tks_init(tks_handle_t *hperh)
{
    uint32_t tmp;

    assert_param(IS_TKS(hperh->perh));
    assert_param(IS_FUNC_STATE(hperh->init.n_scan_key_op));
    assert_param(IS_FUNC_STATE(hperh->init.poll_scan));
    assert_param(IS_FUNC_STATE(hperh->init.key_cap_compen));
    assert_param(IS_TKS_RESULT_TYPE(hperh->init.ret_type));
    assert_param(IS_TKS_DATA_MODE(hperh->init.data_mode));
    assert_param(IS_FUNC_STATE(hperh->init.acc_average));
    assert_param(IS_TKS_CHARGE_DUTY(hperh->init.charge_duty));
    assert_param(IS_TKS_SAMPLING(hperh->init.sampling));
    assert_param(IS_FUNC_STATE(hperh->init.dc));
    assert_param(IS_TKS_DC_MODE(hperh->init.dc_mode));
    assert_param(IS_FUNC_STATE(hperh->init.enhance_time));
    assert_param(IS_FUNC_STATE(hperh->init.cp_cap_compen));
    assert_param(IS_FUNC_STATE(hperh->init.ext_trigger));
    assert_param(IS_TKS_PRE_DIV(hperh->init.pre_div));
    assert_param(IS_TKS_DIV(hperh->init.div));
    assert_param(IS_FUNC_STATE(hperh->init.hyst_en));
    assert_param(IS_TKS_CHARGE_VOL(hperh->init.charge_vol));
    assert_param(IS_TKS_FILT_TIME(hperh->init.flt_time));
    assert_param(IS_TKS_FILT_MODE(hperh->init.flt_mode));
    assert_param(IS_TKS_DISCHG_TIME(hperh->init.disc_time));
    assert_param(IS_TKS_FUNCTION(hperh->init.tks_func));
    assert_param(IS_FUNC_STATE(hperh->init.high_power));
    assert_param(IS_TKS_KEY_VOL(hperh->init.key_vol));
    assert_param(IS_FUNC_STATE(hperh->init.count_en));
    assert_param(IS_FUNC_STATE(hperh->init.count_bkp_en));
    assert_param(IS_TKS_BACKUP_MODE(hperh->init.bkp_mode));
    assert_param(IS_TKS_BKP_HIGH_TIME(hperh->init.bkp_h));
    assert_param(IS_TKS_COMP_HYS(hperh->init.comp_hys));

    tmp = 0x0;
    MODIFY_REG(tmp, TKS_CON0_NCHOE_MSK, hperh->init.n_scan_key_op << TKS_CON0_NCHOE_POS);
    MODIFY_REG(tmp, TKS_CON0_CHSE_MSK, hperh->init.poll_scan << TKS_CON0_CHSE_POS);
    MODIFY_REG(tmp, TKS_CON0_CAPCE_MSK, hperh->init.key_cap_compen << TKS_CON0_CAPCE_POS);
    MODIFY_REG(tmp, TKS_CON0_RESSEL_MSK, hperh->init.ret_type << TKS_CON0_RESSEL_POS);
    MODIFY_REG(tmp, TKS_CON0_RESMD_MSK, hperh->init.data_mode << TKS_CON0_RESMD_POS);
    MODIFY_REG(tmp, TKS_CON0_AVEEN_MSK, hperh->init.acc_average << TKS_CON0_AVEEN_POS);
    MODIFY_REG(tmp, TKS_CON0_DUTY_MSK, hperh->init.charge_duty << TKS_CON0_DUTY_POS);
    MODIFY_REG(tmp, TKS_CON0_SMPSEL_MSK, hperh->init.sampling << TKS_CON0_SMPSEL_POSS);
    MODIFY_REG(tmp, TKS_CON0_DCEN_MSK, hperh->init.dc << TKS_CON0_DCEN_POS);
    MODIFY_REG(tmp, TKS_CON0_DCMOD_MSK, hperh->init.dc_mode << TKS_CON0_DCMOD_POS);
    MODIFY_REG(tmp, TKS_CON0_CNTEN_MSK, hperh->init.enhance_time << TKS_CON0_CNTEN_POS);
    MODIFY_REG(tmp, TKS_CON0_CCSEL_MSK, hperh->init.cp_cap_compen << TKS_CON0_CCSEL_POS);
    MODIFY_REG(tmp, TKS_CON0_EXTEN_MSK, hperh->init.ext_trigger << TKS_CON0_EXTEN_POS);
    MODIFY_REG(tmp, TKS_CON0_CMUDIV_MSK, hperh->init.pre_div << TKS_CON0_CMUDIV_POS);
    MODIFY_REG(tmp, TKS_CON0_TKDIV_MSK, hperh->init.div << TKS_CON0_TKDIV_POSS);
    hperh->perh->CON0 = tmp;

    tmp = 0x0;
    MODIFY_REG(tmp, TKS_CON1_HYSEN_MSK, hperh->init.hyst_en << TKS_CON1_HYSEN_POS);
    MODIFY_REG(tmp, TKS_CON1_CHGVS_MSK, hperh->init.charge_vol << TKS_CON1_CHGVS_POSS);
    MODIFY_REG(tmp, TKS_CON1_FLTSEL_MSK, hperh->init.flt_time << TKS_CON1_FLTSEL_POSS);
    MODIFY_REG(tmp, TKS_CON1_FLTMOD_MSK, hperh->init.flt_mode << TKS_CON1_FLTMOD_POS);
    MODIFY_REG(tmp, TKS_CON1_CxDCT_MSK, hperh->init.disc_time << TKS_CON1_CxDCT_POSS);
    MODIFY_REG(tmp, TKS_CON1_SCHSEL_MSK, hperh->init.tks_func << TKS_CON1_SCHSEL_POS);
    MODIFY_REG(tmp, TKS_CON1_HPEN_MSK, hperh->init.high_power << TKS_CON1_HPEN_POS);
    MODIFY_REG(tmp, TKS_CON1_PWSEL_MSK, hperh->init.key_vol << TKS_CON1_PWSEL_POS);
    MODIFY_REG(tmp, TKS_CON1_ACCEN_MSK, hperh->init.count_en << TKS_CON1_ACCEN_POS);
    MODIFY_REG(tmp, TKS_CON1_BACEN_MSK, hperh->init.count_bkp_en << TKS_CON1_BACEN_POS);
    MODIFY_REG(tmp, TKS_CON1_BACSEL_MSK, hperh->init.bkp_mode << TKS_CON1_BACSEL_POS);
    MODIFY_REG(tmp, TKS_CON1_BACT_MSK, hperh->init.bkp_h << TKS_CON1_BACT_POSS);
    MODIFY_REG(tmp, TKS_CON1_HYSSEL_MSK, hperh->init.comp_hys << TKS_CON1_HYSSEL_POSS);
    hperh->perh->CON1 = tmp;

    SET_BIT(hperh->perh->CON0, TKS_CON0_EN_MSK);
    return;
}
/**
  * @}
  */

/** @defgroup TKS_Public_Functions_Group2 Configure functions
  * @brief    TKS configure functions
  * @{
  */
/**
  * @brief  Enable/Disable the channels.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  enable: Enable/Disable.
  * @retval None
  */
void ald_tks_channel_control(tks_handle_t *hperh, uint32_t enable)
{
    assert_param(IS_TKS(hperh->perh));

    hperh->perh->CHEN = enable;
    return;
}

/**
  * @brief  Get channel status.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @retval Status of the channel.
  */
uint32_t ald_tks_channel_select_get(tks_handle_t *hperh)
{
    assert_param(IS_TKS(hperh->perh));

    return hperh->perh->CHSEL;
}

/**
  * @brief  Set gain.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  gain: Value of the gain.
  * @retval None
  */
void ald_tks_gain_set(tks_handle_t *hperh, uint32_t gain)
{
    assert_param(IS_TKS(hperh->perh));
    assert_param(gain < (1u << 28));

    hperh->perh->GAIN = gain;
    return;
}

/**
  * @brief  Configure software frequency jitter.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  config: Pointer to parameters.
  * @retval None
  */
void ald_tks_sfjt_config(tks_handle_t *hperh, tks_sfjt_config_t *config)
{
    uint32_t tmp = 0;

    assert_param(IS_TKS(hperh->perh));
    assert_param(IS_TKS_SFJT_CLK_CNT(config->sft_h));
    assert_param(IS_TKS_SFJT_CLK_CNT(config->sft_l));
    assert_param(IS_FUNC_STATE(config->en));
    assert_param(IS_FUNC_STATE(config->en_micro));
    assert_param(IS_TKS_MICRO_SFJT(config->jtsel));

    MODIFY_REG(tmp, TKS_SFJTR_SFT1_MSK, config->sft_h << TKS_SFJTR_SFT1_POSS);
    MODIFY_REG(tmp, TKS_SFJTR_SFT2_MSK, config->sft_l << TKS_SFJTR_SFT2_POSS);
    MODIFY_REG(tmp, TKS_SFJTR_SFEN_MSK, config->en << TKS_SFJTR_SFEN_POS);
    MODIFY_REG(tmp, TKS_SFJTR_JTEN_MSK, config->en_micro << TKS_SFJTR_JTEN_POS);
    MODIFY_REG(tmp, TKS_SFJTR_JTSEL_MSK, config->jtsel << TKS_SFJTR_JTSEL_POSS);
    hperh->perh->SFJTR = tmp;

    return;
}
/**
  * @}
  */

/** @defgroup TKS_Public_Functions_Group3 Peripheral Control functions
  *  @brief   TKS control functions
  * @{
  */
/**
  * @brief  Get merge result.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  value: Merge result.
  * @param  acccnt: Jitter count.
  * @retval None
  */
void ald_tks_merge_result_get(tks_handle_t *hperh, uint16_t *value, uint16_t *acccnt)
{
    assert_param(IS_TKS(hperh->perh));

    *value  = (uint16_t)READ_BITS(hperh->perh->SCANRES, TKS_SCANRES_VAL_MSK, TKS_SCANRES_VAL_POSS);
    *acccnt = (uint16_t)READ_BITS(hperh->perh->SCANRES, TKS_SCANRES_ACCCNT_MSK, TKS_SCANRES_ACCCNT_POSS);

    return;
}

/**
  * @brief  Get result.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  ch_idx: Index of channel.
  * @retval Result.
  */
uint16_t ald_tks_result_get(tks_handle_t *hperh, uint8_t ch_idx)
{
    assert_param(IS_TKS(hperh->perh));
    assert_param(ch_idx < 32);

    return (uint16_t)hperh->perh->CHRES[ch_idx];
}

/**
  * @brief  This function handles TKS interrupt request.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @retval None
  */
void ald_tks_irq_handler(tks_handle_t *hperh)
{
    uint32_t tmp0, tmp1;

    assert_param(IS_TKS(hperh->perh));

    tmp0 = hperh->perh->IE;
    tmp1 = hperh->perh->IF;

    if ((tmp0 & TKS_IT_RESOV) && (tmp1 & TKS_IF_RESOV))
    {
        hperh->perh->IFC = TKS_IF_RESOV;

        if (hperh->cbk)
            hperh->cbk(hperh, TKS_ALD_EVENT_RESOV);
    }

    if ((tmp0 & TKS_IT_SCANOV) && (tmp1 & TKS_IF_SCANOV))
    {
        hperh->perh->IFC = TKS_IF_SCANOV;

        if (hperh->cbk)
            hperh->cbk(hperh, TKS_ALD_EVENT_SCANOV);
    }

    if ((tmp0 & TKS_IT_ERR) && (tmp1 & TKS_IF_ERR))
    {
        hperh->perh->IFC = TKS_IF_ERR;

        if (hperh->cbk)
            hperh->cbk(hperh, TKS_ALD_EVENT_ERROR);
    }

    if ((tmp0 & TKS_IT_BREAK) && (tmp1 & TKS_IF_BREAK))
    {
        hperh->perh->IFC = TKS_IF_BREAK;

        if (hperh->cbk)
            hperh->cbk(hperh, TKS_ALD_EVENT_BREAK);
    }

    if ((tmp0 & TKS_IT_DONE) && (tmp1 & TKS_IF_DONE))
    {
        hperh->perh->IFC = TKS_IF_DONE;

        if (hperh->cbk)
            hperh->cbk(hperh, TKS_ALD_EVENT_DONE);
    }

    return;
}

/**
  * @brief  Enable/disable the specified TKS interrupts.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  it: Specifies the TKS interrupt sources to be enabled or disabled.
  *         This parameter can be one of the @ref tks_it_t.
  * @param  state: New state of the specified TKS interrupts.
  *         This parameter can be:
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void ald_tks_interrupt_config(tks_handle_t *hperh, tks_it_t it, type_func_t state)
{
    assert_param(IS_TKS(hperh->perh));
    assert_param(IS_TKS_IT(it));
    assert_param(IS_FUNC_STATE(state));

    if (state == ENABLE)
        SET_BIT(hperh->perh->IE, it);
    else
        CLEAR_BIT(hperh->perh->IE, it);

    return;
}

/**
  * @brief  Get the status of TKS interrupt source.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  it: Specifies the TKS interrupt source.
  *         This parameter can be one of the @ref tks_it_t.
  * @retval Status:
  *           - 0: RESET
  *           - 1: SET
  */
it_status_t ald_tks_get_it_status(tks_handle_t *hperh, tks_it_t it)
{
    assert_param(IS_TKS(hperh->perh));
    assert_param(IS_TKS_IT(it));

    if (hperh->perh->IE & it)
        return SET;

    return RESET;
}

/**
  * @brief  Get the status of TKS interrupt flag.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  flag: Specifies the TKS interrupt flag.
  *         This parameter can be one of the @ref tks_flag_t.
  * @retval Status:
  *           - 0: RESET
  *           - 1: SET
  */
flag_status_t ald_tks_get_flag_status(tks_handle_t *hperh, tks_flag_t flag)
{
    assert_param(IS_TKS(hperh->perh));
    assert_param(IS_TKS_IF(flag));

    if (hperh->perh->IF & flag)
        return SET;

    return RESET;
}

/**
  * @brief  Clear the TKS interrupt flag.
  * @param  hperh: Pointer to a tks_handle_t structure.
  * @param  flag: Specifies the TKS interrupt flag.
  *         This parameter can be one of the @ref tks_flag_t.
  * @retval None
  */
void ald_tks_clear_flag_status(tks_handle_t *hperh, tks_flag_t flag)
{
    assert_param(IS_TKS(hperh->perh));
    assert_param(IS_TKS_IF(flag));

    hperh->perh->IFC = flag;
    return;
}
/**
  * @}
  */
/**
  * @}
  */
#endif /* ALD_TKS */
/**
  * @}
  */
/**
  * @}
  */
