/**
  *********************************************************************************
  *
  * @file    ald_tks.h
  * @brief   Header file of TKS module library.
  *
  * @version V1.0
  * @date    19 Mar 2020
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  *********************************************************************************
  */

#ifndef __ALD_TKS_H__
#define __ALD_TKS_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup TKS
  * @{
  */

/**
  * @defgroup TKS_Public_Macros TKS Public Macros
  * @{
  */
#define TKS_ALD_EVENT_RESOV 0x01
#define TKS_ALD_EVENT_SCANOV    0x02
#define TKS_ALD_EVENT_ERROR 0x04
#define TKS_ALD_EVENT_BREAK 0x08
#define TKS_ALD_EVENT_DONE  0x10

#define TKS_CHANNEL_0       (1u << 0)
#define TKS_CHANNEL_1       (1u << 1)
#define TKS_CHANNEL_2       (1u << 2)
#define TKS_CHANNEL_3       (1u << 3)
#define TKS_CHANNEL_4       (1u << 4)
#define TKS_CHANNEL_5       (1u << 5)
#define TKS_CHANNEL_6       (1u << 6)
#define TKS_CHANNEL_7       (1u << 7)
#define TKS_CHANNEL_8       (1u << 8)
#define TKS_CHANNEL_9       (1u << 9)
#define TKS_CHANNEL_10      (1u << 10)
#define TKS_CHANNEL_11      (1u << 11)
#define TKS_CHANNEL_12      (1u << 12)
#define TKS_CHANNEL_13      (1u << 13)
#define TKS_CHANNEL_14      (1u << 14)
#define TKS_CHANNEL_15      (1u << 15)
#define TKS_CHANNEL_16      (1u << 16)
#define TKS_CHANNEL_17      (1u << 17)
#define TKS_CHANNEL_18      (1u << 18)
#define TKS_CHANNEL_19      (1u << 19)
#define TKS_CHANNEL_20      (1u << 20)
#define TKS_CHANNEL_21      (1u << 21)
#define TKS_CHANNEL_22      (1u << 22)
#define TKS_CHANNEL_23      (1u << 23)
#define TKS_CHANNEL_24      (1u << 24)
#define TKS_CHANNEL_25      (1u << 25)
#define TKS_CHANNEL_26      (1u << 26)
#define TKS_CHANNEL_27      (1u << 27)
#define TKS_CHANNEL_28      (1u << 28)
#define TKS_CHANNEL_29      (1u << 29)
#define TKS_CHANNEL_30      (1u << 30)
#define TKS_CHANNEL_31      (1u << 31)

#define TKS_ENABLE()            (SET_BIT(TKS->CON0, TKS_CON0_EN_MSK))
#define TKS_DISABLE()           (CLEAR_BIT(TKS->CON0, TKS_CON0_EN_MSK))
#define TKS_SCAN_START()        (SET_BIT(TKS->CON0, TKS_CON0_START_MSK))
#define TKS_SCAN_STOP()         (CLEAR_BIT(TKS->CON0, TKS_CON0_START_MSK))
#define TKS_SCAN_STATUS_GET()       (READ_BITS(TKS->CON0, TKS_CON0_BUSY_MSK, TKS_CON0_BUSY_POS))
#define TKS_N_SCAN_KEY_O_ENABLE()   (SET_BIT(TKS->CON0, TKS_CON0_NCHOE_MSK))
#define TKS_N_SCAN_KEY_O_DISABLE()  (CLEAR_BIT(TKS->CON0, TKS_CON0_NCHOE_MSK))
#define TKS_EXT_TRIGGER_ENABLE()    (SET_BIT(TKS->CON0, TKS_CON0_EXTEN_MSK))
#define TKS_EXT_TRIGGER_DISABLE()   (CLEAR_BIT(TKS->CON0, TKS_CON0_EXTEN_MSK))
#define TKS_SCAN_PAUSE()        (SET_BIT(TKS->CON0, TKS_CON0_PAUSE_MSK))
#define TKS_SCAN_RESUME()       (CLEAR_BIT(TKS->CON0, TKS_CON0_PAUSE_MSK))
/**
  * @}
  */

/** @defgroup TKS_Public_Types TKS Public Types
  * @{
  */
/**
  * @brief Charge/Discharge duty
  */
typedef enum
{
    TKS_CHARGE_DUTY_1_4 = 0x0,  /**< Charge/Discharge = 1/3 */
    TKS_CHARGE_DUTY_1_2 = 0x1,  /**< Charge/Discharge = 1/1 */
} tks_charge_duty_t;

/**
  * @brief Sampling count
  */
typedef enum
{
    TKS_SAMPLING_1  = 0x0,  /**< 1 */
    TKS_SAMPLING_4  = 0x1,  /**< 4 */
    TKS_SAMPLING_8  = 0x2,  /**< 8 */
    TKS_SAMPLING_16 = 0x3,  /**< 16 */
} tks_sampling_t;

/**
  * @brief Result type
  */
typedef enum
{
    TKS_RESULT_TYPE_0 = 0x0,    /**< Gain/Count */
    TKS_RESULT_TYPE_1 = 0x1,    /**< Count */
} tks_result_type_t;

/**
  * @brief Data mode
  */
typedef enum
{
    TKS_DATA_MODE_CNT  = 0x0,   /**< Count */
    TKS_DATA_MODE_TIME = 0x1,   /**< Time */
} tks_data_mode_t;

/**
  * @brief Des-Coupling mode
  */
typedef enum
{
    TKS_DC_MODE_SYNC = 0x0, /**< Synchronization */
    TKS_DC_MODE_INV  = 0x1, /**< Inversion */
} tks_dc_mode_t;

/**
  * @brief Pre-dividing
  */
typedef enum
{
    TKS_PRE_DIV_1 = 0x0,    /**< 1/1 */
    TKS_PRE_DIV_3 = 0x1,    /**< 1/3 */
} tks_pre_div_t;

/**
  * @brief dividing
  */
typedef enum
{
    TKS_DIV_1  = 0x0,   /**< 1/1 */
    TKS_DIV_2  = 0x1,   /**< 1/2 */
    TKS_DIV_4  = 0x2,   /**< 1/4 */
    TKS_DIV_8  = 0x3,   /**< 1/8 */
    TKS_DIV_16 = 0x4,   /**< 1/16 */
    TKS_DIV_32 = 0x5,   /**< 1/32 */
} tks_div_t;

/**
  * @brief Charge voltage
  */
typedef enum
{
    TKS_CHARGE_VOL_1_4  = 0x0,  /**< 1.4V */
    TKS_CHARGE_VOL_1_5  = 0x1,  /**< 1.5V */
    TKS_CHARGE_VOL_1_7  = 0x2,  /**< 1.7V */
    TKS_CHARGE_VOL_1_9  = 0x3,  /**< 1.9V */
    TKS_CHARGE_VOL_2_1  = 0x4,  /**< 2.1V */
    TKS_CHARGE_VOL_2_3  = 0x5,  /**< 2.3V */
    TKS_CHARGE_VOL_2_5  = 0x6,  /**< 2.5V */
    TKS_CHARGE_VOL_VDD  = 0x7,  /**< VDD */
} tks_charge_vol_t;

/**
  * @brief Filtering time
  */
typedef enum
{
    TKS_FILTERING_NONE = 0x0,   /**< NONE */
    TKS_FILTERING_2    = 0x1,   /**< 2 Clock */
    TKS_FILTERING_3    = 0x2,   /**< 3 Clock */
    TKS_FILTERING_4    = 0x3,   /**< 4 Clock */
    TKS_FILTERING_5    = 0x4,   /**< 5 Clock */
    TKS_FILTERING_6    = 0x5,   /**< 6 Clock */
    TKS_FILTERING_7    = 0x6,   /**< 7 Clock */
    TKS_FILTERING_8    = 0x7,   /**< 8 Clock */
} tks_filtering_time_t;

/**
  * @brief Filtering mode
  */
typedef enum
{
    TKS_FILTERING_CON = 0x0,    /**< Continuous */
    TKS_FILTERING_ACC = 0x1,    /**< Accumulate */
} tks_filtering_mode_t;

/**
  * @brief Discharge time
  */
typedef enum
{
    TKS_DISCHARGE_32   = 0x0,   /**< 32 Clock */
    TKS_DISCHARGE_64   = 0x1,   /**< 64 Clock */
    TKS_DISCHARGE_128  = 0x2,   /**< 128 Clock */
    TKS_DISCHARGE_256  = 0x3,   /**< 256 Clock */
    TKS_DISCHARGE_512  = 0x4,   /**< 512 Clock */
    TKS_DISCHARGE_1024 = 0x5,   /**< 1024 Clock */
    TKS_DISCHARGE_2048 = 0x6,   /**< 2048 Clock */
    TKS_DISCHARGE_4096 = 0x7,   /**< 4096 Clock */
} tks_discharge_time_t;

/**
  * @brief Function select
  */
typedef enum
{
    TKS_FUNC_0 = 0x0,   /**< Cp->Cx */
    TKS_FUNC_1 = 0x1,   /**< Ci->Cp */
} tks_function_t;

/**
  * @brief Key voltage
  */
typedef enum
{
    TKS_KEY_VOL_CP = 0x0,   /**< VDD */
    TKS_KEY_VOL_VDD  = 0x1, /**< Cp */
} tks_key_vol_t;

/**
  * @brief Backup mode
  */
typedef enum
{
    TKS_BKP_MODE_RISE = 0x0,    /**< Edge of rising */
    TKS_BKP_MODE_HIGH = 0x1,    /**< High level */
} tks_backup_mode_t;

/**
  * @brief Backup high time
  */
typedef enum
{
    TKS_BKP_HIGH_0  = 0x0,  /**< 0 Clock */
    TKS_BKP_HIGH_1  = 0x1,  /**< 1 Clock */
    TKS_BKP_HIGH_2  = 0x2,  /**< 2 Clock */
    TKS_BKP_HIGH_3  = 0x3,  /**< 3 Clock */
    TKS_BKP_HIGH_4  = 0x4,  /**< 4 Clock */
    TKS_BKP_HIGH_5  = 0x5,  /**< 5 Clock */
    TKS_BKP_HIGH_6  = 0x6,  /**< 6 Clock */
    TKS_BKP_HIGH_7  = 0x7,  /**< 7 Clock */
    TKS_BKP_HIGH_8  = 0x8,  /**< 8 Clock */
    TKS_BKP_HIGH_9  = 0x9,  /**< 9 Clock */
    TKS_BKP_HIGH_10 = 0xA,  /**< 10 Clock */
    TKS_BKP_HIGH_11 = 0xB,  /**< 11 Clock */
    TKS_BKP_HIGH_12 = 0xC,  /**< 12 Clock */
    TKS_BKP_HIGH_13 = 0xD,  /**< 13 Clock */
    TKS_BKP_HIGH_14 = 0xE,  /**< 14 Clock */
    TKS_BKP_HIGH_15 = 0xF,  /**< 15 Clock */
} tks_backup_high_time_t;

/**
  * @brief Comparator hysteresis
  */
typedef enum
{
    TKS_COMP_HYS_50  = 0x0, /**< 50mV */
    TKS_COMP_HYS_60  = 0x1, /**< 60mV */
    TKS_COMP_HYS_80  = 0x2, /**< 80mV */
    TKS_COMP_HYS_120 = 0x3, /**< 120mV */
} tks_comp_hys_t;

/**
  * @brief TKS init structure definition
  */
typedef struct
{
    type_func_t n_scan_key_op;  /**< Enable/Disable unscanned key output */
    type_func_t poll_scan;      /**< Enable/Disable poll scan mode */
    type_func_t key_cap_compen; /**< Enable/Disable key capacitance compensation */
    tks_result_type_t ret_type; /**< Type of result data */
    tks_data_mode_t data_mode;  /**< Mode of result data */
    type_func_t acc_average;    /**< Average of cumulative charging time */
    tks_charge_duty_t charge_duty;  /**< Duty of charge/discharge */
    tks_sampling_t sampling;    /**< Number of samples */
    type_func_t dc;         /**< Enable/Disable of des-Coupling */
    tks_dc_mode_t dc_mode;      /**< Mode of des-Coupling */
    type_func_t enhance_time;   /**< Enable/Disable enhanced timing function */
    type_func_t cp_cap_compen;  /**< Enable/Disable Cp capacitance compensation */
    type_func_t ext_trigger;    /**< Enable/Disable external trigger */
    tks_pre_div_t pre_div;      /**< Prescaler clock */
    tks_div_t div;          /**< Division clock */
    type_func_t hyst_en;        /**< Enable/Disable comparator hysteresis */
    tks_charge_vol_t charge_vol;    /**< Select of charge voltage */
    tks_filtering_time_t flt_time;  /**< Select of filtering time */
    tks_filtering_mode_t flt_mode;  /**< Mode of filtering time */
    tks_discharge_time_t disc_time; /**< Select of discharge time */
    tks_function_t tks_func;    /**< Function of TKS */
    type_func_t high_power;     /**< Enable/Disable of high power */
    tks_key_vol_t key_vol;      /**< Select of key voltage */
    type_func_t count_en;       /**< Enable/Disable of comparator count */
    type_func_t count_bkp_en;   /**< Enable/Disable of comparator count backup */
    tks_backup_mode_t bkp_mode; /**< Mode of comparator count backup */
    tks_backup_high_time_t bkp_h;   /**< Comparator count backup high level time */
    tks_comp_hys_t comp_hys;    /**< Select of comparator hysteresis */
} tks_init_t;

/**
  * @brief  TKS handle structure definition
  */
typedef struct tks_handle_s
{
    TKS_TypeDef *perh;  /**< TKS registers base address */
    tks_init_t init;    /**< TKS initializate parameters */
    void (*cbk)(struct tks_handle_s *arg, uint32_t code);   /**< Callback */
} tks_handle_t;

/**
  * @brief Software frequency jitter micro
  */
typedef enum
{
    TKS_SFJT_224 = 0x0, /**< -256--224 */
    TKS_SFJT_112 = 0x1, /**< -128--112 */
    TKS_SFJT_56  = 0x2, /**< -64--56 */
    TKS_SFJT_28  = 0x3, /**< -32--28 */
} tks_mirco_sfjt_t;

/**
  * @brief  TKS Software frequency jitter structure definition
  */
typedef struct tks_sfjt_config_s
{
    uint8_t sft_h;      /**< Numbers of high-level clocks */
    uint8_t sft_l;      /**< Numbers of low-level clocks */
    type_func_t en;     /**< Enable/Disable software frequency jitter */
    type_func_t en_micro;   /**< Enable/Disable micro-jitter */
    tks_mirco_sfjt_t jtsel; /**< Select the value of miro-jitter */
} tks_sfjt_config_t;

/**
  * @brief TKS interrupt types
  */
typedef enum
{
    TKS_IT_RESOV  = (1U << 0),  /**< Result overflow */
    TKS_IT_SCANOV = (1U << 1),  /**< Scan count overflow */
    TKS_IT_ERR    = (1U << 2),  /**< Scan error */
    TKS_IT_BREAK  = (1U << 3),  /**< Break */
    TKS_IT_DONE   = (1U << 4),  /**< Scan complete */
} tks_it_t;

/**
  * @brief TKS flags types
  */
typedef enum
{
    TKS_IF_RESOV  = (1U << 0),  /**< Result overflow */
    TKS_IF_SCANOV = (1U << 1),  /**< Scan count overflow */
    TKS_IF_ERR    = (1U << 2),  /**< Scan error */
    TKS_IF_BREAK  = (1U << 3),  /**< Break */
    TKS_IF_DONE   = (1U << 4),  /**< Scan complete */
} tks_flag_t;
/**
  * @}
  */

/** @defgroup TKS_Private_Macros   TKS Private Macros
  * @{
  */
#define IS_TKS(x)   ((x) == TKS)
#define IS_TKS_RESULT_TYPE(x)   (((x) == TKS_RESULT_TYPE_0) || ((x) == TKS_RESULT_TYPE_1))
#define IS_TKS_DATA_MODE(x) (((x) == TKS_DATA_MODE_CNT) || ((x) == TKS_DATA_MODE_TIME))
#define IS_TKS_CHARGE_DUTY(x)   (((x) == TKS_CHARGE_DUTY_1_4) || ((x) == TKS_CHARGE_DUTY_1_2))
#define IS_TKS_SAMPLING(x)  (((x) == TKS_SAMPLING_1) || \
                             ((x) == TKS_SAMPLING_4) || \
                             ((x) == TKS_SAMPLING_8) || \
                             ((x) == TKS_SAMPLING_16))
#define IS_TKS_DC_MODE(x)   (((x) == TKS_DC_MODE_SYNC) || ((x) == TKS_DC_MODE_INV))
#define IS_TKS_PRE_DIV(x)   (((x) == TKS_PRE_DIV_1) || ((x) == TKS_PRE_DIV_3))
#define IS_TKS_DIV(x)       (((x) == TKS_DIV_1)  || \
                             ((x) == TKS_DIV_2)  || \
                             ((x) == TKS_DIV_4)  || \
                             ((x) == TKS_DIV_8)  || \
                             ((x) == TKS_DIV_16) || \
                             ((x) == TKS_DIV_32))
#define IS_TKS_CHARGE_VOL(x)    (((x) == TKS_CHARGE_VOL_1_4) || \
                                 ((x) == TKS_CHARGE_VOL_1_5) || \
                                 ((x) == TKS_CHARGE_VOL_1_7) || \
                                 ((x) == TKS_CHARGE_VOL_1_9) || \
                                 ((x) == TKS_CHARGE_VOL_2_1) || \
                                 ((x) == TKS_CHARGE_VOL_2_3) || \
                                 ((x) == TKS_CHARGE_VOL_2_5) || \
                                 ((x) == TKS_CHARGE_VOL_VDD))
#define IS_TKS_FILT_TIME(x) (((x) == TKS_FILTERING_NONE) || \
                             ((x) == TKS_FILTERING_2)    || \
                             ((x) == TKS_FILTERING_3)    || \
                             ((x) == TKS_FILTERING_4)    || \
                             ((x) == TKS_FILTERING_5)    || \
                             ((x) == TKS_FILTERING_6)    || \
                             ((x) == TKS_FILTERING_7)    || \
                             ((x) == TKS_FILTERING_8))
#define IS_TKS_FILT_MODE(x) (((x) == TKS_FILTERING_CON) || ((x) == TKS_FILTERING_ACC))
#define IS_TKS_DISCHG_TIME(x)   (((x) == TKS_DISCHARGE_32)   || \
                                 ((x) == TKS_DISCHARGE_64)   || \
                                 ((x) == TKS_DISCHARGE_128)  || \
                                 ((x) == TKS_DISCHARGE_256)  || \
                                 ((x) == TKS_DISCHARGE_512)  || \
                                 ((x) == TKS_DISCHARGE_1024) || \
                                 ((x) == TKS_DISCHARGE_2048) || \
                                 ((x) == TKS_DISCHARGE_4096))
#define IS_TKS_FUNCTION(x)  (((x) == TKS_FUNC_0) || ((x) == TKS_FUNC_1))
#define IS_TKS_KEY_VOL(x)   (((x) == TKS_KEY_VOL_VDD) || ((x) == TKS_KEY_VOL_CP))
#define IS_TKS_BACKUP_MODE(x)   (((x) == TKS_BKP_MODE_RISE) || ((x) == TKS_BKP_MODE_HIGH))
#define IS_TKS_BKP_HIGH_TIME(x) (((x) == TKS_BKP_HIGH_0) || \
                                 ((x) == TKS_BKP_HIGH_1) || \
                                 ((x) == TKS_BKP_HIGH_2) || \
                                 ((x) == TKS_BKP_HIGH_3) || \
                                 ((x) == TKS_BKP_HIGH_4) || \
                                 ((x) == TKS_BKP_HIGH_5) || \
                                 ((x) == TKS_BKP_HIGH_6) || \
                                 ((x) == TKS_BKP_HIGH_7) || \
                                 ((x) == TKS_BKP_HIGH_8) || \
                                 ((x) == TKS_BKP_HIGH_9) || \
                                 ((x) == TKS_BKP_HIGH_10) || \
                                 ((x) == TKS_BKP_HIGH_11) || \
                                 ((x) == TKS_BKP_HIGH_12) || \
                                 ((x) == TKS_BKP_HIGH_13) || \
                                 ((x) == TKS_BKP_HIGH_14) || \
                                 ((x) == TKS_BKP_HIGH_15))
#define IS_TKS_COMP_HYS(x)  (((x) == TKS_COMP_HYS_50) || \
                             ((x) == TKS_COMP_HYS_60) || \
                             ((x) == TKS_COMP_HYS_80) || \
                             ((x) == TKS_COMP_HYS_120))
#define IS_TKS_MICRO_SFJT(x)    (((x) == TKS_SFJT_224) || \
                                 ((x) == TKS_SFJT_112) || \
                                 ((x) == TKS_SFJT_56)  || \
                                 ((x) == TKS_SFJT_28))
#define IS_TKS_SFJT_CLK_CNT(x)  ((x) < 0x80)
#define IS_TKS_IT(x)        (((x) == TKS_IT_RESOV)  || \
                             ((x) == TKS_IT_SCANOV) || \
                             ((x) == TKS_IT_ERR)    || \
                             ((x) == TKS_IT_BREAK)  || \
                             ((x) == TKS_IT_DONE))
#define IS_TKS_IF(x)        (((x) == TKS_IF_RESOV)  || \
                             ((x) == TKS_IF_SCANOV) || \
                             ((x) == TKS_IF_ERR)    || \
                             ((x) == TKS_IF_BREAK)  || \
                             ((x) == TKS_IF_DONE))
/**
  * @}
  */

/** @addtogroup TKS_Public_Functions
  * @{
  */

/** @addtogroup TKS_Public_Functions_Group1
  * @{
  */
/* Initialization functions */
extern void ald_tks_init(tks_handle_t *hperh);
extern void ald_tks_reset(tks_handle_t *hperh);
/**
  * @}
  */
/** @addtogroup TKS_Public_Functions_Group2
  * @{
  */
/* Configure functions */
extern void ald_tks_channel_control(tks_handle_t *hperh, uint32_t enable);
extern uint32_t ald_tks_channel_select_get(tks_handle_t *hperh);
extern void ald_tks_gain_set(tks_handle_t *hperh, uint32_t gain);
extern void ald_tks_sfjt_config(tks_handle_t *hperh, tks_sfjt_config_t *config);
/**
  * @}
  */
/** @addtogroup TKS_Public_Functions_Group3
  * @{
  */
/* Peripheral Control functions */
extern void ald_tks_merge_result_get(tks_handle_t *hperh, uint16_t *value, uint16_t *acccnt);
extern uint16_t ald_tks_result_get(tks_handle_t *hperh, uint8_t ch_idx);
extern void ald_tks_irq_handler(tks_handle_t *hperh);
extern void ald_tks_interrupt_config(tks_handle_t *hperh, tks_it_t it, type_func_t state);
extern it_status_t ald_tks_get_it_status(tks_handle_t *hperh, tks_it_t it);
extern flag_status_t ald_tks_get_flag_status(tks_handle_t *hperh, tks_flag_t flag);
extern void ald_tks_clear_flag_status(tks_handle_t *hperh, tks_flag_t flag);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */

#ifdef __cplusplus
}
#endif
#endif /* __ALD_TKS_H__ */
