/**
  *********************************************************************************
  *
  * @file    ald_pmu.h
  * @brief   Header file of PMU module driver.
  *
  * @version V1.2
  * @date    24 Jun 2022
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jun 2020     AE Team         The first version
  *          09 Mar 2022     AE Team         modified structure pmu_ldo_lpmode_output_t
  *          24 Jun 2022     AE Team         modified function ald_pmu_shutoff_enter
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_PMU_H__
#define __ALD_PMU_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"
#include "ald_syscfg.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup PMU
  * @{
  */

/** @defgroup PMU_Public_Macros PMU Public Macros
  * @{
  */
#define PMU_SRAM0_PWR_ENABLE()                  \
    do {                                \
        SYSCFG_UNLOCK();                    \
        SET_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMPWR_POSS));   \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_SRAM0_PWR_DISABLE()                 \
    do {                                \
        SYSCFG_UNLOCK();                    \
        CLEAR_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMPWR_POSS)); \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_SRAM1_PWR_ENABLE()                  \
    do {                                \
        SYSCFG_UNLOCK();                    \
        SET_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMPWR_POSE));   \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_SRAM1_PWR_DISABLE()                 \
    do {                                \
        SYSCFG_UNLOCK();                    \
        CLEAR_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMPWR_POSE)); \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_SRAM0_RET_ENABLE()                  \
    do {                                \
        SYSCFG_UNLOCK();                    \
        SET_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMRET_POSS));   \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_SRAM0_RET_DISABLE()                 \
    do {                                \
        SYSCFG_UNLOCK();                    \
        CLEAR_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMRET_POSS)); \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_SRAM1_RET_ENABLE()                  \
    do {                                \
        SYSCFG_UNLOCK();                    \
        SET_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMRET_POSE));   \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_SRAM1_RET_DISABLE()                 \
    do {                                \
        SYSCFG_UNLOCK();                    \
        CLEAR_BIT(PMU->PWRCR, BIT(PMU_PWRCR_SRAMRET_POSE)); \
        SYSCFG_LOCK();                      \
    } while (0)
#define PMU_TKRAM_PWR_ENABLE()              \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->PWRCR, PMU_PWRCR_TKRAMPWR_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_TKRAM_PWR_DISABLE()             \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_TKRAMPWR_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_TKRAM_RET_ENABLE()              \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->PWRCR, PMU_PWRCR_TKRAMRET_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_TKRAM_RET_DISABLE()             \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_TKRAMRET_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_ROM_PWR_ENABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->PWRCR, PMU_PWRCR_ROMPWR_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_ROM_PWR_DISABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_ROMPWR_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_ROM_RET_ENABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->PWRCR, PMU_PWRCR_ROMKCS_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_ROM_RET_DISABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_ROMKCS_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_LPSTOP_ENABLE()         \
    do {                        \
        SYSCFG_UNLOCK();            \
        SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);    \
        SYSCFG_LOCK();              \
    } while (0)
#define PMU_LPSTOP_DISABLE()            \
    do {                        \
        SYSCFG_UNLOCK();            \
        CLEAR_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);  \
        SYSCFG_LOCK();              \
    } while (0)
#define PMU_MTSTOP_ENABLE()         \
    do {                        \
        SYSCFG_UNLOCK();            \
        SET_BIT(PMU->CR, PMU_CR_MTSTOP_MSK);    \
        SYSCFG_LOCK();              \
    } while (0)
#define PMU_MTSTOP_DISABLE()            \
    do {                        \
        SYSCFG_UNLOCK();            \
        CLEAR_BIT(PMU->CR, PMU_CR_MTSTOP_MSK);  \
        SYSCFG_LOCK();              \
    } while (0)

#define PMU_GET_LVD_STATUS()    (READ_BITS(PMU->LVDCR, PMU_LVDCR_LVDO_MSK, PMU_LVDCR_LVDO_POS))
/**
  * @}
  */


/** @defgroup PMU_Public_Types PMU Public Types
  * @{
  */
/**
  * @brief Low power mode
  */
typedef enum
{
    PMU_LP_STOP1   = 0x0,   /**< Stop1 */
    PMU_LP_STOP2   = 0x1,   /**< Stop2 */
    PMU_LP_STANDBY = 0x2,   /**< Standby */
    PMU_LP_SHUTOFF = 0x3,   /**< Shutoff */
} pmu_lp_mode_t;

/**
  * @brief Standby wakeup level
  */
typedef enum
{
    PMU_WKPL_HIGH = 0x0,    /**< High */
    PMU_WKPL_LOW  = 0x1,    /**< Low */
} pmu_wakeup_level_t;

/**
  * @brief Standby wakeup pin
  */
typedef enum
{
    PMU_PIN_PA0 = 0x0,  /**< PA0 */
    PMU_PIN_PA1 = 0x1,  /**< PA1 */
    PMU_PIN_PA2 = 0x2,  /**< PA2 */
    PMU_PIN_PA3 = 0x3,  /**< PA3 */
    PMU_PIN_PA4 = 0x4,  /**< PA4 */
    PMU_PIN_PA5 = 0x5,  /**< PA5 */
    PMU_PIN_PA6 = 0x6,  /**< PA6 */
    PMU_PIN_PA7 = 0x7,  /**< PA7 */
} pmu_wakeup_pin_t;

/**
  * @brief Wake up status.
  */
typedef enum
{
    PMU_SR_WUF     = (1U << 0),
    PMU_SR_STANDBY = (1U << 1),
    PMU_SR_SHUTOFF = (1U << 2),
} pmu_status_t;

/**
  * @brief LVD voltage select
  */
typedef enum
{
    PMU_LVD_VOL_SEL_1_8 = 0x0,  /**< 1.8V ~ 1.85V */
    PMU_LVD_VOL_SEL_2_0 = 0x1,  /**< 2.0V ~ 2.05V */
    PMU_LVD_VOL_SEL_2_2 = 0x2,  /**< 2.2V ~ 2.25V */
    PMU_LVD_VOL_SEL_2_4 = 0x3,  /**< 2.4V ~ 2.45V */
    PMU_LVD_VOL_SEL_2_6 = 0x4,  /**< 2.6V ~ 2.65V */
    PMU_LVD_VOL_SEL_2_8 = 0x5,  /**< 2.8V ~ 2.85V */
    PMU_LVD_VOL_SEL_3_0 = 0x6,  /**< 3.0V ~ 3.05V */
    PMU_LVD_VOL_SEL_3_2 = 0x7,  /**< 3.2V ~ 3.25V */
    PMU_LVD_VOL_SEL_3_4 = 0x8,  /**< 3.4V ~ 3.45V */
    PMU_LVD_VOL_SEL_3_6 = 0x9,  /**< 3.6V ~ 3.65V */
    PMU_LVD_VOL_SEL_3_8 = 0xA,  /**< 3.8V ~ 3.85V */
    PMU_LVD_VOL_SEL_4_0 = 0xB,  /**< 4.0V ~ 4.05V */
    PMU_LVD_VOL_SEL_4_2 = 0xC,  /**< 4.2V ~ 4.25V */
    PMU_LVD_VOL_SEL_4_4 = 0xD,  /**< 4.4V ~ 4.45V */
    PMU_LVD_VOL_SEL_4_6 = 0xE,  /**< 4.6V ~ 4.65V */
    PMU_LVD_VOL_SEL_4_8 = 0xF,  /**< 4.8V ~ 4.85V */
} pmu_lvd_voltage_sel_t;

/**
  * @brief LVD trigger mode
  */
typedef enum
{
    PMU_LVD_TRIGGER_RISING_EDGE    = 0x0,   /**< Rising edge */
    PMU_LVD_TRIGGER_FALLING_EDGE   = 0x1,   /**< Falling edge */
    PMU_LVD_TRIGGER_HIGH_LEVEL     = 0x2,   /**< High level */
    PMU_LVD_TRIGGER_LOW_LEVEL      = 0x3,   /**< Low level */
    PMU_LVD_TRIGGER_RISING_FALLING = 0x4,   /**< Rising and falling edge */
} pmu_lvd_trigger_mode_t;

/**
  * @brief LDO output voltage selest in low power mode
  */
typedef enum
{
    PMU_LDO_LPMODE_OUTPUT_1_3 = 0x0,    /**< 1.3V */
    PMU_LDO_LPMODE_OUTPUT_1_4 = 0x1,    /**< 1.4V */
    PMU_LDO_LPMODE_OUTPUT_1_5 = 0x2,    /**< 1.5V */
    PMU_LDO_LPMODE_OUTPUT_1_6 = 0x4,    /**< 1.6V */
} pmu_ldo_lpmode_output_t;
/**
  * @}
  */

/**
  * @defgroup PMU_Private_Macros PMU Private Macros
  * @{
  */
#define IS_PMU_LP_MODE(x)       (((x) == PMU_LP_STOP1)   || \
                                 ((x) == PMU_LP_STOP2)   || \
                                 ((x) == PMU_LP_STANDBY) || \
                                 ((x) == PMU_LP_SHUTOFF))
#define IS_PMU_WAKEUP_LEVEL(x)      (((x) == PMU_WKPL_HIGH) || \
                                     ((x) == PMU_WKPL_LOW))
#define IS_PMU_WAKEUP_PIN(x)        (((x) == PMU_PIN_PA0) || \
                                     ((x) == PMU_PIN_PA1) || \
                                     ((x) == PMU_PIN_PA2) || \
                                     ((x) == PMU_PIN_PA3) || \
                                     ((x) == PMU_PIN_PA4) || \
                                     ((x) == PMU_PIN_PA5) || \
                                     ((x) == PMU_PIN_PA6) || \
                                     ((x) == PMU_PIN_PA7))
#define IS_PMU_STATUS(x)        (((x) == PMU_SR_WUF)     || \
                                 ((x) == PMU_SR_STANDBY) || \
                                 ((x) == PMU_SR_SHUTOFF))
#define IS_PMU_LVD_VOL_SEL(x)       (((x) == PMU_LVD_VOL_SEL_1_8) || \
                                     ((x) == PMU_LVD_VOL_SEL_2_0) || \
                                     ((x) == PMU_LVD_VOL_SEL_2_2) || \
                                     ((x) == PMU_LVD_VOL_SEL_2_4) || \
                                     ((x) == PMU_LVD_VOL_SEL_2_6) || \
                                     ((x) == PMU_LVD_VOL_SEL_2_8) || \
                                     ((x) == PMU_LVD_VOL_SEL_3_0) || \
                                     ((x) == PMU_LVD_VOL_SEL_3_2) || \
                                     ((x) == PMU_LVD_VOL_SEL_3_4) || \
                                     ((x) == PMU_LVD_VOL_SEL_3_6) || \
                                     ((x) == PMU_LVD_VOL_SEL_3_8) || \
                                     ((x) == PMU_LVD_VOL_SEL_4_0) || \
                                     ((x) == PMU_LVD_VOL_SEL_4_2) || \
                                     ((x) == PMU_LVD_VOL_SEL_4_4) || \
                                     ((x) == PMU_LVD_VOL_SEL_4_6) || \
                                     ((x) == PMU_LVD_VOL_SEL_4_8))
#define IS_PMU_LVD_TRIGGER_MODE(x)  (((x) == PMU_LVD_TRIGGER_RISING_EDGE)  || \
                                     ((x) == PMU_LVD_TRIGGER_FALLING_EDGE) || \
                                     ((x) == PMU_LVD_TRIGGER_HIGH_LEVEL)   || \
                                     ((x) == PMU_LVD_TRIGGER_LOW_LEVEL)    || \
                                     ((x) == PMU_LVD_TRIGGER_RISING_FALLING))
#define IS_PMU_LDO_LPMODE_OUTPUT(x) (((x) == PMU_LDO_LPMODE_OUTPUT_1_3) || \
                                     ((x) == PMU_LDO_LPMODE_OUTPUT_1_4) || \
                                     ((x) == PMU_LDO_LPMODE_OUTPUT_1_5) || \
                                     ((x) == PMU_LDO_LPMODE_OUTPUT_1_6))
#define IS_PMU_BKP_RAM_IDX(x)   ((x) < 8)
/**
  * @}
  */

/** @addtogroup PMU_Public_Functions
  * @{
  */
/** @addtogroup PMU_Public_Functions_Group1
  * @{
  */
/* Low power mode select */
__STATIC_INLINE__ void ald_pmu_sleep()
{
    __WFI();
}

__STATIC_INLINE__ void ald_pmu_sleep_deep()
{
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
}

void ald_pmu_stop1_enter(void);
void ald_pmu_stop2_enter(void);
void ald_pmu_standby_enter(pmu_wakeup_pin_t pin, pmu_wakeup_level_t levle);
void ald_pmu_shutoff_enter(pmu_wakeup_pin_t pin, pmu_wakeup_level_t levle);
void ald_pmu_lprun_config(pmu_ldo_lpmode_output_t vol, type_func_t state);
void ald_pmu_wakup_set(uint16_t time);
void ald_pmu_bkp_write_ram(uint8_t idx, uint32_t value);
uint32_t ald_pmu_bkp_read_ram(uint8_t idx);
flag_status_t ald_pmu_get_status(pmu_status_t sr);
void ald_pmu_clear_status(pmu_status_t sr);
/**
  * @}
  */
/** @addtogroup PMU_Public_Functions_Group2
  * @{
  */
/* LVD configure */
void ald_pmu_lvd_config(pmu_lvd_voltage_sel_t sel, pmu_lvd_trigger_mode_t mode, type_func_t state);
void ald_lvd_irq_handler(void);
/**
  * @}
  */
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_PMU_H__ */
