/**
  *********************************************************************************
  *
  * @file    ald_pmu.c
  * @brief   PMU module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "ald_conf.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup PMU PMU
  * @brief PMU module driver
  * @{
  */
#ifdef ALD_PMU


/** @defgroup PMU_Private_Functions PMU Private Functions
  * @{
  */

/**
  * @brief  PMU module interrupt handler
  * @retval None
  */
void ald_lvd_irq_handler(void)
{
    SYSCFG_UNLOCK();
    SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
    SYSCFG_LOCK();

    return;
}
/**
  * @}
  */

/** @defgroup PMU_Public_Functions PMU Public Functions
  * @{
  */

/** @addtogroup PMU_Public_Functions_Group1 Low Power Mode
  * @brief Low power mode select functions
  *
  * @verbatim
  ==============================================================================
              ##### Low power mode select functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Enter stop1 mode.
      (+) Enter stop2 mode.
      (+) Enter standby mode.
      (+) Get wakeup status.
      (+) Clear wakeup status.

    @endverbatim
  * @{
  */

/**
  * @brief  Enter stop1 mode
  * @retval None
  */
void ald_pmu_stop1_enter(void)
{
    SYSCFG_UNLOCK();

    CLEAR_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    CLEAR_BIT(PMU->CR, PMU_CR_LPRUN_MSK);
    MODIFY_REG(PMU->CR, PMU_CR_STOPFM_MSK, 0x2 << PMU_CR_STOPFM_POSS);
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, ALD_PMU_LP_STOP1 << PMU_CR_LPM_POSS);
    PMU->CR &= ~(0x1 << 24);
    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);

    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Enter stop2 mode
  * @retval None
  */
void ald_pmu_stop2_enter(void)
{
    SYSCFG_UNLOCK();
    CLEAR_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    CLEAR_BIT(PMU->CR, PMU_CR_LPRUN_MSK);
    MODIFY_REG(PMU->CR, PMU_CR_STOPFM_MSK, 0x2 << PMU_CR_STOPFM_POSS);
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, ALD_PMU_LP_STOP2 << PMU_CR_LPM_POSS);
    PMU->CR &= ~(0x1 << 24);
    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
}

/**
  * @brief  Enter standby mode
  * @param  port: The port whick wake up the standby mode.
  * @param  level: Wakeup level.
  * @retval None
  */
void ald_pmu_standby_enter(ald_pmu_wakeup_pin_t pin, ald_pmu_wakeup_level_t levle)
{
    uint32_t tmp = 0x5A00;

    SET_BIT(tmp, PMU_BKPCR0_WKPEN_MSK | PMU_BKPCR0_STBWKEN_MSK);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPS_MSK, pin << PMU_BKPCR0_WKPS_POSS);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPL_MSK, levle << PMU_BKPCR0_WKPL_POS);

    SYSCFG_UNLOCK();
    PMU->BKPCR0 = tmp;
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, ALD_PMU_LP_STANDBY << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
}

/**
  * @brief  Enter shutoff mode
  * @retval None
  */
void ald_pmu_shutoff_enter(ald_pmu_wakeup_pin_t pin, ald_pmu_wakeup_level_t levle)
{
    uint32_t tmp = 0x5A00;

    SET_BIT(tmp, PMU_BKPCR0_WKPEN_MSK | PMU_BKPCR0_STBWKEN_MSK);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPS_MSK, pin << PMU_BKPCR0_WKPS_POSS);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPL_MSK, levle << PMU_BKPCR0_WKPL_POS);

    SYSCFG_UNLOCK();
    PMU->BKPCR0 = tmp;
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, ALD_PMU_LP_SHUTOFF << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Get wakup status.
  * @param  sr: Status bit.
  * @retval Status.
  */
flag_status_t ald_pmu_get_status(ald_pmu_status_t sr)
{
    assert_param(IS_PMU_STATUS(sr));

    if (READ_BIT(PMU->SR, sr))
        return SET;

    return RESET;
}

/**
  * @brief  Clear wakup status.
  * @param  sr: Status bit.
  * @retval None
  */
void ald_pmu_clear_status(ald_pmu_status_t sr)
{
    assert_param(IS_PMU_STATUS(sr));
    SYSCFG_UNLOCK();

    if (sr == ALD_PMU_SR_WUF)
        SET_BIT(PMU->CR, PMU_CR_CWUF_MSK);
    else if (sr == ALD_PMU_SR_STANDBYF)
        SET_BIT(PMU->CR, PMU_CR_CSTANDBYF_MSK);
    else
        ;/* do nothing */

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Configure peripheral power
  * @param  perh: The peripheral
  * @param  state: ENABLE/DISABLE
  * @retval None
  */
void ald_pmu_perh_power_config(ald_pmu_perh_power_t perh, type_func_t state)
{
    assert_param(IS_PMU_PERH_POWER(perh));
    assert_param(IS_FUNC_STATE(state));

    SYSCFG_UNLOCK();

    if (state)
        SET_BIT(PMU->PWRCR, perh);
    else
        CLEAR_BIT(PMU->PWRCR, perh);

    SYSCFG_LOCK();
    return;
}
/**
  * @}
  */

/** @addtogroup PMU_Public_Functions_Group2 LVD Configure
  * @brief LVD configure functions
  *
  * @verbatim
  ==============================================================================
              ##### LVD configure functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Configure lvd parameters.

    @endverbatim
  * @{
  */

/**
  * @brief  Configure lvd using specified parameters.
  * @param  sel: LVD threshold voltage.
  * @param  mode: LVD trigger mode.
  * @param  state: New state, ENABLE/DISABLE;
  * @retval None
  */
void ald_pmu_lvd_config(ald_pmu_lvd_voltage_sel_t sel, ald_pmu_lvd_trigger_mode_t mode, type_func_t state)
{
    assert_param(IS_FUNC_STATE(state));
    SYSCFG_UNLOCK();

    if (state)
    {
        assert_param(IS_PMU_LVD_VOL_SEL(sel));
        assert_param(IS_PMU_LVD_TRIGGER_MODE(mode));

        MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDS_MSK, sel << PMU_LVDCR_LVDS_POSS);
        MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDIFS_MSK, mode << PMU_LVDCR_LVDIFS_POSS);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
    }
    else
    {
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
        CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
        CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
    }

    SYSCFG_LOCK();
    return;
}
/**
  * @}
  */


/**
  * @}
  */
#endif /* ALD_PMU */
/**
  * @}
  */

/**
  * @}
  */
