/**
  *********************************************************************************
  *
  * @file    ald_dma.c
  * @brief   DMA module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Apr 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "ald_conf.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup DMA DMA
  * @brief DMA module driver
  * @{
  */

#ifdef ALD_DMA
/** @defgroup DMA_Private_Variables DMA Private Variables
  * @{
  */
ald_dma_call_back_t dma0_cbk[6U];
ald_dma_call_back_t dma1_cbk[6U];
/**
  * @}
  */

/** @defgroup DMA_Public_Functions DMA Public Functions
  * @{
  */

/** @defgroup DMA_Public_Functions_Group1 Initialization functions
  * @brief Initialization functions
  * @{
  */
/**
  * @brief  Reset the DMA register
  * @param  DMAx: Pointer to DMA peripheral
  * @retval None
  */
void ald_dma_reset(DMA_TypeDef *DMAx)
{
    uint32_t i, k;

    assert_param(IS_DMA(DMAx));

    memset((DMAx == DMA0 ? dma0_cbk : dma1_cbk), 0x0U, sizeof(dma0_cbk));
    WRITE_REG(DMAx->IDR, 0xFFFFFFFFU);
    WRITE_REG(DMAx->ICR, 0xFFFFFFFFU);

    for (i = 0; i < DMA_CH_COUNT; ++i)
    {
        WRITE_REG(DMAx->CH[i].CON, 0x0U);
        WRITE_REG(DMAx->CH[i].SAR, 0x0U);
        WRITE_REG(DMAx->CH[i].DAR, 0x0U);
        WRITE_REG(DMAx->CH[i].NDT, 0x0U);
    }

    i = ((DMAx == DMA0) ? 0U : DMA_CH_COUNT);
    k = i + 6U;

    for (; i < k; ++i)
        DMA_MUX->CH_SELCON[i] = 0x0U;

    return;
}

/**
  * @brief  DMA module initialization, this function
  *         is invoked by ald_cmu_init().
  * @param  DMAx: Pointer to DMA peripheral
  * @retval None
  */
void ald_dma_init(DMA_TypeDef *DMAx)
{
    assert_param(IS_DMA(DMAx));

    ald_dma_reset(DMAx);

    if (DMAx == DMA0)
    {
        NVIC_SetPriority(DMA0_IRQn, 2U);
        NVIC_EnableIRQ(DMA0_IRQn);
    }
    else
    {
        NVIC_SetPriority(DMA1_IRQn, 2U);
        NVIC_EnableIRQ(DMA1_IRQn);
    }

    return;
}

/**
  * @brief  Configure dma_config_t structure using default parameter.
  *         User can invoked this function, before configure dma_config_t
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @retval None
  */
void ald_dma_config_struct(ald_dma_handle_t *hperh)
{
    hperh->config.src_data_width    = ALD_DMA_DATA_SIZE_BYTE;
    hperh->config.src_inc           = ALD_DMA_DATA_INC_ENABLE;
    hperh->config.dst_data_width    = ALD_DMA_DATA_SIZE_BYTE;
    hperh->config.dst_inc           = ALD_DMA_DATA_INC_ENABLE;
    hperh->config.R_power           = ALD_DMA_R_POWER_1;
    hperh->config.priority          = ALD_DMA_LOW_PRIORITY;
    hperh->config.dir               = ALD_DMA_DIR_TO_SRAM;
    hperh->config.mem_to_mem        = DISABLE;
    hperh->config.circle_mode       = DISABLE;
    hperh->config.irq_tc            = ENABLE;
    hperh->config.irq_ht            = DISABLE;
    hperh->config.channel           = 0x0U;

    return;
}

/**
  * @brief  Configure DMA channel using dma_config_t structure
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @retval None
  */
void ald_dma_config(ald_dma_handle_t *hperh)
{
    uint32_t ch, tmp = 0x0U;

    assert_param(IS_DMA(hperh->perh));
    assert_param(hperh->config.src != NULL);
    assert_param(hperh->config.dst != NULL);
    assert_param(IS_DMA_DATA_SIZE(hperh->config.size));
    assert_param(IS_DMA_DATASIZE_TYPE(hperh->config.src_data_width));
    assert_param(IS_DMA_DATA_INC_STAT(hperh->config.src_inc));
    assert_param(IS_DMA_DATASIZE_TYPE(hperh->config.dst_data_width));
    assert_param(IS_DMA_DATA_INC_STAT(hperh->config.dst_inc));
    assert_param(IS_DMA_ARBITERCONFIG_TYPE(hperh->config.R_power));
    assert_param(IS_DMA_PRIO_TYPE(hperh->config.priority));
    assert_param(IS_DMA_DIR_TYPE(hperh->config.dir));
    assert_param(IS_FUNC_STATE(hperh->config.mem_to_mem));
    assert_param(IS_FUNC_STATE(hperh->config.circle_mode));
    assert_param(IS_FUNC_STATE(hperh->config.irq_tc));
    assert_param(IS_FUNC_STATE(hperh->config.irq_ht));
    assert_param(IS_DMA_CHANNEL(hperh->config.channel));
    assert_param(IS_DMA_MSIGSEL_TYPE(hperh->config.msigsel));

    if ((hperh->config.circle_mode) && (hperh->config.mem_to_mem != 0x1U))
        SET_BIT(tmp, DMA_CON_CIRC_MSK);

    if (hperh->config.dir == ALD_DMA_DIR_TO_PERH)
        SET_BIT(tmp, DMA_CON_DIR_MSK);

    if (hperh->config.mem_to_mem)
        SET_BIT(tmp, DMA_CON_M2M_MSK);

    if (hperh->config.src_inc)
        SET_BIT(tmp, DMA_CON_SINC_MSK);

    if (hperh->config.dst_inc)
        SET_BIT(tmp, DMA_CON_DINC_MSK);

    MODIFY_REG(tmp, DMA_CON_CHPRI_MSK, (hperh->config.priority) << DMA_CON_CHPRI_POSS);
    MODIFY_REG(tmp, DMA_CON_SDWSEL_MSK, (hperh->config.src_data_width) << DMA_CON_SDWSEL_POSS);
    MODIFY_REG(tmp, DMA_CON_DDWSEL_MSK, (hperh->config.dst_data_width) << DMA_CON_DDWSEL_POSS);
    MODIFY_REG(tmp, DMA_CON_MAX_BURST_MSK, (hperh->config.R_power) << DMA_CON_MAX_BURST_POSS);

    /* IF UART/SPI/I2C, only use single transfer, DMA_CONx.MAX_BURST=4b0000 */
    if ((hperh->config.msel >= ALD_DMA_MSEL_UART0) && (hperh->config.msel <= ALD_DMA_MSEL_I2C1))
    {
        MODIFY_REG(tmp, DMA_CON_MAX_BURST_MSK, 0U << DMA_CON_MAX_BURST_POSS);
    }

    ch = hperh->config.channel;
    ALD_DMA_CH_DISABLE(hperh, ch);
    ALD_DMA_SRC_SET(hperh, ch, (uint32_t)hperh->config.src);
    ALD_DMA_DST_SET(hperh, ch, (uint32_t)hperh->config.dst);
    ALD_DMA_CNT_SET(hperh, ch, hperh->config.size);
    ALD_DMA_CON_SET(hperh, ch, tmp);

    tmp = 0x0U;

    if (hperh->config.irq_tc)
        SET_BIT(tmp, 0x1U << (ch << 0x1U));

    if (hperh->config.irq_ht)
        SET_BIT(tmp, 0x1U << ((ch << 0x1U) + 0x1U));

    ALD_DMA_IF_CLEAR(hperh, tmp);
    ALD_DMA_IT_ENABLE(hperh, tmp);

    tmp = 0x0U;
    MODIFY_REG(tmp, DMA_MUX_MSEL_MSK, (hperh->config.msel << DMA_MUX_MSEL_POSS));
    MODIFY_REG(tmp, DMA_MUX_MSIGSEL_MSK, (hperh->config.msigsel << DMA_MUX_MSIGSEL_POSS));
    /* Only UART/SPI/I2C transfer works */
    MODIFY_REG(tmp, DMA_MUX_SINGLE_MSK, (0x1U << DMA_MUX_SINGLE_POSS));

    if (hperh->perh == DMA0)
    {
        WRITE_REG(DMA_MUX->CH_SELCON[ch], tmp);
        dma0_cbk[ch].cplt_ht_arg = hperh->cplt_ht_arg;
        dma0_cbk[ch].cplt_ht_cbk = hperh->cplt_ht_cbk;
        dma0_cbk[ch].cplt_tc_arg = hperh->cplt_tc_arg;
        dma0_cbk[ch].cplt_tc_cbk = hperh->cplt_tc_cbk;
    }
    else
    {
        WRITE_REG(DMA_MUX->CH_SELCON[ch + DMA_CH_COUNT], tmp);
        dma1_cbk[ch].cplt_ht_arg = hperh->cplt_ht_arg;
        dma1_cbk[ch].cplt_ht_cbk = hperh->cplt_ht_cbk;
        dma1_cbk[ch].cplt_tc_arg = hperh->cplt_tc_arg;
        dma1_cbk[ch].cplt_tc_cbk = hperh->cplt_tc_cbk;
    }

    ALD_DMA_CH_ENABLE(hperh, ch);
    return;
}

/**
  * @brief  Pause the DMA transfer
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @retval None
  */
void ald_dma_pause(ald_dma_handle_t *hperh)
{
    assert_param(IS_DMA(hperh->perh));

    ALD_DMA_CH_DISABLE(hperh, hperh->config.channel);
    return;
}

/**
  * @brief  Restart the DMA transfer
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @retval None
  */
void ald_dma_restart(ald_dma_handle_t *hperh)
{
    assert_param(IS_DMA(hperh->perh));

    ALD_DMA_CH_ENABLE(hperh, hperh->config.channel);
    return;
}

/**
  * @brief  Resume the DMA transfer. Continuaion transfer from breakpoint.
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @retval None
  */
void ald_dma_resume(ald_dma_handle_t *hperh)
{
    uint32_t rem, nr, reg, tmp;

    assert_param(IS_DMA(hperh->perh));

    tmp = ((hperh->perh->CH[hperh->config.channel].NDT >> 16U) & 0xFFFFU);

    rem = (hperh->perh->CH[hperh->config.channel].NDT & 0xFFFFU) - tmp;

    if (hperh->config.src_inc)
    {
        if (hperh->config.src_data_width == ALD_DMA_DATA_SIZE_HALFWORD)
            nr = (rem << 1U);
        else if (hperh->config.src_data_width == ALD_DMA_DATA_SIZE_WORD)
            nr = (rem << 2U);
        else
            nr = rem;

        reg = READ_REG(hperh->perh->CH[hperh->config.channel].SAR);
        WRITE_REG(hperh->perh->CH[hperh->config.channel].SAR, (reg + nr));
    }

    if (hperh->config.dst_inc)
    {
        if (hperh->config.dst_data_width == ALD_DMA_DATA_SIZE_HALFWORD)
            nr = (rem << 1U);
        else if (hperh->config.dst_data_width == ALD_DMA_DATA_SIZE_WORD)
            nr = (rem << 2U);
        else
            nr = rem;

        reg = READ_REG(hperh->perh->CH[hperh->config.channel].DAR);
        WRITE_REG(hperh->perh->CH[hperh->config.channel].DAR, (reg + nr));
    }

    reg = ((hperh->perh->CH[hperh->config.channel].NDT >> 16U) & 0xFFFFU);
    WRITE_REG(hperh->perh->CH[hperh->config.channel].NDT, reg);
    ALD_DMA_CH_ENABLE(hperh, hperh->config.channel);
    return;
}
/**
  * @}
  */
/** @defgroup DMA_Public_Functions_Group3 DMA Control functions
  * @brief DMA control functions
  * @{
  */
/**
  * @brief  Handle the DMA interrupt, Call the callback function.
  * @param  DMAx: Pointer to DMA peripheral
  * @retval None
  */
void ald_dma0_irq_handler(void)
{
    uint32_t i, reg;

    reg = DMA0->IFM;

    for (i = 0; i < 0x6; ++i)
    {
        // Transfer completely callback
        if (reg & (1U << 2 * i))
        {
            if (dma0_cbk[i].cplt_tc_cbk != NULL)
                dma0_cbk[i].cplt_tc_cbk(dma0_cbk[i].cplt_tc_arg);

            WRITE_REG(DMA0->ICR, (1U << (2U * i)));
        }

        // Transfer half callback
        if (reg & (1U << (2U * i + 1)))
        {
            if (dma0_cbk[i].cplt_ht_cbk != NULL)
                dma0_cbk[i].cplt_ht_cbk(dma0_cbk[i].cplt_ht_arg);

            WRITE_REG(DMA0->ICR, (1U << (2U * i + 1U)));
        }
    }

}

/**
  * @brief  Handle the DMA interrupt, Call the callback function.
  * @param  DMAx: Pointer to DMA peripheral
  * @retval None
  */
void ald_dma1_irq_handler(void)
{
    uint32_t i, reg;

    reg = DMA1->IFM;


    for (i = 0; i < 0x6; ++i)
    {
        // Transfer completely callback
        if (reg & (1U << (2U * i)))
        {
            if (dma1_cbk[i].cplt_tc_cbk != NULL)
                dma1_cbk[i].cplt_tc_cbk(dma1_cbk[i].cplt_tc_arg);

            WRITE_REG(DMA1->ICR, (1U << (2U * i)));
        }

        // Transfer half callback
        if (reg & (1U << (2U * i + 1)))
        {
            if (dma1_cbk[i].cplt_ht_cbk != NULL)
                dma1_cbk[i].cplt_ht_cbk(dma1_cbk[i].cplt_ht_arg);

            WRITE_REG(DMA1->ICR, (1U << (2U * i + 1U)));
        }
    }

    return;
}


/**
  * @brief  Configure the interrupt enable or disable
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @param  isr: Type of the interrupt.
  * @param  state: status of channel:
  *           @arg ENABLE: Enable the channel
  *           @arg DISABLE: Disable the channel
  *
  * @retval None
  */
void ald_dma_interrupt_config(ald_dma_handle_t *hperh, ald_dma_it_flag_t it, type_func_t state)
{
    assert_param(IS_DMA(hperh->perh));
    assert_param(IS_DMA_IT_TYPE(it));
    assert_param(IS_FUNC_STATE(state));

    if (state)
        SET_BIT(hperh->perh->IER, 0x1U << (hperh->config.channel * 2 + it));
    else
        SET_BIT(hperh->perh->IDR, 0x1U << (hperh->config.channel * 2 + it));

    return;
}

/**
  * @brief  Check whether the specified channel interrupt
  *         is set or reset
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @param  channel: DMA channel.
  * @param  it: Type of the interrupt.
  * @retval Status:
  *           - SET: interrupt is set
  *           - RESET: interrupt is reset
  */
it_status_t ald_dma_get_it_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it)
{
    assert_param(IS_DMA(hperh->perh));
    assert_param(IS_DMA_IT_TYPE(it));
    assert_param(IS_DMA_CHANNEL(channel));

    if (READ_BIT(hperh->perh->IVS, 0x1U << (channel * 2U + it)))
        return SET;

    return RESET;
}

/**
  * @brief  Check whether the specified channel flag
  *         is set or reset
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @param  channel: DMA channel.
  * @param  isr: Type of the interrupt.
  * @retval Status:
  *           - SET: flag is set
  *           - RESET: flag is reset
  */
flag_status_t ald_dma_get_flag_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it)
{
    assert_param(IS_DMA(hperh->perh));
    assert_param(IS_DMA_IT_TYPE(it));
    assert_param(IS_DMA_CHANNEL(channel));

    if (READ_BIT(hperh->perh->RIF, 0x1U << (channel * 2U + it)))
        return SET;

    return RESET;
}

/**
  * @brief  Check whether the specified channel flag and it
  *         is set or reset
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @param  channel: DMA channel.
  * @param  isr: Type of the interrupt.
  * @retval Status:
  *           - SET: flag is set
  *           - RESET: flag is reset
  */
flag_status_t ald_dma_get_mask_flag_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it)
{
    assert_param(IS_DMA(hperh->perh));
    assert_param(IS_DMA_IT_TYPE(it));
    assert_param(IS_DMA_CHANNEL(channel));

    if (READ_BIT(hperh->perh->IFM, 0x1U << (channel * 2U + it)))
        return SET;

    return RESET;
}


/**
  * @brief  Clear the specified channel pending flag
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @param  isr: Type of the interrupt.
  * @retval None
  */
void ald_dma_clear_flag_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it)
{
    assert_param(IS_DMA(hperh->perh));
    assert_param(IS_DMA_IT_TYPE(it));
    assert_param(IS_DMA_CHANNEL(channel));

    WRITE_REG(hperh->perh->ICR, 0x1U << (channel * 2U + it));
    return;
}

/**
  * @}
  */
/**
  * @}
  */
#endif /* ALD_DMA */
/**
  * @}
  */
/**
  * @}
  */
