/**
  ******************************************************************************
  * @file    ald_dac.c
  * @brief   DAC module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team.
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "ald_conf.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup DAC DAC
  * @brief DAC module driver
  * @{
  */
#ifdef ALD_DAC

/** @defgroup DAC_Public_Functions DAC Public Functions
  * @{
  */
/**
  * @brief  Reset the dac mode.
  * @param  hperh: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_dac_reset(ald_dac_handle_t *hperh)
{
    assert_param(IS_DAC_TYPE(hperh->perh));

    hperh->perh->CON     = 0;
    hperh->perh->SWTRG   = 0;
    hperh->perh->DOUT    = 0;
    hperh->perh->R12BUF  = 0;
    hperh->perh->L12BUF  = 0;
    hperh->perh->R8BUF   = 0;

    return ALD_OK;
}

/**
  * @brief  Initializes the DAC peripheral.
  * @param  hperh: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_dac_init(ald_dac_handle_t *hperh)
{
    uint32_t tmp = 0;

    assert_param(IS_DAC_TYPE(hperh->perh));
    assert_param(IS_DAC_TRIGGER_MODE(hperh->init.soft_trig));
    assert_param(IS_DAC_EXTERNAL_TRIGGER_PIS(hperh->init.pis_ch));
    assert_param(IS_DAC_OUTPUT_MODE(hperh->init.output_mode));
    assert_param(IS_DAC_NOISE_FILTER(hperh->init.noise_mode));
    assert_param(IS_DAC_FILTER_WIDTH(hperh->init.noise_width));

    __LOCK(hperh);
    DAC0_RESET();

    tmp = ((hperh->init.soft_trig << DAC_CON_SWTEN_POSS) | (hperh->init.pis_ch << DAC_CON_EXTS_POSS) |
           (hperh->init.output_mode << DAC_CON_MSEL_POSS) | (hperh->init.noise_mode << DAC_CON_NWM_POSS) |
           (hperh->init.noise_width << DAC_CON_NWBW_POSS) | (hperh->init.output_gpio << DAC_CON_DACOE_POSS));

    if (hperh->init.output_op == DISABLE)
    {
        tmp |= (1U << DAC_CON_OPOFF_POSS);
    }

    WRITE_REG(hperh->perh->CON, tmp);

    DAC0_ENABLE();

    __UNLOCK(hperh);
    return ALD_OK;
}

/**
  * @brief  Enables DAC and starts conversion.
  * @param  hperh: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_dac_start(ald_dac_handle_t *hperh)
{
    assert_param(IS_DAC_TYPE(hperh->perh));

    DAC0_ENABLE();

    return ALD_OK;
}

/**
  * @brief  Disables DAC and starts conversion.
  * @param  hperh: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_dac_stop(ald_dac_handle_t *hperh)
{
    assert_param(IS_DAC_TYPE(hperh->perh));

    DAC0_DISABLE();

    return ALD_OK;
}

/**
  * @brief  Set the specified data holding register value for DAC.
  * @param  hperh: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @param  data_align: The data alignment.
  * @param  value: The value be converted,and the valid value is low 12 bit.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_dac_set_value(ald_dac_handle_t *hperh, uint32_t value)
{
    assert_param(IS_DAC_TYPE(hperh->perh));
    assert_param(IS_DAC_DATA(value));

    if (hperh->init.align_mode == ALD_DAC_ALIGN_R12)
    {
        MODIFY_REG(hperh->perh->R12BUF, DAC_R12BUF_VAL_MSK, value);
    }
    else if (hperh->init.align_mode == ALD_DAC_ALIGN_R8)
    {
        MODIFY_REG(hperh->perh->R8BUF, DAC_R8BUF_VAL_MSK, value);
    }
    else
    {
        MODIFY_REG(hperh->perh->L12BUF, DAC_L12BUF_VAL_MSK, value);
    }

    return ALD_OK;
}

/**
  * @brief  Returns DAC last data output value.
  * @retval DAC data output value
  */
uint32_t ald_dac_get_value(ald_dac_handle_t *hperh)
{
    assert_param(IS_DAC_TYPE(hperh->perh));

    return hperh->perh->DOUT;
}

/**
  * @brief  Software trigger DAC output conversion.
  * @param  hperh: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_dac_set_bit_trig(ald_dac_handle_t *hperh)
{
    assert_param(IS_DAC_TYPE(hperh->perh));

    if (hperh->init.soft_trig == ALD_DAC_WR_BIT_TRG)
    {
        SET_BIT(hperh->perh->SWTRG, DAC_SWTRG_TRG_MSK);
    }

    return ALD_OK;
}

/**
  *@}
  */
#endif /* ALD_DAC */
/**
  *@}
  */
/**
  *@}
  */
