/**
  *********************************************************************************
  *
  * @file    ald_cmu.c
  * @brief   CMU module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  * @verbatim
  ==============================================================================
                        ##### How to use this driver #####
  ==============================================================================
    [..]
     *** System clock configure ***
     =================================
     [..]
       (+) If you don't change system clock, you can using ald_cmu_clock_config_default() API.
           It will select HRC as system clock. The system clock is 4MHz.
       (+) If you want to change system clock, you can using ald_cmu_clock_config() API.
           You can select one of the following as system clock:
             @ref CMU_CLOCK_HRC   4MHz
             @ref CMU_CLOCK_LRC   32768Hz
             @ref CMU_CLOCK_LOSC  32768Hz
             @ref CMU_CLOCK_PLL  48MHz/56Mhz/64Mhz/72MHz/80Mhz/84Mhz/88Mhz/96MHz
             @ref CMU_CLOCK_HOSC  1MHz -- 24MHz
       (+) If you select CMU_CLOCK_PLL as system clock, it must config the PLL
           using ald_cmu_pll_config() API. The input of clock must be 4MHz.
       (+) If you get current clock, you can using ald_cmu_get_clock() API.

     *** BUS division control ***
     ===================================

     MCLK            sys_clk
     -------DIV_SYS-----------+--------------------------System(Core, DMA, Systick ... etc.)
                              |
                              |                  hclk1
                              +------DIV_AHB1------------Peripheral(GPIO, CRC, ... etc.)
                              |
                              |                  hclk2
                              +------DIV_AHB2------------Peripheral(EBI, QSPI, ... etc.)
                              |
                              |                  pclk1
                              +------DIV_APB1------------Peripheral(TIM, UART, ... etc.)
                              |
                              |                  pclk2
                              +------DIV_APB2------------Peripheral(ADC, WWDT, ... etc.)

     [..]
       (+) Configure the division using ald_cmu_div_config() API.
       (+) Get sys_clk using ald_cmu_get_sys_clock() API.
       (+) Get hclk1 using ald_cmu_get_hclk1_clock() API.
       (+) Get pclk1 using ald_cmu_get_pclk1_clock() API.
       (+) Get pclk2 using ald_cmu_get_pclk2_clock() API.

     *** Clock safe configure ***
     ===================================
     [..]
       (+) If you select CMU_CLOCK_HOSC as system clock, you need enable
           clock safe using ald_cmu_hosc_safe_config() API. It will change
           CMU_CLOCK_HRC as system clock, when the outer crystal stoped.
       (+) If you select CMU_CLOCK_LOSC as system clock, you need enable
           clock safe using ald_cmu_losc_safe_config() API. It will change
           CMU_CLOCK_LRC as system clock, when the outer crystal stoped.
       (+) If you select CMU_CLOCK_PLL as system clock, you need enable
           clock safe using ald_cmu_pll_safe_config() API. It will change
           CMU_CLOCK_HRC as system clock, when the pll is lose.
       (+) The ald_cmu_irq_cbk() will be invoked, when CMU interrupt has
           been occurred. You can overwrite this function in application.

     *** Clock output configure ***
     ===================================
     [..]
       (+) Output high-speed clock using ald_cmu_output_high_clock_config() API.
       (+) Output low-speed clock using ald_cmu_output_low_clock_config() API.

     *** Peripheral clock configure ***
     ===================================
     [..]
       (+) Configure buzz clock using ald_cmu_buzz_config() API.
       (+) Selected lptim0 clock using ald_cmu_lptim0_clock_select() API.
       (+) Selected lpuart clock using ald_cmu_lpuart0_clock_select() API.
       (+) Selected lcd clock using ald_cmu_lcd_clock_select() API.
       (+) Selected qspi clock using ald_cmu_qspi_clock_select() API.
       (+) Configure usb clock using ald_cmu_usb_clock_config() API.
       (+) Enable/Disable peripheral clock using ald_cmu_perh_clock_config() API.
       (+) Selected stop1 clock using ald_cmu_stop1_clock_sel() API.

     *** CMU ALD driver macros list ***
     =============================================
     [..]
       Below the list of most used macros in CMU driver.

      (+) CMU_HRC_SEL_BY_SW():         HRC clock config by software.
      (+) CMU_HRC_SEL_BY_CFGW():       HRC clock config by CFG Word.
      (+) CMU_HRC_DIV_1MHZ_ENABLE():   Enable HRC divider to 1MHz.
      (+) CMU_HRC_DIV_1MHZ_DISABLE():  Disable HRC divider to 1MHz.
      (+) CMU_HOSC_DIV_1MHZ_ENABLE():  Enable HOSC divider to 1MHz.
      (+) CMU_HOSC_DIV_1MHZ_DISABLE(): Disable HOSC divider to 1MHz.
      (+) CMU_LOSC_ENABLE():           Enable outer low crystal(32768Hz).
      (+) CMU_LOSC_DISABLE():          Disable outer low crystal(32768Hz).
      (+) CMU_LRC_ENABLE():            Enable LRC(32768Hz).
      (+) CMU_LRC_DISABLE():           Disable LRC(32768Hz).
      (+) CMU_ULRC_ENABLE():           Enable ULRC(10KHz).
      (+) CMU_ULRC_DISABLE():          Disable ULRC(10KHz).
      (+) CMU_LP_LRC_ENABLE():         Enable low power LRC(32768Hz).
      (+) CMU_LP_LRC_DISABLE():        Disable low power LRC(32768Hz).
      (+) CMU_LP_LOSC_ENABLE():        Enable low power LOSC(32768Hz).
      (+) CMU_LP_LOSC_DISABLE():       Disable low power LOSC(32768Hz).
      (+) CMU_LP_HRC_ENABLE():         Enable low power HRC(4MHz).
      (+) CMU_LP_HRC_DISABLE():        Disable low power HRC(4MHz).
      (+) CMU_LP_HOSC_ENABLE():        Enable low power HOSC(1MHz -- 24MHz).
      (+) CMU_LP_HOSC_DISABLE():       Disable low power HOSC(1MHz -- 24MHz).

     [..]
      (@) You can refer to the CMU driver header file for used the macros

  @endverbatim
  ******************************************************************************
  */

#include "ald_conf.h"

/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup CMU CMU
  * @brief CMU module driver
  * @{
  */

/**
  * @defgroup CMU_Private_Variables CMU Private Variables
  * @{
  */
uint32_t __system_clock  = 4000000U;
/**
  * @}
  */

/** @defgroup CMU_Private_Functions CMU Private Functions
  * @{
  */

/**
  * @brief  Update the current system clock. This function
  *         will be invoked, when system clock has changed.
  * @param  clock: The new clock.
  * @retval None
  */
static void cmu_clock_update(uint32_t clock)
{
    __system_clock = clock;

    if (clock > 1000000)
        ald_tick_init(TICK_INT_PRIORITY);

    return;
}

/**
  * @brief  CMU module interrupt handler
  * @retval None
  */
void ald_cmu_irq_handler(void)
{
    /* HOSC stop */
    if (READ_BIT(CMU->HOSMCR, CMU_HOSMCR_STPIF_MSK) && READ_BIT(CMU->HOSMCR, CMU_HOSMCR_STPIE_MSK))
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->HOSMCR, CMU_HOSMCR_STPIF_MSK);
        SYSCFG_LOCK();

        if ((READ_BIT(CMU->HOSMCR, CMU_HOSMCR_CLKS_MSK))
                && ((READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) == 1)
                    || ((READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) == 5))))
            cmu_clock_update(4000000);

        ald_cmu_irq_cbk(ALD_CMU_HOSC_STOP);
    }

    /* HOSC start */
    if (READ_BIT(CMU->HOSMCR, CMU_HOSMCR_STRIF_MSK) && READ_BIT(CMU->HOSMCR, CMU_HOSMCR_STRIE_MSK))
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->HOSMCR, CMU_HOSMCR_STRIF_MSK);
        SYSCFG_LOCK();

        if (!(READ_BIT(CMU->HOSMCR, CMU_HOSMCR_CLKS_MSK))
                && ((READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) == 5)))
            cmu_clock_update((READ_BITS(CMU->HOSCCFG, CMU_HOSCCFG_FREQ_MSK, CMU_HOSCCFG_FREQ_POSS) + 1) * 1000000);

        ald_cmu_irq_cbk(ALD_CMU_HOSC_START);
    }

    /* LOSC stop */
    if (READ_BIT(CMU->LOSMCR, CMU_LOSMCR_STPIF_MSK) && READ_BIT(CMU->LOSMCR, CMU_LOSMCR_STPIE_MSK))
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->LOSMCR, CMU_LOSMCR_STPIF_MSK);
        SYSCFG_LOCK();
        ald_cmu_irq_cbk(ALD_CMU_LOSC_STOP);
    }

    /* LOSC start */
    if (READ_BIT(CMU->LOSMCR, CMU_LOSMCR_STRIF_MSK) && READ_BIT(CMU->LOSMCR, CMU_LOSMCR_STRIE_MSK))
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->LOSMCR, CMU_LOSMCR_STRIF_MSK);
        SYSCFG_LOCK();
        ald_cmu_irq_cbk(ALD_CMU_LOSC_START);
    }

    /* PLL lose */
    if (READ_BIT(CMU->PULMCR, CMU_PULMCR_ULKIF_MSK) && READ_BIT(CMU->PULMCR, CMU_PULMCR_ULKIE_MSK))
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->PULMCR, CMU_PULMCR_ULKIF_MSK);
        SYSCFG_LOCK();

        if (READ_BIT(CMU->PULMCR, CMU_PULMCR_CLKS_MSK)
                && ((READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) == 1)
                    || ((READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) == 5))))
            cmu_clock_update(4000000);

        ald_cmu_irq_cbk(ALD_CMU_PLL_UNLOCK);
    }

    return;
}
/**
  * @}
  */

/** @defgroup CMU_Public_Functions CMU Public Functions
  * @{
  */

/** @defgroup CMU_Public_Functions_Group1 System clock configuration
  * @brief    System clock configuration functions
  *
  * @verbatim
  ==============================================================================
              ##### System clock Configuration functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Configure system clock using default parameters.
      (+) Configure system clock using specified parameters.
      (+) Configure PLL using specified parameters.
      (+) Get system clock.

    @endverbatim
  * @{
  */

/**
  * @brief  Configure system clock using default.
  *         Select CMU_CLOCK_HRC(4MHz) as system clock and
  *         enable CMU_CLOCK_LRC(32768Hz).
  * @retval The status of ALD.
  */
ald_status_t ald_cmu_clock_config_default(void)
{
    uint32_t cnt = 4000, tmp;

    SYSCFG_UNLOCK();

    /* Select HRC */
    MODIFY_REG(CMU->CSR, CMU_CSR_SYS_CMD_MSK, ALD_CMU_CLOCK_HRC << CMU_CSR_SYS_CMD_POSS);

    /* Wait system clock switch */
    while (READ_BIT(CMU->CSR, CMU_CSR_SYS_RDYN_MSK) && (--cnt));

    if (READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) != ALD_CMU_CLOCK_HRC)
    {
        SYSCFG_LOCK();
        return ALD_ERROR;
    }

    WRITE_REG(CMU->CFGR, 0x0);  /* Select 4MHz */

    /* Enable HRC/LRC */
    tmp = READ_REG(CMU->CLKENR);
    SET_BIT(tmp, CMU_CLKENR_HRCEN_MSK | CMU_CLKENR_LRCEN_MSK);
    WRITE_REG(CMU->CLKENR, tmp);

    SYSCFG_LOCK();

    return ALD_OK;
}

/**
  * @brief  Configure system clock using specified parameters
  * @param  clk: The parameter can be one of the following:
  *           @arg @ref CMU_CLOCK_HRC  4MHz
  *           @arg @ref CMU_CLOCK_LRC  32768Hz
  *           @arg @ref CMU_CLOCK_LOSC 32768Hz
  *           @arg @ref CMU_CLOCK_PLL One of @ref cmu_pll_output_t
  *           @arg @ref CMU_CLOCK_HOSC 1MHz -- 4MHz
  * @param  clock: The clock which will be set. the value depends
  *         on the parameter of clk.
  * @retval The status of ALD.
  */
ald_status_t ald_cmu_clock_config(ald_cmu_clock_t clk, uint32_t clock)
{
    uint32_t cnt = 4000;

    assert_param(IS_CMU_CLOCK(clk));
    SYSCFG_UNLOCK();

    switch (clk)
    {
        case ALD_CMU_CLOCK_HRC:
            assert_param(clock == 4000000);

            MODIFY_REG(CMU->CSR, CMU_CSR_SYS_CMD_MSK, ALD_CMU_CLOCK_HRC << CMU_CSR_SYS_CMD_POSS);

            while (READ_BIT(CMU->CSR, CMU_CSR_SYS_RDYN_MSK) && (--cnt));

            if (READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) != ALD_CMU_CLOCK_HRC)
            {
                SYSCFG_LOCK();
                return ALD_ERROR;
            }

            SET_BIT(CMU->CLKENR, CMU_CLKENR_HRCEN_MSK);

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_HRCACT_MSK))) && (--cnt));

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_HRCRDY_MSK))) && (--cnt));

            cmu_clock_update(clock);
            break;

        case ALD_CMU_CLOCK_LRC:
            /* Close SysTick interrupt in lower clock */
            SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

            MODIFY_REG(CMU->CSR, CMU_CSR_SYS_CMD_MSK, ALD_CMU_CLOCK_LRC << CMU_CSR_SYS_CMD_POSS);

            while (READ_BIT(CMU->CSR, CMU_CSR_SYS_RDYN_MSK) && (--cnt));

            if (READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) != ALD_CMU_CLOCK_LRC)
            {
                SYSCFG_LOCK();
                return ALD_ERROR;
            }

            SET_BIT(CMU->CLKENR, CMU_CLKENR_LRCEN_MSK);

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_LRCACT_MSK))) && (--cnt));

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_LRCRDY_MSK))) && (--cnt));

            cmu_clock_update(32768);
            break;

        case ALD_CMU_CLOCK_LOSC:
            /* Close SysTick interrupt in lower clock */
            SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

            MODIFY_REG(CMU->CSR, CMU_CSR_SYS_CMD_MSK, ALD_CMU_CLOCK_LOSC << CMU_CSR_SYS_CMD_POSS);

            while (READ_BIT(CMU->CSR, CMU_CSR_SYS_RDYN_MSK) && (--cnt));

            if (READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) != ALD_CMU_CLOCK_LOSC)
            {
                SYSCFG_LOCK();
                return ALD_ERROR;
            }

            SET_BIT(CMU->CLKENR, CMU_CLKENR_LOSCEN_MSK);

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_LOSCACT_MSK))) && (--cnt));

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_LOSCRDY_MSK))) && (--cnt));

            cmu_clock_update(32768);
            break;

        case ALD_CMU_CLOCK_PLL:
            /* Bypass clock filter */
            MODIFY_REG(CMU->CSR, CMU_CSR_CFT_CMD_MSK, 0xAA << CMU_CSR_CFT_CMD_POSS);
            /* HRC switch to PLL clock */
            MODIFY_REG(CMU->CSR, CMU_CSR_SYS_CMD_MSK, ALD_CMU_CLOCK_PLL << CMU_CSR_SYS_CMD_POSS);

            /* Wait for switch success */
            while (READ_BIT(CMU->CSR, CMU_CSR_SYS_RDYN_MSK) && (--cnt));

            /* Confirm that the current system clock is PLL clock */
            if (READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) != ALD_CMU_CLOCK_PLL)
            {
                SYSCFG_LOCK();
                return ALD_ERROR;
            }

            if (clock == 72000000)
            {
                ald_cmu_div_config(ALD_CMU_PCLK_2, ALD_CMU_DIV_2);
            }

            if (clock == 64000000)
            {
                ald_cmu_div_config(ALD_CMU_PCLK_2, ALD_CMU_DIV_2);
            }

            if (clock == 56000000)
            {
                ald_cmu_div_config(ALD_CMU_PCLK_2, ALD_CMU_DIV_2);
            }

            if (clock == 48000000)
            {
                ald_cmu_div_config(ALD_CMU_PCLK_2, ALD_CMU_DIV_2);
            }

            /* Enable pll lost lock manager */
            SET_BIT(CMU->PULMCR, CMU_PULMCR_EN_MSK);
            /* Switch to HRC when pll lost lock */
            MODIFY_REG(CMU->PULMCR, CMU_PULMCR_MODE_MSK, 0x3 << CMU_PULMCR_MODE_POSS);

            cmu_clock_update(clock);
            break;

        case ALD_CMU_CLOCK_HOSC:
            assert_param(clock <= 16000000);

            MODIFY_REG(CMU->CSR, CMU_CSR_SYS_CMD_MSK, ALD_CMU_CLOCK_HOSC << CMU_CSR_SYS_CMD_POSS);

            while (READ_BIT(CMU->CSR, CMU_CSR_SYS_RDYN_MSK) && (--cnt));

            if (READ_BITS(CMU->CSR, CMU_CSR_SYS_STU_MSK, CMU_CSR_SYS_STU_POSS) != ALD_CMU_CLOCK_HOSC)
            {
                SYSCFG_LOCK();
                return ALD_ERROR;
            }

            SET_BIT(CMU->CLKENR, CMU_CLKENR_HOSCEN_MSK);
            MODIFY_REG(CMU->HOSCCFG, CMU_HOSCCFG_FREQ_MSK, clock / 1000000 - 1);

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_HOSCACT_MSK))) && (--cnt));

            cnt = 4000;

            while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_HOSCRDY_MSK))) && (--cnt));

            cmu_clock_update(clock);
            break;

        default:
            break;
    }

    SYSCFG_LOCK();

    return ALD_OK;
}

/**
  * @brief  Configure PLL using specified parameters.
  * @param  input: The input clock type.
  * @param  output: The output clock which can be 48MHz/56MHz/64MHz/72MHz.
  * @retval None
  */
void ald_cmu_pll_config(ald_cmu_pll_input_t input, ald_cmu_pll_output_t output)
{
    uint32_t cnt = 4000;

    assert_param(IS_CMU_PLL_INPUT(input));
    assert_param(IS_CMU_PLL_OUTPUT(output));

    SYSCFG_UNLOCK();

    if (input == ALD_CMU_PLL_INPUT_HRC)
    {
        SET_BIT(CMU->CLKENR, CMU_CLKENR_HRCEN_MSK);
    }
    else
    {
        SET_BIT(CMU->CLKENR, CMU_CLKENR_HOSCEN_MSK);
        cnt = 20000;

        while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_HOSCRDY_MSK))) && (--cnt));
    }

    /* PLL source clock */
    MODIFY_REG(CMU->PLLCFG, CMU_PLLCFG_PLLRFS_MSK, input << CMU_PLLCFG_PLLRFS_POSS);
    /* PLL multiplied to 48Mhz */
    MODIFY_REG(CMU->PLLCFG, CMU_PLLCFG_PLLOS_MSK, ALD_CMU_PLL_OUTPUT_48M << CMU_PLLCFG_PLLOS_POSS);
    /* Enable PLL */
    SET_BIT(CMU->CLKENR, CMU_CLKENR_PLLEN_MSK);
    /* Wait pll lock */
    cnt = 20000;

    while ((READ_BIT(CMU->PLLCFG, CMU_PLLCFG_PLLLCKN_MSK)) && (--cnt));

    /* PLL multiplied to target frequency */
    MODIFY_REG(CMU->PLLCFG, CMU_PLLCFG_PLLOS_MSK, output << CMU_PLLCFG_PLLOS_POSS);
    /* Wait pll lock */
    cnt = 20000;

    while ((READ_BIT(CMU->PLLCFG, CMU_PLLCFG_PLLLCKN_MSK)) && (--cnt));

    /* Wait pll clock ready */
    cnt = 20000;

    while ((!(READ_BIT(CMU->CLKSR, CMU_CLKSR_PLLRDY_MSK))) && (--cnt));

    SYSCFG_LOCK();
}

/**
  * @brief  Gets current system clock.
  * @retval The value of system clock.
  */
uint32_t ald_cmu_get_clock(void)
{
    return __system_clock;
}

/**
  * @}
  */

/** @defgroup CMU_Public_Functions_Group2 BUS division control
  * @brief    BUS division control functions
  *
  * @verbatim
  ==============================================================================
              ##### BUS division control functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Configure the bus division.
      (+) Get ahb1 clock.
      (+) Get sys bus clock.
      (+) Get apb1 clock.
      (+) Get apb2 clock.

    @endverbatim
  * @{
  */

/**
  * @brief  Configure the bus division.
  * @param  bus: The type of bus:
  *          @arg CMU_HCLK_1
  *          @arg CMU_SYS
  *          @arg CMU_PCLK_1
  *          @arg CMU_PCLK_2
  * @param  div: The value of divider.
  * @retval None
  */
void ald_cmu_div_config(ald_cmu_bus_t bus, ald_cmu_div_t div)
{
    assert_param(IS_CMU_BUS(bus));
    assert_param(IS_CMU_DIV(div));

    SYSCFG_UNLOCK();

    switch (bus)
    {
        case ALD_CMU_SYS:
            MODIFY_REG(CMU->CFGR, CMU_CFGR_SYSDIV_MSK, div << CMU_CFGR_SYSDIV_POSS);

            if ((__system_clock >> div) <= 1000000)
            {
                /* Close SysTick interrupt in lower clock */
                SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
            }
            else
            {
                ald_tick_init(TICK_INT_PRIORITY);
            }

            break;

        case ALD_CMU_PCLK_1:
            MODIFY_REG(CMU->CFGR, CMU_CFGR_PCLK1DIV_MSK, div << CMU_CFGR_PCLK1DIV_POSS);
            break;

        case ALD_CMU_PCLK_2:
            MODIFY_REG(CMU->CFGR, CMU_CFGR_PCLK2DIV_MSK, div << CMU_CFGR_PCLK2DIV_POSS);
            break;

        default:
            break;
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Get SYS clock
  * @retval The value of SYS clock
  */
uint32_t ald_cmu_get_sys_clock(void)
{
    uint32_t sys_div = READ_BITS(CMU->CFGR, CMU_CFGR_SYSDIV_MSK, CMU_CFGR_SYSDIV_POSS);

    return __system_clock >> sys_div;
}

/**
  * @brief  Get APB1 clock.
  * @retval The value of APB1 clock.
  */
uint32_t ald_cmu_get_pclk1_clock(void)
{
    uint32_t sys_div  = READ_BITS(CMU->CFGR, CMU_CFGR_SYSDIV_MSK, CMU_CFGR_SYSDIV_POSS);
    uint32_t apb1_div = READ_BITS(CMU->CFGR, CMU_CFGR_PCLK1DIV_MSK, CMU_CFGR_PCLK1DIV_POSS);

    return (__system_clock >> sys_div) >> apb1_div;
}

/**
  * @brief  Get APB2 clock.
  * @retval The value of APB2 clock.
  */
uint32_t ald_cmu_get_pclk2_clock(void)
{
    uint32_t sys_div  = READ_BITS(CMU->CFGR, CMU_CFGR_SYSDIV_MSK, CMU_CFGR_SYSDIV_POSS);
    uint32_t apb2_div = READ_BITS(CMU->CFGR, CMU_CFGR_PCLK2DIV_MSK, CMU_CFGR_PCLK2DIV_POSS);

    return (__system_clock >> sys_div) >> apb2_div;
}
/**
  * @}
  */

/** @defgroup CMU_Public_Functions_Group3 Clock safe configure
  * @brief    Clock safe configure functions
  *
  * @verbatim
  ==============================================================================
              ##### Clock safe configure functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Enable/Disable outer high crystal safe mode.
      (+) Enable/Disable outer low crystal safe mode.
      (+) Enable/Disable PLL safe mode.
      (+) Interrupt callback function.

    @endverbatim
  * @{
  */

/**
  * @brief  Enable/Disable outer high crystal safe mode.
  * @param  clock: the value of outer crystal frequency.
  * @param  status: The new status.
  * @retval None
  */
void ald_cmu_hosc_safe_config(ald_cmu_hosc_range_t clock, type_func_t status)
{
    assert_param(IS_CMU_HOSC_RANGE(clock));
    assert_param(IS_FUNC_STATE(status));

    SYSCFG_UNLOCK();

    if (status)
    {
        SET_BIT(CMU->HOSMCR, CMU_HOSMCR_STPIF_MSK);
        MODIFY_REG(CMU->HOSMCR, CMU_HOSMCR_FRQS_MSK, clock << CMU_HOSMCR_FRQS_POSS);
        SET_BIT(CMU->HOSMCR, CMU_HOSMCR_EN_MSK);
        SET_BIT(CMU->HOSMCR, CMU_HOSMCR_STPIE_MSK);

        ald_mcu_irq_config(CMU_IRQn, 3, 3, ENABLE);
    }
    else
    {
        CLEAR_BIT(CMU->HOSMCR, CMU_HOSMCR_EN_MSK);
        CLEAR_BIT(CMU->HOSMCR, CMU_HOSMCR_STPIE_MSK);

        if (READ_BIT(CMU->LOSMCR, CMU_LOSMCR_EN_MSK) == 0 && READ_BIT(CMU->PULMCR, CMU_PULMCR_EN_MSK) == 0)
            ald_mcu_irq_config(CMU_IRQn, 3, 3, DISABLE);
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Enable/Disable outer low crystal safe mode.
  * @param  status: The new status.
  * @retval None
  */
void ald_cmu_losc_safe_config(type_func_t status)
{
    assert_param(IS_FUNC_STATE(status));
    SYSCFG_UNLOCK();

    if (status)
    {
        SET_BIT(CMU->LOSMCR, CMU_LOSMCR_STPIF_MSK);
        SET_BIT(CMU->LOSMCR, CMU_LOSMCR_EN_MSK);
        SET_BIT(CMU->LOSMCR, CMU_LOSMCR_STPIE_MSK);

        ald_mcu_irq_config(CMU_IRQn, 3, 3, ENABLE);
    }
    else
    {
        CLEAR_BIT(CMU->LOSMCR, CMU_LOSMCR_EN_MSK);
        CLEAR_BIT(CMU->LOSMCR, CMU_LOSMCR_STPIE_MSK);

        if (READ_BIT(CMU->HOSMCR, CMU_HOSMCR_EN_MSK) == 0 && READ_BIT(CMU->PULMCR, CMU_PULMCR_EN_MSK) == 0)
            ald_mcu_irq_config(CMU_IRQn, 3, 3, DISABLE);
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Enable/Disable PLL safe mode.
  * @param  status: The new status.
  * @retval None
  */
void ald_cmu_pll_safe_config(type_func_t status)
{
    assert_param(IS_FUNC_STATE(status));
    SYSCFG_UNLOCK();

    if (status)
    {
        SET_BIT(CMU->PULMCR, CMU_PULMCR_ULKIF_MSK);
        MODIFY_REG(CMU->PULMCR, CMU_PULMCR_MODE_MSK, 2 << CMU_PULMCR_MODE_POSS);
        SET_BIT(CMU->PULMCR, CMU_PULMCR_EN_MSK);
        SET_BIT(CMU->PULMCR, CMU_PULMCR_ULKIE_MSK);

        ald_mcu_irq_config(CMU_IRQn, 3, 3, ENABLE);
    }
    else
    {
        CLEAR_BIT(CMU->PULMCR, CMU_PULMCR_EN_MSK);
        CLEAR_BIT(CMU->PULMCR, CMU_PULMCR_ULKIE_MSK);

        if (READ_BIT(CMU->HOSMCR, CMU_HOSMCR_EN_MSK) == 0 && READ_BIT(CMU->LOSMCR, CMU_LOSMCR_EN_MSK) == 0)
            ald_mcu_irq_config(CMU_IRQn, 3, 3, DISABLE);
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Get current clock source.
  * @param  type: Type of source: HOSC/LOSC/PLL.
  * @retval Status:
  *              - 0: Current clock is HOSC, LOSC or PLL
  *              - 1: Current clock is HRC, LRC or HRC
  */
uint32_t ald_cmu_current_clock_source_get(ald_cmu_clock_safe_type_t type)
{
    assert_param(IS_CMU_SAFE_CLOCK_TYPE(type));

    if (type == ALD_CMU_SAFE_CLK_HOSC)
        return READ_BITS(CMU->HOSMCR, CMU_HOSMCR_CLKS_MSK, CMU_HOSMCR_CLKS_POS);
    else if (type == ALD_CMU_SAFE_CLK_LOSC)
        return READ_BITS(CMU->LOSMCR, CMU_LOSMCR_CLKS_MSK, CMU_LOSMCR_CLKS_POS);
    else
        return READ_BITS(CMU->PULMCR, CMU_PULMCR_CLKS_MSK, CMU_PULMCR_CLKS_POS);
}

/**
  * @brief  Get clock state.
  * @param  sr: The state type, see @ref cmu_clock_state_t.
  * @retval SET/RESET
  */
flag_status_t ald_cmu_get_clock_state(ald_cmu_clock_state_t sr)
{
    assert_param(IS_CMU_CLOCK_STATE(sr));

    if (READ_BIT(CMU->CLKSR, sr))
        return SET;

    return RESET;
}

/**
  * @brief  Interrupt callback function.
  * @note   This function is declared as __weak to be overwritten in case of other
  *         implementations in user file.
  * @retval None
  */
__weak void ald_cmu_irq_cbk(ald_cmu_security_t se)
{
    return;
}
/**
  * @}
  */

/** @defgroup CMU_Public_Functions_Group4 Clock output configure
  * @brief    Clock output configure functions
  *
  * @verbatim
  ==============================================================================
              ##### Clock output configure functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Configure the high-speed clock output.
      (+) Configure the low-speed clock output.

    @endverbatim
  * @{
  */

/**
  * @brief  Configure the high-speed clock output.
  * @param  sel: Select the source:
  *           @arg CMU_OUTPUT_HIGH_SEL_HOSC
  *           @arg CMU_OUTPUT_HIGH_SEL_LOSC
  *           @arg CMU_OUTPUT_HIGH_SEL_HRC
  *           @arg CMU_OUTPUT_HIGH_SEL_LRC
  *           @arg CMU_OUTPUT_HIGH_SEL_HOSM
  *           @arg CMU_OUTPUT_HIGH_SEL_PLL
  *           @arg CMU_OUTPUT_HIGH_SEL_SYSCLK
  * @param  div: The value of divider:
  *           @arg CMU_OUTPUT_DIV_1
  *           @arg CMU_OUTPUT_DIV_2
  *           @arg CMU_OUTPUT_DIV_4
  *           @arg CMU_OUTPUT_DIV_8
  *           @arg CMU_OUTPUT_DIV_16
  *           @arg CMU_OUTPUT_DIV_32
  *           @arg CMU_OUTPUT_DIV_64
  *           @arg CMU_OUTPUT_DIV_128
  * @param  status: The new status.
  * @retval None
  */
void ald_cmu_output_high_clock_config(ald_cmu_output_high_sel_t sel,
                                      ald_cmu_output_high_div_t div, type_func_t status)
{
    assert_param(IS_CMU_OUTPUT_HIGH_SEL(sel));
    assert_param(IS_CMU_OUTPUT_HIGH_DIV(div));
    assert_param(IS_FUNC_STATE(status));

    SYSCFG_UNLOCK();

    if (status)
    {
        MODIFY_REG(CMU->CLKOCR, CMU_CLKOCR_HSCOS_MSK, sel << CMU_CLKOCR_HSCOS_POSS);
        MODIFY_REG(CMU->CLKOCR, CMU_CLKOCR_HSCODIV_MSK, div << CMU_CLKOCR_HSCODIV_POSS);
        SET_BIT(CMU->CLKOCR, CMU_CLKOCR_HSCOEN_MSK);
    }
    else
    {
        CLEAR_BIT(CMU->CLKOCR, CMU_CLKOCR_HSCOEN_MSK);
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Configure the low-speed clock output.
  * @param  sel: Select the source:
  *           @arg CMU_OUTPUT_LOW_SEL_LOSC
  *           @arg CMU_OUTPUT_LOW_SEL_LRC
  *           @arg CMU_OUTPUT_LOW_SEL_LOSM
  *           @arg CMU_OUTPUT_LOW_SEL_BUZZ
  *           @arg CMU_OUTPUT_LOW_SEL_ULRC
  * @param  status: The new status.
  * @retval None
  */
void ald_cmu_output_low_clock_config(ald_cmu_output_low_sel_t sel, type_func_t status)
{
    assert_param(IS_CMU_OUTPUT_LOW_SEL(sel));
    assert_param(IS_FUNC_STATE(status));

    SYSCFG_UNLOCK();

    if (status)
    {
        MODIFY_REG(CMU->CLKOCR, CMU_CLKOCR_LSCOS_MSK, sel << CMU_CLKOCR_LSCOS_POSS);
        SET_BIT(CMU->CLKOCR, CMU_CLKOCR_LSCOEN_MSK);
    }
    else
    {
        CLEAR_BIT(CMU->CLKOCR, CMU_CLKOCR_LSCOEN_MSK);
    }

    SYSCFG_LOCK();
    return;
}
/**
  * @}
  */

/** @defgroup CMU_Public_Functions_Group5 Peripheral Clock configure
  * @brief    Peripheral clock configure functions
  *
  * @verbatim
  ==============================================================================
              ##### Peripheral clock configure functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Configure buzz clock.
      (+) Select lptim0 clock source.
      (+) Select lpuart0 clock source.
      (+) Enable/Disable peripheral clock.

    @endverbatim
  * @{
  */

/**
  * @brief  Select lptim0 clock source.
  * @param  clock: The clock source:
  *           @arg CMU_LP_PERH_CLOCK_SEL_PCLK2
  *           @arg CMU_LP_PERH_CLOCK_SEL_PLL
  *           @arg CMU_LP_PERH_CLOCK_SEL_HRC
  *           @arg CMU_LP_PERH_CLOCK_SEL_HOSC
  *           @arg CMU_LP_PERH_CLOCK_SEL_LRC
  *           @arg CMU_LP_PERH_CLOCK_SEL_LOSC
  *           @arg CMU_LP_PERH_CLOCK_SEL_ULRC
  *           @arg CMU_LP_PERH_CLOCK_SEL_HRC_1M
  *           @arg CMU_LP_PERH_CLOCK_SEL_HOSC_1M
  *           @arg CMU_LP_PERH_CLOCK_SEL_LOSM
  *           @arg CMU_LP_PERH_CLOCK_SEL_HOSM
  * @retval None
  */
void ald_cmu_lptim0_clock_select(ald_cmu_lp_perh_clock_sel_t clock)
{
    assert_param(IS_CMU_LP_PERH_CLOCK_SEL(clock));

    SYSCFG_UNLOCK();
    MODIFY_REG(CMU->PERICR, CMU_PERICR_LPTIM0CS_MSK, clock << CMU_PERICR_LPTIM0CS_POSS);
    SYSCFG_LOCK();

    return;
}

/**
  * @brief  Select lpuart0 clock source.
  * @param  clock: The clock source:
  *           @arg CMU_LP_PERH_CLOCK_SEL_PCLK2
  *           @arg CMU_LP_PERH_CLOCK_SEL_PLL
  *           @arg CMU_LP_PERH_CLOCK_SEL_HRC
  *           @arg CMU_LP_PERH_CLOCK_SEL_HOSC
  *           @arg CMU_LP_PERH_CLOCK_SEL_LRC
  *           @arg CMU_LP_PERH_CLOCK_SEL_LOSC
  *           @arg CMU_LP_PERH_CLOCK_SEL_ULRC
  *           @arg CMU_LP_PERH_CLOCK_SEL_HRC_1M
  *           @arg CMU_LP_PERH_CLOCK_SEL_HOSC_1M
  *           @arg CMU_LP_PERH_CLOCK_SEL_LOSM
  *           @arg CMU_LP_PERH_CLOCK_SEL_HOSM
  * @retval None
  */
void ald_cmu_lpuart0_clock_select(ald_cmu_lp_perh_clock_sel_t clock)
{
    assert_param(IS_CMU_LP_PERH_CLOCK_SEL(clock));

    SYSCFG_UNLOCK();
    MODIFY_REG(CMU->PERICR, CMU_PERICR_LPUART0CS_MSK, clock << CMU_PERICR_LPUART0CS_POSS);
    SYSCFG_LOCK();

    return;
}

/**
  * @brief  Configure the RTC source.
  * @param  sel: RTC source type.
  * @retval None
  */
void ald_cmu_rtc_clock_select(ald_cmu_rtc_clock_sel_t sel)
{
    uint32_t cnt = 2000;

    assert_param(IS_CMU_RTC_CLOCK_SEL(sel));

    RTC_UNLOCK();
    MODIFY_REG(RTC->PSR, RTC_PSR_RTCCS_MSK, sel << RTC_PSR_RTCCS_POSS);
    RTC_LOCK();

    if (sel == ALD_CMU_RTC_SEL_LOSM)
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->CLKENR, CMU_CLKENR_LOSCEN_MSK);
        SET_BIT(CMU->LOSMCR, CMU_LOSMCR_EN_MSK);
        SET_BIT(CMU->LPENR, CMU_LPENR_LOSCEN_MSK);
        SYSCFG_LOCK();

        cnt = 1000000;

        while ((!(CMU->CLKSR & CMU_CLKSR_LOSCRDY_MSK)) && (cnt--));
    }

    if (sel == ALD_CMU_RTC_SEL_HRC_DIV_1M)
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->CLKENR, CMU_CLKENR_HRC1MEN_MSK);
        SYSCFG_LOCK();
    }

    if (sel == ALD_CMU_RTC_SEL_HOSC_DIV_1M)
    {
        SYSCFG_UNLOCK();
        SET_BIT(CMU->CLKENR, CMU_CLKENR_HOSCEN_MSK);
        SET_BIT(CMU->CLKENR, CMU_CLKENR_HOSC1MEN_MSK);
        SYSCFG_LOCK();

        cnt = 1000;

        while (cnt--);

        cnt = 1000000;

        while ((!(CMU->CLKSR & CMU_CLKSR_HOSCRDY_MSK)) && (cnt--));
    }

    return;
}


/**
  * @brief  Enable/Disable peripheral clock.
  * @param  perh: The type of peripheral, you can see @ref cmu_perh_t
  * @param  status: The new status.
  * @retval None
  */
void ald_cmu_perh_clock_config(ald_cmu_perh_t perh, type_func_t status)
{
    uint32_t idx, pos;

    assert_param(IS_CMU_PERH(perh));
    assert_param(IS_FUNC_STATE(status));

    SYSCFG_UNLOCK();

    if (perh == ALD_CMU_PERH_ALL)
    {
        if (status)
        {
            WRITE_REG(CMU->AHB1ENR, ~0);
            WRITE_REG(CMU->AHB2ENR, ~0);
            WRITE_REG(CMU->APB1ENR, ~0);
            WRITE_REG(CMU->APB2ENR, ~0);
        }
        else
        {
            WRITE_REG(CMU->AHB1ENR, 0);
            WRITE_REG(CMU->AHB2ENR, 0);
            WRITE_REG(CMU->APB1ENR, 0);
            WRITE_REG(CMU->APB2ENR, 0);
        }

        SYSCFG_LOCK();
        return;
    }

    idx = ((uint32_t)perh >> 26) & 0x7;
    pos = perh & ~(0x7 << 26);

    if (status)
    {
        switch (idx)
        {
            case 0:
                SET_BIT(CMU->AHB1ENR, pos);
                break;

            case 1:

                SET_BIT(CMU->AHB2ENR, pos);
                break;

            case 2:
                SET_BIT(CMU->APB1ENR, pos);
                break;

            case 4:
                SET_BIT(CMU->APB2ENR, pos);
                break;

            default:
                break;
        }
    }
    else
    {
        switch (idx)
        {
            case 0:
                CLEAR_BIT(CMU->AHB1ENR, pos);
                break;

            case 1:
                CLEAR_BIT(CMU->AHB2ENR, pos);
                break;

            case 2:
                CLEAR_BIT(CMU->APB1ENR, pos);
                break;

            case 4:
                CLEAR_BIT(CMU->APB2ENR, pos);
                break;

            default:
                break;
        }
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Select stop1 clock.
  * @param  clock: See @ref cmu_stop1_clock_t
  * @retval None
  */
void ald_cmu_stop1_clock_sel(ald_cmu_stop1_clock_t clock)
{
    assert_param(IS_CMU_STOP1_CLOCK(clock));

    SYSCFG_UNLOCK();
    MODIFY_REG(CMU->LPENR, CMU_LPENR_STOP1CS_MSK, clock << CMU_LPENR_STOP1CS_POSS);
    SYSCFG_LOCK();
    return;
}
/**
 * @}
 */
/**
 * @}
 */

/**
 * @}
 */

/**
 * @}
 */
