/**
 *********************************************************************************
 *
 * @file    ald_cache.c
 * @brief   Cache module driver.
 *
 * @version V1.0
 * @date    12 Mar 2024
 * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
 */
#include "ald_conf.h"

/** @addtogroup ES32FXXX_ALD
 * @{
 */

/** @defgroup Cache Cache
 * @brief Cache module driver
 * @{
 */
#ifdef ALD_CACHE


/** @defgroup Cache_Public_Functions Cache Public Functions
 * @brief Init Cache function
 * @{
 */
/**
 * @brief Initializes the Cache according to the specified parameters.
 * @param config: Pointer to ald_cache_config_t structure which contains
 *        the configuration parameters.
 * @retval None
 */
void ald_cache_init(ald_cache_config_t *config)
{
    assert_param(IS_CACHE_POWER_SEL(config->power));
    assert_param(IS_CACHE_INVALIDATE_SEL(config->invlidate));
    assert_param(IS_FUNC_STATE(config->statistic));
    assert_param(IS_FUNC_STATE(config->prefetch));

    MODIFY_REG(CACHE->CCR, CACHE_CCR_STATISTIC_EN_MSK, config->statistic << CACHE_CCR_STATISTIC_EN_POS);
    MODIFY_REG(CACHE->CCR, CACHE_CCR_SET_PREFETCH_MSK, config->prefetch << CACHE_CCR_SET_PREFETCH_POS);
    MODIFY_REG(CACHE->CCR, CACHE_CCR_SET_MAN_INV_MSK, config->invlidate << CACHE_CCR_SET_MAN_INV_POS);
    MODIFY_REG(CACHE->CCR, CACHE_CCR_SET_MAN_POW_MSK, config->power << CACHE_CCR_SET_MAN_POW_POS);
}

/**
 * @brief Enable the cache.
 * @retval None
 */
void ald_cache_enable(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_EN_MSK);
}

/**
 * @brief Disable the cache.
 * @retval None
 */
void ald_cache_disable(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_EN_MSK);
}

/**
 * @brief Get cache power or invalidating status.
 * @param flag: Specifies the cache status flag.
 *        This parameter can be one of the @ref ald_inv_pow_stat_t.
 * @retval Status:
 *         - 0: RESET
 *         - 1: SET
 */
flag_status_t ald_get_cache_inv_pow_status(ald_inv_pow_stat_t flag)
{
    assert_param(IS_CACHE_INV_POWER_FLAG(flag));

    if (READ_BIT(CACHE->SR, flag))
        return SET;

    return RESET;
}

/**
 * @brief Get the current status of the cache.
 * @retval Cache current status @ref ald_cache_stat_t
 */
ald_cache_stat_t ald_get_cache_status(void)
{
    return (ald_cache_stat_t)READ_BITS(CACHE->SR, CACHE_SR_CS_MSK, CACHE_SR_CS_POSS);
}

/**
  * @brief  Enable/disable the specified cache interrupts.
  * @param  it: Specifies the CACHE interrupt sources to be enabled or disabled.
  *         This parameter can be one of the @ref ald_cache_it_t.
  * @param  state: New state of the specified CACHE interrupts.
  *         This parameter can be:
  *           @arg ENABLE
  *           @arg DISABLE
  * @retval None
  */
void ald_cache_it_config(ald_cache_it_t it, type_func_t state)
{
    assert_param(IS_CACHE_IT_SEL(it));
    assert_param(IS_FUNC_STATE(state));

    if (state == ENABLE)
        CLEAR_BIT(CACHE->IRQMASK, it);
    else
        SET_BIT(CACHE->IRQMASK, it);
}

/**
  * @brief  Get the status of CACHE interrupt source.
  * @param  it: Specifies the CACHE interrupt source.
  *         This parameter can be one of the @ref ald_cache_flag_t.
  * @retval Status:
  *           - 0: RESET
  *           - 1: SET
  */
flag_status_t ald_cache_get_flag_status(ald_cache_flag_t flag)
{
    assert_param(IS_CACHE_IF_SEL(flag));

    if (READ_BIT(CACHE->IRQSTAT, flag))
        return SET;

    return RESET;
}

/**
  * @brief  Get the number of cache hits during cache look up.
  * @retval uint32_t cache hit count.
  */
uint32_t ald_cache_get_hits(void)
{
    return READ_REG(CACHE->CSHR);
}

/**
  * @brief  Get the number of cache misses during cache look up.
  * @retval uint32_t cache miss count.
  */
uint32_t ald_cache_get_miss_count(void)
{
    return READ_REG(CACHE->CSMR);
}

/**
  * @brief  Clear cache hits counter.
  * @retval None.
  */
void ald_cache_clear_hits(void)
{
    WRITE_REG(CACHE->CSHR, 0x0);
}

/**
  * @brief  Clear cache miss counter.
  * @retval None.
  */
void ald_cache_clear_miss_count(void)
{
    WRITE_REG(CACHE->CSMR, 0x0);
}

/**
  * @brief  Manual SRAM power request.
  * @retval None.
  */
void ald_cache_manual_pow(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_POW_REQ_MSK);
}

/**
  * @brief  Manual invalidate request..
  * @retval None.
  */
void ald_cache_manual_inv(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_EN_MSK);
    SET_BIT(CACHE->CCR, CACHE_CCR_INV_REQ_MSK);
    SET_BIT(CACHE->CCR, CACHE_CCR_EN_MSK);
}

/**
 * @}
 */

#endif /* ALD_WDT */
/**
 * @}
 */

/**
 * @}
 */
