/**
  ******************************************************************************
  * @file    ald_adc.c
  * @brief   This file provides firmware functions to manage the following
  *          functionalities of the Analog to Digital Convertor (ADC)
  *          peripheral:
  *           + Initialization functions
  *             ++ Initialization and Configuration of ADC
  *           + Operation functions
  *             ++ Start, stop, get result of conversions of normal
  *                group, using 3 possible modes: polling, interruption or DMA.
  *           + Control functions
  *             ++ Channels configuration on normal group
  *             ++ Channels configuration on insert group
  *             ++ Analog Watchdog configuration
  *           + State functions
  *             ++ ADC state machine management
  *             ++ Interrupts and flags management
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team.
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


#include "ald_conf.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup ADC ADC
  * @brief ADC module driver
  * @{
  */

#ifdef ALD_ADC

/** @addtogroup ADC_Private_Functions
  * @{
  */
#ifdef ALD_DMA
    static int8_t  bit6, bit7, bit8, bit9, bit10, bit11;
    static void adc_dma_normal_conv_cplt(void *arg);
#endif

volatile uint16_t __sys_vol = 0x0;
volatile int32_t gain_tense, offset_tense;
volatile uint32_t CFG_TRIM = 0x0;
volatile float temp_k = 0x0;

static float ald_get_temp_k(void);

/**
  * @}
  */

/** @defgroup ADC_Public_Functions ADC Public Functions
  * @{
  */

/** @defgroup ADC_Public_Functions_Group1 Initialization functions
  * @brief    Initialization and Configuration functions
  * @{
  */
/**
  * @brief  Get the system voltage
  * @param  hperh: Pointer to a adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status: 0 is success, others failed.
  */
static int adc_system_vol_get(ald_adc_handle_t *hperh)
{
    uint16_t i, max, min, sum, ret;
    uint32_t reg, tmp;
    ald_adc_nch_conf_t nm_config;

    /* Initialize adc */
    ADC_DISABLE(hperh);
    ald_adc_reset(hperh);
    MODIFY_REG(hperh->perh->CON1, ADC_CON1_ALIGN_MSK, ALD_ADC_DATAALIGN_RIGHT << ADC_CON1_ALIGN_POS);
    MODIFY_REG(hperh->perh->CON0, ADC_CON0_RSEL_MSK, ALD_ADC_CONV_BIT_12 << ADC_CON0_RSEL_POSS);
    CLEAR_BIT(hperh->perh->CON0, ADC_CON0_NCHDCEN_MSK);
    CLEAR_BIT(hperh->perh->CON0, ADC_CON0_ICHDCEN_MSK);
    MODIFY_REG(hperh->perh->CON1, ADC_CON1_CM_MSK, ENABLE << ADC_CON1_CM_POS);
    MODIFY_REG(hperh->perh->CON0, ADC_CON0_SCANEN_MSK, DISABLE << ADC_CON0_SCANEN_POS);

    reg = 0x10000000U;
    MODIFY_REG(reg, ADC_CCR_CKDIV_MSK, ALD_ADC_CKDIV_128 << ADC_CCR_CKDIV_POSS);
    MODIFY_REG(reg, ADC_CCR_VRNSEL_MSK, ALD_ADC_NEG_REF_VSS << ADC_CCR_VRNSEL_POS);
    MODIFY_REG(reg, ADC_CCR_VRPSEL_MSK, ALD_ADC_POS_REF_VDD << ADC_CCR_VRPSEL_POSS);

    CLEAR_BIT(reg, ADC_CCR_PMSEL_MSK);

    WRITE_REG(hperh->perh->CCR, reg);
    ADC_ENABLE(hperh);

    /* Initialize normal convert channel */
    nm_config.ch   = ALD_ADC_CHANNEL_16;
    nm_config.idx  = ALD_ADC_NCH_IDX_1;
    nm_config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(hperh, &nm_config);

    max = 0x0;
    min = 0xFFFF;
    sum = 0x0;
    ret = 0x0;

    for (i = 0; i < 3; ++i)
    {
        ald_adc_normal_start(hperh);

        if (ald_adc_normal_poll_for_conversion(hperh, 5000) == ALD_OK)
        {
            tmp = ald_adc_normal_get_value(hperh);
            max = (tmp > max) ? tmp : max;
            min = (tmp < min) ? tmp : min;
            sum = sum + tmp;
        }
        else
        {
            ret = 0x1;
        }
    }

    if (ret)
    {
        __sys_vol = 5000;
        return -1;
    }

    tmp = sum - max - min;
    __sys_vol = (1200 * 4095) / tmp;
    ADC_DISABLE(hperh);
    ald_adc_reset(hperh);

    if ((__sys_vol < 1500) || (__sys_vol > 6000))
        return -2;

    return 0;
}

#define GAIN_CAL      (*(int32_t *)(0x809B0))
#define OFFSET_CAL    (*(int32_t *)(0x809B4))
#define GAIN_CAL2      (*(int32_t *)(0x80AC0))
#define OFFSET_CAL2    (*(int32_t *)(0x80AC4))
#define GAIN5V_CAL    (*(int32_t *)(0x80BB0))
#define OFFSET5V_CAL  (*(int32_t *)(0x80BB4))
#define GAIN3V3_CAL   (*(int32_t *)(0x80BC0))
#define OFFSET3V3_CAL (*(int32_t *)(0x80BC4))
#define OFFSET5V_BAC  (*(int32_t *)(0x80BA0))
#define OFFSET3V3_BAC (*(int32_t *)(0x80BA4))
#define Cost_Ima      6
#define CHIP_ID       (*(uint32_t *)(0x80BF0))
#define CHIP_NUM      (*(uint32_t *)(0x80990))


#define BIT6_CAL  (*(int8_t *)(0x809A0))
#define BIT7_CAL  (*(int8_t *)(0x809A1))
#define BIT8_CAL  (*(int8_t *)(0x809A2))
#define BIT9_CAL  (*(int8_t *)(0x809A3))
#define BIT10_CAL (*(int8_t *)(0x809A4))
#define BIT11_CAL (*(int8_t *)(0x809A5))

#define DATA_CAL(val) ((val>>6)*BIT6_CAL+(val>>7)*BIT7_CAL+(val>>8)*BIT8_CAL+(val>>9)*BIT9_CAL+(val>>10)*BIT10_CAL+(val>>11)*BIT11_CAL)
#define DATA_CAL_FAST(val) ((val>>11)*BIT11_CAL)
int32_t gain_trim = 0, offset_trim = 0;

/**
  * @brief  Get Trim Gain and Offset
  * @retval None.
  */
uint32_t ald_get_trim_gain_offset(void)
{
    float k = 0, b = 0, delta_k = 0, delta_b = 0;
    float tmp = 0;
    uint32_t CFG_NUM = 0x55AA0000;

    if (CHIP_NUM == 0xFFAA0055)
    {
        bit6 = BIT6_CAL;
        bit7 = BIT7_CAL;
        bit8 = BIT8_CAL;
        bit9 = BIT9_CAL;
        bit10 = BIT10_CAL;
        bit11 = BIT11_CAL;
    }
    else
    {
        bit6 = 2;
        bit7 = 1;
        bit8 = 0;
        bit9 = 0;
        bit10 = 0;
        bit11 = 8;
    }

    tmp = (59 * (bit6 - 0) + 29 * (bit7 - 0) + 15 * (bit8 - 0) + 7 * (bit9 - 0) + 3 * (bit10 - 0) + 1 * (bit11 - 0)) / 4600.0;

    if ((__sys_vol < 5050) && (__sys_vol > 4950))
    {
        k = (0.819 * 65536 / GAIN5V_CAL) + tmp;
        b = OFFSET5V_CAL * (-1.0) / GAIN5V_CAL;
        b = b + (2 * (bit6 - 0) + (bit7 - 0) - 200 * tmp);
    }
    else if ((__sys_vol < 3350) && (__sys_vol > 3250))
    {
        k = (0.819 * 65536 / GAIN3V3_CAL) + tmp;
        b = OFFSET3V3_CAL * (-1.0) / GAIN3V3_CAL;
        b = b + (2 * (bit6 - 0) + (bit7 - 0) - 200 * tmp);
    }
    else
    {
        delta_k = ((0.819 * 65536 / GAIN5V_CAL) - (0.819 * 65536 / GAIN3V3_CAL)) / 1700.0;
        delta_b = (OFFSET5V_CAL * (-1.0) / GAIN5V_CAL - OFFSET3V3_CAL * (-1.0) / GAIN3V3_CAL) / 1700.0;
        k = (0.819 * 65536 / GAIN3V3_CAL) + tmp + (__sys_vol - 3300) * delta_k;
        b = OFFSET3V3_CAL * (-1.0) / GAIN3V3_CAL;
        b = b + (2 * (bit6 - 0) + (bit7 - 0) - 200 * tmp);
        b = b + (__sys_vol - 3300) * delta_b;
    }

    gain_trim = (int32_t)(0.819 / k * 2048);

    if (gain_trim > 2048)
    {
        CFG_NUM |= 1 << 15;
        CFG_NUM |= (gain_trim - 2048) << 8 ;
    }
    else if (gain_trim < 2048)
    {
        CFG_NUM |= (2048 - gain_trim) << 8;
    }
    else
        CFG_NUM &= 0xFFFF00FF;

    offset_trim = (int32_t)((-1) * (0.819 / k) * b + 0.5);

    if (offset_trim > 0)
    {
        CFG_NUM |= offset_trim << 0;
    }
    else if (offset_trim < 0)
    {
        CFG_NUM |= 1 << 7;
        offset_trim = 0xffffffff - offset_trim + 1;
        CFG_NUM |= offset_trim << 0;
    }
    else
        CFG_NUM &= 0xFFFFFF00;

    CFG_TRIM = CFG_NUM;
    return CFG_NUM;
}



/**
  * @brief  Unlock test mode.
  * @retval None.
  */
static void test_unlock(void)
{
#define SYSCFG_PORT      (*(volatile uint32_t *)0x40080000)
#define SYSCFG_TESTKEY   (*(volatile uint32_t *)0x40080100)

    SYSCFG_PORT    = 0x55AA6996;
    SYSCFG_TESTKEY = 0x5a962814;
    SYSCFG_TESTKEY = 0xe7cb69a5;
}

/**
  * @brief  Lock test mode.
  * @retval None.
  */
static void test_lock(void)
{
#define SYSCFG_PORT      (*(volatile uint32_t *)0x40080000)
#define SYSCFG_TESTKEY   (*(volatile uint32_t *)0x40080100)

    SYSCFG_TESTKEY = 0xDEADBEEF;
    SYSCFG_TESTKEY = 0xDEADBEEF;
}

/**
  * @brief  Config ADC GAIN/OFFSET trim value.
  * @retval None.
  */
static void ald_adc_trim_config(void)
{
#define MAP_GAIN_OFFSET     (*(volatile uint32_t *)0x40083C5C)
    uint32_t info_gain_offset   = *(volatile uint32_t *)0x80970;
    test_unlock();

    /* Info GAIN/OFFSET empty */
    if (info_gain_offset == 0xFFFFFFFF)
    {
        MAP_GAIN_OFFSET = ald_get_trim_gain_offset();
    }
    else
    {
        CFG_TRIM = info_gain_offset;

        if ((__sys_vol < 3350) && (__sys_vol > 3250))
        {
            MAP_GAIN_OFFSET = ald_get_trim_gain_offset();
        }
    }

    test_lock();
}

/**
  * @brief  Initializes the ADC peripheral and normal group according to
  *         parameters specified in structure "ald_adc_handle_t".
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_init(ald_adc_handle_t *hperh)
{
    uint32_t  ret = 0;

    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_DATA_ALIGN_TYPE(hperh->init.align));
    assert_param(IS_FUNC_STATE(hperh->init.scan));
    assert_param(IS_ADC_CLK_DIV_TYPE(hperh->init.div));
    assert_param(IS_ADC_NEG_REF_VOLTAGE_TYPE(hperh->init.n_ref));
    assert_param(IS_POS_REF_VOLTAGE_TYPE(hperh->init.p_ref));
    assert_param(IS_ADC_CONV_BIT_TYPE(hperh->init.data_bit));
    assert_param(IS_ADC_NCH_NR_TYPE(hperh->init.nch_nr));
    assert_param(IS_ADC_DISC_NR_TYPE(hperh->init.disc_nr));
    assert_param(IS_FUNC_STATE(hperh->init.cont));
    assert_param(IS_ADC_NCHESEL_MODE_TYPE(hperh->init.nche_sel));

    ret = ((adc_system_vol_get(hperh)) == 0x0) ? 0x0 : 0x1;

    if (hperh->state ==  ALD_ADC_STATE_RESET)
    {
        hperh->error_code = ALD_ADC_ERROR_NONE;
        hperh->lock       = UNLOCK;
    }

    ADC_DISABLE(hperh);
    ald_adc_reset(hperh);
    hperh->state = ALD_ADC_STATE_BUSY;
    MODIFY_REG(hperh->perh->CON1, ADC_CON1_ALIGN_MSK, hperh->init.align << ADC_CON1_ALIGN_POS);
    MODIFY_REG(hperh->perh->CON0, ADC_CON0_RSEL_MSK, hperh->init.data_bit << ADC_CON0_RSEL_POSS);

    /* Enable discontinuous mode only if continuous mode is disable */
    if (hperh->init.disc == ALD_ADC_NCH_DISC_EN)
    {
        hperh->init.scan = ENABLE;
        hperh->init.cont = DISABLE;
        SET_BIT(hperh->perh->CON0, ADC_CON0_NCHDCEN_MSK);
        MODIFY_REG(hperh->perh->CON0, ADC_CON0_ETRGN_MSK, hperh->init.disc_nr << ADC_CON0_ETRGN_POSS);
    }
    else if (hperh->init.disc == ALD_ADC_ICH_DISC_EN)
    {
        hperh->init.scan = ENABLE;
        hperh->init.cont = DISABLE;
        SET_BIT(hperh->perh->CON0, ADC_CON0_ICHDCEN_MSK);
        MODIFY_REG(hperh->perh->CON0, ADC_CON0_ETRGN_MSK, hperh->init.disc_nr << ADC_CON0_ETRGN_POSS);
    }
    else
    {
        CLEAR_BIT(hperh->perh->CON0, ADC_CON0_NCHDCEN_MSK);
        CLEAR_BIT(hperh->perh->CON0, ADC_CON0_ICHDCEN_MSK);
    }

    if ((hperh->init.scan == ENABLE) || (hperh->init.disc == ALD_ADC_NCH_DISC_EN))
        MODIFY_REG(hperh->perh->CHSL, ADC_CHSL_NSL_MSK, hperh->init.nch_nr << ADC_CHSL_NSL_POSS);

    MODIFY_REG(hperh->perh->CON1, ADC_CON1_CM_MSK, hperh->init.cont << ADC_CON1_CM_POS);
    MODIFY_REG(hperh->perh->CON0, ADC_CON0_SCANEN_MSK, hperh->init.scan << ADC_CON0_SCANEN_POS);

    ADC0->CCR = 0;
    MODIFY_REG(ADC0->CCR, ADC_CCR_PMSEL_MSK,  DISABLE << ADC_CCR_PMSEL_POS);
    MODIFY_REG(ADC0->CCR, ADC_CCR_VRNSEL_MSK, hperh->init.n_ref << ADC_CCR_VRNSEL_POS);
    MODIFY_REG(ADC0->CCR, ADC_CCR_VRPSEL_MSK, hperh->init.p_ref << ADC_CCR_VRPSEL_POSS);

    MODIFY_REG(hperh->perh->CCR, ADC_CCR_CKDIV_MSK, hperh->init.div << ADC_CCR_CKDIV_POSS);
    /* Enable adc calibration */
    SET_BIT(ADC0->CCR, ADC_CCR_TRMEN_MSK);

    MODIFY_REG(hperh->perh->CON1, ADC_CON1_NCHESEL_MSK, hperh->init.nche_sel << ADC_CON1_NCHESEL_POS);
    ADC_ENABLE(hperh);

    hperh->error_code = ALD_ADC_ERROR_NONE;
    hperh->state      = ALD_ADC_STATE_READY;

    ald_adc_trim_config();
    temp_k = ald_get_temp_k();

    return (ret == 0) ? ALD_OK : ALD_ERROR;
}

/**
  * @brief  Deinitialize the ADC peripheral registers to their default reset
  *         values.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_reset(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    ADC_DISABLE(hperh);
    WRITE_REG(hperh->perh->CLR, 0x30F);
    WRITE_REG(hperh->perh->CON0, 0x0);
    WRITE_REG(hperh->perh->CON1, 0x0);
    WRITE_REG(hperh->perh->CCR, 0x0);
    WRITE_REG(hperh->perh->WDTH, 0xFFF);
    WRITE_REG(hperh->perh->WDTL, 0x0);
    WRITE_REG(hperh->perh->ICHOFF1, 0x0);
    WRITE_REG(hperh->perh->ICHOFF2, 0x0);
    WRITE_REG(hperh->perh->ICHOFF3, 0x0);
    WRITE_REG(hperh->perh->ICHOFF4, 0x0);
    WRITE_REG(hperh->perh->ICHS, 0x0);
    WRITE_REG(hperh->perh->NCHS1, 0x0);
    WRITE_REG(hperh->perh->NCHS2, 0x0);
    WRITE_REG(hperh->perh->NCHS3, 0x0);
    WRITE_REG(hperh->perh->NCHS4, 0x0);
    WRITE_REG(hperh->perh->SMPT1, 0x0);
    WRITE_REG(hperh->perh->SMPT2, 0x0);
    WRITE_REG(hperh->perh->CHSL, 0x0);

    hperh->state      = ALD_ADC_STATE_RESET;
    hperh->error_code = ALD_ADC_ERROR_NONE;
    return ALD_OK;
}
/**
  * @}
  */

/** @defgroup ADC_Public_Functions_Group2 IO operation functions
 *  @brief    Input and Output operation functions
 *  @{
 */

/**
  * @brief  Enables ADC, starts conversion of normal group.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_normal_start(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    ADC_ENABLE(hperh);
    WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_NCH | ALD_ADC_FLAG_NCHS);
    SET_BIT(hperh->perh->CON1, ADC_CON1_NCHTRG_MSK);

    return ALD_OK;
}

/**
  * @brief  Config ADC normal channel pis tirgger channel.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  pch: PIS channel 0 ~ 15.
  * @param  edge: PIS trigger edge @ald_adc_ets_t.
  * @retval None.
  */
void ald_adc_normal_channel_pis_trig_config(ald_adc_handle_t *hperh, uint8_t pch, ald_adc_ets_t edge)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_PIS_NUM(pch));
    assert_param(IS_PIS_TRIGGER_EDGE(edge));

    MODIFY_REG(ADC0->CON1, ADC_CON1_NETP_MSK, (pch & 0xF) << ADC_CON1_NETP_POSS);
    MODIFY_REG(ADC0->CON1, ADC_CON1_NETS_MSK, (edge & 0x3) << ADC_CON1_NETS_POSS);
}

/**
  * @brief  Config ADC insert channel pis tirgger channel.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  pch: PIS channel 0 ~ 15.
  * @param  edge: PIS trigger edge @ald_adc_ets_t.
  * @retval None.
  */
void ald_adc_insert_channel_pis_trig_config(ald_adc_handle_t *hperh, uint8_t pch, ald_adc_ets_t edge)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_PIS_NUM(pch));
    assert_param(IS_PIS_TRIGGER_EDGE(edge));

    MODIFY_REG(ADC0->CON1, ADC_CON1_IETP_MSK, (pch & 0xF) << ADC_CON1_IETP_POSS);
    MODIFY_REG(ADC0->CON1, ADC_CON1_IETS_MSK, (edge & 0x3) << ADC_CON1_IETS_POSS);
}

/**
  * @brief  Stop ADC conversion of normal group (and insert channels in
  *         case of auto_injection mode), disable ADC peripheral.
  * @note:  ADC peripheral disable is forcing stop of potential
  *         conversion on insert group. If insert group is under use, it
  *         should be preliminarily stopped using ald_adc_insert_stop function.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_normal_stop(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    ADC_DISABLE(hperh);
    hperh->state = ALD_ADC_STATE_READY;

    return ALD_OK;
}

/**
  * @brief  Wait for normal group conversion to be completed.
  * @note   This function cannot be used in a particular setup: ADC configured  in DMA mode.
  *         In this case, DMA resets the flag EOC and polling cannot be performed on each conversion.
  * @note   When use this function,you should be pay attention to the hperh->init.reocs_mode,
  *         if it is ADC_REOCS_MODE_ALL, it means the function will wait all normal rank conversion  finished.
  *         if it is ADC_REOCS_MODE_ONE, it means the funcion will wait every normal rank conversion finished.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  timeout: timeout value in millisecond.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_normal_poll_for_conversion(ald_adc_handle_t *hperh, uint32_t timeout)
{
    uint32_t _tick;

    assert_param(IS_ADC_TYPE(hperh->perh));

    _tick = ald_get_tick();

    while (!(READ_BIT(hperh->perh->STAT, ADC_STAT_NCHE_MSK)))
    {
        if (timeout != ALD_MAX_DELAY)
        {
            if ((timeout == 0) || ((ald_get_tick() - _tick) > timeout))
            {
                hperh->state = ALD_ADC_STATE_TIMEOUT;
                return ALD_TIMEOUT;
            }
        }
    }

    WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_NCHS | ALD_ADC_FLAG_NCH);
    return ALD_OK;
}

/**
  * @brief  Poll for conversion event.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  event_type: the ADC event type.
  *          This parameter can be one of the following values:
  *            ADC_awd_event: ADC Analog watchdog event.
  * @param  timeout: timeout value in millisecond.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_poll_for_event(ald_adc_handle_t *hperh, ald_adc_event_type_t event_type, uint32_t timeout)
{
    uint32_t _tick;

    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_EVENT_TYPE(event_type));

    _tick = ald_get_tick();

    while (ald_adc_get_flag_status(hperh, (ald_adc_flag_t)event_type) == RESET)
    {
        if (timeout != ALD_MAX_DELAY)
        {
            if ((timeout == 0) || ((ald_get_tick() - _tick) > timeout))
            {
                hperh->state = ALD_ADC_STATE_TIMEOUT;
                return ALD_TIMEOUT;
            }
        }
    }

    CLEAR_BIT(hperh->state, ALD_ADC_STATE_BUSY_WDG);
    return ALD_OK;
}

/**
  * @brief  Enables ADC, starts conversion of normal group with interruption.
  *         Interruptions enabled in this function:
  *          - REOC (end of conversion of normal group)
  *         Each of these interruptions has its dedicated callback function.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_normal_start_by_it(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    SET_BIT(hperh->state, ALD_ADC_STATE_BUSY_N);
    ADC_ENABLE(hperh);
    WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_NCH);
    ald_adc_interrupt_config(hperh, ALD_ADC_IT_NCH, ENABLE);
    SET_BIT(hperh->perh->CON1, ADC_CON1_NCHTRG_MSK);

    return ALD_OK;
}

/**
  * @brief  Stop ADC conversion of normal group (and insert group in
  *         case of auto_injection mode), disable interrution of
  *         end-of-conversion, disable ADC peripheral.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_normal_stop_by_it(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    ADC_DISABLE(hperh);
    ald_adc_interrupt_config(hperh, ALD_ADC_IT_NCH, DISABLE);
    CLEAR_BIT(hperh->state, ALD_ADC_STATE_BUSY_N);

    return ALD_OK;
}

#ifdef ALD_DMA
/**
  * @brief  Enables ADC, starts conversion of normal group and transfers result
  *         through DMA.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  buf: The destination Buffer address.
  * @param  size: The length of data to be transferred from ADC peripheral to memory.
  * @param  channel: The DMA channel
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_start_by_dma(ald_adc_handle_t *hperh, uint16_t *buf, uint16_t size, uint8_t channel)
{
    if ((buf == NULL) || (size == 0))
        return ALD_ERROR;

    assert_param(IS_ADC_TYPE(hperh->perh));

    SET_BIT(hperh->state, ALD_ADC_STATE_BUSY_N);

    if (hperh->hdma.perh == NULL)
    {
        hperh->hdma.perh = DMA0;
    }

    hperh->hdma.cplt_tc_cbk = adc_dma_normal_conv_cplt;
    hperh->hdma.cplt_tc_arg = hperh;

    ald_dma_config_struct(&hperh->hdma);
    hperh->hdma.config.src              = (void *)&hperh->perh->NCHDR;
    hperh->hdma.config.dst              = (void *)buf;
    hperh->hdma.config.size             = size;
    hperh->hdma.config.src_data_width   = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh->hdma.config.src_inc          = ALD_DMA_DATA_INC_DISABLE;
    hperh->hdma.config.dst_data_width   = ALD_DMA_DATA_SIZE_HALFWORD;
    hperh->hdma.config.dst_inc          = ALD_DMA_DATA_INC_ENABLE;
    hperh->hdma.config.msel             = ALD_DMA_MSEL_ADC0;
    hperh->hdma.config.msigsel          = ALD_DMA_MSIGSEL_ADC;
    hperh->hdma.config.R_power          = ALD_DMA_R_POWER_1;
    hperh->hdma.config.dir              = ALD_DMA_DIR_TO_SRAM;
    hperh->hdma.config.channel          = channel;
    hperh->hdma.config.circle_mode      = ENABLE;
    ald_dma_config(&hperh->hdma);

    ADC_ENABLE(hperh);
    SET_BIT(hperh->perh->CON1, ADC_CON1_DMA_MSK);
    SET_BIT(hperh->perh->CON1, ADC_CON1_NCHTRG_MSK);

    return ALD_OK;
}

/**
  * @brief  Stop ADC conversion of normal group (and insert group in
  *         case of auto_insert mode), disable ADC DMA transfer, disable
  *         ADC peripheral.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  channel: The DMA channel
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_stop_by_dma(ald_adc_handle_t *hperh, uint8_t channel)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    ADC_DISABLE(hperh);
    CLEAR_BIT(hperh->perh->CON1, ADC_CON1_DMA_MSK);
    ald_dma_pause(&hperh->hdma);
    CLEAR_BIT(hperh->state, ALD_ADC_STATE_BUSY_N);

    return ALD_OK;
}

/**
  * @brief  DMA transfer complete callback.
  * @param  arg: argument of the call back.
  * @retval None
  */
static void adc_dma_timer_trigger_cplt(void *arg)
{
    adc_timer_config_t *hperh = (adc_timer_config_t *)arg;

    ald_timer_base_stop(&hperh->h_timer);
    CLEAR_BIT(hperh->h_adc.perh->CON1, ADC_CON1_DMA_MSK);
    ADC_DISABLE(&hperh->h_adc);
    ald_dma_pause(&hperh->h_dma);
    CLEAR_BIT(hperh->h_adc.state, ALD_ADC_STATE_BUSY_N);

    if (hperh->h_adc.normal_cplt_cbk)
        hperh->h_adc.normal_cplt_cbk(&hperh->h_adc);

    return;
}

/**
  * @brief  Config timer trigger adc insert channel conversion.
  * @param  config: Pointer to a adc_timer_config_t structure that
  *         contains the configuration information for the specified function.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_timer_trigger_insert(adc_timer_config_t *config)
{
    config->h_pis.perh               = PIS;
    config->h_pis.init.producer_clk  = ALD_PIS_CLK_PCLK1;
    config->h_pis.init.producer_edge = ALD_PIS_EDGE_NONE;
    config->h_pis.init.consumer_clk  = ALD_PIS_CLK_PCLK2;

    if (config->p_timer == AD16C4T0)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER0_UPDATA;
    else if (config->p_timer == GP32C4T0)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER2_TRGOUT;
    else if (config->p_timer == GP32C4T1)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER3_TRGOUT;
    else if (config->p_timer == GP16C4T0)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER6_TRGOUT;
    else if (config->p_timer == GP16C4T1)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER7_TRGOUT;
    else
        return ALD_ERROR;

    if (config->p_adc == ADC0)
        config->h_pis.init.consumer_trig = ALD_PIS_CH7_ADC0_INSERT;
    else
        return ALD_ERROR;

    ald_pis_create(&config->h_pis);

    /* Initialize TIMER */
    config->h_timer.perh           = config->p_timer;
    config->h_timer.init.prescaler = 0;
    config->h_timer.init.mode      = ALD_TIMER_CNT_MODE_UP;
    config->h_timer.init.period    = ((ald_cmu_get_pclk1_clock() / 1000000) * config->time);
    config->h_timer.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    config->h_timer.init.re_cnt    = 0;
    ald_timer_base_init(&config->h_timer);

    config->h_adc.perh            = config->p_adc;
    config->h_adc.init.align      = ALD_ADC_DATAALIGN_RIGHT;
    config->h_adc.init.scan       = DISABLE;
    config->h_adc.init.cont       = DISABLE;
    config->h_adc.init.ich_nr     = ALD_ADC_ICH_NR_1;
    config->h_adc.init.disc       = ALD_ADC_ALL_DISABLE;
    config->h_adc.init.disc_nr    = ALD_ADC_DISC_NR_1;
    config->h_adc.init.data_bit   = ALD_ADC_CONV_BIT_12;
    config->h_adc.init.div        = ALD_ADC_CKDIV_8;
    config->h_adc.init.nche_sel   = ALD_ADC_NCHESEL_MODE_ONE;
    config->h_adc.init.n_ref      = config->n_ref;
    config->h_adc.init.p_ref      = config->p_ref;
    config->h_adc.normal_cplt_cbk = config->cplt_cbk;
    config->h_adc.insert_cplt_cbk = NULL;
    config->h_adc.wdg_cbk         = NULL;
    config->h_adc.error_cbk       = NULL;
    config->h_adc.ovr_cbk         = NULL;
    ald_adc_init(&config->h_adc);

    config->h_adc.perh->CON1   |= 0x00100000;   /* rising edge trigger insert channel convert */
    config->i_config.ch         = config->adc_ch;
    config->i_config.idx        = ALD_ADC_ICH_IDX_1;
    config->i_config.samp       = ALD_ADC_SAMPLETIME_4;
    config->i_config.nr         = ALD_ADC_ICH_NR_1;
    config->i_config.auto_m     = DISABLE;
    ald_adc_insert_channel_config(&config->h_adc, &config->i_config);

    ADC_ENABLE(&config->h_adc);
    ald_timer_base_start(&config->h_timer);

    return ALD_OK;
}

/**
  * @brief  Config Timer trigger adc function
  * @param  config: Pointer to a adc_timer_config_t structure that
  *         contains the configuration information for the specified function.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_timer_trigger_adc_by_dma(adc_timer_config_t *config)
{
    config->h_pis.perh               = PIS;
    config->h_pis.init.producer_clk  = ALD_PIS_CLK_PCLK1;
    config->h_pis.init.producer_edge = ALD_PIS_EDGE_NONE;
    config->h_pis.init.consumer_clk  = ALD_PIS_CLK_PCLK2;

    if (config->p_timer == AD16C4T0)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER0_UPDATA;
    else if (config->p_timer == GP32C4T0)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER2_TRGOUT;
    else if (config->p_timer == GP32C4T1)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER3_TRGOUT;
    else if (config->p_timer == GP16C4T0)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER6_TRGOUT;
    else if (config->p_timer == GP16C4T1)
        config->h_pis.init.producer_src  = ALD_PIS_TIMER7_TRGOUT;
    else
        return ALD_ERROR;

    if (config->p_adc == ADC0)
    {
        config->h_pis.init.consumer_trig = ALD_PIS_CH6_ADC0_NORMAL;
    }
    else
        return ALD_ERROR;

    ald_pis_create(&config->h_pis);

    /* Initialize TIMER */
    config->h_timer.perh           = config->p_timer;
    config->h_timer.init.prescaler = 0;
    config->h_timer.init.mode      = ALD_TIMER_CNT_MODE_UP;
    config->h_timer.init.period    = ((ald_cmu_get_pclk1_clock() / 1000000) * config->time);
    config->h_timer.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    config->h_timer.init.re_cnt    = 0;
    ald_timer_base_init(&config->h_timer);

    config->h_adc.perh            = config->p_adc;
    config->h_adc.init.align      = ALD_ADC_DATAALIGN_RIGHT;
    config->h_adc.init.scan       = ENABLE;
    config->h_adc.init.cont       = ENABLE;
    config->h_adc.init.nch_nr     = ALD_ADC_NCH_NR_1;
    config->h_adc.init.disc       = ALD_ADC_ALL_DISABLE;
    config->h_adc.init.disc_nr    = ALD_ADC_DISC_NR_1;
    config->h_adc.init.data_bit   = ALD_ADC_CONV_BIT_12;
    config->h_adc.init.div        = ALD_ADC_CKDIV_8;
    config->h_adc.init.nche_sel   = ALD_ADC_NCHESEL_MODE_ONE;
    config->h_adc.init.n_ref      = config->n_ref;
    config->h_adc.init.p_ref      = config->p_ref;
    config->h_adc.normal_cplt_cbk = config->cplt_cbk;
    config->h_adc.insert_cplt_cbk = NULL;
    config->h_adc.wdg_cbk         = NULL;
    config->h_adc.error_cbk       = NULL;
    config->h_adc.ovr_cbk         = NULL;
    ald_adc_init(&config->h_adc);

    config->config.ch   = config->adc_ch;
    config->config.idx  = ALD_ADC_NCH_IDX_1;
    config->config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&config->h_adc, &config->config);
    /* Config ADC0 pis trigger channel */
    ald_adc_normal_channel_pis_trig_config(&config->h_adc, (ALD_PIS_CH6_ADC0_NORMAL & 0xF), ALD_ADC_ETS_RISE);

    config->h_dma.cplt_tc_cbk = adc_dma_timer_trigger_cplt;
    config->h_dma.cplt_tc_arg = config;

    ald_dma_config_struct(&config->h_dma);
    config->h_dma.perh                   = DMA0;
    config->h_dma.config.src             = (void *)&config->h_adc.perh->NCHDR;
    config->h_dma.config.dst             = (void *)config->buf;
    config->h_dma.config.size            = config->size;
    config->h_dma.config.src_data_width  = ALD_DMA_DATA_SIZE_HALFWORD;
    config->h_dma.config.src_inc         = ALD_DMA_DATA_INC_DISABLE;
    config->h_dma.config.dst_data_width  = ALD_DMA_DATA_SIZE_HALFWORD;
    config->h_dma.config.dst_inc         = ALD_DMA_DATA_INC_ENABLE;
    config->h_dma.config.msel            = ALD_DMA_MSEL_ADC0;
    config->h_dma.config.msigsel         = ALD_DMA_MSIGSEL_ADC;
    config->h_dma.config.R_power         = ALD_DMA_R_POWER_1;
    config->h_dma.config.dir             = ALD_DMA_DIR_TO_SRAM;
    config->h_dma.config.channel         = config->dma_ch;
    ald_dma_config(&config->h_dma);

    SET_BIT(config->h_adc.perh->CON1, ADC_CON1_DMA_MSK);
    ADC_ENABLE(&config->h_adc);
    ald_timer_base_start(&config->h_timer);

    return ALD_OK;
}
#endif

/**
  * @brief  Get ADC normal group conversion result.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval ADC group normal conversion data
  */
uint32_t ald_adc_normal_get_value(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    return hperh->perh->NCHDR;
}

/**
  * @brief  Enables ADC, starts conversion of insert group.
  *         Interruptions enabled in this function: None.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_insert_start(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    ADC_ENABLE(hperh);
    WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_ICH);

    if (!(READ_BIT(hperh->perh->CON0, ADC_CON0_IAUTO_MSK)))
        SET_BIT(hperh->perh->CON1, ADC_CON1_ICHTRG_MSK);

    return ALD_OK;
}

/**
  * @brief  Stop conversion of insert channels. Disable ADC peripheral if
  *         no normal conversion is on going.
  * @note   If ADC must be disabled and if conversion is on going on
  *         normal group, function ald_adc_normal_stop must be used to stop both
  *         insert and normal groups, and disable the ADC.
  * @note   If insert group mode auto-injection is enabled,
  *         function ald_adc_normal_stop must be used.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_insert_stop(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    ADC_DISABLE(hperh);
    hperh->state = ALD_ADC_STATE_READY;
    return ALD_OK;
}

/**
  * @brief  Wait for insert group conversion to be completed.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  timeout: timeout value in millisecond.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_insert_poll_for_conversion(ald_adc_handle_t *hperh, uint32_t timeout)
{
    uint32_t _tick;

    assert_param(IS_ADC_TYPE(hperh->perh));

    _tick = ald_get_tick();

    while (!(READ_BIT(hperh->perh->STAT, ADC_STAT_ICHE_MSK)))
    {
        if (timeout != ALD_MAX_DELAY)
        {
            if ((timeout == 0) || ((ald_get_tick() - _tick) > timeout))
            {
                hperh->state |= ALD_ADC_STATE_TIMEOUT;
                return ALD_TIMEOUT;
            }
        }
    }

    WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_ICHS | ALD_ADC_FLAG_ICH);
    return ALD_OK;
}

/**
  * @brief  Enables ADC, starts conversion of insert group with interruption.
  *          - JEOC (end of conversion of insert group)
  *         Each of these interruptions has its dedicated callback function.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval Status, see @ref ald_status_t..
  */
ald_status_t ald_adc_insert_start_by_it(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    SET_BIT(hperh->state, ALD_ADC_STATE_BUSY_I);
    ADC_ENABLE(hperh);
    WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_ICHS | ALD_ADC_FLAG_ICH);
    ald_adc_interrupt_config(hperh, ALD_ADC_IT_ICH, ENABLE);

    if (!(READ_BIT(hperh->perh->CON0, ADC_CON0_IAUTO_MSK)))
        SET_BIT(hperh->perh->CON1, ADC_CON1_ICHTRG_MSK);

    return ALD_OK;
}

/**
  * @brief  Stop conversion of insert channels, disable interruption of
  *         end-of-conversion. Disable ADC peripheral if no normal conversion
  *         is on going.
  * @note   If ADC must be disabled and if conversion is on going on
  *         normal group, function ald_adc_normal_stop must be used to stop both
  *         insert and normal groups, and disable the ADC.
  * @note   If insert group mode auto-injection is enabled,
  *         function ald_adc_normal_stop must be used.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval None
  */
ald_status_t ald_adc_insert_stop_by_it(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    CLEAR_BIT(hperh->state, ALD_ADC_STATE_BUSY_I);
    ADC_DISABLE(hperh);
    ald_adc_interrupt_config(hperh, ALD_ADC_IT_ICH, DISABLE);
    return ALD_OK;
}

/**
  * @brief  Get ADC insert group conversion result.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  idx: Index of converted ADC insert channel.
  * @retval ADC group insert conversion data
  */
uint32_t ald_adc_insert_get_value(ald_adc_handle_t *hperh, adc_ich_idx_t idx)
{
    uint32_t tmp;

    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_ICH_IDX_TYPE(idx));

    switch (idx)
    {
        case ALD_ADC_ICH_IDX_1:
            tmp = hperh->perh->ICHDR1;
            break;

        case ALD_ADC_ICH_IDX_2:
            tmp = hperh->perh->ICHDR2;
            break;

        case ALD_ADC_ICH_IDX_3:
            tmp = hperh->perh->ICHDR3;
            break;

        case ALD_ADC_ICH_IDX_4:
            tmp = hperh->perh->ICHDR4;
            break;

        default:
            tmp = hperh->perh->ICHDR1;
            break;
    }

    return tmp;
}

/**
  * @brief  Handles ADC interrupt request
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval None
  */
void ald_adc_irq_handler(ald_adc_handle_t *hperh)
{
    assert_param(IS_ADC_TYPE(hperh->perh));

    if (ald_adc_get_it_status(hperh, ALD_ADC_IT_NCH) && ald_adc_get_flag_status(hperh, ALD_ADC_FLAG_NCH))
    {
        WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_NCH | ALD_ADC_FLAG_NCHS);
        CLEAR_BIT(hperh->state, ALD_ADC_STATE_BUSY_N);

        if (hperh->normal_cplt_cbk)
            hperh->normal_cplt_cbk(hperh);
    }

    if (ald_adc_get_it_status(hperh, ALD_ADC_IT_ICH) && ald_adc_get_flag_status(hperh, ALD_ADC_FLAG_ICH))
    {
        WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_ICH | ALD_ADC_FLAG_ICHS);
        CLEAR_BIT(hperh->state, ALD_ADC_STATE_BUSY_I);

        if (hperh->insert_cplt_cbk)
            hperh->insert_cplt_cbk(hperh);
    }

    if (ald_adc_get_it_status(hperh, ALD_ADC_IT_AWD) && ald_adc_get_flag_status(hperh, ALD_ADC_FLAG_AWD))
    {
        CLEAR_BIT(hperh->state, ALD_ADC_STATE_BUSY_WDG);
        WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_AWD);

        if (hperh->wdg_cbk)
            hperh->wdg_cbk(hperh);
    }

    if (ald_adc_get_it_status(hperh, ALD_ADC_IT_OVR) && ald_adc_get_flag_status(hperh, ALD_ADC_FLAG_OVR))
    {
        WRITE_REG(hperh->perh->CLR, ALD_ADC_FLAG_OVR);
        hperh->error_code |= ALD_ADC_ERROR_OVR;
        hperh->state      |= ALD_ADC_STATE_ERROR;

        if (hperh->ovr_cbk)
            hperh->ovr_cbk(hperh);
    }
}

/**
  * @}
  */

/** @defgroup ADC_Public_Functions_Group3 Peripheral Control functions
 *  @brief    Peripheral Control functions
 *  @{
 */

/**
  * @brief  Configures the the selected channel to be linked to the normal
  *         group.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  config: Structure of ADC channel for normal group.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_normal_channel_config(ald_adc_handle_t *hperh, ald_adc_nch_conf_t *config)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_CHANNELS_TYPE(config->ch));
    assert_param(IS_ADC_NCH_IDX_TYPE(config->idx));
    assert_param(IS_ADC_SAMPLING_TIMES_TYPE(config->samp));

    if (config->idx <= ALD_ADC_NCH_IDX_4)
    {
        hperh->perh->NCHS1 &= ~(0x1f << (uint32_t)((config->idx - 1) << 3));
        hperh->perh->NCHS1 |= (config->ch << (uint32_t)((config->idx - 1) << 3));
    }
    else if (config->idx <= ALD_ADC_NCH_IDX_8)
    {
        hperh->perh->NCHS2 &= ~(0x1f << (uint32_t)((config->idx - 5) << 3));
        hperh->perh->NCHS2 |= (config->ch << (uint32_t)((config->idx - 5) << 3));
    }
    else if (config->idx <= ALD_ADC_NCH_IDX_12)
    {
        hperh->perh->NCHS3 &= ~(0x1f << (uint32_t)((config->idx - 9) << 3));
        hperh->perh->NCHS3 |= (config->ch << (uint32_t)((config->idx - 9) << 3));
    }
    else
    {
        hperh->perh->NCHS4 &= ~(0x1f << (uint32_t)((config->idx - 13) << 3));
        hperh->perh->NCHS4 |= (config->ch << (uint32_t)((config->idx - 13) << 3));
    }

    if (config->ch < 8)
    {
        hperh->perh->SMPT1 &= ~(0x0f << (uint32_t)(config->ch << 2));
        hperh->perh->SMPT1 |= config->samp << (uint32_t)(config->ch << 2);
    }
    else if (config->ch < 16)
    {
        hperh->perh->SMPT2 &= ~(0x0f << (uint32_t)((config->ch - 8) << 2));
        hperh->perh->SMPT2 |= config->samp << (uint32_t)((config->ch - 8) << 2);
    }
    else
    {
        hperh->perh->SMPT3 &= ~(0x0f << (uint32_t)((config->ch - 16) << 2));
        hperh->perh->SMPT3 |= config->samp << (uint32_t)((config->ch - 16) << 2);
    }

    return ALD_OK;
}

/**
  * @brief  Configures the the selected channel to be linked to the insert
  *         group.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  config: Structure of ADC channel for insert group.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ald_adc_insert_channel_config(ald_adc_handle_t *hperh, ald_adc_ich_conf_t *config)
{
    ald_status_t tmp_status = ALD_OK;

    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_CHANNELS_TYPE(config->ch));
    assert_param(IS_ADC_ICH_IDX_TYPE(config->idx));
    assert_param(IS_ADC_SAMPLING_TIMES_TYPE(config->samp));
    assert_param(IS_ADC_IST_OFFSET_TYPE(config->offset));
    assert_param(IS_ADC_ICH_NR_TYPE(config->nr));
    assert_param(IS_FUNC_STATE(config->auto_m));

    MODIFY_REG(hperh->perh->CHSL, ADC_CHSL_ISL_MSK, config->nr << ADC_CHSL_ISL_POSS);
    hperh->perh->ICHS &= ~(0x1f << (uint32_t)((config->idx - 1) << 3));
    hperh->perh->ICHS |= config->ch << (uint32_t)((config->idx - 1) << 3);

    if (config->auto_m == ENABLE)
        SET_BIT(hperh->perh->CON0, ADC_CON0_IAUTO_MSK);
    else
        CLEAR_BIT(hperh->perh->CON0, ADC_CON0_IAUTO_MSK);

    if (hperh->init.disc == ALD_ADC_ICH_DISC_EN)
    {
        if (config->auto_m == DISABLE)
        {
            SET_BIT(hperh->perh->CON0, ADC_CON0_ICHDCEN_MSK);
        }
        else
        {
            hperh->state      |= ALD_ADC_STATE_ERROR;
            hperh->error_code |= ALD_ADC_ERROR_INTERNAL;
            tmp_status         = ALD_ERROR;
        }
    }

    if (config->ch < 8)
    {
        hperh->perh->SMPT1 &=  ~(0x0f << (uint32_t)(config->ch << 2));
        hperh->perh->SMPT1 |= config->samp << (uint32_t)(config->ch << 2);
    }
    else if (config->ch < 16)
    {
        hperh->perh->SMPT2 &=  ~(0x0f << (uint32_t)((config->ch - 8) << 2));
        hperh->perh->SMPT2 |= config->samp << (uint32_t)((config->ch - 8) << 2);
    }
    else
    {
        hperh->perh->SMPT3 &=  ~(0x0f << (uint32_t)((config->ch - 16) << 2));
        hperh->perh->SMPT3 |= config->samp << (uint32_t)((config->ch - 16) << 2);
    }

    switch (config->idx)
    {
        case ALD_ADC_ICH_IDX_1:
            hperh->perh->ICHOFF1 = config->offset;
            break;

        case ALD_ADC_ICH_IDX_2:
            hperh->perh->ICHOFF2 = config->offset;
            break;

        case ALD_ADC_ICH_IDX_3:
            hperh->perh->ICHOFF3 = config->offset;
            break;

        case ALD_ADC_ICH_IDX_4:
            hperh->perh->ICHOFF4 = config->offset;
            break;

        default:
            break;
    }

    return tmp_status;
}

/**
  * @brief  Configures the analog watchdog.
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @param  config: Structure of ADC analog watchdog configuration
  * @retval ALD status
  */
ald_status_t ald_adc_analog_wdg_config(ald_adc_handle_t *hperh, adc_analog_wdg_conf_t *config)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_ANALOG_WTD_MODE_TYPE(config->mode));
    assert_param(IS_FUNC_STATE(config->interrupt));
    assert_param(IS_HTR_TYPE(config->high_thrd));
    assert_param(IS_LTR_TYPE(config->low_thrd));

    if ((config->mode == ALD_ADC_ANAWTD_SING_NM)
            || (config->mode == ALD_ADC_ANAWTD_SING_IST)
            || (config->mode == ALD_ADC_ANAWTD_SING_NMIST))
        assert_param(IS_ADC_CHANNELS_TYPE(config->ch));

    if (config->interrupt == DISABLE)
        ald_adc_interrupt_config(hperh, ALD_ADC_IT_AWD, DISABLE);
    else
        ald_adc_interrupt_config(hperh, ALD_ADC_IT_AWD, ENABLE);

    CLEAR_BIT(hperh->perh->CON0, ADC_CON0_ICHWDTEN_MSK);
    CLEAR_BIT(hperh->perh->CON0, ADC_CON0_NCHWDEN_MSK);
    CLEAR_BIT(hperh->perh->CON0, ADC_CON0_AWDSGL_MSK);
    hperh->perh->CON0 |= config->mode;

    if (READ_BIT(hperh->perh->CON0, ADC_CON0_AWDSGL_MSK))
        MODIFY_REG(hperh->perh->CON0, ADC_CON0_AWDCH_MSK, config->ch << ADC_CON0_AWDCH_POSS);

    WRITE_REG(hperh->perh->WDTL, config->low_thrd);
    WRITE_REG(hperh->perh->WDTH, config->high_thrd);
    SET_BIT(hperh->state, ALD_ADC_STATE_BUSY_WDG);

    return ALD_OK;
}

/**
  * @brief  Enables or disables the specified ADC interrupts.
  * @param  hperh: Pointer to a ald_adc_handle_t structure.
  * @param  it: Specifies the ADC interrupt sources to be enabled or disabled.
  *         This parameter can be one of the @ref adc_it_t.
  * @param  state: New status
  *           - ENABLE
  *           - DISABLE
  * @retval None
  */
void ald_adc_interrupt_config(ald_adc_handle_t *hperh, ald_adc_it_t it, type_func_t state)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_IT_TYPE(it));
    assert_param(IS_FUNC_STATE(state));

    if (state == ENABLE)
        SET_BIT(hperh->perh->CON0, it);
    else
        CLEAR_BIT(hperh->perh->CON0, it);

    return;
}

/**
  * @brief  Checks whether the specified ADC interrupt has occurred or not.
  * @param  hperh: Pointer to a ald_adc_handle_t structure.
  * @param  it: Specifies the ADC interrupt source to check.
  *         This parameter can be one of the @ref adc_it_t.
  * @retval Status
  *           - SET
  *           - RESET
  */
it_status_t ald_adc_get_it_status(ald_adc_handle_t *hperh, ald_adc_it_t it)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_IT_TYPE(it));

    if (READ_BIT(hperh->perh->CON0, it))
        return SET;

    return RESET;
}

/** @brief  Check whether the specified ADC flag is set or not.
  * @param  hperh: Pointer to a ald_adc_handle_t structure.
  * @param  flag: specifies the flag to check.
  *         This parameter can be one of the @ref adc_flag_t.
  * @retval Status
  *           - SET
  *           - RESET
  */
flag_status_t ald_adc_get_flag_status(ald_adc_handle_t *hperh, ald_adc_flag_t flag)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_FLAGS_TYPE(flag));

    if (READ_BIT(hperh->perh->STAT, flag))
        return SET;

    return RESET;
}

/** @brief  Clear the specified ADC pending flags.
  * @param  hperh: Pointer to a ald_adc_handle_t structure.
  * @param  flag: specifies the flag to check.
  *         This parameter can be one of the @ref adc_flag_t.
  * @retval None
  */
void ald_adc_clear_flag_status(ald_adc_handle_t *hperh, ald_adc_flag_t flag)
{
    assert_param(IS_ADC_TYPE(hperh->perh));
    assert_param(IS_ADC_FLAGS_TYPE(flag));

    WRITE_REG(hperh->perh->CLR, flag);
    return;
}
/**
  * @}
  */

/** @defgroup ADC_Public_Functions_Group4 Peripheral State functions
 *  @brief    Peripheral State functions
 *  @{
 */

/**
  * @brief  return the ADC state
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval state
  */
uint32_t ald_adc_get_state(ald_adc_handle_t *hperh)
{
    return hperh->state;
}

/**
  * @brief  Return the ADC error code
  * @param  hperh: Pointer to a ald_adc_handle_t structure that contains
  *         the configuration information for the specified ADC module.
  * @retval ADC error Code
  */
uint32_t ald_adc_get_error(ald_adc_handle_t *hperh)
{
    return hperh->error_code;
}

/**
  * @brief  Get adc adjust gain and offset
  * @param  vol_src: Voltage reference source for ADC
  * @param  vol: Value of the supply voltage, in mV
  * @param  gain: Address where the gain value is stored
  * @param  offset: Address where the offset vaule is stored
  * @param  adc_mode: Trim mode for ADC
  * @retval None
  */
void ald_get_adc_gain_offset(ald_adc_pos_ref_t vol_src, int32_t *gain, int32_t *offset)
{
    float delta_gain = 0;
    float delta_offset = 0;
    float gain_fast_5V = 0;
    float gain_fast_3P3 = 0;
    float offset_fast_5V = 0;
    float offset_fast_3P3 = 0;
    float k = 0, b = 0;
    float tmp = 0;
    float R = 0;

    /* Disable adc calibration */
    CLEAR_BIT(ADC0->CCR, ADC_CCR_TRMEN_MSK);

    if (CHIP_NUM == 0xFFAA0055)
    {
        bit6 = BIT6_CAL;
        bit7 = BIT7_CAL;
        bit8 = BIT8_CAL;
        bit9 = BIT9_CAL;
        bit10 = BIT10_CAL;
        bit11 = BIT11_CAL;
    }
    else
    {
        bit6 = 2;
        bit7 = 1;
        bit8 = 0;
        bit9 = 0;
        bit10 = 0;
        bit11 = 8;
    }

    R  = ((OFFSET5V_CAL - OFFSET5V_BAC) + (OFFSET3V3_CAL - OFFSET3V3_BAC)) / 2 / 12;

    /* Internal vol ref */
    if (vol_src == ALD_ADC_POS_REF_VDD)
    {
        gain_fast_5V = GAIN5V_CAL;
        gain_fast_3P3 = GAIN3V3_CAL;
        offset_fast_5V = OFFSET5V_CAL + R * Cost_Ima;
        offset_fast_3P3 = OFFSET3V3_CAL + R * Cost_Ima;

        if ((__sys_vol < 5050) && (__sys_vol > 4950))
        {
            *gain = (int32_t)(gain_fast_5V);
            *offset = (int32_t)(offset_fast_5V);
            gain_tense = *gain;
            offset_tense = *offset;
            return;
        }
        else if ((__sys_vol < 3350) && (__sys_vol > 3250))
        {
            *gain = (int32_t)(gain_fast_3P3);
            *offset = (int32_t)(offset_fast_3P3);
            gain_tense = *gain;
            offset_tense = *offset;
            return;
        }
        else
        {
            delta_gain = (gain_fast_5V - gain_fast_3P3) / (5000.0 - 3300.0);
            delta_offset = (offset_fast_5V - offset_fast_3P3) / (5000.0 - 3300.0);
            *gain = (int32_t)(gain_fast_3P3 + (__sys_vol - 3300) * delta_gain);
            *offset = (int32_t)(offset_fast_3P3 + (__sys_vol - 3300) * delta_offset);
            gain_tense = *gain;
            offset_tense = *offset;
            return;
        }
    }
    else if (vol_src == ALD_ADC_POS_REF_VREEFP)
    {
        tmp = (59 * (BIT6_CAL - bit6) + 29 * (BIT7_CAL - bit7) + 15 * (BIT8_CAL - bit8) + 7 * (BIT9_CAL - bit9) + 3 * (BIT10_CAL - bit10) + 1 * (BIT11_CAL - bit11)) / 4600.0;
        k = (0.819 * 65536 / GAIN_CAL) + tmp;
        b = OFFSET_CAL * (-1.0) / GAIN_CAL ;
        b = b + (2 * (BIT6_CAL - bit6) + (BIT7_CAL - bit7) - 200 * tmp);
        *gain = (int32_t)(0.819 / k * 65536);
        *offset = (int32_t)(-1) * (0.819 / k * 65536) * b;
        gain_tense = *gain;
        offset_tense = *offset;
        return;
    }
    else
    {
        *gain = 65536;
        *offset = 0;
        gain_tense = *gain;
        offset_tense = *offset;
        return;
    }

}

/**
  * @brief  Adc value adjust
  * @param  gain: ADC adjust gain value
  * @param  offset: ADC adjust offset value
  * @param  adc_val: Starting address where the original ADC data is stored
  * @param  len: Original ADC data length
  * @param  adc_mode: Trim mode for ADC
  * @retval result @ald_status_t
  */

ald_status_t ald_adc_value_adjust(int32_t gain, int32_t offset, uint16_t *adc_val, uint16_t len)
{
    uint16_t i = 0,  data_cal = 0;
    int16_t tmp = 0 ;
    uint16_t re_value = 0;

    if (CHIP_NUM == 0xFFAA0055)
    {
        bit6 = BIT6_CAL;
        bit7 = BIT7_CAL;
        bit8 = BIT8_CAL;
        bit9 = BIT9_CAL;
        bit10 = BIT10_CAL;
        bit11 = BIT11_CAL;
    }
    else
    {
        bit6 = 2;
        bit7 = 1;
        bit8 = 0;
        bit9 = 0;
        bit10 = 0;
        bit11 = 8;
    }

    data_cal = bit6 + bit7 + bit8 + bit9 + bit10 + bit11;

    for (i = 0; i < len; ++i)
    {
        tmp =  *(adc_val + i);

        if ((tmp > (2048 - data_cal)) && (tmp < 2048))
            tmp  = 2048;

        data_cal = (tmp >> 6) * bit6 + (tmp >> 7) * bit7 + (tmp >> 8) * bit8 + (tmp >> 9) * bit9 + (tmp >> 10) * bit10 + (tmp >> 11) * bit11;
        re_value = tmp - data_cal;
        tmp = (re_value * gain + offset + 32768) >> 16;

        if (tmp > 4095)
        {
            tmp = 4095;
        }
        else if (tmp < 0)
        {
            tmp = 0;
        }

        *(adc_val + i) = tmp;
    }

    return ALD_OK;
}

/**
  * @brief  Get adc Tense k
  * @param  None
  * @retval K:Indicates the proportionality between ADC output conversion and temperature
  */
#define TS_30       (*(uint16_t *)(0x809D0))
#define TS_85       (*(uint16_t *)(0x80DD0))
#define TS_110      (*(uint16_t *)(0x80B20))
#define TS_temp_30  (*(uint16_t *)(0x809D2))
#define TS_temp_85  (*(uint16_t *)(0x80DD2))
#define TS_temp_110 (*(uint16_t *)(0x80B22))


#define TS_30_3V3       (*(uint16_t *)(0x809D8))
#define TS_temp_30_3V3  (*(uint16_t *)(0x809DA))
#define TS_110_3V3      (*(uint16_t *)(0x80B28))
#define TS_temp_110_3V3 (*(uint16_t *)(0x80B2A))


float ald_get_temp_k(void)
{
    float k, k_3v3, k_5v, k_delta;

    k_5v = 2.85604;
    k_3v3 = 4.292308;

    if ((__sys_vol < 5100) && (__sys_vol > 4900))
    {
        if ((CHIP_NUM == 0xFFAA0055) && (TS_110 != 0xffff))  //CP 汾 CHIP_NUM == 0xFFAA0055110궨
        {
            if (READ_BIT(ADC0->CCR, ADC_CCR_VRNSEL_MSK)) //vdd
                k_5v = 2.9137;
            else
                k_5v = 2.9315;

            return k_5v;
        }
        else
        {
            k = k_5v;
            return k;
        }
    }
    else if ((__sys_vol < 3400) && (__sys_vol > 3200))
    {
        if ((CHIP_NUM == 0xFFAA0055) && (TS_110 != 0xffff))  //CP 汾 110궨
        {
            if (READ_BIT(ADC0->CCR, ADC_CCR_VRNSEL_MSK)) //vdd
                k_3v3 = 4.4151;
            else
                k_3v3 = 4.3877;

            return k_3v3;
        }
        else
        {
            k = k_3v3 ;
            return k;
        }
    }
    else
    {
        k_delta = (k_5v - k_3v3) / (5000.0 - 3300.0);
        k  = k_3v3 + k_delta * (__sys_vol - 3300.0);
        return k;
    }
}

/**
  * @brief  Get ADC tsensor conversion result.
  * @param  adc_value: ADC normal/insert group conversion result.
  * @param  k: Indicates the proportionality between ADC output conversion and temperature.
  * @retval ADC tsensor conversion data
  */
float ald_adc_tsensor_get_temperature(uint16_t adc_value)
{
    float k_3p3, k_5v, k_delta, k_lt;
    uint16_t TS_B, ad_value, Code_30, TS_BN, data_cal;
    uint16_t TS_temp;
    float Temperature, Temp_30;
    uint32_t d_gain, d_offset;

    if (CHIP_NUM == 0xFFAA0055)
    {
        bit6 = BIT6_CAL;
        bit7 = BIT7_CAL;
        bit8 = BIT8_CAL;
        bit9 = BIT9_CAL;
        bit10 = BIT10_CAL;
        bit11 = BIT11_CAL;
    }
    else
    {
        bit6 = 2;
        bit7 = 1;
        bit8 = 0;
        bit9 = 0;
        bit10 = 0;
        bit11 = 8;
    }

    /*trim num*/
    if (READ_BIT(ADC0->CCR, ADC_CCR_TRMEN_MSK))
    {
        d_gain  = (CFG_TRIM & 0x00007f00) >> 8;
        d_offset = CFG_TRIM & 0x0000007f;
        d_gain = adc_value * d_gain / 2048; //roughly calculate

        if ((CFG_TRIM  & (1 << 15)))
            ad_value = adc_value - d_gain;
        else ad_value = adc_value + d_gain;

        if ((CFG_TRIM  & (1 << 7)))
            ad_value += d_offset;
        else ad_value -= d_offset;

        data_cal = (ad_value >> 6) * bit6 + (ad_value >> 7) * bit7 + (ad_value >> 8) * bit8 + (ad_value >> 9) * bit9 + (ad_value >> 10) * bit10 + (ad_value >> 11) * bit11;
        ad_value = ad_value - data_cal;
    }
    else
    {
        ad_value = (adc_value * 65536 - offset_tense + gain_tense / 2) / gain_tense  ;
    }

    if ((CHIP_NUM == 0xFFAA0055) && (TS_110 != 0xffff))  //CP 汾 110궨

    {
        if (READ_BIT(ADC0->CCR, ADC_CCR_VRNSEL_MSK)) //vdd
        {
            k_5v = 2.8178;
            k_3p3 = 4.2411;
        }
        else
        {
            k_5v = 2.7803;
            k_3p3 = 4.2439;
        }
    }
    else
    {
        k_5v = 2.716667;
        k_3p3 = 4.15;
    }

    if ((__sys_vol < 5100) && (__sys_vol > 4900)) //Ӧõѹ5v
    {
        if ((CHIP_NUM == 0xFFAA0055) && (TS_110 != 0xffff))  //CP 汾 110궨
        {
            Code_30 = (TS_30 * GAIN_CAL + OFFSET_CAL + 32768) >> 16;
            Temp_30 = (TS_temp_30 >> 8) + (TS_temp_30 & 0x00FF) / 256.0;

            if (READ_BIT(ADC0->CCR, ADC_CCR_VRNSEL_MSK))
                ad_value = (ad_value  * GAIN5V_CAL + OFFSET5V_CAL + 32768) >> 16;
            else
                ad_value = (ad_value  * GAIN_CAL + OFFSET_CAL + 32768) >> 16;
        }
        else
        {
            TS_B = TS_30;
            TS_BN = ~(*((uint16_t *)(0x80000 + 0x9D4)));

            if (TS_B != TS_BN)
            {
                TS_B = TS_BN;
                Code_30 = TS_B & 0xFFFF;
                TS_temp = ~(*((uint16_t *)(0x80000 + 0x9D6)));
                Temp_30 = (TS_temp >> 8) + (TS_temp & 0x00FF) / 256.0;
            }
            else
            {
                Code_30 = TS_B & 0xFFFF;
                TS_temp = TS_temp_30;
                Temp_30 = (TS_temp >> 8) + (TS_temp & 0x00FF) / 256.0;
            }
        }
    }
    else if ((__sys_vol < 3400) && (__sys_vol > 3200))
    {
        if ((CHIP_NUM == 0xFFAA0055) && (TS_110 != 0xffff))  //CP 汾 110궨
        {
            Code_30 = (TS_30_3V3 * GAIN_CAL + OFFSET_CAL + 32768) >> 16;
            Temp_30 = (TS_temp_30 >> 8) + (TS_temp_30 & 0x00FF) / 256.0;

            if (READ_BIT(ADC0->CCR, ADC_CCR_VRNSEL_MSK))
                ad_value = (ad_value  * GAIN3V3_CAL + OFFSET3V3_CAL + 32768) >> 16;
            else
                ad_value = (ad_value  * GAIN_CAL + OFFSET_CAL + 32768) >> 16;
        }
        else if (TS_30_3V3 != 0xffff)  //3.3VֻҪб궨ֵ,CP汾2024/11
        {
            Code_30 = TS_30_3V3;
            Temp_30 = (TS_temp_30_3V3 >> 8) + (TS_temp_30_3V3 & 0x00FF) / 256.0;
        }
        else
        {
            TS_B = TS_30;
            TS_BN = ~(*((uint16_t *)(0x80000 + 0x9D4)));

            if (TS_B != TS_BN)
            {
                TS_B = TS_BN;
                Code_30 = TS_B & 0xFFFF;
                TS_temp = ~(*((uint16_t *)(0x80000 + 0x9D6)));
                Temp_30 = (TS_temp >> 8) + (TS_temp & 0x00FF) / 256.0;
            }
            else
            {
                Code_30 = TS_B & 0xFFFF;
                TS_temp = TS_temp_30;
                Temp_30 = (TS_temp >> 8) + (TS_temp & 0x00FF) / 256.0;
            }

            Code_30 = Code_30 / 3.3 * 5;
        }
    }
    else
    {
        if ((CHIP_NUM == 0xFFAA0055) && (TS_110 != 0xffff))  //CP 汾 110궨 5V/3.3V
        {
            Code_30 = (TS_30_3V3 * GAIN3V3_CAL + OFFSET3V3_CAL + 32768) >> 16;
            Temp_30 = (TS_temp_30 >> 8) + (TS_temp_30 & 0x00FF) / 256.0;
            Code_30 = Code_30 * 3300  / __sys_vol;  //3.3V code 㵱ǰϵͳѹµ30궨ֵ

            if (READ_BIT(ADC0->CCR, ADC_CCR_VRNSEL_MSK) == 1)
                ad_value = (ad_value  * gain_tense + offset_tense + 32768) >> 16;
            else if (READ_BIT(ADC0->CCR, ADC_CCR_VRNSEL_MSK) == 0)
                ad_value = (ad_value  * GAIN_CAL + OFFSET_CAL + 32768) >> 16;
        }

        if (TS_30_3V3 != 0xffff)
        {
            Code_30 = TS_30_3V3;
            Temp_30 = (TS_temp_30_3V3 >> 8) + (TS_temp_30_3V3 & 0x00FF) / 256.0;
            Code_30 = Code_30 * 3300  / __sys_vol;
        }
        else
        {
            TS_B = TS_30;
            TS_BN = ~(*((uint16_t *)(0x80000 + 0x9D4)));

            if (TS_B != TS_BN)
            {
                TS_B = TS_BN;
                Code_30 = TS_B & 0xFFFF;
                TS_temp = ~(*((uint16_t *)(0x80000 + 0x9D6)));
                Temp_30 = (TS_temp >> 8) + (TS_temp & 0x00FF) / 256.0;
            }
            else
            {
                Code_30 = TS_B & 0xFFFF;
                TS_temp = TS_temp_30;
                Temp_30 = (TS_temp >> 8) + (TS_temp & 0x00FF) / 256.0;
            }

            Code_30 = Code_30 * 5000 / __sys_vol;
        }
    }

    // fitting function
    if (ad_value >= Code_30)
    {
        Temperature = (ad_value - Code_30) / temp_k + Temp_30 ; //ht
    }
    else
    {
        if ((__sys_vol < 5050) && (__sys_vol > 4950))
            k_lt  = k_5v;
        else if ((__sys_vol < 3350) && (__sys_vol > 3250))
            k_lt  = k_3p3;
        else
        {
            k_delta = (k_5v - k_3p3) / (5000.0 - 3300.0);
            k_lt  = k_3p3 + k_delta * (__sys_vol - 3300);
        }

        Temperature = Temp_30 - (Code_30 - ad_value) / k_lt ; //lt
    }

    return Temperature;
}

/**
  *@}
  */

/**
  *@}
  */

/** @defgroup ADC_Private_Functions ADC Private Functions
  * @{
  */
#ifdef ALD_DMA
/**
  * @brief  DMA transfer complete callback.
  * @param  arg: argument of the call back.
  * @retval None
  */
static void adc_dma_normal_conv_cplt(void *arg)
{
    ald_adc_handle_t *hperh = (ald_adc_handle_t *)arg;

    if (hperh->normal_cplt_cbk)
        hperh->normal_cplt_cbk(hperh);
}
#endif
/**
  *@}
  */

#endif /* ALD_ADC */

/**
  *@}
  */

/**
  *@}
  */
