/**
  ******************************************************************************
  * @file    ald_rtc.h
  * @brief   Header file of RTC Module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_RTC_H__
#define __ALD_RTC_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup RTC
  * @{
  */

/** @defgroup RTC_Public_Types RTC Public Types
  * @{
  */

/**
  * @brief Hours format
  */
typedef enum
{
    ALD_RTC_HOUR_FORMAT_24 = 0x0U,  /**< 24-hours format */
    ALD_RTC_HOUR_FORMAT_12 = 0x1U,  /**< 12-hours format */
} ald_rtc_hour_format_t;

/**
  * @brief Output mode
  */
typedef enum
{
    ALD_RTC_OUTPUT_DISABLE = 0x0U,  /**< Disable output */
    ALD_RTC_OUTPUT_ALARM_A = 0x1U,  /**< Output alarm_a signal */
    ALD_RTC_OUTPUT_ALARM_B = 0x2U,  /**< Output alarm_b signal */
    ALD_RTC_OUTPUT_WAKEUP  = 0x3U,  /**< Output wakeup signal */
} ald_rtc_output_select_t;

/**
  * @brief Output polarity
  */
typedef enum
{
    ALD_RTC_OUTPUT_POLARITY_HIGH = 0x0U,    /**< Polarity is high */
    ALD_RTC_OUTPUT_POLARITY_LOW  = 0x1U,    /**< Polarity is low */
} ald_rtc_output_polarity_t;

/**
  * @brief Initialization structure
  */
typedef struct
{
    ald_rtc_hour_format_t hour_format;          /**< Hours format */
    uint32_t asynch_pre_div;                    /**< Asynchronous predivider value */
    uint32_t synch_pre_div;                     /**< Synchronous predivider value */
    ald_rtc_output_select_t output;             /**< Output signal type */
    ald_rtc_output_polarity_t output_polarity;  /**< Output polarity */
} ald_rtc_init_t;

/**
  * @brief Time structure
  */
typedef struct
{
    uint8_t hour;       /**< Hours */
    uint8_t minute;     /**< Minutes */
    uint8_t second;     /**< Seconds */
    uint16_t sub_sec;   /**< Sub-seconds */
} ald_rtc_time_t;

/**
  * @brief Date structure
  */
typedef struct
{
    uint8_t week;   /**< Weeks */
    uint8_t day;    /**< days */
    uint8_t month;  /**< months */
    uint8_t year;   /**< years */
} ald_rtc_date_t;

/**
  * @brief Data format
  */
typedef enum
{
    ALD_RTC_FORMAT_DEC = 0U,    /**< Decimal */
    ALD_RTC_FORMAT_BCD = 1U,    /**< BSD */
} ald_rtc_format_t;

/**
  * @brief Index of alarm
  */
typedef enum
{
    ALD_RTC_ALARM_A = 0x0U, /**< Alarm-A */
    ALD_RTC_ALARM_B = 0x1U, /**< Alarm-B */
} ald_rtc_alarm_idx_t;

/**
  * @brief Alarm mask
  */
typedef enum
{
    ALD_RTC_ALARM_MASK_NONE     = 0x0U,     /**< Mask is disable */
    ALD_RTC_ALARM_MASK_WEEK_DAY = (1U << 30),   /**< Mask week or day */
    ALD_RTC_ALARM_MASK_HOUR     = (1U << 23),   /**< Mask hour */
    ALD_RTC_ALARM_MASK_MINUTE   = (1U << 15),   /**< Mask minute */
    ALD_RTC_ALARM_MASK_SECOND   = (1U << 7),    /**< Mask second */
    ALD_RTC_ALARM_MASK_ALL      = 0x40808080U,  /**< Mask all */
} ald_rtc_alarm_mask_t;

/**
  * @brief Alarm sub-second mask
  */
typedef enum
{
    ALD_RTC_ALARM_SS_MASK_NONE  = 0xFU, /**< Mask is disable */
    ALD_RTC_ALARM_SS_MASK_14_1  = 0x1U, /**< Mask bit(1-14) */
    ALD_RTC_ALARM_SS_MASK_14_2  = 0x2U, /**< Mask bit(2-14) */
    ALD_RTC_ALARM_SS_MASK_14_3  = 0x3U, /**< Mask bit(3-14) */
    ALD_RTC_ALARM_SS_MASK_14_4  = 0x4U, /**< Mask bit(4-14) */
    ALD_RTC_ALARM_SS_MASK_14_5  = 0x5U, /**< Mask bit(5-14) */
    ALD_RTC_ALARM_SS_MASK_14_6  = 0x6U, /**< Mask bit(6-14) */
    ALD_RTC_ALARM_SS_MASK_14_7  = 0x7U, /**< Mask bit(7-14) */
    ALD_RTC_ALARM_SS_MASK_14_8  = 0x8U, /**< Mask bit(8-14) */
    ALD_RTC_ALARM_SS_MASK_14_9  = 0x9U, /**< Mask bit(9-14) */
    ALD_RTC_ALARM_SS_MASK_14_10 = 0xAU, /**< Mask bit(10-14) */
    ALD_RTC_ALARM_SS_MASK_14_11 = 0xBU, /**< Mask bit(11-14) */
    ALD_RTC_ALARM_SS_MASK_14_12 = 0xCU, /**< Mask bit(12-14) */
    ALD_RTC_ALARM_SS_MASK_14_13 = 0xDU, /**< Mask bit(13-14) */
    ALD_RTC_ALARM_SS_MASK_14    = 0xEU, /**< Mask bit14 */
    ALD_RTC_ALARM_SS_MASK_ALL   = 0x0U, /**< Mask bit(0-14) */
} ald_rtc_sub_second_mask_t;

/**
  * @brief Alarm select week or day */
typedef enum
{
    ALD_RTC_SELECT_DAY  = 0x0U, /**< Alarm select day */
    ALD_RTC_SELECT_WEEK = 0x1U, /**< Alarm select week */
} ald_rtc_week_day_sel_t;

/**
  * @brief Alarm structure
  */
typedef struct
{
    ald_rtc_alarm_idx_t idx;            /**< Index of alarm */
    ald_rtc_time_t time;                /**< Time structure */
    uint32_t mask;                      /**< Alarm mask */
    ald_rtc_sub_second_mask_t ss_mask;  /**< Alarm sub-second mask */
    ald_rtc_week_day_sel_t sel;         /**< Select week or day */

    union
    {
        uint8_t week;                   /**< Alarm select week */
        uint8_t day;                    /**< Alarm select day */
    };
} ald_rtc_alarm_t;

/**
  * @brief Time stamp signel select
  */
typedef enum
{
    ALD_RTC_TS_SIGNAL_SEL_TAMPER0 = 0U, /**< Select tamper0 */
    ALD_RTC_TS_SIGNAL_SEL_TAMPER1 = 1U, /**< Select tamper1 */
} ald_rtc_ts_signal_sel_t;

/**
  * @brief Time stamp trigger style
  */
typedef enum
{
    ALD_RTC_TS_RISING_EDGE  = 0U,   /**< Rising edge */
    ALD_RTC_TS_FALLING_EDGE = 1U,   /**< Falling edge */
} ald_rtc_ts_trigger_style_t;

/**
  * @brief Index of tamper
  */
typedef enum
{
    ALD_RTC_TAMPER_0 = 0U,  /**< Tamper0 */
    ALD_RTC_TAMPER_1 = 1U,  /**< Tamper1 */
} ald_rtc_tamper_idx_t;

/**
  * @brief Tamper trigger type
  */
typedef enum
{
    ALD_RTC_TAMPER_TRIGGER_LOW  = 0U,   /**< High trigger */
    ALD_RTC_TAMPER_TRIGGER_HIGH = 1U,   /**< Low trigger */
} ald_rtc_tamper_trigger_t;

/**
  * @brief Tamper sampling frequency
  */
typedef enum
{
    ALD_RTC_TAMPER_SAMPLING_FREQ_32768 = 0U,    /**< RTCCLK / 32768 */
    ALD_RTC_TAMPER_SAMPLING_FREQ_16384 = 1U,    /**< RTCCLK / 16384 */
    ALD_RTC_TAMPER_SAMPLING_FREQ_8192  = 2U,    /**< RTCCLK / 8192 */
    ALD_RTC_TAMPER_SAMPLING_FREQ_4096  = 3U,    /**< RTCCLK / 4096 */
    ALD_RTC_TAMPER_SAMPLING_FREQ_2048  = 4U,    /**< RTCCLK / 2048 */
    ALD_RTC_TAMPER_SAMPLING_FREQ_1024  = 5U,    /**< RTCCLK / 1024 */
    ALD_RTC_TAMPER_SAMPLING_FREQ_512   = 6U,    /**< RTCCLK / 512 */
    ALD_RTC_TAMPER_SAMPLING_FREQ_256   = 7U,    /**< RTCCLK / 256 */
} ald_rtc_tamper_sampling_freq_t;

/**
  * @brief Tamper filter time
  */
typedef enum
{
    ALD_RTC_TAMPER_DURATION_1 = 0U, /**< Duration 1 sampling */
    ALD_RTC_TAMPER_DURATION_2 = 1U, /**< Duration 2 sampling */
    ALD_RTC_TAMPER_DURATION_4 = 2U, /**< Duration 4 sampling */
    ALD_RTC_TAMPER_DURATION_8 = 3U, /**< Duration 8 sampling */
} ald_rtc_tamper_duration_t;

/**
  * @brief Tamper structure
  */
typedef struct
{
    ald_rtc_tamper_idx_t idx;               /**< Index of tamper */
    ald_rtc_tamper_trigger_t trig;          /**< Trigger type */
    ald_rtc_tamper_sampling_freq_t freq;    /**< Sampling frequency */
    ald_rtc_tamper_duration_t dur;          /**< Filter time */
    type_func_t ts;                         /**< Enable/Disable trigger time stamp event */
} ald_rtc_tamper_t;

/**
  * @brief Wake-up clock
  */
typedef enum
{
    ALD_RTC_WAKEUP_CLOCK_DIV_16   = 0U, /**< RTCCLK / 16 */
    ALD_RTC_WAKEUP_CLOCK_DIV_8    = 1U, /**< RTCCLK / 8 */
    ALD_RTC_WAKEUP_CLOCK_DIV_4    = 2U, /**< RTCCLK / 4 */
    ALD_RTC_WAKEUP_CLOCK_DIV_2    = 3U, /**< RTCCLK / 2 */
    ALD_RTC_WAKEUP_CLOCK_1HZ      = 4U, /**< 1Hz */
    ALD_RTC_WAKEUP_CLOCK_1HZ_PULS = 6U, /**< 1Hz and WUT + 65536 */
} ald_rtc_wakeup_clock_t;

/**
  * @brief RTC clock output type
  */
typedef enum
{
    ALD_RTC_CLOCK_OUTPUT_32768 = 0U,    /**< 32768Hz */
    ALD_RTC_CLOCK_OUTPUT_1024  = 1U,    /**< 1024Hz */
    ALD_RTC_CLOCK_OUTPUT_32    = 2U,    /**< 32Hz */
    ALD_RTC_CLOCK_OUTPUT_1     = 3U,    /**< 1Hz */
    ALD_RTC_CLOCK_OUTPUT_CAL_1 = 4U,    /**< 1Hz after calibration */
} ald_rtc_clock_output_t;

/**
  * @ Calibration frequency
  */
typedef enum
{
    ALD_RTC_CALI_FREQ_10_SEC = 0U,  /**< Calibrate every 10 seconds */
    ALD_RTC_CALI_FREQ_20_SEC = 1U,  /**< Calibrate every 20 seconds */
    ALD_RTC_CALI_FREQ_1_MIN  = 2U,  /**< Calibrate every 1 minute */
    ALD_RTC_CALI_FREQ_2_MIN  = 3U,  /**< Calibrate every 2 minutes */
    ALD_RTC_CALI_FREQ_5_MIN  = 4U,  /**< Calibrate every 5 minutes */
    ALD_RTC_CALI_FREQ_10_MIN = 5U,  /**< Calibrate every 10 minutes */
    ALD_RTC_CALI_FREQ_20_MIN = 6U,  /**< Calibrate every 20 minutes */
    ALD_RTC_CALI_FREQ_1_SEC  = 7U,  /**< Calibrate every 1 second */
} ald_rtc_cali_freq_t;

/**
  * @brief Temperature compensate type
  */
typedef enum
{
    ALD_RTC_CALI_TC_NONE          = 0U, /**< Temperature compensate disable */
    ALD_RTC_CALI_TC_AUTO_BY_HW    = 1U, /**< Temperature compensate by hardware */
    ALD_RTC_CALI_TC_AUTO_BY_SF    = 2U, /**< Temperature compensate by software */
    ALD_RTC_CALI_TC_AUTO_BY_HW_SF = 3U, /**< Temperature compensate by hardware, trigger by software */
} ald_rtc_cali_tc_t;

/**
  * @ Calculate frequency
  */
typedef enum
{
    ALD_RTC_CALI_CALC_FREQ_10_SEC = 0U, /**< Calculate every 10 seconds */
    ALD_RTC_CALI_CALC_FREQ_20_SEC = 1U, /**< Calculate every 20 seconds */
    ALD_RTC_CALI_CALC_FREQ_1_MIN  = 2U, /**< Calculate every 1 minute */
    ALD_RTC_CALI_CALC_FREQ_2_MIN  = 3U, /**< Calculate every 2 minutes */
    ALD_RTC_CALI_CALC_FREQ_5_MIN  = 4U, /**< Calculate every 5 minutes */
    ALD_RTC_CALI_CALC_FREQ_10_MIN = 5U, /**< Calculate every 10 minutes */
    ALD_RTC_CALI_CALC_FREQ_20_MIN = 6U, /**< Calculate every 20 minutes */
    ALD_RTC_CALI_CALC_FREQ_1_HOUR = 7U, /**< Calculate every 1 hour */
} ald_rtc_cali_calc_freq_t;

/**
  * @brief Calibration algorithm
  */
typedef enum
{
    ALD_RTC_CALI_CALC_4 = 0U,   /**< 4-polynomial */
    ALD_RTC_CALI_CALC_2 = 1U,   /**< 2-parabola */
} ald_rtc_cali_calc_t;

/**
  * @brief Calibration structure
  */
typedef struct
{
    ald_rtc_cali_freq_t cali_freq;          /**< calibrate frequency */
    ald_rtc_cali_tc_t tc;                   /**< Temperature compensate type */
    ald_rtc_cali_calc_freq_t calc_freq;     /**< Calculate frequency */
    ald_rtc_cali_calc_t calc;               /**< algorithm */
    type_func_t acc;                        /**< Enable/Disable decimal accumulate */
} ald_rtc_cali_t;

/**
  * @brief Interrupt type
  */
typedef enum
{
    ALD_RTC_IT_SEC  = (1U << 0),    /**< Second */
    ALD_RTC_IT_MIN  = (1U << 1),    /**< Minute */
    ALD_RTC_IT_HR   = (1U << 2),    /**< Hour */
    ALD_RTC_IT_DAY  = (1U << 3),    /**< Day */
    ALD_RTC_IT_MON  = (1U << 4),    /**< Month */
    ALD_RTC_IT_YR   = (1U << 5),    /**< Year */
    ALD_RTC_IT_ALMA = (1U << 8),    /**< Alarm-A */
    ALD_RTC_IT_ALMB = (1U << 9),    /**< Alarm-B */
    ALD_RTC_IT_TS   = (1U << 10),   /**< Time stamp */
    ALD_RTC_IT_TSOV = (1U << 11),   /**< Time stamp overflow */
    ALD_RTC_IT_TP0  = (1U << 12),   /**< Tamper-0 */
    ALD_RTC_IT_TP1  = (1U << 13),   /**< Tamper-1 */
    ALD_RTC_IT_RSC  = (1U << 16),   /**< Synchronous complete */
    ALD_RTC_IT_SFC  = (1U << 17),   /**< Shift complete */
    ALD_RTC_IT_WU   = (1U << 18),   /**< Wake-up */
    ALD_RTC_IT_TCC  = (1U << 24),   /**< Temperature compensate complete */
    ALD_RTC_IT_TCE  = (1U << 25),   /**< Temperature compensate error */
} ald_rtc_it_t;

/**
  * @brief Interrupt flag
  */
typedef enum
{
    ALD_RTC_IF_SEC  = (1U << 0),    /**< Second */
    ALD_RTC_IF_MIN  = (1U << 1),    /**< Minute */
    ALD_RTC_IF_HR   = (1U << 2),    /**< Hour */
    ALD_RTC_IF_DAY  = (1U << 3),    /**< Day */
    ALD_RTC_IF_MON  = (1U << 4),    /**< Month */
    ALD_RTC_IF_YR   = (1U << 5),    /**< Year */
    ALD_RTC_IF_ALMA = (1U << 8),    /**< Alarm-A */
    ALD_RTC_IF_ALMB = (1U << 9),    /**< Alarm-B */
    ALD_RTC_IF_TS   = (1U << 10),   /**< Time stamp */
    ALD_RTC_IF_TSOV = (1U << 11),   /**< Time stamp overflow */
    ALD_RTC_IF_TP0  = (1U << 12),   /**< Tamper-0 */
    ALD_RTC_IF_TP1  = (1U << 13),   /**< Tamper-1 */
    ALD_RTC_IF_RSC  = (1U << 16),   /**< Synchronous complete */
    ALD_RTC_IF_SFC  = (1U << 17),   /**< Shift complete */
    ALD_RTC_IF_WU   = (1U << 18),   /**< Wake-up */
    ALD_RTC_IF_TCC  = (1U << 24),   /**< Temperature compensate complete */
    ALD_RTC_IF_TCE  = (1U << 25),   /**< Temperature compensate error */
} ald_rtc_flag_t;
/**
  * @}
  */

/** @defgroup RTC_Public_Macro RTC Public Macros
  * @{
  */
#define RTC_UNLOCK()        (WRITE_REG(RTC->WPR, 0x55AAAA55U))
#define RTC_LOCK()      (WRITE_REG(RTC->WPR, 0x0U))
#define RTC_BY_PASS_ENABLE()            \
    do {                        \
        RTC_UNLOCK();               \
        SET_BIT(RTC->CON, RTC_CON_SHDBP_MSK);   \
        RTC_LOCK();             \
    } while (0)
#define RTC_BY_PASS_DISABLE()           \
    do {                        \
        RTC_UNLOCK();               \
        CLEAR_BIT(RTC->CON, RTC_CON_SHDBP_MSK); \
        RTC_LOCK();             \
    } while (0)
#define RTC_SUMMER_TIME_ENABLE()        \
    do {                        \
        RTC_UNLOCK();               \
        SET_BIT(RTC->CON, RTC_CON_ADD1H_MSK);   \
        RTC_LOCK();             \
    } while (0)
#define RTC_SUMMER_TIME_DISABLE()       \
    do {                        \
        RTC_UNLOCK();               \
        CLEAR_BIT(RTC->CON, RTC_CON_ADD1H_MSK); \
        RTC_LOCK();             \
    } while (0)
#define RTC_WINTER_TIME_ENABLE()        \
    do {                        \
        RTC_UNLOCK();               \
        SET_BIT(RTC->CON, RTC_CON_SUB1H_MSK);   \
        RTC_LOCK();             \
    } while (0)
#define RTC_WINTER_TIME_DISABLE()       \
    do {                        \
        RTC_UNLOCK();               \
        CLEAR_BIT(RTC->CON, RTC_CON_SUB1H_MSK); \
        RTC_LOCK();             \
    } while (0)
/**
 * @}
 */

/** @defgroup RTC_Private_Macros RTC Private Macros
  * @{
  */
#define RTC_CALI_UNLOCK()   (WRITE_REG(RTC->CALWPR, 0x699655AAU))
#define RTC_CALI_LOCK()     (WRITE_REG(RTC->CALWPR, 0x0U))
#define ALARM_MASK_ALL      0x40808080
#define RTC_TIMEOUT_VALUE   100

#define IS_SHIFT_SUB_SS(x)  ((x) < (1U << 15))
#define IS_RTC_HOUR_FORMAT(x)   (((x) == ALD_RTC_HOUR_FORMAT_24) || \
                                 ((x) == ALD_RTC_HOUR_FORMAT_12))
#define IS_RTC_OUTPUT_SEL(x)    (((x) == ALD_RTC_OUTPUT_DISABLE) || \
                                 ((x) == ALD_RTC_OUTPUT_ALARM_A) || \
                                 ((x) == ALD_RTC_OUTPUT_ALARM_B) || \
                                 ((x) == ALD_RTC_OUTPUT_WAKEUP))
#define IS_RTC_OUTPUT_POLARITY(x)   (((x) == ALD_RTC_OUTPUT_POLARITY_HIGH) || \
                                     ((x) == ALD_RTC_OUTPUT_POLARITY_LOW))
#define IS_RTC_ALARM(x)     (((x) == ALD_RTC_ALARM_A) || \
                             ((x) == ALD_RTC_ALARM_B))
#define IS_RTC_ALARM_SEL(x) (((x) == ALD_RTC_SELECT_DAY) || \
                             ((x) == ALD_RTC_SELECT_WEEK))
#define IS_RTC_ALARM_MASK(x)    (((x) == ALD_RTC_ALARM_MASK_NONE)     || \
                                 ((x) == ALD_RTC_ALARM_MASK_WEEK_DAY) || \
                                 ((x) == ALD_RTC_ALARM_MASK_HOUR)     || \
                                 ((x) == ALD_RTC_ALARM_MASK_MINUTE)   || \
                                 ((x) == ALD_RTC_ALARM_MASK_SECOND)   || \
                                 ((x) == ALD_RTC_ALARM_MASK_ALL))
#define IS_RTC_ALARM_SS_MASK(x) (((x) == ALD_RTC_ALARM_SS_MASK_NONE)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_1)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_2)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_3)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_4)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_5)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_6)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_7)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_8)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_9)  || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_10) || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_11) || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_12) || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14_13) || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_14)    || \
                                 ((x) == ALD_RTC_ALARM_SS_MASK_ALL))
#define IS_RTC_TS_SIGNAL(x) (((x) == ALD_RTC_TS_SIGNAL_SEL_TAMPER0) || \
                             ((x) == ALD_RTC_TS_SIGNAL_SEL_TAMPER1))
#define IS_RTC_TS_STYLE(x)  (((x) == ALD_RTC_TS_RISING_EDGE) || \
                             ((x) == ALD_RTC_TS_FALLING_EDGE))
#define IS_RTC_FORMAT(x)    (((x) == ALD_RTC_FORMAT_DEC) || \
                             ((x) == ALD_RTC_FORMAT_BCD))
#define IS_RTC_TAMPER(x)    (((x) == ALD_RTC_TAMPER_0) || \
                             ((x) == ALD_RTC_TAMPER_1))
#define IS_RTC_TAMPER_TRIGGER(x)    (((x) == ALD_RTC_TAMPER_TRIGGER_LOW) || \
                                     ((x) == ALD_RTC_TAMPER_TRIGGER_HIGH))
#define IS_RTC_TAMPER_SAMPLING_FREQ(x)  (((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_32768) || \
        ((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_16384) || \
        ((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_8192)  || \
        ((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_4096)  || \
        ((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_2048)  || \
        ((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_1024)  || \
        ((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_512)   || \
        ((x) == ALD_RTC_TAMPER_SAMPLING_FREQ_256))
#define IS_RTC_TAMPER_DURATION(x)   (((x) == ALD_RTC_TAMPER_DURATION_1) || \
                                     ((x) == ALD_RTC_TAMPER_DURATION_2) || \
                                     ((x) == ALD_RTC_TAMPER_DURATION_4) || \
                                     ((x) == ALD_RTC_TAMPER_DURATION_8))
#define IS_RTC_WAKEUP_CLOCK(x)  (((x) == ALD_RTC_WAKEUP_CLOCK_DIV_16) || \
                                 ((x) == ALD_RTC_WAKEUP_CLOCK_DIV_8)  || \
                                 ((x) == ALD_RTC_WAKEUP_CLOCK_DIV_4)  || \
                                 ((x) == ALD_RTC_WAKEUP_CLOCK_DIV_2)  || \
                                 ((x) == ALD_RTC_WAKEUP_CLOCK_1HZ)    || \
                                 ((x) == ALD_RTC_WAKEUP_CLOCK_1HZ_PULS))
#define IS_RTC_CLOCK_OUTPUT(x)  (((x) == ALD_RTC_CLOCK_OUTPUT_32768) || \
                                 ((x) == ALD_RTC_CLOCK_OUTPUT_1024)  || \
                                 ((x) == ALD_RTC_CLOCK_OUTPUT_32)    || \
                                 ((x) == ALD_RTC_CLOCK_OUTPUT_1)     || \
                                 ((x) == ALD_RTC_CLOCK_OUTPUT_CAL_1))
#define IS_RTC_CALI_FREQ(x) (((x) == ALD_RTC_CALI_FREQ_10_SEC) || \
                             ((x) == ALD_RTC_CALI_FREQ_20_SEC) || \
                             ((x) == ALD_RTC_CALI_FREQ_1_MIN)  || \
                             ((x) == ALD_RTC_CALI_FREQ_2_MIN)  || \
                             ((x) == ALD_RTC_CALI_FREQ_5_MIN)  || \
                             ((x) == ALD_RTC_CALI_FREQ_10_MIN) || \
                             ((x) == ALD_RTC_CALI_FREQ_20_MIN) || \
                             ((x) == ALD_RTC_CALI_FREQ_1_SEC))
#define IS_RTC_CALI_TC(x)   (((x) == ALD_RTC_CALI_TC_NONE)       || \
                             ((x) == ALD_RTC_CALI_TC_AUTO_BY_HW) || \
                             ((x) == ALD_RTC_CALI_TC_AUTO_BY_SF) || \
                             ((x) == ALD_RTC_CALI_TC_AUTO_BY_HW_SF))
#define IS_RTC_CALC_FREQ(x) (((x) == ALD_RTC_CALI_CALC_FREQ_10_SEC) || \
                             ((x) == ALD_RTC_CALI_CALC_FREQ_20_SEC) || \
                             ((x) == ALD_RTC_CALI_CALC_FREQ_1_MIN)  || \
                             ((x) == ALD_RTC_CALI_CALC_FREQ_2_MIN)  || \
                             ((x) == ALD_RTC_CALI_CALC_FREQ_5_MIN)  || \
                             ((x) == ALD_RTC_CALI_CALC_FREQ_10_MIN) || \
                             ((x) == ALD_RTC_CALI_CALC_FREQ_20_MIN) || \
                             ((x) == ALD_RTC_CALI_CALC_FREQ_1_HOUR))
#define IS_RTC_CALI_CALC(x) (((x) == ALD_RTC_CALI_CALC_4) || \
                             ((x) == ALD_RTC_CALI_CALC_2))
#define IS_RTC_IT(x)        (((x) == ALD_RTC_IT_SEC)  || \
                             ((x) == ALD_RTC_IT_MIN)  || \
                             ((x) == ALD_RTC_IT_HR)   || \
                             ((x) == ALD_RTC_IT_DAY)  || \
                             ((x) == ALD_RTC_IT_MON)  || \
                             ((x) == ALD_RTC_IT_YR)   || \
                             ((x) == ALD_RTC_IT_ALMA) || \
                             ((x) == ALD_RTC_IT_ALMB) || \
                             ((x) == ALD_RTC_IT_TS)   || \
                             ((x) == ALD_RTC_IT_TSOV) || \
                             ((x) == ALD_RTC_IT_TP0)  || \
                             ((x) == ALD_RTC_IT_TP1)  || \
                             ((x) == ALD_RTC_IT_RSC)  || \
                             ((x) == ALD_RTC_IT_SFC)  || \
                             ((x) == ALD_RTC_IT_WU)   || \
                             ((x) == ALD_RTC_IT_TCC)  || \
                             ((x) == ALD_RTC_IT_TCE))
#define IS_RTC_IF(x)        (((x) == ALD_RTC_IF_SEC)  || \
                             ((x) == ALD_RTC_IF_MIN)  || \
                             ((x) == ALD_RTC_IF_HR)   || \
                             ((x) == ALD_RTC_IF_DAY)  || \
                             ((x) == ALD_RTC_IF_MON)  || \
                             ((x) == ALD_RTC_IF_YR)   || \
                             ((x) == ALD_RTC_IF_ALMA) || \
                             ((x) == ALD_RTC_IF_ALMB) || \
                             ((x) == ALD_RTC_IF_TS)   || \
                             ((x) == ALD_RTC_IF_TSOV) || \
                             ((x) == ALD_RTC_IF_TP0)  || \
                             ((x) == ALD_RTC_IF_TP1)  || \
                             ((x) == ALD_RTC_IF_RSC)  || \
                             ((x) == ALD_RTC_IF_SFC)  || \
                             ((x) == ALD_RTC_IF_WU)   || \
                             ((x) == ALD_RTC_IF_TCC)  || \
                             ((x) == ALD_RTC_IF_TCE))
#define IS_RTC_SECOND(x)    ((x) < 60)
#define IS_RTC_MINUTE(x)    ((x) < 60)
#define IS_RTC_HOUR(x)      ((x) < 24)
#define IS_RTC_DAY(x)       (((x) > 0) && ((x) < 32))
#define IS_RTC_MONTH(x)     (((x) > 0) && ((x) < 13))
#define IS_RTC_YEAR(x)      ((x) < 100)

#define IS_RTC_BKPR_RAM_IDX(x)  ((x) < 32)
/**
  * @}
  */

/** @addtogroup RTC_Public_Functions
  * @{
  */

/** @addtogroup RTC_Public_Functions_Group1
  *  @{
  */
/* Initialization functions */
void ald_rtc_reset(void);
void ald_rtc_init(ald_rtc_init_t *init);
/**
  * @}
  */
/** @addtogroup RTC_Public_Functions_Group2
  * @{
  */
/* Time and date operation functions */
ald_status_t ald_rtc_set_time(ald_rtc_time_t *time, ald_rtc_format_t format);
ald_status_t ald_rtc_set_date(ald_rtc_date_t *date, ald_rtc_format_t format);
void ald_rtc_get_time(ald_rtc_time_t *time, ald_rtc_format_t format);
void ald_rtc_get_date(ald_rtc_date_t *date, ald_rtc_format_t format);
int32_t ald_rtc_get_date_time(ald_rtc_date_t *date, ald_rtc_time_t *time, ald_rtc_format_t format);
/**
  * @}
  */
/** @addtogroup RTC_Public_Functions_Group3
  * @{
  */
/* Alarm functions */
void ald_rtc_set_alarm(ald_rtc_alarm_t *alarm, ald_rtc_format_t format);
void ald_rtc_get_alarm(ald_rtc_alarm_t *alarm, ald_rtc_format_t format);
/**
  * @}
  */
/** @addtogroup RTC_Public_Functions_Group4
  * @{
  */
/* Time stamp functions */
void ald_rtc_set_time_stamp(ald_rtc_ts_signal_sel_t sel, ald_rtc_ts_trigger_style_t style);
void ald_rtc_cancel_time_stamp(void);
void ald_rtc_get_time_stamp(ald_rtc_time_t *ts_time, ald_rtc_date_t *ts_date, ald_rtc_format_t format);
/**
  * @}
  */
/** @addtogroup RTC_Public_Functions_Group5
  * @{
  */
/* Tamper functions */
void ald_rtc_set_tamper(ald_rtc_tamper_t *tamper);
void ald_rtc_cancel_tamper(ald_rtc_tamper_idx_t idx);
/**
  * @}
  */
/** @addtogroup RTC_Public_Functions_Group6
  * @{
  */
/* Wakeup functions */
void ald_rtc_set_wakeup(ald_rtc_wakeup_clock_t clock, uint16_t value);
void ald_rtc_cancel_wakeup(void);
uint16_t ald_rtc_get_wakeup_timer_value(void);
/**
  * @}
  */
/** @addtogroup RTC_Public_Functions_Group7
  * @{
  */
/* Clock output functions */
ald_status_t ald_rtc_set_clock_output(ald_rtc_clock_output_t clock);
void ald_rtc_cancel_clock_output(void);
/**
  * @}
  */
/** @addtogroup RTC_Public_Functions_Group8
  * @{
  */
/* Control functions */
void ald_rtc_interrupt_config(ald_rtc_it_t it, type_func_t state);
void ald_rtc_alarm_cmd(ald_rtc_alarm_idx_t idx, type_func_t state);
ald_status_t ald_rtc_set_shift(type_func_t add_1s, uint16_t sub_ss);
void ald_rtc_set_cali(ald_rtc_cali_t *config);
void ald_rtc_cancel_cali(void);
ald_status_t ald_rtc_get_cali_status(void);
void ald_rtc_write_temp(uint16_t temp);
it_status_t ald_rtc_get_it_status(ald_rtc_it_t it);
flag_status_t ald_rtc_get_flag_status(ald_rtc_flag_t flag);
void ald_rtc_clear_flag_status(ald_rtc_flag_t flag);
/**
  * @}
  */

/** @addtogroup RTC_Public_Functions_Group9
  * @{
  */
void ald_rtc_bkpc_write_ram(uint8_t idx, uint32_t value);
uint32_t ald_rtc_bkpc_read_ram(uint8_t idx);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif
#endif
