/**
  *********************************************************************************
  *
  * @file    ald_rmu.h
  * @brief   Header file of RMU module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_RMU_H__
#define __ALD_RMU_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"

/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup RMU
  * @{
  */

/** @defgroup RMU_Public_Types RMU Public Types
  * @{
  */
/**
  * @brief RMU BOR fliter
  */
typedef enum
{
    ALD_RMU_BORFLT_1 = 0x1U,    /**< 1 cycle  */
    ALD_RMU_BORFLT_2 = 0x2U,    /**< 2 cycles */
    ALD_RMU_BORFLT_3 = 0x3U,    /**< 3 cycles */
    ALD_RMU_BORFLT_4 = 0x4U,    /**< 4 cycles */
    ALD_RMU_BORFLT_5 = 0x5U,    /**< 5 cycles */
    ALD_RMU_BORFLT_6 = 0x6U,    /**< 6 cycles */
    ALD_RMU_BORFLT_7 = 0x7U,    /**< 7 cycles */
} ald_rmu_bor_filter_t;

/**
  * @brief RMU BOR voltage
  */
typedef enum
{
    ALD_RMU_VOL_1_8  = 0x0U,    /**< 1.8V */
    ALD_RMU_VOL_2_0  = 0x1U,    /**< 2.0V */
    ALD_RMU_VOL_2_4  = 0x2U,    /**< 2.4V */
    ALD_RMU_VOL_2_8  = 0x3U,    /**< 2.8V */
    ALD_RMU_VOL_3_0  = 0x4U,    /**< 3.0V */
    ALD_RMU_VOL_3_5  = 0x5U,    /**< 3.5V */
    ALD_RMU_VOL_3_9  = 0x6U,    /**< 3.9V */
    ALD_RMU_VOL_4_5  = 0x7U,    /**< 4.5V */
} ald_rmu_bor_vol_t;

/**
  * @brief RMU reset status
  */
typedef enum
{
    ALD_RMU_RST_POR    = (1U << 0),     /**< POR */
    ALD_RMU_RST_WAKEUP = (1U << 1),     /**< WAKEUP */
    ALD_RMU_RST_BOR    = (1U << 2),     /**< BOR */
    ALD_RMU_RST_NMRST  = (1U << 3),     /**< NMRST */
    ALD_RMU_RST_IWDT   = (1U << 4),     /**< IWDT */
    ALD_RMU_RST_WWDT   = (1U << 5),     /**< WWDT */
    ALD_RMU_RST_LOCKUP = (1U << 6),     /**< LOCKUP */
    ALD_RMU_RST_CHIP   = (1U << 7),     /**< CHIP */
    ALD_RMU_RST_MCU    = (1U << 8),     /**< MCU */
    ALD_RMU_RST_CPU    = (1U << 9),     /**< CPU */
    ALD_RMU_RST_CFG    = (1U << 10),    /**< CFG */
    ALD_RMU_RST_CFGERR = (1U << 16),    /**< CFG error */
    ALD_RMU_RST_ALL    = (0xFFFFFU),    /**< ALL */
} ald_rmu_state_t;

/**
  * @brief RMU periperal select bit
  * @verbatim
      In this module, for the convenience of code maintenance,
      TIMERx is used to indicate the sequence of the timer peripheral.
      Different product series TIMERx represent different meanings:
      1. For ES32F36xx series:
      TIMER0 ----> AD16C4T0
      TIMER2 ----> GP32C4T0
      TIMER3 ----> GP32C4T1
      TIMER4 ----> BS16T0
      TIMER5 ----> BS16T1
      TIMER6 ----> GP16C4T0
      TIMER7 ----> GP16C4T1
    @endverbatim
  */
typedef enum
{
    ALD_RMU_PERH_GPIO    = (1U << 0),               /**< AHB1: GPIO */
    ALD_RMU_PERH_CRC     = (1U << 1),               /**< AHB1: CRC */
    ALD_RMU_PERH_CALC    = (1U << 2),               /**< AHB1: CALC */
    ALD_RMU_PERH_PIS     = (1U << 5),               /**< AHB1: PIS */
    ALD_RMU_PERH_CHIP    = (1U << 0)  | (1U << 27), /**< AHB2: CHIP */
    ALD_RMU_PERH_CPU     = (1U << 1)  | (1U << 27), /**< AHB2: CPU */
    ALD_RMU_PERH_DMA0    = (1U << 2)  | (1U << 27), /**< AHB2: DMA0 */
    ALD_RMU_PERH_DMA1    = (1U << 3)  | (1U << 27), /**< AHB2: DMA1 */
    ALD_RMU_PERH_DMAMUX  = (1U << 4)  | (1U << 27), /**< AHB2: DMAMUX */
    ALD_RMU_PERH_TIMER0  = (1U << 0)  | (1U << 28), /**< APB1: TIMER0 */
    ALD_RMU_PERH_TIMER2  = (1U << 2)  | (1U << 28), /**< APB1: TIMER2 */
    ALD_RMU_PERH_TIMER3  = (1U << 3)  | (1U << 28), /**< APB1: TIMER3 */
    ALD_RMU_PERH_TIMER4  = (1U << 4)  | (1U << 28), /**< APB1: TIMER4 */
    ALD_RMU_PERH_TIMER5  = (1U << 5)  | (1U << 28), /**< APB1: TIMER5 */
    ALD_RMU_PERH_TIMER6  = (1U << 6)  | (1U << 28), /**< APB1: TIMER6 */
    ALD_RMU_PERH_TIMER7  = (1U << 7)  | (1U << 28), /**< APB1: TIMER7 */
    ALD_RMU_PERH_UART0   = (1U << 8)  | (1U << 28), /**< APB1: UART0 */
    ALD_RMU_PERH_UART1   = (1U << 9)  | (1U << 28), /**< APB1: UART1 */
    ALD_RMU_PERH_UART2   = (1U << 10) | (1U << 28), /**< APB1: UART2 */
    ALD_RMU_PERH_UART3   = (1U << 11) | (1U << 28), /**< APB1: UART3 */
    ALD_RMU_PERH_UART4   = (1U << 12) | (1U << 28), /**< APB1: UART4 */
    ALD_RMU_PERH_UART5   = (1U << 13) | (1U << 28), /**< APB1: UART5 */
    ALD_RMU_PERH_SPI0    = (1U << 16) | (1U << 28), /**< APB1: SPI0 */
    ALD_RMU_PERH_SPI1    = (1U << 17) | (1U << 28), /**< APB1: SPI1 */
    ALD_RMU_PERH_I2C0    = (1U << 20) | (1U << 28), /**< APB1: I2C0 */
    ALD_RMU_PERH_I2C1    = (1U << 21) | (1U << 28), /**< APB1: I2C1 */
    ALD_RMU_PERH_CAN0    = (1U << 24) | (1U << 28), /**< APB1: CAN0 */
    ALD_RMU_PERH_LPTIM0  = (1U << 0)  | (1U << 29), /**< APB2: LPTIM0 */
    ALD_RMU_PERH_LPUART0 = (1U << 2)  | (1U << 29), /**< APB2: LPUART */
    ALD_RMU_PERH_ADC0    = (1U << 4)  | (1U << 29), /**< APB2: ADC0 */
    ALD_RMU_PERH_WWDT    = (1U << 12) | (1U << 29), /**< APB2: WWDT */
    ALD_RMU_PERH_IWDT    = (1U << 14) | (1U << 29), /**< APB2: IWDT */
    ALD_RMU_PERH_RTC     = (1U << 15) | (1U << 29), /**< APB2: RTC */
    ALD_RMU_PERH_DAC0    = (1U << 16) | (1U << 29), /**< APB2: DAC0 */
    ALD_RMU_PERH_DBGCON  = (1U << 19) | (1U << 29), /**< APB2: DBGCON */
} ald_rmu_peripheral_t;
/**
  * @}
  */

/**
  * @defgroup RMU_Private_Macros RMU Private Macros
  * @{
  */
#define IS_RMU_BORFLT(x)    (((x) == ALD_RMU_BORFLT_1) || \
                             ((x) == ALD_RMU_BORFLT_2) || \
                             ((x) == ALD_RMU_BORFLT_3) || \
                             ((x) == ALD_RMU_BORFLT_4) || \
                             ((x) == ALD_RMU_BORFLT_5) || \
                             ((x) == ALD_RMU_BORFLT_6) || \
                             ((x) == ALD_RMU_BORFLT_7))
#define IS_RMU_BORVOL(x)    (((x) == ALD_RMU_VOL_1_8) || \
                             ((x) == ALD_RMU_VOL_2_0) || \
                             ((x) == ALD_RMU_VOL_2_4) || \
                             ((x) == ALD_RMU_VOL_2_8) || \
                             ((x) == ALD_RMU_VOL_3_0) || \
                             ((x) == ALD_RMU_VOL_3_5) || \
                             ((x) == ALD_RMU_VOL_3_9) || \
                             ((x) == ALD_RMU_VOL_4_5))
#define IS_RMU_STATE(x)     (((x) == ALD_RMU_RST_POR)    || \
                             ((x) == ALD_RMU_RST_WAKEUP) || \
                             ((x) == ALD_RMU_RST_BOR)    || \
                             ((x) == ALD_RMU_RST_NMRST)  || \
                             ((x) == ALD_RMU_RST_IWDT)   || \
                             ((x) == ALD_RMU_RST_WWDT)   || \
                             ((x) == ALD_RMU_RST_LOCKUP) || \
                             ((x) == ALD_RMU_RST_CHIP)   || \
                             ((x) == ALD_RMU_RST_MCU)    || \
                             ((x) == ALD_RMU_RST_CPU)    || \
                             ((x) == ALD_RMU_RST_CFG)    || \
                             ((x) == ALD_RMU_RST_CFGERR)    || \
                             ((x) == ALD_RMU_RST_ALL))
#define IS_RMU_STATE_CLEAR(x)   (((x) == ALD_RMU_RST_POR)    || \
                                 ((x) == ALD_RMU_RST_WAKEUP) || \
                                 ((x) == ALD_RMU_RST_BOR)    || \
                                 ((x) == ALD_RMU_RST_NMRST)  || \
                                 ((x) == ALD_RMU_RST_IWDT)   || \
                                 ((x) == ALD_RMU_RST_WWDT)   || \
                                 ((x) == ALD_RMU_RST_LOCKUP) || \
                                 ((x) == ALD_RMU_RST_CHIP)   || \
                                 ((x) == ALD_RMU_RST_MCU)    || \
                                 ((x) == ALD_RMU_RST_CPU)    || \
                                 ((x) == ALD_RMU_RST_CFG)    || \
                                 ((x) == ALD_RMU_RST_ALL))
#define IS_RMU_PERH(x)      (((x) == ALD_RMU_PERH_GPIO)    || \
                             ((x) == ALD_RMU_PERH_CRC)     || \
                             ((x) == ALD_RMU_PERH_CALC)    || \
                             ((x) == ALD_RMU_PERH_PIS)     || \
                             ((x) == ALD_RMU_PERH_CHIP)    || \
                             ((x) == ALD_RMU_PERH_CPU)     || \
                             ((x) == ALD_RMU_PERH_TIMER0)  || \
                             ((x) == ALD_RMU_PERH_TIMER2)  || \
                             ((x) == ALD_RMU_PERH_TIMER3)  || \
                             ((x) == ALD_RMU_PERH_TIMER4)  || \
                             ((x) == ALD_RMU_PERH_TIMER5)  || \
                             ((x) == ALD_RMU_PERH_TIMER6)  || \
                             ((x) == ALD_RMU_PERH_TIMER7)  || \
                             ((x) == ALD_RMU_PERH_UART0)   || \
                             ((x) == ALD_RMU_PERH_UART1)   || \
                             ((x) == ALD_RMU_PERH_UART2)   || \
                             ((x) == ALD_RMU_PERH_UART3)   || \
                             ((x) == ALD_RMU_PERH_UART4)   || \
                             ((x) == ALD_RMU_PERH_UART5)   || \
                             ((x) == ALD_RMU_PERH_SPI0)    || \
                             ((x) == ALD_RMU_PERH_SPI1)    || \
                             ((x) == ALD_RMU_PERH_I2C0)    || \
                             ((x) == ALD_RMU_PERH_I2C1)    || \
                             ((x) == ALD_RMU_PERH_CAN0)    || \
                             ((x) == ALD_RMU_PERH_LPTIM0)  || \
                             ((x) == ALD_RMU_PERH_LPUART0) || \
                             ((x) == ALD_RMU_PERH_ADC0)    || \
                             ((x) == ALD_RMU_PERH_DAC0)    || \
                             ((x) == ALD_RMU_PERH_WWDT)    || \
                             ((x) == ALD_RMU_PERH_IWDT)    || \
                             ((x) == ALD_RMU_PERH_RTC))
/**
  * @}
  */

/** @addtogroup RMU_Public_Functions
  * @{
  */
void ald_rmu_bor_config(ald_rmu_bor_filter_t flt, ald_rmu_bor_vol_t vol, type_func_t state);
uint32_t ald_rmu_get_reset_status(ald_rmu_state_t state);
void ald_rmu_clear_reset_status(ald_rmu_state_t state);
void ald_rmu_reset_periperal(ald_rmu_peripheral_t perh);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_RMU_H__ */
