/**
  *********************************************************************************
  *
  * @file    ald_pmu.h
  * @brief   Header file of PMU module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_PMU_H__
#define __ALD_PMU_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"
#include "ald_syscfg.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup PMU
  * @{
  */

/** @defgroup PMU_Public_Macros PMU Public Macros
  * @{
  */
#define PMU_LPSTOP_ENABLE()             \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->CR0, PMU_CR0_LPSTOP_MSK);      \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_LPSTOP_DISABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->CR0, PMU_CR0_LPSTOP_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_FLASH_MODE_IDLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->CR0, PMU_CR0_SFPD_MSK);      \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_FLASH_MODE_WAIT()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->CR0, PMU_CR0_SFPD_MSK);        \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_MTSTOP_ENABLE()             \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->CR0, PMU_CR0_MTSTOP_MSK);      \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_MTSTOP_DISABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->CR0, PMU_CR0_MTSTOP_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_VREF_ENABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(PMU->VREFCR, PMU_VREFCR_VREFEN_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define PMU_VREF_DISABLE()              \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(PMU->VREFCR, PMU_VREFCR_VREFEN_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)

#define PMU_GET_LVD_STATUS()    (READ_BITS(PMU->LVDCR, PMU_LVDCR_LVDO_MSK, PMU_LVDCR_LVDO_POS))
/**
  * @}
  */


/** @defgroup PMU_Public_Types PMU Public Types
  * @{
  */
/**
  * @brief Low power mode
  */
typedef enum
{
    ALD_PMU_LP_STOP1   = 0x0,   /**< Stop1 */
    ALD_PMU_LP_STOP2   = 0x1,   /**< Stop2 */
    ALD_PMU_LP_STANDBY = 0x2,   /**< Standby */
    ALD_PMU_LP_SHUTOFF = 0x3,   /**< Shutoff */
} ald_pmu_lp_mode_t;

/**
  * @brief Standby wakeup level
  */
typedef enum
{
    ALD_PMU_WKPL_HIGH = 0x0,    /**< High */
    ALD_PMU_WKPL_LOW  = 0x1,    /**< Low */
} ald_pmu_wakeup_level_t;

/**
  * @brief Standby wakeup pin
  */
typedef enum
{
    ALD_PMU_PIN_PA0 = 0x0,  /**< PA0 */
    ALD_PMU_PIN_PA1 = 0x1,  /**< PA1 */
    ALD_PMU_PIN_PA2 = 0x2,  /**< PA2 */
    ALD_PMU_PIN_PA3 = 0x3,  /**< PA3 */
    ALD_PMU_PIN_PA4 = 0x4,  /**< PA4 */
    ALD_PMU_PIN_PA5 = 0x5,  /**< PA5 */
    ALD_PMU_PIN_PA6 = 0x6,  /**< PA6 */
    ALD_PMU_PIN_PA7 = 0x7,  /**< PA7 */
} ald_pmu_wakeup_pin_t;

/**
  * @brief PMU status
  */
typedef enum
{
    ALD_PMU_SR_WUF      = (0x1U << 0),    /**< WakeUp status */
    ALD_PMU_SR_STANDBYF = (0x1U << 1),    /**< Standby status */
    ALD_PMU_SR_SHUTOFF  = (0x1U << 2),    /**< RUN/LPRUAN status */
} ald_pmu_status_t;

/**
  * @brief LVD voltage select
  */
typedef enum
{
    ALD_PMU_LVD_VOL_SEL_2_1 = 0x0U,  /**< 2.1V */
    ALD_PMU_LVD_VOL_SEL_2_3 = 0x1U,  /**< 2.3V */
    ALD_PMU_LVD_VOL_SEL_2_5 = 0x2U,  /**< 2.5V */
    ALD_PMU_LVD_VOL_SEL_2_7 = 0x3U,  /**< 2.7V */
    ALD_PMU_LVD_VOL_SEL_2_9 = 0x4U,  /**< 2.9V */
    ALD_PMU_LVD_VOL_SEL_3_5 = 0x5U,  /**< 3.5V */
    ALD_PMU_LVD_VOL_SEL_3_9 = 0x6U,  /**< 3.9V */
    ALD_PMU_LVD_VOL_SEL_4_5 = 0x7U,  /**< 4.5V */
} ald_pmu_lvd_voltage_sel_t;


/**
  * @brief LDO output voltage selest in low power mode
  */
typedef enum
{
    PMU_LDO_LPMODE_OUTPUT_1_3 = 0x0,    /**< 1.3V */
    PMU_LDO_LPMODE_OUTPUT_1_4 = 0x1,    /**< 1.4V */
    PMU_LDO_LPMODE_OUTPUT_1_5 = 0x2,    /**< 1.5V */
    PMU_LDO_LPMODE_OUTPUT_1_6 = 0x4,    /**< 1.6V */
} ald_pmu_ldo_lpmode_output_t;

/**
  * @brief LVD trigger mode
  */
typedef enum
{
    ALD_PMU_LVD_TRIGGER_RISING_EDGE    = 0x0U,  /**< Rising edge */
    ALD_PMU_LVD_TRIGGER_FALLING_EDGE   = 0x1U,  /**< Falling edge */
    ALD_PMU_LVD_TRIGGER_HIGH_LEVEL     = 0x2U,  /**< High level */
    ALD_PMU_LVD_TRIGGER_LOW_LEVEL      = 0x3U,  /**< Low level */
    ALD_PMU_LVD_TRIGGER_RISING_FALLING = 0x4U,  /**< Rising and falling edge */
} ald_pmu_lvd_trigger_mode_t;

/**
  * @brief LDO output voltage selest in low power mode
  */
typedef enum
{
    ALD_PMU_POWER_SRAM0  = (0x1U << 1),         /**< SRAM0 */
    ALD_PMU_POWER_SRAM1  = (0x1U << 2),         /**< SRAM1 */
    ALD_PMU_POWER_SRAM2  = (0x1U << 3),         /**< SRAM2 */
    ALD_PMU_POWER_SRAM3  = (0x1U << 4),         /**< SRAM3 */
    ALD_PMU_POWER_CAN    = (0x1U << 16),        /**< CAN */
    ALD_PMU_POWER_ROM    = (0x1U << 26),        /**< ROM */
} ald_pmu_perh_power_t;

/**
  * @}
  */

/**
  * @defgroup PMU_Private_Macros PMU Private Macros
  * @{
  */
#define IS_PMU_LP_MODE(x)       (((x) == ALD_PMU_LP_STOP1)   || \
                                 ((x) == ALD_PMU_LP_STOP2)   || \
                                 ((x) == ALD_PMU_LP_SHUTOFF) || \
                                 ((x) == ALD_PMU_LP_STANDBY))
#define IS_PMU_STATUS(x)        (((x) == ALD_PMU_SR_WUF) || \
                                 ((x) == ALD_PMU_SR_STANDBYF) || \
                                 ((x) == ALD_PMU_SR_SHUTOFF))
#define IS_PMU_LVD_VOL_SEL(x)       (((x) == ALD_PMU_LVD_VOL_SEL_2_1) || \
                                     ((x) == ALD_PMU_LVD_VOL_SEL_2_3) || \
                                     ((x) == ALD_PMU_LVD_VOL_SEL_2_5) || \
                                     ((x) == ALD_PMU_LVD_VOL_SEL_2_7) || \
                                     ((x) == ALD_PMU_LVD_VOL_SEL_2_9) || \
                                     ((x) == ALD_PMU_LVD_VOL_SEL_3_5) || \
                                     ((x) == ALD_PMU_LVD_VOL_SEL_3_9) || \
                                     ((x) == ALD_PMU_LVD_VOL_SEL_4_5))
#define IS_PMU_LVD_TRIGGER_MODE(x)  (((x) == ALD_PMU_LVD_TRIGGER_RISING_EDGE)  || \
                                     ((x) == ALD_PMU_LVD_TRIGGER_FALLING_EDGE) || \
                                     ((x) == ALD_PMU_LVD_TRIGGER_HIGH_LEVEL)   || \
                                     ((x) == ALD_PMU_LVD_TRIGGER_LOW_LEVEL)    || \
                                     ((x) == ALD_PMU_LVD_TRIGGER_RISING_FALLING))
#define IS_PMU_LDO_LPMODE_OUTPUT(x) (((x) == ALD_PMU_LDO_LPMODE_OUTPUT_1_3) || \
                                     ((x) == ALD_PMU_LDO_LPMODE_OUTPUT_1_4) || \
                                     ((x) == ALD_PMU_LDO_LPMODE_OUTPUT_1_5) || \
                                     ((x) == ALD_PMU_LDO_LPMODE_OUTPUT_1_6))
#define IS_PMU_PERH_POWER(x)        (((x) == ALD_PMU_POWER_SRAM0) || \
                                     ((x) == ALD_PMU_POWER_SRAM1) || \
                                     ((x) == ALD_PMU_POWER_SRAM2) || \
                                     ((x) == ALD_PMU_POWER_SRAM3) || \
                                     ((x) == ALD_PMU_POWER_CAN)   || \
                                     ((x) == ALD_PMU_POWER_ROM))

/**
  * @}
  */

/** @addtogroup PMU_Public_Functions
  * @{
  */
/** @addtogroup PMU_Public_Functions_Group1
  * @{
  */
/* Low power mode select */
__STATIC_INLINE__ void ald_pmu_sleep()
{
    __WFI();
}

__STATIC_INLINE__ void ald_pmu_sleep_deep()
{
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
}

void ald_pmu_stop1_enter(void);
void ald_pmu_stop2_enter(void);
void ald_pmu_standby_enter(ald_pmu_wakeup_pin_t pin, ald_pmu_wakeup_level_t levle);
void ald_pmu_shutoff_enter(ald_pmu_wakeup_pin_t pin, ald_pmu_wakeup_level_t levle);
flag_status_t ald_pmu_get_status(ald_pmu_status_t sr);
void ald_pmu_clear_status(ald_pmu_status_t sr);
void ald_pmu_perh_power_config(ald_pmu_perh_power_t perh, type_func_t state);
/**
  * @}
  */
/** @addtogroup PMU_Public_Functions_Group2
  * @{
  */
/* LVD configure */
void ald_pmu_lvd_config(ald_pmu_lvd_voltage_sel_t sel, ald_pmu_lvd_trigger_mode_t mode, type_func_t state);
void ald_lvd_irq_handler(void);
/**
  * @}
  */
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_PMU_H__ */
