/**
  *********************************************************************************
  *
  * @file    ald_pis.h
  * @brief   Header file of PIS driver.
  *
  * @version V1.0
  * @date    12 May 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 May 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_PIS_H__
#define __ALD_PIS_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup PIS
  * @{
  */

/** @defgroup PIS_Public_Types PIS Public Types
  * @verbatim
      In PIS module, for the convenience of code maintenance,
      TIMERx is used to indicate the sequence of the timer peripheral.
      Different product series TIMERx represent different meanings:
      TIMER0 ----> AD16C4T0
      TIMER2 ----> GP32C4T0
      TIMER3 ----> GP32C4T1
      TIMER4 ----> BS16T0
      TIMER5 ----> BS16T1
      TIMER6 ----> GP16C4T0
      TIMER7 ----> GP16C4T1
    @endverbatim
  * @{
  */

/**
  * @brief Producer entry
  */
typedef enum
{
    ALD_PIS_SOFTWARE_LEVEL       = 0x00U,    /**< Software level trigger */
    ALD_PIS_SOFTWARE_PULSE       = 0x01U,    /**< Software pulse trigger */
    ALD_PIS_GPIOA_PIN0           = 0x10U,    /**< Pin0, level,support async */
    ALD_PIS_GPIOA_PIN1           = 0x11U,    /**< Pin1, level,support async */
    ALD_PIS_GPIOA_PIN2           = 0x12U,    /**< Pin2, level,support async */
    ALD_PIS_GPIOA_PIN3           = 0x13U,    /**< Pin3, level,support async */
    ALD_PIS_GPIOA_PIN4           = 0x14U,    /**< Pin4, level,support async */
    ALD_PIS_GPIOA_PIN5           = 0x15U,    /**< Pin5, level,support async */
    ALD_PIS_GPIOA_PIN6           = 0x16U,    /**< Pin6, level,support async */
    ALD_PIS_GPIOA_PIN7           = 0x17U,    /**< Pin7, level,support async */
    ALD_PIS_GPIOA_PIN8           = 0x18U,    /**< Pin8, level,support async */
    ALD_PIS_GPIOA_PIN9           = 0x19U,    /**< Pin9, level,support async */
    ALD_PIS_GPIOA_PIN10          = 0x1aU,    /**< Pin10, level,support async */
    ALD_PIS_GPIOA_PIN11          = 0x1bU,    /**< Pin11, level,support async */
    ALD_PIS_GPIOA_PIN12          = 0x1cU,    /**< Pin12, level,support async */
    ALD_PIS_GPIOA_PIN13          = 0x1dU,    /**< Pin13, level,support async */
    ALD_PIS_GPIOA_PIN14          = 0x1eU,    /**< Pin14, level,support async */
    ALD_PIS_GPIOA_PIN15          = 0x1fU,    /**< Pin15, level,support async */
    ALD_PIS_GPIOB_PIN0           = 0x20U,    /**< Pin0, level,support async */
    ALD_PIS_GPIOB_PIN1           = 0x21U,    /**< Pin1, level,support async */
    ALD_PIS_GPIOB_PIN2           = 0x22U,    /**< Pin2, level,support async */
    ALD_PIS_GPIOB_PIN3           = 0x23U,    /**< Pin3, level,support async */
    ALD_PIS_GPIOB_PIN4           = 0x24U,    /**< Pin4, level,support async */
    ALD_PIS_GPIOB_PIN5           = 0x25U,    /**< Pin5, level,support async */
    ALD_PIS_GPIOB_PIN6           = 0x26U,    /**< Pin6, level,support async */
    ALD_PIS_GPIOB_PIN7           = 0x27U,    /**< Pin7, level,support async */
    ALD_PIS_GPIOB_PIN8           = 0x28U,    /**< Pin8, level,support async */
    ALD_PIS_GPIOB_PIN9           = 0x29U,    /**< Pin9, level,support async */
    ALD_PIS_GPIOB_PIN10          = 0x2aU,    /**< Pin10, level,support async */
    ALD_PIS_GPIOB_PIN11          = 0x2bU,    /**< Pin11, level,support async */
    ALD_PIS_GPIOB_PIN12          = 0x2cU,    /**< Pin12, level,support async */
    ALD_PIS_GPIOB_PIN13          = 0x2dU,    /**< Pin13, level,support async */
    ALD_PIS_GPIOB_PIN14          = 0x2eU,    /**< Pin14, level,support async */
    ALD_PIS_GPIOB_PIN15          = 0x2fU,    /**< Pin15, level,support async */
    ALD_PIS_GPIOC_PIN0           = 0x30U,    /**< Pin0, level,support async */
    ALD_PIS_GPIOC_PIN1           = 0x31U,    /**< Pin1, level,support async */
    ALD_PIS_GPIOC_PIN2           = 0x32U,    /**< Pin2, level,support async */
    ALD_PIS_GPIOC_PIN3           = 0x33U,    /**< Pin3, level,support async */
    ALD_PIS_GPIOC_PIN4           = 0x34U,    /**< Pin4, level,support async */
    ALD_PIS_GPIOC_PIN5           = 0x35U,    /**< Pin5, level,support async */
    ALD_PIS_GPIOC_PIN6           = 0x36U,    /**< Pin6, level,support async */
    ALD_PIS_GPIOC_PIN7           = 0x37U,    /**< Pin7, level,support async */
    ALD_PIS_GPIOC_PIN8           = 0x38U,    /**< Pin8, level,support async */
    ALD_PIS_GPIOC_PIN9           = 0x39U,    /**< Pin9, level,support async */
    ALD_PIS_GPIOC_PIN10          = 0x3aU,    /**< Pin10, level,support async */
    ALD_PIS_GPIOC_PIN11          = 0x3bU,    /**< Pin11, level,support async */
    ALD_PIS_GPIOC_PIN12          = 0x3cU,    /**< Pin12, level,support async */
    ALD_PIS_GPIOC_PIN13          = 0x3dU,    /**< Pin13, level,support async */
    ALD_PIS_GPIOC_PIN14          = 0x3eU,    /**< Pin14, level,support async */
    ALD_PIS_GPIOC_PIN15          = 0x3fU,    /**< Pin15, level,support async */
    ALD_PIS_GPIOD_PIN0           = 0x40U,    /**< Pin0, level,support async */
    ALD_PIS_GPIOD_PIN1           = 0x41U,    /**< Pin1, level,support async */
    ALD_PIS_GPIOD_PIN2           = 0x42U,    /**< Pin2, level,support async */
    ALD_PIS_GPIOD_PIN3           = 0x43U,    /**< Pin3, level,support async */
    ALD_PIS_GPIOD_PIN4           = 0x44U,    /**< Pin4, level,support async */
    ALD_PIS_GPIOD_PIN5           = 0x45U,    /**< Pin5, level,support async */
    ALD_PIS_GPIOD_PIN6           = 0x46U,    /**< Pin6, level,support async */
    ALD_PIS_GPIOD_PIN7           = 0x47U,    /**< Pin7, level,support async */
    ALD_PIS_GPIOD_PIN8           = 0x48U,    /**< Pin8, level,support async */
    ALD_PIS_GPIOD_PIN9           = 0x49U,    /**< Pin9, level,support async */
    ALD_PIS_GPIOD_PIN10          = 0x4aU,    /**< Pin10, level,support async */
    ALD_PIS_GPIOD_PIN11          = 0x4bU,    /**< Pin11, level,support async */
    ALD_PIS_GPIOD_PIN12          = 0x4cU,    /**< Pin12, level,support async */
    ALD_PIS_GPIOD_PIN13          = 0x4dU,    /**< Pin13, level,support async */
    ALD_PIS_GPIOD_PIN14          = 0x4eU,    /**< Pin14, level,support async */
    ALD_PIS_GPIOD_PIN15          = 0x4fU,    /**< Pin15, level,support async */
    ALD_PIS_GPIOF_PIN0           = 0x60U,    /**< Pin0, level,support async */
    ALD_PIS_GPIOF_PIN1           = 0x61U,    /**< Pin1, level,support async */
    ALD_PIS_GPIOF_PIN2           = 0x62U,    /**< Pin2, level,support async */
    ALD_PIS_GPIOF_PIN3           = 0x63U,    /**< Pin3, level,support async */
    ALD_PIS_GPIOF_PIN4           = 0x64U,    /**< Pin4, level,support async */
    ALD_PIS_GPIOF_PIN5           = 0x65U,    /**< Pin5, level,support async */
    ALD_PIS_GPIOF_PIN6           = 0x66U,    /**< Pin6, level,support async */
    ALD_PIS_GPIOF_PIN7           = 0x67U,    /**< Pin7, level,support async */
    ALD_PIS_GPIOF_PIN8           = 0x68U,    /**< Pin8, level,support async */
    ALD_PIS_GPIOF_PIN9           = 0x69U,    /**< Pin9, level,support async */
    ALD_PIS_GPIOF_PIN10          = 0x6aU,    /**< Pin10, level,support async */
    ALD_PIS_GPIOF_PIN11          = 0x6bU,    /**< Pin11, level,support async */
    ALD_PIS_GPIOF_PIN12          = 0x6cU,    /**< Pin12, level,support async */
    ALD_PIS_GPIOF_PIN13          = 0x6dU,    /**< Pin13, level,support async */
    ALD_PIS_GPIOF_PIN14          = 0x6eU,    /**< Pin14, level,support async */
    ALD_PIS_GPIOF_PIN15          = 0x6fU,    /**< Pin15, level,support async */
    ALD_PIS_GPIOH_PIN0           = 0x70U,    /**< Pin0, level,support async */
    ALD_PIS_GPIOH_PIN1           = 0x71U,    /**< Pin1, level,support async */
    ALD_PIS_GPIOH_PIN2           = 0x72U,    /**< Pin2, level,support async */
    ALD_PIS_GPIOH_PIN3           = 0x73U,    /**< Pin3, level,support async */
    ALD_PIS_GPIOH_PIN4           = 0x74U,    /**< Pin4, level,support async */
    ALD_PIS_GPIOH_PIN5           = 0x75U,    /**< Pin5, level,support async */
    ALD_PIS_GPIOH_PIN6           = 0x76U,    /**< Pin6, level,support async */
    ALD_PIS_GPIOH_PIN7           = 0x77U,    /**< Pin7, level,support async */
    ALD_PIS_GPIOH_PIN8           = 0x78U,    /**< Pin8, level,support async */
    ALD_PIS_GPIOH_PIN9           = 0x79U,    /**< Pin9, level,support async */
    ALD_PIS_GPIOH_PIN10          = 0x7aU,    /**< Pin10, level,support async */
    ALD_PIS_GPIOH_PIN11          = 0x7bU,    /**< Pin11, level,support async */
    ALD_PIS_GPIOH_PIN12          = 0x7cU,    /**< Pin12, level,support async */
    ALD_PIS_GPIOH_PIN13          = 0x7dU,    /**< Pin13, level,support async */
    ALD_PIS_GPIOH_PIN14          = 0x7eU,    /**< Pin14, level,support async */
    ALD_PIS_GPIOH_PIN15          = 0x7fU,    /**< Pin15, level,support async */
    ALD_PIS_UART0_RESERVE0       = 0x80U,    /**< Uart0 reserve bit 0 */
    ALD_PIS_UART0_RESERVE1       = 0x81U,    /**< Uart0 reserve bit 1*/
    ALD_PIS_UART0_IRDAOUT        = 0x82U,    /**< Uart0 irdaout, level,support async */
    ALD_PIS_UART0_RTSOUT         = 0x83U,    /**< Uart0 rtsout, level,support async */
    ALD_PIS_UART0_TXOUT          = 0x84U,    /**< Uart0 txout, level,support async */
    ALD_PIS_UART0_SEND_INT_PULSE = 0x85U,    /**< Uart0 syn send, pulse,support async */
    ALD_PIS_UART0_RECV_INT_PULSE = 0x86U,    /**< Uart0 syn recv, pulse,support async */
    ALD_PIS_UART1_RESERVE0       = 0x90U,    /**< Uart1 reserve bit 0 */
    ALD_PIS_UART1_RESERVE1       = 0x91U,    /**< Uart1 reserve bit 1*/
    ALD_PIS_UART1_IRDA           = 0x92U,    /**< Uart1 irdaout, level,support async */
    ALD_PIS_UART1_RTS            = 0x93U,    /**< Uart1 rtsout, level,support async */
    ALD_PIS_UART1_TXOUT          = 0x94U,    /**< Uart1 txout, level,support async */
    ALD_PIS_UART1_SEND_INT_PULSE = 0x95U,    /**< Uart1 syn send, pulse,support async */
    ALD_PIS_UART1_RECV_INT_PULSE = 0x96U,    /**< Uart1 syn recv, pulse,support async */
    ALD_PIS_UART2_RESERVE0       = 0xa0U,    /**< Uart2 reserve bit 0 */
    ALD_PIS_UART2_RESERVE1       = 0xa1U,    /**< Uart2 reserve bit 1*/
    ALD_PIS_UART2_RTS            = 0xa3U,    /**< Uart2 rtsout, level,support async */
    ALD_PIS_UART2_TXOUT          = 0xa4U,    /**< Uart2 txout, level,support async */
    ALD_PIS_UART2_SEND_INT_PULSE = 0xa5U,    /**< Uart2 syn send, pulse,support async */
    ALD_PIS_UART2_RECV_INT_PULSE = 0xa6U,    /**< Uart2 syn recv, pulse,support async */
    ALD_PIS_UART3_RESERVE0       = 0xb0U,    /**< Uart3 reserve bit 0 */
    ALD_PIS_UART3_RESERVE1       = 0xb1U,    /**< Uart3 reserve bit 1*/
    ALD_PIS_UART3_RTS            = 0xb3U,    /**< Uart3 rtsout, level,support async */
    ALD_PIS_UART3_TXOUT          = 0xb4U,    /**< Uart3 txout, level,support async */
    ALD_PIS_UART3_SEND_INT_PULSE = 0xb5U,    /**< Uart3 syn send, pulse,support async */
    ALD_PIS_UART3_RECV_INT_PULSE = 0xb6U,    /**< Uart3 syn recv, pulse,support async */
    ALD_PIS_UART4_RESERVE0       = 0xc0U,    /**< Uart4 reserve bit 0 */
    ALD_PIS_UART4_RESERVE1       = 0xc1U,    /**< Uart4 reserve bit 1*/
    ALD_PIS_UART4_RTS            = 0xc3U,    /**< Uart4 rtsout, level,support async */
    ALD_PIS_UART4_TXOUT          = 0xc4U,    /**< Uart4 txout, level,support async */
    ALD_PIS_UART4_SEND_INT_PULSE = 0xc5U,    /**< Uart4 syn send, pulse,support async */
    ALD_PIS_UART4_RECV_INT_PULSE = 0xc6U,    /**< Uart4 syn recv, pulse,support async */
    ALD_PIS_UART5_RESERVE0       = 0xd0U,    /**< Uart5 reserve bit 0 */
    ALD_PIS_UART5_RESERVE1       = 0xd1U,    /**< Uart5 reserve bit 1*/
    ALD_PIS_UART5_RTS            = 0xd3U,    /**< Uart5 rtsout, level,support async */
    ALD_PIS_UART5_TXOUT          = 0xd4U,    /**< Uart5 txout, level,support async */
    ALD_PIS_UART5_SEND_INT_PULSE = 0xd5U,    /**< Uart5 syn send, pulse,support async */
    ALD_PIS_UART5_RECV_INT_PULSE = 0xd6U,    /**< Uart5 syn recv, pulse,support async */
    ALD_PIS_SPI0_RECV            = 0xe0U,    /**< Spi0 recv, plck1 pulse */
    ALD_PIS_SPI0_SEND            = 0xe1U,    /**< Spi0 send, plck1 pulse */
    ALD_PIS_SPI0_NE              = 0xe2U,    /**< Spi0 ne, plck1 level */
    ALD_PIS_SPI1_RECV            = 0xf0U,    /**< Spi1 recv, plck1 pulse */
    ALD_PIS_SPI1_SEND            = 0xf1U,    /**< Spi1 send, plck1 pulse */
    ALD_PIS_SPI1_NE              = 0xf2U,    /**< Spi1 ne, plck1 level */
    ALD_PIS_I2C0_RECV            = 0x100U,   /**< I2c0 recv, plck1 level */
    ALD_PIS_I2C0_SEND            = 0x101U,   /**< I2c0 send, plck1 level */
    ALD_PIS_I2C1_RECV            = 0x110U,   /**< I2c1 recv, plck1 level */
    ALD_PIS_I2C1_SEND            = 0x111U,   /**< I2c1 send, plck1 level */
    ALD_PIS_TIMER0_UPDATA        = 0x120U,   /**< AD16C4T0 updata, plck1 pulse */
    ALD_PIS_TIMER0_TRGOUT        = 0x121U,   /**< AD16C4T0 trgout, plck1 pulse */
    ALD_PIS_TIMER0_INPUT_1       = 0x122U,   /**< AD16C4T0 chan1 input catch, plck1 pulse */
    ALD_PIS_TIMER0_OUTPUT_1      = 0x123U,   /**< AD16C4T0 chan1 output compare, plck1 pulse */
    ALD_PIS_TIMER0_INPUT_2       = 0x124U,   /**< AD16C4T0 chan2 input catch, plck1 pulse */
    ALD_PIS_TIMER0_OUTPUT_2      = 0x125U,   /**< AD16C4T0 chan2 output compare, plck1 pulse */
    ALD_PIS_TIMER0_INPUT_3       = 0x126U,   /**< AD16C4T0 chan3 input catch, plck1 pulse */
    ALD_PIS_TIMER0_OUTPUT_3      = 0x127U,   /**< AD16C4T0 chan3 output compare, plck1 pulse */
    ALD_PIS_TIMER0_INPUT_4       = 0x128U,   /**< AD16C4T0 chan4 input catch, plck1 pulse */
    ALD_PIS_TIMER0_OUTPUT_4      = 0x129U,   /**< AD16C4T0 chan4 output compare, plck1 pulse */
    ALD_PIS_TIMER2_TRGOUT        = 0x141U,   /**< GP32C4T0 trgout, plck1 pulse */
    ALD_PIS_TIMER2_INPUT_1       = 0x142U,   /**< GP32C4T0 chan1 input catch, plck1 pulse */
    ALD_PIS_TIMER2_OUTPUT_1      = 0x143U,   /**< GP32C4T0 chan1 output compare, plck1 pulse */
    ALD_PIS_TIMER2_INPUT_2       = 0x144U,   /**< GP32C4T0 chan2 input catch, plck1 pulse */
    ALD_PIS_TIMER2_OUTPUT_2      = 0x145U,   /**< GP32C4T0 chan2 output compare, plck1 pulse */
    ALD_PIS_TIMER2_INPUT_3       = 0x146U,   /**< GP32C4T0 chan3 input catch, plck1 pulse */
    ALD_PIS_TIMER2_OUTPUT_3      = 0x147U,   /**< GP32C4T0 chan3 output compare, plck1 pulse */
    ALD_PIS_TIMER2_INPUT_4       = 0x148U,   /**< GP32C4T0 chan4 input catch, plck1 pulse */
    ALD_PIS_TIMER2_OUTPUT_4      = 0x149U,   /**< GP32C4T0 chan4 output compare, plck1 pulse */
    ALD_PIS_TIMER3_TRGOUT        = 0x151U,   /**< GP32C4T1 trgout, plck1 pulse */
    ALD_PIS_TIMER3_INPUT_1       = 0x152U,   /**< GP32C4T1 chan1 input catch, plck1 pulse */
    ALD_PIS_TIMER3_OUTPUT_1      = 0x153U,   /**< GP32C4T1 chan1 output compare, plck1 pulse */
    ALD_PIS_TIMER3_INPUT_2       = 0x154U,   /**< GP32C4T1 chan2 input catch, plck1 pulse */
    ALD_PIS_TIMER3_OUTPUT_2      = 0x155U,   /**< GP32C4T1 chan2 output compare, plck1 pulse */
    ALD_PIS_TIMER3_INPUT_3       = 0x156U,   /**< GP32C4T1 chan3 input catch, plck1 pulse */
    ALD_PIS_TIMER3_OUTPUT_3      = 0x157U,   /**< GP32C4T1 chan3 output compare, plck1 pulse */
    ALD_PIS_TIMER3_INPUT_4       = 0x158U,   /**< GP32C4T1 chan4 input catch, plck1 pulse */
    ALD_PIS_TIMER3_OUTPUT_4      = 0x159U,   /**< GP32C4T1 chan4 output compare, plck1 pulse */
    ALD_PIS_TIMER6_TRGOUT        = 0x1D1U,   /**< GP16C4T0 trgout, plck1 pulse */
    ALD_PIS_TIMER6_INPUT_1       = 0x1D2U,   /**< GP16C4T0 chan1 input catch, plck1 pulse */
    ALD_PIS_TIMER6_OUTPUT_1      = 0x1D3U,   /**< GP16C4T0 chan1 output compare, plck1 pulse */
    ALD_PIS_TIMER6_INPUT_2       = 0x1D4U,   /**< GP16C4T0 chan2 input catch, plck1 pulse */
    ALD_PIS_TIMER6_OUTPUT_2      = 0x1D5U,   /**< GP16C4T0 chan2 output compare, plck1 pulse */
    ALD_PIS_TIMER6_INPUT_3       = 0x1D6U,   /**< GP16C4T0 chan3 input catch, plck1 pulse */
    ALD_PIS_TIMER6_OUTPUT_3      = 0x1D7U,   /**< GP16C4T0 chan3 output compare, plck1 pulse */
    ALD_PIS_TIMER6_INPUT_4       = 0x1D8U,   /**< GP16C4T0 chan4 input catch, plck1 pulse */
    ALD_PIS_TIMER6_OUTPUT_4      = 0x1D9U,   /**< GP16C4T0 chan4 output compare, plck1 pulse */
    ALD_PIS_TIMER7_TRGOUT        = 0x1E1U,   /**< GP16C4T1 trgout, plck1 pulse */
    ALD_PIS_TIMER7_INPUT_1       = 0x1E2U,   /**< GP16C4T1 chan1 input catch, plck1 pulse */
    ALD_PIS_TIMER7_OUTPUT_1      = 0x1E3U,   /**< GP16C4T1 chan1 output compare, plck1 pulse */
    ALD_PIS_TIMER7_INPUT_2       = 0x1E4U,   /**< GP16C4T1 chan2 input catch, plck1 pulse */
    ALD_PIS_TIMER7_OUTPUT_2      = 0x1E5U,   /**< GP16C4T1 chan2 output compare, plck1 pulse */
    ALD_PIS_TIMER7_INPUT_3       = 0x1E6U,   /**< GP16C4T1 chan3 input catch, plck1 pulse */
    ALD_PIS_TIMER7_OUTPUT_3      = 0x1E7U,   /**< GP16C4T1 chan3 output compare, plck1 pulse */
    ALD_PIS_TIMER7_INPUT_4       = 0x1E8U,   /**< GP16C4T1 chan4 input catch, plck1 pulse */
    ALD_PIS_TIMER7_OUTPUT_4      = 0x1E9U,   /**< GP16C4T1 chan4 output compare, plck1 pulse */
    ALD_PIS_RTC_CLOCK            = 0x160U,   /**< Rtc clock, pulse,support async */
    ALD_PIS_RTC_ALARM            = 0x161U,   /**< Rtc alarm, pulse,support async */
    ALD_PIS_LPTIMER0_SYN_UPDATA  = 0x170U,   /**< Lptimer0 syn updata, pulse,support async */
    ALD_PIS_LPTIMER0_ASY_UPDATA  = 0x171U,   /**< Lptimer0 asy updata, pulse,support async */
    ALD_PIS_LPUART0_ASY_RECV     = 0x180U,   /**< Lpuart0 asy recv, pulse,support async */
    ALD_PIS_LPUART0_ASY_SEND     = 0x181U,   /**< Lpuart0 asy send, pulse,support async */
    ALD_PIS_LPUART0_SYN_RECV     = 0x182U,   /**< Lpuart0 syn recv, pulse,support async */
    ALD_PIS_LPUART0_SYN_SEND     = 0x183U,   /**< Lpuart0 syn recv, pulse,support async */
    ALD_PIS_DMA_CH0              = 0x190U,   /**< Dma ch0 transmit completely, support async */
    ALD_PIS_DMA_CH1              = 0x191U,   /**< Dma ch1 transmit completely, support async */
    ALD_PIS_DMA_CH2              = 0x192U,   /**< Dma ch2 transmit completely, support async */
    ALD_PIS_DMA_CH3              = 0x193U,   /**< Dma ch3 transmit completely, support async */
    ALD_PIS_DMA_CH4              = 0x194U,   /**< Dma ch4 transmit completely, support async */
    ALD_PIS_DMA_CH5              = 0x195U,   /**< Dma ch5 transmit completely, support async */
    ALD_PIS_DMA_CH6              = 0x196U,   /**< Dma ch6 transmit completely, support async */
    ALD_PIS_DMA_CH7              = 0x197U,   /**< Dma ch7 transmit completely, support async */
    ALD_PIS_DMA_CH8              = 0x198U,   /**< Dma ch8 transmit completely, support async */
    ALD_PIS_DMA_CH9              = 0x199U,   /**< Dma ch9 transmit completely, support async */
    ALD_PIS_DMA_CH10             = 0x19AU,   /**< Dma chA transmit completely, support async */
    ALD_PIS_DMA_CH11             = 0x19BU,   /**< Dma chB transmit completely, support async */
    ALD_PIS_DMA_CH               = 0x19FU,   /**< Dma channel transmit completely, support async */
    ALD_PIS_LVD                  = 0x1A0U,   /**< Lvd compare output */
    ALD_PIS_LVDEX                = 0x1A1U,   /**< Lvdex compare output */
    ALD_PIS_TIMER4_TRGOUT        = 0X1B1U,   /**< BS16T0, trgout */
    ALD_PIS_TIMER5_TRGOUT        = 0X1C1U,   /**< BS16T1, trgout */
    ALD_PIS_ADC0_INSERT          = 0x1F0U,   /**< Adc0 insert, pclk2 pulse,support async */
    ALD_PIS_ADC0_NORMAL          = 0x1F1U,   /**< Adc0 normal, pclk2 pulse,support async */
    ALD_PIS_ADC0_WDT             = 0x1F2U,   /**< Adc0 watch dog */
    ALD_PIS_SYS_HSCO             = 0x210U,   /**< System high clock */
    ALD_PIS_SYS_LSCO             = 0x211U,   /**< System low clock */
    ALD_PIS_SYS_BUZ              = 0x212U,   /**< System low clock */
} ald_pis_src_t;

/**
  * @brief Consumer entry
  */
typedef enum
{
    ALD_PIS_CH0_TIMER0_ITR0    = 0x0090U,    /**< AD16C4T0 internal input chan0 */
    ALD_PIS_CH1_TIMER0_ITR1    = 0x0091U,    /**< AD16C4T0 internal input chan1 */
    ALD_PIS_CH2_TIMER0_ITR2    = 0x0092U,    /**< AD16C4T0 internal input chan2 */
    ALD_PIS_CH3_TIMER0_ITR3    = 0x0093U,    /**< AD16C4T0 internal input chan3 */
    ALD_PIS_CH4_TIMER0_ITR4    = 0x0094U,    /**< AD16C4T0 internal input chan4 */
    ALD_PIS_CH5_TIMER0_ITR5    = 0x0095U,    /**< AD16C4T0 internal input chan5 */
    ALD_PIS_CH6_TIMER0_ITR6    = 0x0096U,    /**< AD16C4T0 internal input chan6 */
    ALD_PIS_CH7_TIMER0_ITR7    = 0x0097U,    /**< AD16C4T0 internal input chan7 */
    ALD_PIS_CH8_TIMER0_ITR8    = 0x0098U,    /**< AD16C4T0 internal input chan8 */
    ALD_PIS_CH9_TIMER0_ITR9    = 0x0099U,    /**< AD16C4T0 internal input chan9 */
    ALD_PIS_CH10_TIMER0_ITR10  = 0x009AU,    /**< AD16C4T0 internal input chan10 */
    ALD_PIS_CH11_TIMER0_ITR11  = 0x009BU,    /**< AD16C4T0 internal input chan11 */
    ALD_PIS_CH12_TIMER0_ITR12  = 0x009CU,    /**< AD16C4T0 internal input chan0 */
    ALD_PIS_CH13_TIMER0_ITR13  = 0x009DU,    /**< AD16C4T0 internal input chan1 */
    ALD_PIS_CH14_TIMER0_ITR14  = 0x009EU,    /**< AD16C4T0 internal input chan2 */
    ALD_PIS_CH15_TIMER0_ITR15  = 0x009FU,    /**< AD16C4T0 internal input chan3 */
    ALD_PIS_CH0_TIMER2_ITR0    = 0x0090U,    /**< GP32C4T0 internal input chan0 */
    ALD_PIS_CH1_TIMER2_ITR1    = 0x0091U,    /**< GP32C4T0 internal input chan1 */
    ALD_PIS_CH2_TIMER2_ITR2    = 0x0092U,    /**< GP32C4T0 internal input chan2 */
    ALD_PIS_CH3_TIMER2_ITR3    = 0x0093U,    /**< GP32C4T0 internal input chan3 */
    ALD_PIS_CH4_TIMER2_ITR4    = 0x0094U,    /**< GP32C4T0 internal input chan4 */
    ALD_PIS_CH5_TIMER2_ITR5    = 0x0095U,    /**< GP32C4T0 internal input chan5 */
    ALD_PIS_CH6_TIMER2_ITR6    = 0x0096U,    /**< GP32C4T0 internal input chan6 */
    ALD_PIS_CH7_TIMER2_ITR7    = 0x0097U,    /**< GP32C4T0 internal input chan7 */
    ALD_PIS_CH8_TIMER2_ITR8    = 0x0098U,    /**< GP32C4T0 internal input chan8 */
    ALD_PIS_CH9_TIMER2_ITR8    = 0x0099U,    /**< GP32C4T0 internal input chan9 */
    ALD_PIS_CH10_TIMER2_ITR10  = 0x009AU,    /**< GP32C4T0 internal input chan10 */
    ALD_PIS_CH11_TIMER2_ITR11  = 0x009BU,    /**< GP32C4T0 internal input chan11 */
    ALD_PIS_CH12_TIMER2_ITR12  = 0x009CU,    /**< GP32C4T0 internal input chan12 */
    ALD_PIS_CH13_TIMER2_ITR13  = 0x009DU,    /**< GP32C4T0 internal input chan13 */
    ALD_PIS_CH14_TIMER2_ITR14  = 0x009EU,    /**< GP32C4T0 internal input chan14 */
    ALD_PIS_CH15_TIMER2_ITR15  = 0x009FU,    /**< GP32C4T0 internal input chan15 */
    ALD_PIS_CH5_TIMER2_CH1IN   = 0x1005U,    /**< GP32C4T0 input chan1 */
    ALD_PIS_CH6_TIMER2_CH2IN   = 0x1106U,    /**< GP32C4T0 input chan2 */
    ALD_PIS_CH7_TIMER2_CH3IN   = 0x1207U,    /**< GP32C4T0 input chan3 */
    ALD_PIS_CH8_TIMER2_CH4IN   = 0x1308U,    /**< GP32C4T0 input chan4 */
    ALD_PIS_CH0_TIMER3_ITR0    = 0x0090U,    /**< GP32C4T1 internal input chan0 */
    ALD_PIS_CH1_TIMER3_ITR1    = 0x0091U,    /**< GP32C4T1 internal input chan1 */
    ALD_PIS_CH2_TIMER3_ITR2    = 0x0092U,    /**< GP32C4T1 internal input chan2 */
    ALD_PIS_CH3_TIMER3_ITR3    = 0x0093U,    /**< GP32C4T1 internal input chan3 */
    ALD_PIS_CH4_TIMER3_ITR4    = 0x0094U,    /**< GP32C4T1 internal input chan4 */
    ALD_PIS_CH5_TIMER3_ITR5    = 0x0095U,    /**< GP32C4T1 internal input chan5 */
    ALD_PIS_CH6_TIMER3_ITR6    = 0x0096U,    /**< GP32C4T1 internal input chan6 */
    ALD_PIS_CH7_TIMER3_ITR7    = 0x0097U,    /**< GP32C4T1 internal input chan7 */
    ALD_PIS_CH8_TIMER3_ITR8    = 0x0098U,    /**< GP32C4T1 internal input chan8 */
    ALD_PIS_CH9_TIMER3_ITR9    = 0x0099U,    /**< GP32C4T1 internal input chan9 */
    ALD_PIS_CH10_TIMER3_ITR10  = 0x009AU,    /**< GP32C4T1 internal input chan10 */
    ALD_PIS_CH11_TIMER3_ITR11  = 0x009BU,    /**< GP32C4T1 internal input chan11*/
    ALD_PIS_CH12_TIMER3_ITR12  = 0x009CU,    /**< GP32C4T1 internal input chan12 */
    ALD_PIS_CH13_TIMER3_ITR13  = 0x009DU,    /**< GP32C4T1 internal input chan13 */
    ALD_PIS_CH14_TIMER3_ITR14  = 0x009EU,    /**< GP32C4T1 internal input chan14 */
    ALD_PIS_CH15_TIMER3_ITR15  = 0x009FU,    /**< GP32C4T1 internal input chan15 */
    ALD_PIS_CH5_TIMER3_CH1IN   = 0x1805U,    /**< GP32C4T1 input chan1 */
    ALD_PIS_CH6_TIMER3_CH2IN   = 0x1906U,    /**< GP32C4T1 input chan2 */
    ALD_PIS_CH7_TIMER3_CH3IN   = 0x1A07U,    /**< GP32C4T1 input chan3 */
    ALD_PIS_CH8_TIMER3_CH4IN   = 0x1B08U,    /**< GP32C4T1 input chan4 */
    ALD_PIS_CH0_ADC0_NORMAL    = 0x0090U,    /**< ADC0 normal transform */
    ALD_PIS_CH1_ADC0_NORMAL    = 0x0091U,    /**< ADC0 normal transform */
    ALD_PIS_CH2_ADC0_NORMAL    = 0x0092U,    /**< ADC0 normal transform */
    ALD_PIS_CH3_ADC0_NORMAL    = 0x0093U,    /**< ADC0 normal transform */
    ALD_PIS_CH4_ADC0_NORMAL    = 0x0094U,    /**< ADC0 normal transform */
    ALD_PIS_CH5_ADC0_NORMAL    = 0x0095U,    /**< ADC0 normal transform */
    ALD_PIS_CH6_ADC0_NORMAL    = 0x0096U,    /**< ADC0 normal transform */
    ALD_PIS_CH7_ADC0_NORMAL    = 0x0097U,    /**< ADC0 normal transform */
    ALD_PIS_CH8_ADC0_NORMAL    = 0x0098U,    /**< ADC0 normal transform */
    ALD_PIS_CH9_ADC0_NORMAL    = 0x0099U,    /**< ADC0 normal transform */
    ALD_PIS_CH10_ADC0_NORMAL   = 0x009AU,    /**< ADC0 normal transform */
    ALD_PIS_CH11_ADC0_NORMAL   = 0x009BU,    /**< ADC0 normal transform */
    ALD_PIS_CH12_ADC0_NORMAL   = 0x009CU,    /**< ADC0 normal transform */
    ALD_PIS_CH13_ADC0_NORMAL   = 0x009DU,    /**< ADC0 normal transform */
    ALD_PIS_CH14_ADC0_NORMAL   = 0x009EU,    /**< ADC0 normal transform */
    ALD_PIS_CH15_ADC0_NORMAL   = 0x009FU,    /**< ADC0 normal transform */
    ALD_PIS_CH0_ADC0_INSERT    = 0x0090U,    /**< ADC0 insert transform */
    ALD_PIS_CH1_ADC0_INSERT    = 0x0091U,    /**< ADC0 insert transform */
    ALD_PIS_CH2_ADC0_INSERT    = 0x0092U,    /**< ADC0 insert transform */
    ALD_PIS_CH3_ADC0_INSERT    = 0x0093U,    /**< ADC0 insert transform */
    ALD_PIS_CH4_ADC0_INSERT    = 0x0094U,    /**< ADC0 insert transform */
    ALD_PIS_CH5_ADC0_INSERT    = 0x0095U,    /**< ADC0 insert transform */
    ALD_PIS_CH6_ADC0_INSERT    = 0x0096U,    /**< ADC0 insert transform */
    ALD_PIS_CH7_ADC0_INSERT    = 0x0097U,    /**< ADC0 insert transform */
    ALD_PIS_CH8_ADC0_INSERT    = 0x0098U,    /**< ADC0 insert transform */
    ALD_PIS_CH9_ADC0_INSERT    = 0x0099U,    /**< ADC0 insert transform */
    ALD_PIS_CH10_ADC0_INSERT   = 0x009AU,    /**< ADC0 insert transform */
    ALD_PIS_CH11_ADC0_INSERT   = 0x009BU,    /**< ADC0 insert transform */
    ALD_PIS_CH12_ADC0_INSERT   = 0x009CU,    /**< ADC0 insert transform */
    ALD_PIS_CH13_ADC0_INSERT   = 0x009DU,    /**< ADC0 insert transform */
    ALD_PIS_CH14_ADC0_INSERT   = 0x009EU,    /**< ADC0 insert transform */
    ALD_PIS_CH15_ADC0_INSERT   = 0x009FU,    /**< ADC0 insert transform */
    ALD_PIS_CH0_DAC_CONVERT    = 0x0090U,    /**< DAC convert */
    ALD_PIS_CH1_DAC_CONVERT    = 0x0091U,    /**< DAC convert */
    ALD_PIS_CH2_DAC_CONVERT    = 0x0092U,    /**< DAC convert */
    ALD_PIS_CH3_DAC_CONVERT    = 0x0093U,    /**< DAC convert */
    ALD_PIS_CH4_DAC_CONVERT    = 0x0094U,    /**< DAC convert */
    ALD_PIS_CH5_DAC_CONVERT    = 0x0095U,    /**< DAC convert */
    ALD_PIS_CH6_DAC_CONVERT    = 0x0096U,    /**< DAC convert */
    ALD_PIS_CH7_DAC_CONVERT    = 0x0097U,    /**< DAC convert */
    ALD_PIS_CH8_DAC_CONVERT    = 0x0098U,    /**< DAC convert */
    ALD_PIS_CH9_DAC_CONVERT    = 0x0099U,    /**< DAC convert */
    ALD_PIS_CH10_DAC_CONVERT   = 0x009AU,    /**< DAC convert */
    ALD_PIS_CH11_DAC_CONVERT   = 0x009BU,    /**< DAC convert */
    ALD_PIS_CH12_DAC_CONVERT   = 0x009CU,    /**< DAC convert */
    ALD_PIS_CH13_DAC_CONVERT   = 0x009DU,    /**< DAC convert */
    ALD_PIS_CH14_DAC_CONVERT   = 0x009EU,    /**< DAC convert */
    ALD_PIS_CH15_DAC_CONVERT   = 0x009FU,    /**< DAC convert */
    ALD_PIS_CH0_LPTIM0_EXT0    = 0x0090U,    /**< Lptim0 external trigger 0 */
    ALD_PIS_CH1_LPTIM0_EXT1    = 0x0091U,    /**< Lptim0 external trigger 1 */
    ALD_PIS_CH2_LPTIM0_EXT2    = 0x0092U,    /**< Lptim0 external trigger 2 */
    ALD_PIS_CH3_LPTIM0_EXT3    = 0x0093U,    /**< Lptim0 external trigger 3 */
    ALD_PIS_CH4_LPTIM0_EXT4    = 0x0094U,    /**< Lptim0 external trigger 4 */
    ALD_PIS_CH5_LPTIM0_EXT5    = 0x0095U,    /**< Lptim0 external trigger 5 */
    ALD_PIS_CH6_LPTIM0_EXT6    = 0x0096U,    /**< Lptim0 external trigger 6 */
    ALD_PIS_CH7_LPTIM0_EXT7    = 0x0097U,    /**< Lptim0 external trigger 7 */
    ALD_PIS_CH0_DMA_REQUEST    = 0x0090U,    /**< DMA request */
    ALD_PIS_CH1_DMA_REQUEST    = 0x0091U,    /**< DMA request */
    ALD_PIS_CH2_DMA_REQUEST    = 0x0092U,    /**< DMA request */
    ALD_PIS_CH3_DMA_REQUEST    = 0x0093U,    /**< DMA request */
    ALD_PIS_CH4_DMA_REQUEST    = 0x0094U,    /**< DMA request */
    ALD_PIS_CH5_DMA_REQUEST    = 0x0095U,    /**< DMA request */
    ALD_PIS_CH6_DMA_REQUEST    = 0x0096U,    /**< DMA request */
    ALD_PIS_CH7_DMA_REQUEST    = 0x0097U,    /**< DMA request */
    ALD_PIS_CH8_DMA_REQUEST    = 0x0098U,    /**< DMA request */
    ALD_PIS_CH9_DMA_REQUEST    = 0x0099U,    /**< DMA request */
    ALD_PIS_CH10_DMA_REQUEST   = 0x009AU,    /**< DMA request */
    ALD_PIS_CH11_DMA_REQUEST   = 0x009BU,    /**< DMA request */
    ALD_PIS_CH12_DMA_REQUEST   = 0x009CU,    /**< DMA request */
    ALD_PIS_CH13_DMA_REQUEST   = 0x009DU,    /**< DMA request */
    ALD_PIS_CH14_DMA_REQUEST   = 0x009EU,    /**< DMA request */
    ALD_PIS_CH15_DMA_REQUEST   = 0x009FU,    /**< DMA request */
    ALD_PIS_CH15_LPUART0_RXD   = 0x081FU,    /**< Lpuart Rx data */
    ALD_PIS_CH14_UART5_RXD     = 0x051EU,    /**< Uart5 Rx data */
    ALD_PIS_CH13_UART4_RXD     = 0x041DU,    /**< Uart4 Rx data */
    ALD_PIS_CH12_UART3_RXD     = 0x031CU,    /**< Uart3 Rx data */
    ALD_PIS_CH11_UART2_RXD     = 0x021BU,    /**< Uart2 Rx data */
    ALD_PIS_CH10_UART1_RXD     = 0x011AU,    /**< Uart1 Rx data */
    ALD_PIS_CH9_UART0_RXD      = 0x0019U,    /**< Uart0 Rx data  */
    ALD_PIS_CH8_SPI1_CLK       = 0x0F18U,    /**< Spi1 clk */
    ALD_PIS_CH7_SPI1_RX        = 0x0E17U,    /**< Spi1 rx */
    ALD_PIS_CH6_SPI0_CLK       = 0x0D16U,    /**< SPI0 CLK */
    ALD_PIS_CH5_SPI0_RX        = 0x0C15U,    /**< SPI0 RX */
    ALD_PIS_CH4_TIMER0_CH4IN   = 0x0304U,    /**< AD16C4T0 input chan4 */
    ALD_PIS_CH3_TIMER0_CH3IN   = 0x0203U,    /**< AD16C4T0 input chan3 */
    ALD_PIS_CH2_TIMER0_CH2IN   = 0x0102U,    /**< AD16C4T0 input chan2 */
    ALD_PIS_CH1_TIMER0_CH1IN   = 0x0001U,    /**< AD16C4T0 input chan1 */
    ALD_PIS_CH0_TIMER0_BRKIN   = 0x0400U,    /**< AD16C4T0 break in */
    ALD_PIS_GPIO_CH0_OUTPUT    = 0x0040U,    /**< PIS channel0 OER output */
    ALD_PIS_GPIO_CH1_OUTPUT    = 0x0141U,    /**< PIS channel1 OER output */
    ALD_PIS_GPIO_CH2_OUTPUT    = 0x0242U,    /**< PIS channel2 OER output */
    ALD_PIS_GPIO_CH3_OUTPUT    = 0x0343U,    /**< PIS channel3 OER output */
    ALD_PIS_TRIG_RESERVE       = 0xFFFFU,    /**< Other Consumer */
} ald_pis_trig_t;

/**
  * @brief Clock select
  */
typedef enum
{
    ALD_PIS_CLK_PCLK1    = 0U,  /**< APB1 peripherals clock */
    ALD_PIS_CLK_PCLK2    = 1U,  /**< APB2 peripherals clock */
    ALD_PIS_CLK_SYS      = 2U,  /**< AHB peripherals clock */
    ALD_PIS_CLK_RESERVE  = 3U,  /**< reserve clock */
} ald_pis_clock_t;

/**
  * @brief Level select
  */
typedef enum
{
    ALD_PIS_EDGE_NONE    = 0U,  /**< None edge */
    ALD_PIS_EDGE_UP      = 1U,  /**< Up edge */
    ALD_PIS_EDGE_DOWN    = 2U,  /**< Down edge */
    ALD_PIS_EDGE_UP_DOWN = 3U,  /**< Up and down edge */
} ald_pis_edge_t;

/**
  * @brief Output style
  */
typedef enum
{
    ALD_PIS_OUT_LEVEL = 0U, /**< Level */
    ALD_PIS_OUT_PULSE = 1U, /**< Pulse */
} ald_pis_output_t;

/**
  * @brief Sync select
  */
typedef enum
{
    ALD_PIS_SYN_DIRECT          = 0U,   /**< Direct */
    ALD_PIS_SYN_LEVEL_ASY_APB1  = 1U,   /**< Producer level signal and Consumer in APB1 */
    ALD_PIS_SYN_LEVEL_ASY_APB2  = 2U,   /**< Producer level signal and Consumer in APB2 */
    ALD_PIS_SYN_LEVEL_ASY_AHB   = 3U,   /**< Producer level signal and Consumer in AHB */
    ALD_PIS_SYN_PULSE_ASY_APB1  = 4U,   /**< Producer Pulse signal and Consumer in APB1 */
    ALD_PIS_SYN_PULSE_ASY_APB2  = 5U,   /**< Producer Pulse signal and Consumer in APB2 */
    ALD_PIS_SYN_PULSE_ASY_AHB   = 6U,   /**< Producer Pulse signal and Consumer in AHB */
} ald_pis_syncsel_t;

/**
  * @brief Pis channel
  */
typedef enum
{
    ALD_PIS_CH_0 = 0U,   /**< Channel 0 */
    ALD_PIS_CH_1 = 1U,   /**< Channel 1 */
    ALD_PIS_CH_2 = 2U,   /**< Channel 2 */
    ALD_PIS_CH_3 = 3U,   /**< Channel 3 */
    ALD_PIS_CH_4 = 4U,   /**< Channel 4 */
    ALD_PIS_CH_5 = 5U,   /**< Channel 5 */
    ALD_PIS_CH_6 = 6U,   /**< Channel 6 */
    ALD_PIS_CH_7 = 7U,   /**< Channel 7 */
    ALD_PIS_CH_8 = 8U,   /**< Channel 8 */
    ALD_PIS_CH_9 = 9U,   /**< Channel 9 */
    ALD_PIS_CH_10 = 10U, /**< Channel 10 */
    ALD_PIS_CH_11 = 11U, /**< Channel 11 */
    ALD_PIS_CH_12 = 12U, /**< Channel 12 */
    ALD_PIS_CH_13 = 13U, /**< Channel 13 */
    ALD_PIS_CH_14 = 14U, /**< Channel 14 */
    ALD_PIS_CH_15 = 15U, /**< Channel 15 */
} ald_pis_ch_t;

/**
  * @brief Pis output channel
  */
typedef enum
{
    ALD_PIS_OUT_CH_0 = 0U,  /**< Channel 0 */
    ALD_PIS_OUT_CH_1 = 1U,  /**< Channel 1 */
    ALD_PIS_OUT_CH_2 = 2U,  /**< Channel 2 */
    ALD_PIS_OUT_CH_3 = 3U,  /**< Channel 3 */
} ald_pis_out_ch_t;

/**
  * @brief Indirect value,no care of it.
  */
typedef enum
{
    ALD_PIS_CON_0    = 0U,  /**< Con 0 */
    ALD_PIS_CON_1    = 1U,  /**< Con 1 */
    ALD_PIS_OER      = 4U,  /**< OER */
    ALD_PIS_CON_NONE = 9U,  /**< None */
} ald_pis_con_t;

/**
  * @brief PIS state structures definition
  */
typedef enum
{
    ALD_PIS_STATE_RESET   = 0x00U,  /**< Peripheral is not initialized */
    ALD_PIS_STATE_READY   = 0x01U,  /**< Peripheral Initialized and ready for use */
    ALD_PIS_STATE_BUSY    = 0x02U,  /**< An internal process is ongoing */
    ALD_PIS_STATE_TIMEOUT = 0x03U,  /**< timeout state */
    ALD_PIS_STATE_ERROR   = 0x04U,  /**< error */
} ald_pis_state_t;

/**
  * @brief PIS modulate target
  */
typedef enum
{
    ALD_PIS_UART0_TX   = 0U,    /**< Modulate uart0 tx */
    ALD_PIS_UART1_TX   = 1U,    /**< Modulate uart1 tx */
    ALD_PIS_UART2_TX   = 2U,    /**< Modulate uart2 tx */
    ALD_PIS_UART3_TX   = 3U,    /**< Modulate uart3 tx */
    ALD_PIS_LPUART0_TX = 4U,    /**< Modulate lpuart0 tx */
} ald_pis_modu_targ_t;

/**
  * @brief PIS modulate level
  */
typedef enum
{
    ALD_PIS_LOW_LEVEL  = 0U,    /**< Modulate low level */
    ALD_PIS_HIGH_LEVEL = 1U,    /**< Modulate high level */
} ald_pis_modu_level_t;

/**
  * @brief PIS modulate source
  */
typedef enum
{
    ALD_PIS_SRC_NONE   = 0U,    /**< Stop modulate */
    ALD_PIS_SRC_TIMER0 = 1U,    /**< Modulate source is TIMER0 */
    ALD_PIS_SRC_TIMER2 = 3U,    /**< Modulate source is TIMER2 */
    ALD_PIS_SRC_TIMER3 = 4U,    /**< Modulate source is TIMER3 */
    ALD_PIS_SRC_TIMER6 = 5U,    /**< Modulate source is TIMER6 */
    ALD_PIS_SRC_TIMER7 = 6U,    /**< Modulate source is TIMER7 */
    ALD_PIS_SRC_LPTIM0 = 7U,    /**< Modulate source is LPTIM0 */
    ALD_PIS_SRC_BUZ    = 8U,    /**< Modulate source is buz */
} ald_pis_modu_src_t;

/**
  * @brief PIS modulate channel
  */
typedef enum
{
    ALD_PIS_TIMER_CH1 = 0U, /**< Src is TIMERx and choose channel 1 */
    ALD_PIS_TIMER_CH2 = 1U, /**< Src is TIMERx and choose channel 2 */
    ALD_PIS_TIMER_CH3 = 2U, /**< Src is TIMERx and choose channel 3 */
    ALD_PIS_TIMER_CH4 = 3U, /**< Src is TIMERx and choose channel 4 */
} ald_pis_modu_channel_t;

/**
  * @brief PIS input channel chose
  */
typedef enum
{
    ALD_PIS_NONE_INPUT = 0U,    /**< Consumer input none */
    ALD_PIS_PORT_INPUT = 1U,    /**< Consumer input choose external port */
    ALD_PIS_CHAN_INPUT = 2U,    /**< Consumer input choose pis channel */
} ald_pis_input_sel_t;

/**
  * @brief PIS init structure definition
  */
typedef struct
{
    ald_pis_src_t producer_src;         /**< Producer entry */
    ald_pis_output_t producer_signal;   /**< Producer signal mode */
    ald_pis_clock_t producer_clk;       /**< Producer module clock */
    ald_pis_edge_t producer_edge;       /**< Producer module pin output edge */
    ald_pis_trig_t consumer_trig;       /**< Consumer entry */
    ald_pis_clock_t consumer_clk;       /**< Consumer clock */
    ald_pis_input_sel_t input_chan;     /**< Consumer input channel */
} ald_pis_init_t;

/**
  * @brief PIS modulate config structure definition
  */
typedef struct
{
    ald_pis_modu_targ_t target;     /**< Modulate target */
    ald_pis_modu_level_t level;     /**< Modulate level */
    ald_pis_modu_src_t src;         /**< Modulate src */
    ald_pis_modu_channel_t channel; /**< Modulate channel */
} ald_pis_modulate_config_t;

/**
  * @brief  PIS Handle Structure definition
  */
typedef struct pis_handle_s
{
    PIS_TypeDef *perh;              /**< Register base address */
    ald_pis_init_t init;            /**< PIS required parameters */
    ald_pis_ch_t consumer_ch;       /**< Indirect value, no care of it */
    ald_pis_con_t consumer_con;     /**< Indirect value, no care of it */
    uint8_t consumer_pos;           /**< Indirect value, no care of it */
    uint32_t check_info;            /**< When destroy a handle ,user need check whether is right that ready to destroy */
    lock_state_t lock;              /**< Locking object */
    ald_pis_state_t state;          /**< PIS operation state */
} ald_pis_handle_t;
/**
  * @}
  */

/** @defgroup PIS_Private_Macros PIS Private Macros
  * @{
  */
#define IS_PIS(x)       (((x) == PIS))
#define IS_PIS_SRC(x)       (((x) == ALD_PIS_SOFTWARE_LEVEL)      || \
                             ((x) == ALD_PIS_SOFTWARE_PULSE)      || \
                             ((x) == ALD_PIS_GPIOA_PIN0)          || \
                             ((x) == ALD_PIS_GPIOA_PIN1)          || \
                             ((x) == ALD_PIS_GPIOA_PIN2)          || \
                             ((x) == ALD_PIS_GPIOA_PIN3)          || \
                             ((x) == ALD_PIS_GPIOA_PIN4)          || \
                             ((x) == ALD_PIS_GPIOA_PIN5)          || \
                             ((x) == ALD_PIS_GPIOA_PIN6)          || \
                             ((x) == ALD_PIS_GPIOA_PIN7)          || \
                             ((x) == ALD_PIS_GPIOA_PIN8)          || \
                             ((x) == ALD_PIS_GPIOA_PIN9)          || \
                             ((x) == ALD_PIS_GPIOA_PIN10)         || \
                             ((x) == ALD_PIS_GPIOA_PIN11)         || \
                             ((x) == ALD_PIS_GPIOA_PIN12)         || \
                             ((x) == ALD_PIS_GPIOA_PIN13)         || \
                             ((x) == ALD_PIS_GPIOA_PIN14)         || \
                             ((x) == ALD_PIS_GPIOA_PIN15)         || \
                             ((x) == ALD_PIS_GPIOB_PIN0)          || \
                             ((x) == ALD_PIS_GPIOB_PIN1)          || \
                             ((x) == ALD_PIS_GPIOB_PIN2)          || \
                             ((x) == ALD_PIS_GPIOB_PIN3)          || \
                             ((x) == ALD_PIS_GPIOB_PIN4)          || \
                             ((x) == ALD_PIS_GPIOB_PIN5)          || \
                             ((x) == ALD_PIS_GPIOB_PIN6)          || \
                             ((x) == ALD_PIS_GPIOB_PIN7)          || \
                             ((x) == ALD_PIS_GPIOB_PIN8)          || \
                             ((x) == ALD_PIS_GPIOB_PIN9)          || \
                             ((x) == ALD_PIS_GPIOB_PIN10)         || \
                             ((x) == ALD_PIS_GPIOB_PIN11)         || \
                             ((x) == ALD_PIS_GPIOB_PIN12)         || \
                             ((x) == ALD_PIS_GPIOB_PIN13)         || \
                             ((x) == ALD_PIS_GPIOB_PIN14)         || \
                             ((x) == ALD_PIS_GPIOB_PIN15)         || \
                             ((x) == ALD_PIS_GPIOC_PIN0 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN1 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN2 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN3 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN4 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN5 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN6 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN7 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN8 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN9 )         || \
                             ((x) == ALD_PIS_GPIOC_PIN10)         || \
                             ((x) == ALD_PIS_GPIOC_PIN11)         || \
                             ((x) == ALD_PIS_GPIOC_PIN12)         || \
                             ((x) == ALD_PIS_GPIOC_PIN13)         || \
                             ((x) == ALD_PIS_GPIOC_PIN14)         || \
                             ((x) == ALD_PIS_GPIOC_PIN15)         || \
                             ((x) == ALD_PIS_GPIOD_PIN0 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN1 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN2 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN3 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN4 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN5 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN6 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN7 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN8 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN9 )         || \
                             ((x) == ALD_PIS_GPIOD_PIN10)         || \
                             ((x) == ALD_PIS_GPIOD_PIN11)         || \
                             ((x) == ALD_PIS_GPIOD_PIN12)         || \
                             ((x) == ALD_PIS_GPIOD_PIN13)         || \
                             ((x) == ALD_PIS_GPIOD_PIN14)         || \
                             ((x) == ALD_PIS_GPIOD_PIN15)         || \
                             ((x) == ALD_PIS_GPIOF_PIN0 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN1 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN2 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN3 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN4 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN5 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN6 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN7 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN8 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN9 )         || \
                             ((x) == ALD_PIS_GPIOF_PIN10)         || \
                             ((x) == ALD_PIS_GPIOF_PIN11)         || \
                             ((x) == ALD_PIS_GPIOF_PIN12)         || \
                             ((x) == ALD_PIS_GPIOF_PIN13)         || \
                             ((x) == ALD_PIS_GPIOF_PIN14)         || \
                             ((x) == ALD_PIS_GPIOF_PIN15)         || \
                             ((x) == ALD_PIS_GPIOH_PIN0 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN1 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN2 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN3 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN4 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN5 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN6 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN7 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN8 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN9 )         || \
                             ((x) == ALD_PIS_GPIOH_PIN10)         || \
                             ((x) == ALD_PIS_GPIOH_PIN11)         || \
                             ((x) == ALD_PIS_GPIOH_PIN12)         || \
                             ((x) == ALD_PIS_GPIOH_PIN13)         || \
                             ((x) == ALD_PIS_GPIOH_PIN14)         || \
                             ((x) == ALD_PIS_GPIOH_PIN15)         || \
                             ((x) == ALD_PIS_ADC0_INSERT)         || \
                             ((x) == ALD_PIS_ADC0_NORMAL)         || \
                             ((x) == ALD_PIS_ADC0_WDT   )         || \
                             ((x) == ALD_PIS_LVD        )         || \
                             ((x) == ALD_PIS_LVDEX               )|| \
                             ((x) == ALD_PIS_UART0_RESERVE0      )|| \
                             ((x) == ALD_PIS_UART0_RESERVE1      )|| \
                             ((x) == ALD_PIS_UART0_IRDAOUT       )|| \
                             ((x) == ALD_PIS_UART0_RTSOUT        )|| \
                             ((x) == ALD_PIS_UART0_TXOUT         )|| \
                             ((x) == ALD_PIS_UART0_SEND_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART0_RECV_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART1_RESERVE0      )|| \
                             ((x) == ALD_PIS_UART1_RESERVE1      )|| \
                             ((x) == ALD_PIS_UART1_IRDA          )|| \
                             ((x) == ALD_PIS_UART1_RTS           )|| \
                             ((x) == ALD_PIS_UART1_TXOUT         )|| \
                             ((x) == ALD_PIS_UART1_SEND_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART1_RECV_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART2_RESERVE0      )|| \
                             ((x) == ALD_PIS_UART2_RESERVE1      )|| \
                             ((x) == ALD_PIS_UART2_RTS           )|| \
                             ((x) == ALD_PIS_UART2_TXOUT         )|| \
                             ((x) == ALD_PIS_UART2_SEND_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART2_RECV_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART3_RESERVE0      )|| \
                             ((x) == ALD_PIS_UART3_RESERVE1      )|| \
                             ((x) == ALD_PIS_UART3_RTS           )|| \
                             ((x) == ALD_PIS_UART3_TXOUT         )|| \
                             ((x) == ALD_PIS_UART3_SEND_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART3_RECV_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART4_RESERVE0      )|| \
                             ((x) == ALD_PIS_UART4_RESERVE1      )|| \
                             ((x) == ALD_PIS_UART4_RTS           )|| \
                             ((x) == ALD_PIS_UART4_TXOUT         )|| \
                             ((x) == ALD_PIS_UART4_SEND_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART4_RECV_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART5_RESERVE0      )|| \
                             ((x) == ALD_PIS_UART5_RESERVE1      )|| \
                             ((x) == ALD_PIS_UART5_RTS           )|| \
                             ((x) == ALD_PIS_UART5_TXOUT         )|| \
                             ((x) == ALD_PIS_UART5_SEND_INT_PULSE)|| \
                             ((x) == ALD_PIS_UART5_RECV_INT_PULSE)|| \
                             ((x) == ALD_PIS_SPI0_RECV           )|| \
                             ((x) == ALD_PIS_SPI0_SEND           )|| \
                             ((x) == ALD_PIS_SPI0_NE             )|| \
                             ((x) == ALD_PIS_SPI1_RECV           )|| \
                             ((x) == ALD_PIS_SPI1_SEND           )|| \
                             ((x) == ALD_PIS_SPI1_NE             )|| \
                             ((x) == ALD_PIS_I2C0_RECV           )|| \
                             ((x) == ALD_PIS_I2C0_SEND           )|| \
                             ((x) == ALD_PIS_I2C1_RECV           )|| \
                             ((x) == ALD_PIS_I2C1_SEND           )|| \
                             ((x) == ALD_PIS_TIMER0_UPDATA       )|| \
                             ((x) == ALD_PIS_TIMER0_TRGOUT       )|| \
                             ((x) == ALD_PIS_TIMER0_INPUT_1      )|| \
                             ((x) == ALD_PIS_TIMER0_OUTPUT_1     )|| \
                             ((x) == ALD_PIS_TIMER0_INPUT_2      )|| \
                             ((x) == ALD_PIS_TIMER0_OUTPUT_2     )|| \
                             ((x) == ALD_PIS_TIMER0_INPUT_3      )|| \
                             ((x) == ALD_PIS_TIMER0_OUTPUT_3     )|| \
                             ((x) == ALD_PIS_TIMER0_INPUT_4      )|| \
                             ((x) == ALD_PIS_TIMER0_OUTPUT_4     )|| \
                             ((x) == ALD_PIS_TIMER2_TRGOUT       )|| \
                             ((x) == ALD_PIS_TIMER2_INPUT_1      )|| \
                             ((x) == ALD_PIS_TIMER2_OUTPUT_1     )|| \
                             ((x) == ALD_PIS_TIMER2_INPUT_2      )|| \
                             ((x) == ALD_PIS_TIMER2_OUTPUT_2     )|| \
                             ((x) == ALD_PIS_TIMER2_INPUT_3      )|| \
                             ((x) == ALD_PIS_TIMER2_OUTPUT_3     )|| \
                             ((x) == ALD_PIS_TIMER2_INPUT_4      )|| \
                             ((x) == ALD_PIS_TIMER2_OUTPUT_4     )|| \
                             ((x) == ALD_PIS_TIMER3_TRGOUT       )|| \
                             ((x) == ALD_PIS_TIMER3_INPUT_1      )|| \
                             ((x) == ALD_PIS_TIMER3_OUTPUT_1     )|| \
                             ((x) == ALD_PIS_TIMER3_INPUT_2      )|| \
                             ((x) == ALD_PIS_TIMER3_OUTPUT_2     )|| \
                             ((x) == ALD_PIS_TIMER3_INPUT_3      )|| \
                             ((x) == ALD_PIS_TIMER3_OUTPUT_3     )|| \
                             ((x) == ALD_PIS_TIMER3_INPUT_4      )|| \
                             ((x) == ALD_PIS_TIMER3_OUTPUT_4     )|| \
                             ((x) == ALD_PIS_RTC_CLOCK           )|| \
                             ((x) == ALD_PIS_RTC_ALARM           )|| \
                             ((x) == ALD_PIS_LPTIMER0_SYN_UPDATA )|| \
                             ((x) == ALD_PIS_LPTIMER0_ASY_UPDATA )|| \
                             ((x) == ALD_PIS_LPUART0_ASY_RECV    )|| \
                             ((x) == ALD_PIS_LPUART0_ASY_SEND    )|| \
                             ((x) == ALD_PIS_LPUART0_SYN_RECV    )|| \
                             ((x) == ALD_PIS_LPUART0_SYN_SEND    )|| \
                             ((x) == ALD_PIS_TIMER4_TRGOUT       )|| \
                             ((x) == ALD_PIS_TIMER5_TRGOUT       )|| \
                             ((x) == ALD_PIS_TIMER6_TRGOUT       )|| \
                             ((x) == ALD_PIS_TIMER6_INPUT_1      )|| \
                             ((x) == ALD_PIS_TIMER6_OUTPUT_1     )|| \
                             ((x) == ALD_PIS_TIMER6_INPUT_2      )|| \
                             ((x) == ALD_PIS_TIMER6_OUTPUT_2     )|| \
                             ((x) == ALD_PIS_TIMER6_INPUT_3      )|| \
                             ((x) == ALD_PIS_TIMER6_OUTPUT_3     )|| \
                             ((x) == ALD_PIS_TIMER6_INPUT_4      )|| \
                             ((x) == ALD_PIS_TIMER6_OUTPUT_4     )|| \
                             ((x) == ALD_PIS_TIMER7_TRGOUT       )|| \
                             ((x) == ALD_PIS_TIMER7_INPUT_1      )|| \
                             ((x) == ALD_PIS_TIMER7_OUTPUT_1     )|| \
                             ((x) == ALD_PIS_TIMER7_INPUT_2      )|| \
                             ((x) == ALD_PIS_TIMER7_OUTPUT_2     )|| \
                             ((x) == ALD_PIS_TIMER7_INPUT_3      )|| \
                             ((x) == ALD_PIS_TIMER7_OUTPUT_3     )|| \
                             ((x) == ALD_PIS_TIMER7_INPUT_4      )|| \
                             ((x) == ALD_PIS_TIMER7_OUTPUT_4     )|| \
                             ((x) == ALD_PIS_ADC0_INSERT         )|| \
                             ((x) == ALD_PIS_ADC0_NORMAL         )|| \
                             ((x) == ALD_PIS_ADC0_WDT            )|| \
                             ((x) == ALD_PIS_SYS_HSCO            )|| \
                             ((x) == ALD_PIS_SYS_LSCO            )|| \
                             ((x) == ALD_PIS_SYS_BUZ             )|| \
                             ((x) == ALD_PIS_SYS_HSCO            )|| \
                             ((x) == ALD_PIS_SYS_LSCO            )|| \
                             ((x) == ALD_PIS_SYS_BUZ             )|| \
                             ((x) == ALD_PIS_DMA_CH0             )|| \
                             ((x) == ALD_PIS_DMA_CH1             )|| \
                             ((x) == ALD_PIS_DMA_CH2             )|| \
                             ((x) == ALD_PIS_DMA_CH3             )|| \
                             ((x) == ALD_PIS_DMA_CH4             )|| \
                             ((x) == ALD_PIS_DMA_CH5             )|| \
                             ((x) == ALD_PIS_DMA_CH6             )|| \
                             ((x) == ALD_PIS_DMA_CH7             )|| \
                             ((x) == ALD_PIS_DMA_CH8             )|| \
                             ((x) == ALD_PIS_DMA_CH9             )|| \
                             ((x) == ALD_PIS_DMA_CH10            )|| \
                             ((x) == ALD_PIS_DMA_CH11            )|| \
                             ((x) == ALD_PIS_DMA_CH))
#define IS_PIS_TRIG(x)      (((x) ==  ALD_PIS_CH0_TIMER0_ITR0    )  || \
                             ((x) ==  ALD_PIS_CH1_TIMER0_ITR1    )  || \
                             ((x) ==  ALD_PIS_CH2_TIMER0_ITR2    )  || \
                             ((x) ==  ALD_PIS_CH3_TIMER0_ITR3    )  || \
                             ((x) ==  ALD_PIS_CH4_TIMER0_ITR4    )  || \
                             ((x) ==  ALD_PIS_CH5_TIMER0_ITR5    )  || \
                             ((x) ==  ALD_PIS_CH6_TIMER0_ITR6    )  || \
                             ((x) ==  ALD_PIS_CH7_TIMER0_ITR7    )  || \
                             ((x) ==  ALD_PIS_CH8_TIMER0_ITR8    )  || \
                             ((x) ==  ALD_PIS_CH9_TIMER0_ITR9    )  || \
                             ((x) ==  ALD_PIS_CH10_TIMER0_ITR10  )  || \
                             ((x) ==  ALD_PIS_CH11_TIMER0_ITR11  )  || \
                             ((x) ==  ALD_PIS_CH12_TIMER0_ITR12  )  || \
                             ((x) ==  ALD_PIS_CH13_TIMER0_ITR13  )  || \
                             ((x) ==  ALD_PIS_CH14_TIMER0_ITR14  )  || \
                             ((x) ==  ALD_PIS_CH15_TIMER0_ITR15  )  || \
                             ((x) ==  ALD_PIS_CH0_TIMER2_ITR0    )  || \
                             ((x) ==  ALD_PIS_CH1_TIMER2_ITR1    )  || \
                             ((x) ==  ALD_PIS_CH2_TIMER2_ITR2    )  || \
                             ((x) ==  ALD_PIS_CH3_TIMER2_ITR3    )  || \
                             ((x) ==  ALD_PIS_CH4_TIMER2_ITR4    )  || \
                             ((x) ==  ALD_PIS_CH5_TIMER2_ITR5    )  || \
                             ((x) ==  ALD_PIS_CH6_TIMER2_ITR6    )  || \
                             ((x) ==  ALD_PIS_CH7_TIMER2_ITR7    )  || \
                             ((x) ==  ALD_PIS_CH8_TIMER2_ITR8    )  || \
                             ((x) ==  ALD_PIS_CH9_TIMER2_ITR8    )  || \
                             ((x) ==  ALD_PIS_CH10_TIMER2_ITR10  )  || \
                             ((x) ==  ALD_PIS_CH11_TIMER2_ITR11  )  || \
                             ((x) ==  ALD_PIS_CH12_TIMER2_ITR12  )  || \
                             ((x) ==  ALD_PIS_CH13_TIMER2_ITR13  )  || \
                             ((x) ==  ALD_PIS_CH14_TIMER2_ITR14  )  || \
                             ((x) ==  ALD_PIS_CH15_TIMER2_ITR15  )  || \
                             ((x) ==  ALD_PIS_CH5_TIMER2_CH1IN   )  || \
                             ((x) ==  ALD_PIS_CH6_TIMER2_CH2IN   )  || \
                             ((x) ==  ALD_PIS_CH7_TIMER2_CH3IN   )  || \
                             ((x) ==  ALD_PIS_CH8_TIMER2_CH4IN   )  || \
                             ((x) ==  ALD_PIS_CH0_TIMER3_ITR0    )  || \
                             ((x) ==  ALD_PIS_CH1_TIMER3_ITR1    )  || \
                             ((x) ==  ALD_PIS_CH2_TIMER3_ITR2    )  || \
                             ((x) ==  ALD_PIS_CH3_TIMER3_ITR3    )  || \
                             ((x) ==  ALD_PIS_CH4_TIMER3_ITR4    )  || \
                             ((x) ==  ALD_PIS_CH5_TIMER3_ITR5    )  || \
                             ((x) ==  ALD_PIS_CH6_TIMER3_ITR6    )  || \
                             ((x) ==  ALD_PIS_CH7_TIMER3_ITR7    )  || \
                             ((x) ==  ALD_PIS_CH8_TIMER3_ITR8    )  || \
                             ((x) ==  ALD_PIS_CH9_TIMER3_ITR9    )  || \
                             ((x) ==  ALD_PIS_CH10_TIMER3_ITR10  )  || \
                             ((x) ==  ALD_PIS_CH11_TIMER3_ITR11  )  || \
                             ((x) ==  ALD_PIS_CH12_TIMER3_ITR12  )  || \
                             ((x) ==  ALD_PIS_CH13_TIMER3_ITR13  )  || \
                             ((x) ==  ALD_PIS_CH14_TIMER3_ITR14  )  || \
                             ((x) ==  ALD_PIS_CH15_TIMER3_ITR15  )  || \
                             ((x) ==  ALD_PIS_CH5_TIMER3_CH1IN   )  || \
                             ((x) ==  ALD_PIS_CH6_TIMER3_CH2IN   )  || \
                             ((x) ==  ALD_PIS_CH7_TIMER3_CH3IN   )  || \
                             ((x) ==  ALD_PIS_CH8_TIMER3_CH4IN   )  || \
                             ((x) ==  ALD_PIS_CH0_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH1_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH2_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH3_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH4_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH5_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH6_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH7_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH8_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH9_ADC0_NORMAL    )  || \
                             ((x) ==  ALD_PIS_CH10_ADC0_NORMAL   )  || \
                             ((x) ==  ALD_PIS_CH11_ADC0_NORMAL   )  || \
                             ((x) ==  ALD_PIS_CH12_ADC0_NORMAL   )  || \
                             ((x) ==  ALD_PIS_CH13_ADC0_NORMAL   )  || \
                             ((x) ==  ALD_PIS_CH14_ADC0_NORMAL   )  || \
                             ((x) ==  ALD_PIS_CH15_ADC0_NORMAL   )  || \
                             ((x) ==  ALD_PIS_CH0_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH1_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH2_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH3_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH4_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH5_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH6_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH7_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH8_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH9_ADC0_INSERT    )  || \
                             ((x) ==  ALD_PIS_CH10_ADC0_INSERT   )  || \
                             ((x) ==  ALD_PIS_CH11_ADC0_INSERT   )  || \
                             ((x) ==  ALD_PIS_CH12_ADC0_INSERT   )  || \
                             ((x) ==  ALD_PIS_CH13_ADC0_INSERT   )  || \
                             ((x) ==  ALD_PIS_CH14_ADC0_INSERT   )  || \
                             ((x) ==  ALD_PIS_CH15_ADC0_INSERT   )  || \
                             ((x) ==  ALD_PIS_CH0_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH1_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH2_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH3_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH4_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH5_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH6_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH7_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH8_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH9_DAC_CONVERT    )  || \
                             ((x) ==  ALD_PIS_CH10_DAC_CONVERT   )  || \
                             ((x) ==  ALD_PIS_CH11_DAC_CONVERT   )  || \
                             ((x) ==  ALD_PIS_CH12_DAC_CONVERT   )  || \
                             ((x) ==  ALD_PIS_CH13_DAC_CONVERT   )  || \
                             ((x) ==  ALD_PIS_CH14_DAC_CONVERT   )  || \
                             ((x) ==  ALD_PIS_CH15_DAC_CONVERT   )  || \
                             ((x) ==  ALD_PIS_CH0_LPTIM0_EXT0    )  || \
                             ((x) ==  ALD_PIS_CH1_LPTIM0_EXT1    )  || \
                             ((x) ==  ALD_PIS_CH2_LPTIM0_EXT2    )  || \
                             ((x) ==  ALD_PIS_CH3_LPTIM0_EXT3    )  || \
                             ((x) ==  ALD_PIS_CH4_LPTIM0_EXT4    )  || \
                             ((x) ==  ALD_PIS_CH5_LPTIM0_EXT5    )  || \
                             ((x) ==  ALD_PIS_CH6_LPTIM0_EXT6    )  || \
                             ((x) ==  ALD_PIS_CH7_LPTIM0_EXT7    )  || \
                             ((x) ==  ALD_PIS_CH0_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH1_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH2_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH3_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH4_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH5_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH6_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH7_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH8_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH9_DMA_REQUEST    )  || \
                             ((x) ==  ALD_PIS_CH10_DMA_REQUEST   )  || \
                             ((x) ==  ALD_PIS_CH11_DMA_REQUEST   )  || \
                             ((x) ==  ALD_PIS_CH12_DMA_REQUEST   )  || \
                             ((x) ==  ALD_PIS_CH13_DMA_REQUEST   )  || \
                             ((x) ==  ALD_PIS_CH14_DMA_REQUEST   )  || \
                             ((x) ==  ALD_PIS_CH15_DMA_REQUEST   )  || \
                             ((x) ==  ALD_PIS_CH15_LPUART0_RXD   )  || \
                             ((x) ==  ALD_PIS_CH14_UART5_RXD     )  || \
                             ((x) ==  ALD_PIS_CH13_UART4_RXD     )  || \
                             ((x) ==  ALD_PIS_CH12_UART3_RXD     )  || \
                             ((x) ==  ALD_PIS_CH11_UART2_RXD     )  || \
                             ((x) ==  ALD_PIS_CH10_UART1_RXD     )  || \
                             ((x) ==  ALD_PIS_CH9_UART0_RXD      )  || \
                             ((x) ==  ALD_PIS_CH8_SPI1_CLK       )  || \
                             ((x) ==  ALD_PIS_CH7_SPI1_RX        )  || \
                             ((x) ==  ALD_PIS_CH6_SPI0_CLK       )  || \
                             ((x) ==  ALD_PIS_CH5_SPI0_RX        )  || \
                             ((x) ==  ALD_PIS_CH4_TIMER0_CH4IN   )  || \
                             ((x) ==  ALD_PIS_CH3_TIMER0_CH3IN   )  || \
                             ((x) ==  ALD_PIS_CH2_TIMER0_CH2IN   )  || \
                             ((x) ==  ALD_PIS_CH1_TIMER0_CH1IN   )  || \
                             ((x) ==  ALD_PIS_CH0_TIMER0_BRKIN   )  || \
                             ((x) ==  ALD_PIS_GPIO_CH0_OUTPUT    )  || \
                             ((x) ==  ALD_PIS_GPIO_CH1_OUTPUT    )  || \
                             ((x) ==  ALD_PIS_GPIO_CH2_OUTPUT    )  || \
                             ((x) ==  ALD_PIS_GPIO_CH3_OUTPUT    ))
#define IS_PIS_CLOCK(x)     (((x) == ALD_PIS_CLK_PCLK1)  || \
                             ((x) == ALD_PIS_CLK_PCLK2)  || \
                             ((x) == ALD_PIS_CLK_SYS))
#define IS_PIS_SIGNAL_MODE(x)   (((x) == ALD_PIS_OUT_LEVEL)  || \
                                 ((x) == ALD_PIS_OUT_PULSE))
#define IS_PIS_EDGE(x)      (((x) == ALD_PIS_EDGE_NONE) || \
                             ((x) == ALD_PIS_EDGE_UP)   || \
                             ((x) == ALD_PIS_EDGE_DOWN) || \
                             ((x) == ALD_PIS_EDGE_UP_DOWN))
#define IS_PIS_OUTPUT(x)    (((x) == ALD_PIS_OUT_LEVEL) || \
                             ((x) == ALD_PIS_OUT_PULSE))
#define IS_PIS_OUPUT_CH(x)  (((x) == ALD_PIS_OUT_CH_0) || \
                             ((x) == ALD_PIS_OUT_CH_1) || \
                             ((x) == ALD_PIS_OUT_CH_2) || \
                             ((x) == ALD_PIS_OUT_CH_3))
#define IS_PIS_MODU_TARGET(x)   (((x) == ALD_PIS_UART0_TX) || \
                                 ((x) == ALD_PIS_UART1_TX) || \
                                 ((x) == ALD_PIS_UART2_TX) || \
                                 ((x) == ALD_PIS_UART3_TX) || \
                                 ((x) == ALD_PIS_LPUART0_TX))
#define IS_PIS_MODU_LEVEL(x)    (((x) == ALD_PIS_LOW_LEVEL) || \
                                 ((x) == ALD_PIS_HIGH_LEVEL))
#define IS_PIS_MODU_SRC(x)  (((x) == ALD_PIS_SRC_NONE)     || \
                             ((x) == ALD_PIS_SRC_TIMER0)   || \
                             ((x) == ALD_PIS_SRC_TIMER2)   || \
                             ((x) == ALD_PIS_SRC_TIMER3)   || \
                             ((x) == ALD_PIS_SRC_TIMER6)   || \
                             ((x) == ALD_PIS_SRC_TIMER7)   || \
                             ((x) == ALD_PIS_SRC_LPTIM0)   || \
                             ((x) == ALD_PIS_SRC_BUZ))
#define IS_PIS_MODU_CHANNEL(x)  (((x) == ALD_PIS_TIMER_CH1) || \
                                 ((x) == ALD_PIS_TIMER_CH2) || \
                                 ((x) == ALD_PIS_TIMER_CH3) || \
                                 ((x) == ALD_PIS_TIMER_CH4))

#define IS_PIS_CH(x)        (((x) == ALD_PIS_CH_0) || \
                             ((x) == ALD_PIS_CH_1) || \
                             ((x) == ALD_PIS_CH_2) || \
                             ((x) == ALD_PIS_CH_3) || \
                             ((x) == ALD_PIS_CH_4) || \
                             ((x) == ALD_PIS_CH_5) || \
                             ((x) == ALD_PIS_CH_6) || \
                             ((x) == ALD_PIS_CH_7) || \
                             ((x) == ALD_PIS_CH_8) || \
                             ((x) == ALD_PIS_CH_9) || \
                             ((x) == ALD_PIS_CH_10) || \
                             ((x) == ALD_PIS_CH_11) || \
                             ((x) == ALD_PIS_CH_12) || \
                             ((x) == ALD_PIS_CH_13) || \
                             ((x) == ALD_PIS_CH_14) || \
                             ((x) == ALD_PIS_CH_15))
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions
  * @{
  */

/** @addtogroup PIS_Public_Functions_Group1
  * @{
  */
ald_status_t ald_pis_create(ald_pis_handle_t *hperh);
ald_status_t ald_pis_destroy(ald_pis_handle_t *hperh);
ald_status_t ald_pis_software_signal(ald_pis_handle_t *hperh, ald_pis_ch_t ch, ald_pis_output_t out);
ald_status_t ald_pis_software_signal_disable(ald_pis_handle_t *hperh, ald_pis_ch_t ch, ald_pis_output_t out);
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions_Group2
  * @{
  */
ald_status_t ald_pis_output_start(ald_pis_handle_t *hperh, ald_pis_out_ch_t ch);
ald_status_t ald_pis_output_stop(ald_pis_handle_t *hperh, ald_pis_out_ch_t ch);
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions_Group3
  * @{
  */
ald_pis_state_t ald_pis_get_state(ald_pis_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup PIS_Public_Functions_Group4
  * @{
  */
ald_status_t ald_pis_modu_config(ald_pis_handle_t *hperh, ald_pis_modulate_config_t *config);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __ALD_PIS_H__ */
