/**
  *********************************************************************************
  *
  * @file    ald_lptim.c
  * @brief   LPTIM module driver.
  *      This is the common part of the LPTIM initialization
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_LPTIM_H__
#define __ALD_LPTIM_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"
#include "ald_cmu.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup LPTIM
  * @{
  */

/** @defgroup LPTIM_Public_Types LPTIM Public Types
  * @{
  */

/**
  * @brief LPTIM clock select
  */
typedef enum
{
    ALD_LPTIM_CKSEL_INTERNAL = 0U,  /**< Select internal clock */
    ALD_LPTIM_CKSEL_EXTERNAL = 1U,  /**< Select external clock */
} ald_lptim_cksel_t;

/**
  * @brief LPTIM clock pol
  */
typedef enum
{
    ALD_LPTIM_CKPOL_RISING  = 0U,   /**< using rising edge */
    ALD_LPTIM_CKPOL_FALLING = 1U,   /**< using falling edge */
} ald_lptim_ckpol_t;

/**
  * @brief LPTIM clock fliter
  */
typedef enum
{
    ALD_LPTIM_CKFLT_0 = 0U, /**< not clock filter */
    ALD_LPTIM_CKFLT_2 = 1U, /**< 2 cycle filter */
    ALD_LPTIM_CKFLT_4 = 2U, /**< 4 cycle filter */
    ALD_LPTIM_CKFLT_8 = 3U, /**< 8 cycle filter */
} ald_lptim_ckflt_t;

/**
  * @brief LPTIM trigger fliter
  */
typedef enum
{
    ALD_LPTIM_TRGFLT_0 = 0U,    /**< not clock filter */
    ALD_LPTIM_TRGFLT_2 = 1U,    /**< 2 cycle filter */
    ALD_LPTIM_TRGFLT_4 = 2U,    /**< 4 cycle filter */
    ALD_LPTIM_TRGFLT_8 = 3U,    /**< 8 cycle filter */
} ald_lptim_trgflt_t;

/**
  * @brief LPTIM prescaler
  */
typedef enum
{
    ALD_LPTIM_PRESC_1   = 0U,   /**< No prescaler is used */
    ALD_LPTIM_PRESC_2   = 1U,   /**< Clock is divided by 2 */
    ALD_LPTIM_PRESC_4   = 2U,   /**< Clock is divided by 4 */
    ALD_LPTIM_PRESC_8   = 3U,   /**< Clock is divided by 8 */
    ALD_LPTIM_PRESC_16  = 4U,   /**< Clock is divided by 16 */
    ALD_LPTIM_PRESC_32  = 5U,   /**< Clock is divided by 32 */
    ALD_LPTIM_PRESC_64  = 6U,   /**< Clock is divided by 64 */
    ALD_LPTIM_PRESC_128 = 7U,   /**< Clock is divided by 128 */
} ald_lptim_presc_t;

/**
  * @brief LPTIM trig select
  */
typedef enum
{
    ALD_LPTIM_TRIGSEL_EXT0 = 0U,    /**< Trigger select external channel 0 */
    ALD_LPTIM_TRIGSEL_EXT1 = 1U,    /**< Trigger select external channel 1 */
    ALD_LPTIM_TRIGSEL_EXT2 = 2U,    /**< Trigger select external channel 2 */
    ALD_LPTIM_TRIGSEL_EXT3 = 3U,    /**< Trigger select external channel 3 */
    ALD_LPTIM_TRIGSEL_EXT4 = 4U,    /**< Trigger select external channel 4 */
    ALD_LPTIM_TRIGSEL_EXT5 = 5U,    /**< Trigger select external channel 5 */
    ALD_LPTIM_TRIGSEL_EXT6 = 6U,    /**< Trigger select external channel 6 */
    ALD_LPTIM_TRIGSEL_EXT7 = 7U,    /**< Trigger select external channel 7 */
} ald_lptim_trigsel_t;

/**
  * @brief LPTIM start mode select
  */
typedef enum
{
    ALD_LPTIM_MODE_SINGLE     = 0U, /**< Start single mode */
    ALD_LPTIM_MODE_CONTINUOUS = 1U, /**< Start continuous mode */
} ald_lptim_mode_t;

/**
  * @brief LPTIM trig en
  */
typedef enum
{
    ALD_LPTIM_TRIGEN_SW      = 0U,  /**< software trigger */
    ALD_LPTIM_TRIGEN_RISING  = 1U,  /**< rising edge trigger */
    ALD_LPTIM_TRIGEN_FALLING = 2U,  /**< falling edge trigger */
    ALD_LPTIM_TRIGEN_BOTH    = 3U,  /**< rising and falling edge trigger */
} ald_lptim_trigen_t;

/**
  * @brief LPTIM wave
  */
typedef enum
{
    ALD_LPTIM_WAVE_NONE   = 0U, /**< Output close */
    ALD_LPTIM_WAVE_TOGGLE = 1U, /**< Output toggle */
    ALD_LPTIM_WAVE_PULSE  = 2U, /**< Output pulse */
    ALD_LPTIM_WAVE_PWM    = 3U, /**< Output PWM */
} ald_lptim_wave_t;

/**
  * @brief LPTIM interrupt
  */
typedef enum
{
    ALD_LPTIM_IT_CMPMAT  = 1U,  /**< Compare interrupt bit */
    ALD_LPTIM_IT_ARRMAT  = 2U,  /**< Update interrupt bit */
    ALD_LPTIM_IT_EXTTRIG = 4U,  /**< external trigger interrupt bit */
} ald_lptim_it_t;

/**
  * @brief LPTIM Interrupt flag
  */
typedef enum
{
    ALD_LPTIM_FLAG_CMPMAT  = 1U,    /**< Compare interrupt flag */
    ALD_LPTIM_FLAG_ARRMAT  = 2U,    /**< Update interrupt flag */
    ALD_LPTIM_FLAG_EXTTRIG = 4U,    /**< Update interrupt flag */
} ald_lptim_flag_t;

/**
  * @brief LPTIM state structures definition
  */
typedef enum
{
    ALD_LPTIM_STATE_RESET   = 0x00U,    /**< Peripheral not yet initialized or disabled */
    ALD_LPTIM_STATE_READY   = 0x01U,    /**< Peripheral Initialized and ready for use */
    ALD_LPTIM_STATE_BUSY    = 0x02U,    /**< An internal process is ongoing */
    ALD_LPTIM_STATE_TIMEOUT = 0x03U,    /**< timeout state */
    ALD_LPTIM_STATE_ERROR   = 0x04U,    /**< Reception process is ongoing */
} ald_lptim_state_t;

/**
  * @brief LPTIM Init Structure definition
  */
typedef struct
{
    ald_lptim_presc_t psc;      /**< Specifies the prescaler value */
    uint16_t arr;           /**< Specifies the update value */
    uint16_t cmp;           /**< Specifies the compare value */
    ald_cmu_lp_perh_clock_sel_t clock;  /**< Specifies the clock choose */
    ald_lptim_mode_t mode;      /**< Specifies the start mode */
} ald_lptim_init_t;

/**
  * @brief LPTIM trigger Structure definition
  */
typedef struct
{
    ald_lptim_trigen_t mode;    /**< Specifies the trigger mode */
    ald_lptim_trigsel_t sel;    /**< Specifies the trigger source select */
} ald_lptim_trigger_init_t;

/**
  * @brief LPTIM trigger Structure definition
  */
typedef struct
{
    ald_lptim_cksel_t sel;  /**< Specifies the clock select */
    ald_lptim_ckpol_t polarity; /**< Specifies the clock polarity */
} ald_lptim_clock_source_init_t;

/**
  * @brief  LPTIM Handle Structure definition
  */
typedef struct lptim_handle_s
{
    LP16T_TypeDef *perh;        /**< Register base address */
    ald_lptim_init_t init;      /**< LPTIM Time required parameters */
    lock_state_t lock;          /**< Locking object */
    ald_lptim_state_t state;    /**< LPTIM operation state */

    void (*trig_cbk)(struct lptim_handle_s *arg);   /**< Trigger callback */
    void (*update_cbk)(struct lptim_handle_s *arg); /**< Update callback */
    void (*cmp_cbk)(struct lptim_handle_s *arg);    /**< Compare callback */
} ald_lptim_handle_t;
/**
  * @}
  */

/** @defgroup LPTIM_Public_Macros LPTIM Public Macros
  * @{
  */
#define LPTIM_ENABLE(x)         (SET_BIT((x)->perh->CON1, LP16T_CON1_ENABLE_MSK))
#define LPTIM_DISABLE(x)        (CLEAR_BIT((x)->perh->CON1, LP16T_CON1_ENABLE_MSK))
#define LPTIM_CNTSTART(x)       (SET_BIT((x)->perh->CON1, LP16T_CON1_CNTSTRT_MSK))
#define LPTIM_SNGSTART(x)       (SET_BIT((x)->perh->CON1, LP16T_CON1_SNGSTRT_MSK))
#define LPTIM_UPDATE_ENABLE(x)      (SET_BIT((x)->perh->UPDATE, LP16T_UPDATE_UDIS_MSK))
#define LPTIM_UPDATE_DISABLE(x)     (CLEAR_BIT((x)->perh->UPDATE, LP16T_UPDATE_UDIS_MSK))
#define LPTIM_PRELOAD_IMM(x)        (SET_BIT((x)->perh->CR0, LP16T_CON0_PRELOAD_MSK))
#define LPTIM_PRELOAD_WAIT(x)       (CLEAR_BIT((x)->perh->CR0, LP16T_CON0_PRELOAD_MSK))
#define LPTIM_WAVEPOL_NORMAL(x)     (MODIFY_REG((x)->perh->CR0, LP16T_CON0_WAVE_MSK, 0 << LP16T_CON0_WAVE_POSS))
#define LPTIM_WAVEPOL_INVERSE(x)    (MODIFY_REG((x)->perh->CR0, LP16T_CON0_WAVE_MSK, 1 << LP16T_CON0_WAVE_POSS))
/**
  * @}
  */

/** @defgroup LPTIM_Private_Macros   LPTIM Private Macros
  * @{
  */
#define IS_LPTIM(x)     ((x) == LPTIM)
#define IS_LPTIM_CKSEL(x)   (((x) == ALD_LPTIM_CKSEL_INTERNAL) || \
                             ((x) == ALD_LPTIM_CKSEL_EXTERNAL))
#define IS_LPTIM_CKPOL(x)   (((x) == ALD_LPTIM_CKPOL_RISING) || \
                             ((x) == ALD_LPTIM_CKPOL_FALLING))
#define IS_LPTIM_MODE(x)    (((x) == ALD_LPTIM_MODE_SINGLE) || \
                             ((x) == ALD_LPTIM_MODE_CONTINUOUS))
#define IS_LPTIM_CKFLT(x)   (((x) == ALD_LPTIM_CKFLT_0) || \
                             ((x) == ALD_LPTIM_CKFLT_2) || \
                             ((x) == ALD_LPTIM_CKFLT_4) || \
                             ((x) == ALD_LPTIM_CKFLT_8))
#define IS_LPTIM_TRGFLT(x)  (((x) == ALD_LPTIM_TRGFLT_0) || \
                             ((x) == ALD_LPTIM_TRGFLT_2) || \
                             ((x) == ALD_LPTIM_TRGFLT_4) || \
                             ((x) == ALD_LPTIM_TRGFLT_8))
#define IS_LPTIM_PRESC(x)   (((x) == ALD_LPTIM_PRESC_1)  || \
                             ((x) == ALD_LPTIM_PRESC_2)  || \
                             ((x) == ALD_LPTIM_PRESC_4)  || \
                             ((x) == ALD_LPTIM_PRESC_8)  || \
                             ((x) == ALD_LPTIM_PRESC_16) || \
                             ((x) == ALD_LPTIM_PRESC_32) || \
                             ((x) == ALD_LPTIM_PRESC_64) || \
                             ((x) == ALD_LPTIM_PRESC_128))
#define IS_LPTIM_TRIGSEL(x) (((x) == ALD_LPTIM_TRIGSEL_EXT0) || \
                             ((x) == ALD_LPTIM_TRIGSEL_EXT1) || \
                             ((x) == ALD_LPTIM_TRIGSEL_EXT2) || \
                             ((x) == ALD_LPTIM_TRIGSEL_EXT3) || \
                             ((x) == ALD_LPTIM_TRIGSEL_EXT4) || \
                             ((x) == ALD_LPTIM_TRIGSEL_EXT5) || \
                             ((x) == ALD_LPTIM_TRIGSEL_EXT6) || \
                             ((x) == ALD_LPTIM_TRIGSEL_EXT7))
#define IS_LPTIM_TRIGEN(x)  (((x) == ALD_LPTIM_TRIGEN_SW)      || \
                             ((x) == ALD_LPTIM_TRIGEN_RISING)  || \
                             ((x) == ALD_LPTIM_TRIGEN_FALLING) || \
                             ((x) == ALD_LPTIM_TRIGEN_BOTH))
#define IS_LPTIM_IT(x)      (((x) == ALD_LPTIM_IT_CMPMAT) || \
                             ((x) == ALD_LPTIM_IT_ARRMAT) || \
                             ((x) == ALD_LPTIM_IT_EXTTRIG))
#define IS_LPTIM_FLAG(x)    (((x) == ALD_LPTIM_FLAG_CMPMAT) || \
                             ((x) == ALD_LPTIM_FLAG_ARRMAT) || \
                             ((x) == ALD_LPTIM_FLAG_EXTTRIG))
/**
  * @}
  */

/** @addtogroup LPTIM_Public_Functions
  * @{
  */

/** @addtogroup LPTIM_Public_Functions_Group1
  * @{
  */
void ald_lptim_reset(ald_lptim_handle_t *hperh);
void ald_lptim_trigger_config(ald_lptim_handle_t *hperh, ald_lptim_trigger_init_t *config);
void ald_lptim_clock_source_config(ald_lptim_handle_t *hperh, ald_lptim_clock_source_init_t *config);
void ald_lptim_trigger_filter_config(ald_lptim_handle_t *hperh, ald_lptim_trgflt_t flt);
void ald_lptim_clock_filter_config(ald_lptim_handle_t *hperh, ald_lptim_ckflt_t flt);
/**
  * @}
  */

/** @addtogroup LPTIM_Public_Functions_Group2
  * @{
  */
ald_status_t ald_lptim_base_init(ald_lptim_handle_t *hperh);
void ald_lptim_base_start(ald_lptim_handle_t *hperh);
void ald_lptim_base_stop(ald_lptim_handle_t *hperh);
void ald_lptim_base_start_by_it(ald_lptim_handle_t *hperh);
void ald_lptim_base_stop_by_it(ald_lptim_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup LPTIM_Public_Functions_Group3
  * @{
  */
ald_status_t ald_lptim_toggle_init(ald_lptim_handle_t *hperh);
void ald_lptim_toggle_start(ald_lptim_handle_t *hperh);
void ald_lptim_toggle_stop(ald_lptim_handle_t *hperh);
void ald_lptim_toggle_start_by_it(ald_lptim_handle_t *hperh);
void ald_lptim_toggle_stop_by_it(ald_lptim_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup LPTIM_Public_Functions_Group4
  * @{
  */
ald_status_t ald_lptim_pulse_init(ald_lptim_handle_t *hperh);
void ald_lptim_pulse_start(ald_lptim_handle_t *hperh);
void ald_lptim_pulse_stop(ald_lptim_handle_t *hperh);
void ald_lptim_pulse_start_by_it(ald_lptim_handle_t *hperh);
void ald_lptim_pulse_stop_by_it(ald_lptim_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup LPTIM_Public_Functions_Group5
  * @{
  */
ald_status_t ald_lptim_pwm_init(ald_lptim_handle_t *hperh);
void ald_lptim_pwm_start(ald_lptim_handle_t *hperh);
void ald_lptim_pwm_stop(ald_lptim_handle_t *hperh);
void ald_lptim_pwm_start_by_it(ald_lptim_handle_t *hperh);
void ald_lptim_pwm_stop_by_it(ald_lptim_handle_t *hperh);
/**
  * @}
  */

/** @addtogroup LPTIM_Public_Functions_Group6
  * @{
  */
void ald_lptim_irq_handler(ald_lptim_handle_t *hperh);
void ald_lptim_interrupt_config(ald_lptim_handle_t *hperh, ald_lptim_it_t it, type_func_t state);
it_status_t ald_lptim_get_it_status(ald_lptim_handle_t *hperh, ald_lptim_it_t it);
flag_status_t ald_lptim_get_flag_status(ald_lptim_handle_t *hperh, ald_lptim_flag_t flag);
void ald_lptim_clear_flag_status(ald_lptim_handle_t *hperh, ald_lptim_flag_t flag);
/**
  * @}
  */

/** @addtogroup LPTIM_Public_Functions_Group7
  * @{
  */
ald_lptim_state_t ald_lptim_get_state(ald_lptim_handle_t *hperh);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_LPTIM_H__ */
