/**
  *********************************************************************************
  *
  * @file    ald_dma.h
  * @brief   DMA module Library.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_DMA_H__
#define __ALD_DMA_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */
/** @addtogroup DMA
  * @{
  */
/**
  * @defgroup DMA_Public_Macros DMA Public Macros
  * @{
  */
#define DMA_CH_COUNT    6U

#define ALD_DMA_CH_ENABLE(hperh, ch)     SET_BIT((hperh)->perh->CH[(ch)].CON, DMA_CON_CHEN_MSK)
#define ALD_DMA_CH_DISABLE(hperh, ch)    CLEAR_BIT((hperh)->perh->CH[(ch)].CON, DMA_CON_CHEN_MSK)
#define ALD_DMA_CON_SET(hperh, ch, ctrl) WRITE_REG((hperh)->perh->CH[(ch)].CON, (ctrl))
#define ALD_DMA_SRC_SET(hperh, ch, addr) WRITE_REG((hperh)->perh->CH[(ch)].SAR, (addr))
#define ALD_DMA_DST_SET(hperh, ch, addr) WRITE_REG((hperh)->perh->CH[(ch)].DAR, (addr))
#define ALD_DMA_CNT_SET(hperh, ch, cnt)  WRITE_REG((hperh)->perh->CH[(ch)].NDT, ((cnt) & 0xFFFFU))
#define ALD_DMA_NRT_GET(hperh, ch, cnt)  READ_BITS((hperh)->perh->CH[(ch)].NDT, DMA_NDT_NRDT_MSK, DMA_NDT_NRDT_POSS)

#define ALD_DMA_IT_ENABLE(hperh, x)      WRITE_REG((hperh)->perh->IER, (x))
#define ALD_DMA_IT_DISABLE(hperh, x)     WRITE_REG((hperh)->perh->IDR, (x))
#define ALD_DMA_IF_GET(hperh, x)         ((READ_BIT((hperh)->perh->RIF, (x)) == 0U) ? 0U : 1U)
#define ALD_DMA_IFM_GET(hperh, x)        ((READ_BIT((hperh)->perh->IFM, (x)) == 0U) ? 0U : 1U)
#define ALD_DMA_IF_CLEAR(hperh, x)       WRITE_REG((hperh)->perh->ICR, (x))
/**
  * @}
  */

/**
  * @defgroup DMA_Public_Types DMA Public Types
  * @{
  */

/**
  * @brief Input source to DMA channel
  */
typedef enum
{
    ALD_DMA_MSEL_NONE    = 0x0U,     /**< NONE */
    ALD_DMA_MSEL_GPIO    = 0x1U,     /**< GPIO */
    ALD_DMA_MSEL_DAC0    = 0x4U,     /**< DAC0 */
    ALD_DMA_MSEL_ADC0    = 0x6U,     /**< ADC0 */
    ALD_DMA_MSEL_CRC     = 0x7U,     /**< CRC */
    ALD_DMA_MSEL_UART0   = 0x8U,     /**< UART0 */
    ALD_DMA_MSEL_UART1   = 0x9U,     /**< UART1 */
    ALD_DMA_MSEL_UART2   = 0xAU,     /**< UART2 */
    ALD_DMA_MSEL_UART3   = 0xBU,     /**< UART3 */
    ALD_DMA_MSEL_UART4   = 0xCU,     /**< UART4 */
    ALD_DMA_MSEL_UART5   = 0xDU,     /**< UART5 */
    ALD_DMA_MSEL_SPI0    = 0xEU,     /**< SPI0 */
    ALD_DMA_MSEL_SPI1    = 0xFU,     /**< SPI1 */
    ALD_DMA_MSEL_I2C0    = 0x10U,    /**< I2C0 */
    ALD_DMA_MSEL_I2C1    = 0x11U,    /**< I2C1 */
    ALD_DMA_MSEL_TIMER0  = 0x12U,    /**< TIMER0 */
    ALD_DMA_MSEL_TIMER2  = 0x14U,    /**< TIMER2 */
    ALD_DMA_MSEL_TIMER3  = 0x15U,    /**< TIMER3 */
    ALD_DMA_MSEL_LPUART0 = 0x18U,    /**< LPUART0 */
    ALD_DMA_MSEL_TIMER4  = 0x1BU,    /**< TIMER4 */
    ALD_DMA_MSEL_TIMER5  = 0x1CU,    /**< TIMER5 */
    ALD_DMA_MSEL_TIMER6  = 0x1DU,    /**< TIMER6 */
    ALD_DMA_MSEL_TIMER7  = 0x1EU,    /**< TIMER7 */
    ALD_DMA_MSEL_PIS     = 0x20U,    /**< PIS */
} ald_dma_msel_t;

/**
  * @brief Input signal to DMA channel
  */
typedef enum
{
    ALD_DMA_MSIGSEL_NONE           = 0x0U,  /**< External interrupt 0 */
    ALD_DMA_MSIGSEL_EXTI_0         = 0x0U,  /**< External interrupt 0 */
    ALD_DMA_MSIGSEL_EXTI_1         = 0x1U,  /**< External interrupt 1 */
    ALD_DMA_MSIGSEL_EXTI_2         = 0x2U,  /**< External interrupt 2 */
    ALD_DMA_MSIGSEL_EXTI_3         = 0x3U,  /**< External interrupt 3 */
    ALD_DMA_MSIGSEL_EXTI_4         = 0x4U,  /**< External interrupt 4 */
    ALD_DMA_MSIGSEL_EXTI_5         = 0x5U,  /**< External interrupt 5 */
    ALD_DMA_MSIGSEL_EXTI_6         = 0x6U,  /**< External interrupt 6 */
    ALD_DMA_MSIGSEL_EXTI_7         = 0x7U,  /**< External interrupt 7 */
    ALD_DMA_MSIGSEL_EXTI_8         = 0x8U,  /**< External interrupt 8 */
    ALD_DMA_MSIGSEL_EXTI_9         = 0x9U,  /**< External interrupt 9 */
    ALD_DMA_MSIGSEL_EXTI_10        = 0xAU,  /**< External interrupt 10 */
    ALD_DMA_MSIGSEL_EXTI_11        = 0xBU,  /**< External interrupt 11 */
    ALD_DMA_MSIGSEL_EXTI_12        = 0xCU,  /**< External interrupt 12 */
    ALD_DMA_MSIGSEL_EXTI_13        = 0xDU,  /**< External interrupt 13 */
    ALD_DMA_MSIGSEL_EXTI_14        = 0xEU,  /**< External interrupt 14 */
    ALD_DMA_MSIGSEL_EXTI_15        = 0xFU,  /**< External interrupt 15 */
    ALD_DMA_MSIGSEL_CRC_WRITE      = 0x0U,  /**< CRC write mode */
    ALD_DMA_MSIGSEL_DAC0_CH0       = 0x0U,  /**< DAC0 channel 0 complete */
    ALD_DMA_MSIGSEL_DAC0_CH1       = 0x1U,  /**< DAC0 channel 1 complete */
    ALD_DMA_MSIGSEL_ADC            = 0x0U,  /**< ADC mode */
    ALD_DMA_MSIGSEL_UART_TXEMPTY   = 0x0U,  /**< UART transmit */
    ALD_DMA_MSIGSEL_UART_RNR       = 0x1U,  /**< UART receive */
    ALD_DMA_MSIGSEL_SPI_RNR        = 0x1U,  /**< SPI receive */
    ALD_DMA_MSIGSEL_SPI_TXEMPTY    = 0x0U,  /**< SPI transmit */
    ALD_DMA_MSIGSEL_I2C_RNR        = 0x1U,  /**< I2C receive */
    ALD_DMA_MSIGSEL_I2C_TXEMPTY    = 0x0U,  /**< I2C transmit */
    ALD_DMA_MSIGSEL_LPUART_RNR     = 0x1U,  /**< LPUART receive */
    ALD_DMA_MSIGSEL_LPUART_TXEMPTY = 0x0U,  /**< LPUART transmit */
    ALD_DMA_MSIGSEL_TIMER_CH1      = 0x0U,  /**< TIM channal 1 */
    ALD_DMA_MSIGSEL_TIMER_CH2      = 0x1U,  /**< TIM channal 2 */
    ALD_DMA_MSIGSEL_TIMER_CH3      = 0x2U,  /**< TIM channal 3 */
    ALD_DMA_MSIGSEL_TIMER_CH4      = 0x3U,  /**< TIM channal 4 */
    ALD_DMA_MSIGSEL_TIMER_TRI      = 0x4U,  /**< TIM trigger */
    ALD_DMA_MSIGSEL_TIMER_COMP     = 0x5U,  /**< TIM compare */
    ALD_DMA_MSIGSEL_TIMER_UPDATE   = 0x6U,  /**< TIM update */
    ALD_DMA_MSIGSEL_PIS_CH0        = 0x0U,  /**< PIS channal 0 */
    ALD_DMA_MSIGSEL_PIS_CH1        = 0x1U,  /**< PIS channal 1 */
    ALD_DMA_MSIGSEL_PIS_CH2        = 0x2U,  /**< PIS channal 2 */
    ALD_DMA_MSIGSEL_PIS_CH3        = 0x3U,  /**< PIS channal 3 */
    ALD_DMA_MSIGSEL_PIS_CH4        = 0x4U,  /**< PIS channal 4 */
    ALD_DMA_MSIGSEL_PIS_CH5        = 0x5U,  /**< PIS channal 5 */
    ALD_DMA_MSIGSEL_PIS_CH6        = 0x6U,  /**< PIS channal 6 */
    ALD_DMA_MSIGSEL_PIS_CH7        = 0x7U,  /**< PIS channal 7 */
    ALD_DMA_MSIGSEL_PIS_CH8        = 0x8U,  /**< PIS channal 8 */
    ALD_DMA_MSIGSEL_PIS_CH9        = 0x9U,  /**< PIS channal 9 */
    ALD_DMA_MSIGSEL_PIS_CH10       = 0xAU,  /**< PIS channal 10 */
    ALD_DMA_MSIGSEL_PIS_CH11       = 0xBU,  /**< PIS channal 11 */
    ALD_DMA_MSIGSEL_PIS_CH12       = 0xCU,  /**< PIS channal 12 */
    ALD_DMA_MSIGSEL_PIS_CH13       = 0xDU,  /**< PIS channal 13 */
    ALD_DMA_MSIGSEL_PIS_CH14       = 0xEU,  /**< PIS channal 14 */
    ALD_DMA_MSIGSEL_PIS_CH15       = 0xFU,  /**< PIS channal 15 */
} ald_dma_msigsel_t;

/**
  * @brief Direction
  */
typedef enum
{
    ALD_DMA_DIR_TO_SRAM = 0x0U,  /**< From peripheral to sram */
    ALD_DMA_DIR_TO_PERH = 0x1U,  /**< From sram to peripheral */
} ald_dma_dir_t;

/**
  * @brief Data size
  */
typedef enum
{
    ALD_DMA_DATA_SIZE_BYTE     = 0x0U,  /**< Byte */
    ALD_DMA_DATA_SIZE_HALFWORD = 0x1U,  /**< Halfword */
    ALD_DMA_DATA_SIZE_WORD     = 0x2U,  /**< Word */
} ald_dma_data_size_t;

/**
  * @brief Control how many DMA transfers can occur
  *        before the controller re-arbitrates
  */
typedef enum
{
    ALD_DMA_R_POWER_1    = 0x0U,  /**< Arbitrates after each DMA transfer */
    ALD_DMA_R_POWER_2    = 0x1U,  /**< Arbitrates after 2 DMA transfer */
    ALD_DMA_R_POWER_4    = 0x2U,  /**< Arbitrates after 4 DMA transfer */
    ALD_DMA_R_POWER_8    = 0x3U,  /**< Arbitrates after 8 DMA transfer */
    ALD_DMA_R_POWER_16   = 0x4U,  /**< Arbitrates after 16 DMA transfer */
    ALD_DMA_R_POWER_32   = 0x5U,  /**< Arbitrates after 32 DMA transfer */
    ALD_DMA_R_POWER_64   = 0x6U,  /**< Arbitrates after 64 DMA transfer */
    ALD_DMA_R_POWER_128  = 0x7U,  /**< Arbitrates after 128 DMA transfer */
    ALD_DMA_R_POWER_256  = 0x8U,  /**< Arbitrates after 256 DMA transfer */
    ALD_DMA_R_POWER_512  = 0x9U,  /**< Arbitrates after 512 DMA transfer */
    ALD_DMA_R_POWER_1024 = 0xAU,  /**< Arbitrates after 1024 DMA transfer */
} ald_dma_arbiter_config_t;

/**
  * @brief Priority of DMA channel
  */
typedef enum
{
    ALD_DMA_LOW_PRIORITY     = 0x0U,
    ALD_DMA_MEDIUM_PRIORITY  = 0x1U,
    ALD_DMA_HIGH_PRIORITY    = 0x2U,
    ALD_DMA_HIGHEST_PRIORITY = 0x3U,
} ald_dma_priority_t;

/**
  * @brief data increment
  */
typedef enum
{
    ALD_DMA_DATA_INC_DISABLE  = 0x0U,
    ALD_DMA_DATA_INC_ENABLE   = 0x1U,
} ald_dma_data_inc_t;

/**
  * @brief Interrupt flag trigger mode
  */
typedef enum
{
    ALD_DMA_IT_FLAG_TC  = 0x0U,  /**< DMA transfer complete interrupt */
    ALD_DMA_IT_FLAG_HT  = 0x1U,  /**< DMA transfer half complete interrupt */
} ald_dma_it_flag_t;

/**
  * @brief Callback function pointer and param
  */
typedef struct
{
    void (*cplt_tc_cbk)(void *arg);   /**< DMA transfers complete callback */
    void (*cplt_ht_cbk)(void *arg);   /**< DMA transfers half complete callback */
    void *cplt_tc_arg;                /**< The parameter of cplt_tc_cbk() */
    void *cplt_ht_arg;                /**< The parameter of cplt_ht_cbk() */
} ald_dma_call_back_t;

/**
  * @brief DMA channal configure structure
  */
typedef struct
{
    void *src;                              /**< Source data begin pointer */
    void *dst;                              /**< Destination data begin pointer */
    uint16_t size;                          /**< The total number of DMA transfers that DMA cycle contains */
    ald_dma_data_size_t src_data_width;     /**< Data width, @ref dma_data_width_t */
    ald_dma_data_inc_t src_inc;             /**< Source increment type */
    ald_dma_data_size_t dst_data_width;     /**< Data width, @ref dma_data_width_t */
    ald_dma_data_inc_t dst_inc;             /**< Destination increment type */
    ald_dma_arbiter_config_t R_power;       /**< Control how many DMA transfers can occur before re-arbitrates */
    ald_dma_priority_t priority;            /**< Channel priority */
    ald_dma_dir_t dir;                      /**< Direction */
    type_func_t mem_to_mem;                 /**< Use primary descriptor or alternate descriptor */
    type_func_t circle_mode;                /**< Enable/Disable the useburst setting for this channel */
    type_func_t irq_tc;                     /**< Enable/disable interrupt complete */
    type_func_t irq_ht;                     /**< Enable/disable interrupt half */
    ald_dma_msel_t msel;                    /**< Input source to DMA channel @ref dma_msel_t */
    ald_dma_msigsel_t msigsel;              /**< Input signal to DMA channel @ref dma_msigsel_t */
    uint8_t channel;                        /**< Channel index */
} ald_dma_config_t;

/**
  * @brief DMA handle structure definition
  */
typedef struct
{
    DMA_TypeDef *perh;               /**< DMA registers base address */
    ald_dma_config_t config;         /**< Channel configure structure */
    void (*cplt_tc_cbk)(void *arg);  /**< DMA transfers complete callback */
    void (*cplt_ht_cbk)(void *arg);  /**< DMA transfers half complete callback */
    void *cplt_tc_arg;               /**< The parameter of cplt_tc_cbk() */
    void *cplt_ht_arg;               /**< The parameter of cplt_ht_cbk() */
} ald_dma_handle_t;
/**
  * @}
  */

/**
  * @defgroup DMA_Private_Macros DMA Private Macros
  * @{
  */
#define IS_DMA_MSIGSEL_TYPE(x)  ((x) <= 0xFU)
#define IS_DMA_IT_TYPE(x)             (((x) == ALD_DMA_IT_FLAG_TC)          || \
                                       ((x) == ALD_DMA_IT_FLAG_HT))
#define IS_DMA_DIR_TYPE(x)            (((x) == ALD_DMA_DIR_TO_SRAM)         || \
                                       ((x) == ALD_DMA_DIR_TO_PERH))
#define IS_DMA_PRIO_TYPE(x)           (((x) == ALD_DMA_HIGHEST_PRIORITY)    || \
                                       ((x) == ALD_DMA_HIGH_PRIORITY)       || \
                                       ((x) == ALD_DMA_MEDIUM_PRIORITY)     || \
                                       ((x) == ALD_DMA_LOW_PRIORITY))
#define IS_DMA_DATASIZE_TYPE(x)       (((x) == ALD_DMA_DATA_SIZE_BYTE)      || \
                                       ((x) == ALD_DMA_DATA_SIZE_HALFWORD)  || \
                                       ((x) == ALD_DMA_DATA_SIZE_WORD))
#define IS_DMA_ARBITERCONFIG_TYPE(x)  (((x) == ALD_DMA_R_POWER_1)   || \
                                       ((x) == ALD_DMA_R_POWER_2)   || \
                                       ((x) == ALD_DMA_R_POWER_4)   || \
                                       ((x) == ALD_DMA_R_POWER_8)   || \
                                       ((x) == ALD_DMA_R_POWER_16)  || \
                                       ((x) == ALD_DMA_R_POWER_32)  || \
                                       ((x) == ALD_DMA_R_POWER_64)  || \
                                       ((x) == ALD_DMA_R_POWER_128) || \
                                       ((x) == ALD_DMA_R_POWER_256) || \
                                       ((x) == ALD_DMA_R_POWER_512) || \
                                       ((x) == ALD_DMA_R_POWER_1024))
#define IS_DMA_DATA_INC_STAT(x)        (((x) == ALD_DMA_DATA_INC_ENABLE)          || \
                                        ((x) == ALD_DMA_DATA_INC_DISABLE))

#define IS_DMA(x)              (((x) == DMA0) || ((x) == DMA1))
#define IS_DMA_CHANNEL(x)      ((x) < 6U)
#define IS_DMA_DATA_SIZE(x)    ((x) < 65535U)
/**
  * @}
  */

/**
  * @addtogroup DMA_Public_Functions
  * @{
  */
/** @addtogroup DMA_Public_Functions_Group1
  * @{
  */
/* Initialization functions */
extern void ald_dma_reset(DMA_TypeDef *DMAx);
extern void ald_dma_init(DMA_TypeDef *DMAx);
extern void ald_dma_config_struct(ald_dma_handle_t *hperh);
/**
  * @}
  */
/** @addtogroup DMA_Public_Functions_Group2
  * @{
  */
/* Configure DMA operation functions */
extern void ald_dma_config(ald_dma_handle_t *hperh);
extern void ald_dma_pause(ald_dma_handle_t *hperh);
extern void ald_dma_restart(ald_dma_handle_t *hperh);
extern void ald_dma_resume(ald_dma_handle_t *hperh);
/**
  * @}
  */
/** @addtogroup DMA_Public_Functions_Group3
  * @{
  */
/* DMA control functions */
extern void ald_dma0_irq_handler(void);
extern void ald_dma1_irq_handler(void);
extern void ald_dma_interrupt_config(ald_dma_handle_t *hperh, ald_dma_it_flag_t it, type_func_t state);
extern it_status_t ald_dma_get_it_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it);
extern flag_status_t ald_dma_get_flag_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it);
extern flag_status_t ald_dma_get_mask_flag_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it);
extern void ald_dma_clear_flag_status(ald_dma_handle_t *hperh, uint8_t channel, ald_dma_it_flag_t it);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif
#endif  /*__ALD_DMA_H__ */
