/**
  ******************************************************************************
 * @file    ald_dac.h
 * @brief   Header file of DAC Module library.
 *
 * @version V1.0
 * @date    12 Mar 2024
 * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
 */

#ifndef __ALD_DAC_H__
#define __ALD_DAC_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"

/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup DAC
  * @{
  */

/** @defgroup DAC_Pubulic_Types DAC Pubulic Types
  * @{
  */
/**
  * @brief DAC noise filter width
  */
typedef enum
{
    ALD_DAC_NWBW_1   = 0U,  /**< DAC nwbw 1 */
    ALD_DAC_NWBW_2   = 1U,  /**< DAC nwbw 2 */
    ALD_DAC_NWBW_3   = 2U,  /**< DAC nwbw 3 */
    ALD_DAC_NWBW_4   = 3U,  /**< DAC nwbw 4 */
    ALD_DAC_NWBW_5   = 4U,  /**< DAC nwbw 5 */
    ALD_DAC_NWBW_6   = 5U,  /**< DAC nwbw 6 */
    ALD_DAC_NWBW_7   = 6U,  /**< DAC nwbw 7 */
    ALD_DAC_NWBW_8   = 7U,  /**< DAC nwbw 8 */
    ALD_DAC_NWBW_9   = 8U,  /**< DAC nwbw 9 */
    ALD_DAC_NWBW_10  = 9U,  /**< DAC nwbw 10 */
    ALD_DAC_NWBW_11  = 10U, /**< DAC nwbw 11 */
    ALD_DAC_NWBW_12  = 11U, /**< DAC nwbw 12 */
} ald_dac_nwbw_t;

/**
  * @brief DAC noise filter mode
  */
typedef enum
{
    ALD_DAC_NWM_None      = 0U, /**< DAC no filter */
    ALD_DAC_NWM_LFSR      = 1U, /**< DAC LFSR mode */
    ALD_DAC_NWM_TRINOISE  = 2U, /**< DAC TRINOISE mode */
} ald_dac_nwm_mode_t;

/**
  * @brief DAC pis input channel
  */
typedef enum
{
    ALD_DAC_PIS_CH_0  = 0U,     /**< PIS channel 0 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_1  = 1U,     /**< PIS channel 1 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_2  = 2U,     /**< PIS channel 2 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_3  = 3U,     /**< PIS channel 3 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_4  = 4U,     /**< PIS channel 4 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_5  = 5U,     /**< PIS channel 5 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_6  = 6U,     /**< PIS channel 6 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_7  = 7U,     /**< PIS channel 7 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_8  = 8U,     /**< PIS channel 8 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_9  = 9U,     /**< PIS channel 9 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_10 = 10U,    /**< PIS channel 10 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_11 = 11U,    /**< PIS channel 11 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_12 = 12U,    /**< PIS channel 12 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_13 = 13U,    /**< PIS channel 13 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_14 = 14U,    /**< PIS channel 14 triggers DAC channel conversion */
    ALD_DAC_PIS_CH_15 = 15U,    /**< PIS channel 15 triggers DAC channel conversion */
} ald_dac_pis_ch_t;

/**
  * @brief DAC trigger mode
  */
typedef enum
{
    ALD_DAC_WR_DBUF_TRG  = 0U,  /**< Write DAC value trigger output */
    ALD_DAC_WR_BIT_TRG   = 1U,  /**< Set trigger bit to output */
} ald_dac_soft_trig_t;

/**
  * @brief DAC output mode
  */
typedef enum
{
    ALD_DAC_OUTPUT_NORMAL  = 0U,    /**< DAC NORMAL MODE OUTPUT */
    ALD_DAC_OUTPUT_ENHANCE = 1U,    /**< DAC ENHANCED MODE OUTPUT */
} ald_dac_out_mode_t;

/**
  * @brief DAC data_alignment
  */
typedef enum
{
    ALD_DAC_ALIGN_R12  = 0U,    /**< DAC data right-aligned 12bit */
    ALD_DAC_ALIGN_L12  = 1U,    /**< DAC data left-aligned 12bit */
    ALD_DAC_ALIGN_R8   = 2U,    /**< DAC data right-aligned 8bit */
} ald_dac_data_align_t;

/**
  * @brief DAC init structure definition
  */
typedef struct
{
    ald_dac_soft_trig_t  soft_trig;   /**< Trig mode */
    type_func_t  pis_trig;            /**< External PIS trigger */
    ald_dac_pis_ch_t  pis_ch;         /**< External pis channel */
    ald_dac_out_mode_t output_mode;   /**< Select output mode */
    type_func_t  output_op;           /**< Refresh interval select */
    type_func_t  output_gpio;         /**< DAC output to GPIO */
    ald_dac_nwm_mode_t  noise_mode;   /**< Noise wave mode */
    ald_dac_nwbw_t  noise_width;      /**< Noise wave width */
    ald_dac_data_align_t align_mode;  /**< DAC output data align mode */
} ald_dac_init_t;

/**
  * @brief  DAC handle Structure definition
  */
typedef struct dac_handle_s
{
    DAC_TypeDef *perh;  /**< Register base address */
    ald_dac_init_t init;    /**< DAC initialize parameters */
    lock_state_t lock;  /**< DAC locking object */
} ald_dac_handle_t;
/**
  * @}
  */

/** @defgroup DAC_Public_Macros DAC Public Macros
  * @{
  */
#define DAC0_ENABLE()   (SET_BIT(DAC0->CON, DAC_CON_DACEN_MSK))
#define DAC0_DISABLE()  (CLEAR_BIT(DAC0->CON, DAC_CON_DACEN_MSK))
#define DAC0_RESET()    (SET_BIT(DAC0->SWTRG, DAC_SWTRG_RST_MSK))

/**
  * @}
  */

/** @defgroup DAC_Private_Macros DAC Private Macros
  * @{
  */
#define IS_DAC_TYPE(x)          ((x) == DAC0)
#define IS_DAC_TRIGGER_MODE(x)      (((x) == ALD_DAC_WR_DBUF_TRG) || \
                                     ((x) == ALD_DAC_WR_BIT_TRG))
#define IS_DAC_OUTPUT_MODE(x)       (((x) == ALD_DAC_OUTPUT_NORMAL) || \
                                     ((x) == ALD_DAC_OUTPUT_ENHANCE))
#define IS_DAC_NOISE_FILTER(x)  (((x) == ALD_DAC_NWM_None)  || \
                                 ((x) == ALD_DAC_NWM_LFSR) || \
                                 ((x) == ALD_DAC_NWM_TRINOISE))
#define IS_DAC_FILTER_WIDTH(x)      (((x) == ALD_DAC_NWBW_1) || \
                                     ((x) == ALD_DAC_NWBW_2)  || \
                                     ((x) == ALD_DAC_NWBW_3)  || \
                                     ((x) == ALD_DAC_NWBW_4)  || \
                                     ((x) == ALD_DAC_NWBW_5)  || \
                                     ((x) == ALD_DAC_NWBW_6)  || \
                                     ((x) == ALD_DAC_NWBW_7)  || \
                                     ((x) == ALD_DAC_NWBW_8)  || \
                                     ((x) == ALD_DAC_NWBW_9)  || \
                                     ((x) == ALD_DAC_NWBW_10) || \
                                     ((x) == ALD_DAC_NWBW_11) || \
                                     ((x) == ALD_DAC_NWBW_12))
#define IS_DAC_EXTERNAL_TRIGGER_PIS(x)  (((x) == ALD_DAC_PIS_CH_0)  || \
        ((x) == ALD_DAC_PIS_CH_1)  || \
        ((x) == ALD_DAC_PIS_CH_2)  || \
        ((x) == ALD_DAC_PIS_CH_3)  || \
        ((x) == ALD_DAC_PIS_CH_4)  || \
        ((x) == ALD_DAC_PIS_CH_5)  || \
        ((x) == ALD_DAC_PIS_CH_6)  || \
        ((x) == ALD_DAC_PIS_CH_7)  || \
        ((x) == ALD_DAC_PIS_CH_8)  || \
        ((x) == ALD_DAC_PIS_CH_9)  || \
        ((x) == ALD_DAC_PIS_CH_10) || \
        ((x) == ALD_DAC_PIS_CH_11))
#define IS_DAC_DATA_ALIGN(x)    (((x) == ALD_DAC_ALIGN_R12)  || \
                                 ((x) == ALD_DAC_ALIGN_L12) || \
                                 ((x) == ALD_DAC_ALIGN_R8))
#define IS_DAC_DATA(DATA) ((DATA) <= 0xFFF0U)

/**
  * @}
  */

/** @addtogroup DAC_Public_Functions
  * @{
  */
ald_status_t ald_dac_reset(ald_dac_handle_t *hperh);
ald_status_t ald_dac_init(ald_dac_handle_t *hperh);
ald_status_t ald_dac_stop(ald_dac_handle_t *hperh);
ald_status_t ald_dac_set_value(ald_dac_handle_t *hperh, uint32_t value);
uint32_t ald_dac_get_value(ald_dac_handle_t *hperh);
ald_status_t ald_dac_set_bit_trig(ald_dac_handle_t *hperh);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
extern "C"
}
#endif

#endif /* __ALD_DAC_H */
