/**
  *********************************************************************************
  *
  * @file    ald_cmu.h
  * @brief   Header file of CMU module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_CMU_H__
#define __ALD_CMU_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"
#include "ald_syscfg.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup CMU
  * @{
  */

/** @defgroup CMU_Public_Macros CMU Public Macros
  * @{
  */
#define CMU_HRC_SEL_BY_SW()             \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->CFGR, CMU_CFGR_HRCFCS_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_HRC_SEL_BY_CFGW()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->CFGR, CMU_CFGR_HRCFCS_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_HRC_DIV_1MHZ_ENABLE()           \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->CLKENR, CMU_CLKENR_HRC1MEN_MSK);   \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_HRC_DIV_1MHZ_DISABLE()          \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->CLKENR, CMU_CLKENR_HRC1MEN_MSK); \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_HOSC_DIV_1MHZ_ENABLE()          \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->CLKENR, CMU_CLKENR_HOSC1MEN_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_HOSC_DIV_1MHZ_DISABLE()         \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->CLKENR, CMU_CLKENR_HOSC1MEN_MSK);\
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LOSC_ENABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->CLKENR, CMU_CLKENR_LOSCEN_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LOSC_DISABLE()              \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->CLKENR, CMU_CLKENR_LOSCEN_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LRC_ENABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->CLKENR, CMU_CLKENR_LRCEN_MSK); \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LRC_DISABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->CLKENR, CMU_CLKENR_LRCEN_MSK);   \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_ULRC_ENABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->CLKENR, CMU_CLKENR_ULRCEN_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_ULRC_DISABLE()              \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->CLKENR, CMU_CLKENR_ULRCEN_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)

/* Low power mode control */
#define CMU_LP_LRC_ENABLE()             \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->LPENR, CMU_LPENR_LRCEN_MSK);   \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LP_LRC_DISABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->LPENR, CMU_LPENR_LRCEN_MSK); \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LP_LOSC_ENABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->LPENR, CMU_LPENR_LOSCEN_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LP_LOSC_DISABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->LPENR, CMU_LPENR_LOSCEN_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LP_HRC_ENABLE()             \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->LPENR, CMU_LPENR_HRCEN_MSK);   \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LP_HRC_DISABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->LPENR, CMU_LPENR_HRCEN_MSK); \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LP_HOSC_ENABLE()                \
    do {                            \
        SYSCFG_UNLOCK();                \
        SET_BIT(CMU->LPENR, CMU_LPENR_HOSCEN_MSK);  \
        SYSCFG_LOCK();                  \
    } while (0)
#define CMU_LP_HOSC_DISABLE()               \
    do {                            \
        SYSCFG_UNLOCK();                \
        CLEAR_BIT(CMU->LPENR, CMU_LPENR_HOSCEN_MSK);    \
        SYSCFG_LOCK();                  \
    } while (0)
/**
  * @}
  */


/** @defgroup CMU_Public_Types CMU Public Types
  * @{
  */
/**
  * @brief CMU state structure definition
  */
typedef enum
{
    ALD_CMU_CLOCK_HRC  = 0x1U,  /**< HRC */
    ALD_CMU_CLOCK_LRC  = 0x2U,  /**< LRC */
    ALD_CMU_CLOCK_LOSC = 0x3U,  /**< LOSC */
    ALD_CMU_CLOCK_PLL  = 0x4U,  /**< PLL  */
    ALD_CMU_CLOCK_HOSC = 0x5U,  /**< HOSC */
} ald_cmu_clock_t;

/**
  * @brief PLL output clock
  */
typedef enum
{
    ALD_CMU_PLL_OUTPUT_48M = 0x0U,  /**< x12 (48MHz) */
    ALD_CMU_PLL_OUTPUT_56M = 0x1U,  /**< x14 (56MHz) */
    ALD_CMU_PLL_OUTPUT_64M = 0x2U,  /**< x16 (64MHz) */
    ALD_CMU_PLL_OUTPUT_72M = 0x3U,  /**< x18 (72MHz) */
} ald_cmu_pll_output_t;

/**
  * @brief PLL referance clock
  */
typedef enum
{
    ALD_CMU_PLL_INPUT_HRC    = 0x0U,    /**< HRC  / 6 */
    ALD_CMU_PLL_INPUT_HOSC   = 0x2U,    /**< HOSC / 1 */
    ALD_CMU_PLL_INPUT_HOSC_2 = 0x3U,    /**< HOSC / 2 */
    ALD_CMU_PLL_INPUT_HOSC_3 = 0x4U,    /**< HOSC / 3 */
    ALD_CMU_PLL_INPUT_HOSC_4 = 0x5U,    /**< HOSC / 4 */
    ALD_CMU_PLL_INPUT_HOSC_5 = 0x6U,    /**< HOSC / 5 */
    ALD_CMU_PLL_INPUT_HOSC_6 = 0x7U,    /**< HOSC / 6 */
} ald_cmu_pll_input_t;

/**
  * @brief HOSC range
  */
typedef enum
{
    ALD_CMU_HOSC_2M  = 0x0U,    /**< 0~2MHz */
    ALD_CMU_HOSC_4M  = 0x1U,    /**< 2~4MHz */
    ALD_CMU_HOSC_8M  = 0x2U,    /**< 4~8MHz */
    ALD_CMU_HOSC_16M = 0x3U,    /**< 8~16MHz */
    ALD_CMU_HOSC_24M = 0x4U,    /**< 16~24MHz */
} ald_cmu_hosc_range_t;

/**
  * @brief Safe clock source type
  */
typedef enum
{
    ALD_CMU_SAFE_CLK_HOSC = 0x0U,   /**< HOSC */
    ALD_CMU_SAFE_CLK_LOSC = 0x1U,   /**< LOSC */
    ALD_CMU_SAFE_CLK_PLL  = 0x2U,   /**< PLL */
} ald_cmu_clock_safe_type_t;

/**
  * @brief Frequency division select bit
  */
typedef enum
{
    ALD_CMU_DIV_1    = 0x0U,    /**< Division by 1 */
    ALD_CMU_DIV_2    = 0x1U,    /**< Division by 2 */
    ALD_CMU_DIV_4    = 0x2U,    /**< Division by 4 */
    ALD_CMU_DIV_8    = 0x3U,    /**< Division by 8 */
    ALD_CMU_DIV_16   = 0x4U,    /**< Division by 16 */
    ALD_CMU_DIV_32   = 0x5U,    /**< Division by 32 */
    ALD_CMU_DIV_64   = 0x6U,    /**< Division by 64 */
    ALD_CMU_DIV_128  = 0x7U,    /**< Division by 128 */
    ALD_CMU_DIV_256  = 0x8U,    /**< Division by 256 */
    ALD_CMU_DIV_512  = 0x9U,    /**< Division by 512 */
    ALD_CMU_DIV_1024 = 0xAU,    /**< Division by 1024 */
    ALD_CMU_DIV_2048 = 0xBU,    /**< Division by 2048 */
    ALD_CMU_DIV_4096 = 0xCU,    /**< Division by 4096 */
} ald_cmu_div_t;

/**
  * @brief Bus type
  */
typedef enum
{
    ALD_CMU_SYS    = 0x2U,  /**< SYS bus */
    ALD_CMU_PCLK_1 = 0x3U,  /**< APB1 bus */
    ALD_CMU_PCLK_2 = 0x4U,  /**< APB2 bus */
} ald_cmu_bus_t;

/**
  * @brief Output high clock select
  */
typedef enum
{
    ALD_CMU_OUTPUT_HIGH_SEL_HOSC   = 0x0U,  /**< Select HOSC */
    ALD_CMU_OUTPUT_HIGH_SEL_LOSC   = 0x1U,  /**< Select LOSC */
    ALD_CMU_OUTPUT_HIGH_SEL_HRC    = 0x2U,  /**< Select HRC */
    ALD_CMU_OUTPUT_HIGH_SEL_LRC    = 0x3U,  /**< Select LRC */
    ALD_CMU_OUTPUT_HIGH_SEL_PLL    = 0x4U,  /**< Select PLL */
    ALD_CMU_OUTPUT_HIGH_SEL_SYSCLK = 0x5U,  /**< Select SYSCLK */
    ALD_CMU_OUTPUT_HIGH_SEL_PCLK1  = 0x6U,  /**< Select PCLK1 */
    ALD_CMU_OUTPUT_HIGH_SEL_PCLK2  = 0x7U,  /**< Select PCLK2 */
    ALD_CMU_OUTPUT_HIGH_SEL_LOSM   = 0x9U,  /**< Select LOSM */
    ALD_CMU_OUTPUT_HIGH_SEL_HOSM   = 0xCU,  /**< Select HOSM */
} ald_cmu_output_high_sel_t;

/**
  * @brief Output frequency division
  */
typedef enum
{
    ALD_CMU_OUTPUT_DIV_1   = 0x0U,  /**< Division by 1 */
    ALD_CMU_OUTPUT_DIV_2   = 0x1U,  /**< Division by 2 */
    ALD_CMU_OUTPUT_DIV_4   = 0x2U,  /**< Division by 4 */
    ALD_CMU_OUTPUT_DIV_8   = 0x3U,  /**< Division by 8 */
    ALD_CMU_OUTPUT_DIV_16  = 0x4U,  /**< Division by 16 */
    ALD_CMU_OUTPUT_DIV_32  = 0x5U,  /**< Division by 32 */
    ALD_CMU_OUTPUT_DIV_64  = 0x6U,  /**< Division by 64 */
    ALD_CMU_OUTPUT_DIV_128 = 0x7U,  /**< Division by 128 */
} ald_cmu_output_high_div_t;

/**
  * @brief Output low clock select
  */
typedef enum
{
    ALD_CMU_OUTPUT_LOW_SEL_LOSC = 0x0U, /**< Select LOSC */
    ALD_CMU_OUTPUT_LOW_SEL_LRC  = 0x1U, /**< Select LRC */
    ALD_CMU_OUTPUT_LOW_SEL_LOSM = 0x2U, /**< Select LOSM */
    ALD_CMU_OUTPUT_LOW_SEL_BUZZ = 0x3U, /**< Select BUZZ */
    ALD_CMU_OUTPUT_LOW_SEL_ULRC = 0x4U, /**< Select ULRC */
} ald_cmu_output_low_sel_t;

/**
  * @brief BUZZ frequency division
  */
typedef enum
{
    ALD_CMU_BUZZ_DIV_2   = 0x0U,    /**< Division by 2 */
    ALD_CMU_BUZZ_DIV_4   = 0x1U,    /**< Division by 4 */
    ALD_CMU_BUZZ_DIV_8   = 0x2U,    /**< Division by 8 */
    ALD_CMU_BUZZ_DIV_16  = 0x3U,    /**< Division by 16 */
    ALD_CMU_BUZZ_DIV_32  = 0x4U,    /**< Division by 32 */
    ALD_CMU_BUZZ_DIV_64  = 0x5U,    /**< Division by 64 */
    ALD_CMU_BUZZ_DIV_128 = 0x6U,    /**< Division by 128 */
    ALD_CMU_BUZZ_DIV_256 = 0x7U,    /**< Division by 256 */
} ald_cmu_buzz_div_t;

/**
  * @brief Low power peripheral clock select
  */
typedef enum
{
    ALD_CMU_LP_PERH_CLOCK_SEL_PCLK2   = 0x0U,   /**< Select PCLK2 */
    ALD_CMU_LP_PERH_CLOCK_SEL_PLL     = 0x1U,   /**< Select PLL */
    ALD_CMU_LP_PERH_CLOCK_SEL_HRC     = 0x3U,   /**< Select HRC */
    ALD_CMU_LP_PERH_CLOCK_SEL_HOSC    = 0x4U,   /**< Select HOSC */
    ALD_CMU_LP_PERH_CLOCK_SEL_LRC     = 0x5U,   /**< Select LRC */
    ALD_CMU_LP_PERH_CLOCK_SEL_LOSC    = 0x6U,   /**< Select LOSC */
    ALD_CMU_LP_PERH_CLOCK_SEL_ULRC    = 0x7U,   /**< Select ULRC */
    ALD_CMU_LP_PERH_CLOCK_SEL_HRC_1M  = 0x8U,   /**< Select HRC down to 1MHz */
    ALD_CMU_LP_PERH_CLOCK_SEL_HOSC_1M = 0x9U,   /**< Select HOSC down to 1MHz  */
    ALD_CMU_LP_PERH_CLOCK_SEL_LOSM    = 0xAU,   /**< Select LOSM */
    ALD_CMU_LP_PERH_CLOCK_SEL_HOSM    = 0xBU,   /**< Select HOSM */
} ald_cmu_lp_perh_clock_sel_t;

/**
  * @brief RTC Source select
  */
typedef enum
{
    ALD_CMU_RTC_SEL_LOSM        = 0x0U,  /**< LOSM */
    ALD_CMU_RTC_SEL_LRC         = 0x1U,  /**< LRC */
    ALD_CMU_RTC_SEL_HRC_DIV_1M  = 0x2U,  /**< HRC divide to 1MHz */
    ALD_CMU_RTC_SEL_HOSC_DIV_1M = 0x3U,  /**< HOSC divide to 1MHz */
} ald_cmu_rtc_clock_sel_t;

/**
  * @brief Peripheral clock enable/disable
  * @verbatim
      In this module, for the convenience of code maintenance,
      TIMERx is used to indicate the sequence of the timer peripheral.
      Different product series TIMERx represent different meanings:
      1. For ES32F361x_362x series:
      TIMER0 ----> AD16C4T0
      TIMER2 ----> GP32C4T0
      TIMER3 ----> GP32C4T1
      TIMER4 ----> BS16T0
      TIMER5 ----> BS16T1
      TIMER6 ----> GP16C4T0
      TIMER7 ----> GP16C4T1
    @endverbatim
  */
typedef enum
{
    ALD_CMU_PERH_GPIO    = (1U << 0),               /**< GPIO */
    ALD_CMU_PERH_CRC     = (1U << 1),               /**< CRC */
    ALD_CMU_PERH_CALC    = (1U << 2),               /**< CALC */
    ALD_CMU_PERH_PIS     = (1U << 5),               /**< PIS */
    ALD_CMU_PERH_DMA0    = (1U << 2)  | (1U << 26), /**< DMA0 */
    ALD_CMU_PERH_DMA1    = (1U << 3)  | (1U << 26), /**< DMA1 */
    ALD_CUM_PERH_DMAMUX  = (1U << 4)  | (1U << 26), /**< DMAMUX */
    ALD_CMU_PERH_TIMER0  = (1U << 0)  | (1U << 27), /**< TIMER0 */
    ALD_CMU_PERH_TIMER2  = (1U << 2)  | (1U << 27), /**< TIMER2 */
    ALD_CMU_PERH_TIMER3  = (1U << 3)  | (1U << 27), /**< TIMER3 */
    ALD_CMU_PERH_TIMER4  = (1U << 4)  | (1U << 27), /**< TIMER4 */
    ALD_CMU_PERH_TIMER5  = (1U << 5)  | (1U << 27), /**< TIMER5 */
    ALD_CMU_PERH_TIMER6  = (1U << 6)  | (1U << 27), /**< TIMER6 */
    ALD_CMU_PERH_TIMER7  = (1U << 7)  | (1U << 27), /**< TIMER7 */
    ALD_CMU_PERH_UART0   = (1U << 8)  | (1U << 27), /**< UART0 */
    ALD_CMU_PERH_UART1   = (1U << 9)  | (1U << 27), /**< UART1 */
    ALD_CMU_PERH_UART2   = (1U << 10) | (1U << 27), /**< UART2 */
    ALD_CMU_PERH_UART3   = (1U << 11) | (1U << 27), /**< UART3 */
    ALD_CMU_PERH_UART4   = (1U << 12) | (1U << 27), /**< UART4 */
    ALD_CMU_PERH_UART5   = (1U << 13) | (1U << 27), /**< UART5 */
    ALD_CMU_PERH_SPI0    = (1U << 16) | (1U << 27), /**< SPI0 */
    ALD_CMU_PERH_SPI1    = (1U << 17) | (1U << 27), /**< SPI1 */
    ALD_CMU_PERH_I2C0    = (1U << 20) | (1U << 27), /**< I2C0 */
    ALD_CMU_PERH_I2C1    = (1U << 21) | (1U << 27), /**< I2C1 */
    ALD_CMU_PERH_CAN     = (1U << 24) | (1U << 27), /**< CAN */
    ALD_CMU_PERH_LPTIM0  = (1U << 0)  | (1U << 28), /**< LPTIM0 */
    ALD_CMU_PERH_LPUART0 = (1U << 2)  | (1U << 28), /**< LPUART0 */
    ALD_CMU_PERH_ADC0    = (1U << 4)  | (1U << 28), /**< ADC0 */
    ALD_CMU_PERH_WWDT    = (1U << 12) | (1U << 28), /**< WWDT */
    ALD_CMU_PERH_IWDT    = (1U << 14) | (1U << 28), /**< IWDT */
    ALD_CMU_PERH_RTC     = (1U << 15) | (1U << 28), /**< RTC */
    ALD_CMU_PERH_DAC0    = (1U << 16) | (1U << 28), /**< DAC0 */
    ALD_CMU_PERH_DBGC    = (1U << 19) | (1U << 28), /**< DBGC */
    ALD_CMU_PERH_ALL     = (0x7FFFFFFFU),           /**< ALL */
} ald_cmu_perh_t;

/**
  * @brief CMU interrupt type
  */
typedef enum
{
    ALD_CMU_LOSC_STOP    = 0x0U,    /**< LOSC STOP INTERRUPT */
    ALD_CMU_HOSC_STOP    = 0x1U,    /**< HOSC STOP INTERRUPT */
    ALD_CMU_PLL_UNLOCK   = 0x2U,    /**< PLL UNLOCK INTERRUPT */
    ALD_CMU_LOSC_START   = 0x3U,    /**< LOSC START INTERRUPT */
    ALD_CMU_HOSC_START   = 0x4U,    /**< HOSC START INTERRUPT */
} ald_cmu_security_t;

/**
  * @brief CMU clock state type
  */
typedef enum
{
    ALD_CMU_CLOCK_STATE_HOSCACT = (1U << 0),    /**< HOSC active */
    ALD_CMU_CLOCK_STATE_LOSCACT = (1U << 1),    /**< LOSC active */
    ALD_CMU_CLOCK_STATE_HRCACT  = (1U << 2),    /**< HRC active */
    ALD_CMU_CLOCK_STATE_LRCACT  = (1U << 3),    /**< LRC active */
    ALD_CMU_CLOCK_STATE_ULRCACT = (1U << 4),    /**< ULRC active */
    ALD_CMU_CLOCK_STATE_PLLACT  = (1U << 8),    /**< PLL active */
    ALD_CMU_CLOCK_STATE_HOSCRDY = (1U << 16),   /**< HOSC ready */
    ALD_CMU_CLOCK_STATE_LOSCRDY = (1U << 17),   /**< LOSC ready */
    ALD_CMU_CLOCK_STATE_HRCRDY  = (1U << 18),   /**< HRC ready */
    ALD_CMU_CLOCK_STATE_LRCRDY  = (1U << 19),   /**< LRC ready */
    ALD_CMU_CLOCK_STATE_PLLRDY  = (1U << 24),   /**< PLL ready */
} ald_cmu_clock_state_t;

/**
  * @brief Stop1 clock select type
  */
typedef enum
{
    ALD_CMU_STOP1_CLOCK_LRC     = 0x0U, /**< LRC */
    ALD_CMU_STOP1_CLOCK_ULRC    = 0x1U, /**< ULRC */
    ALD_CMU_STOP1_CLOCK_HRC     = 0x2U, /**< HRC */
    ALD_CMU_STOP1_CLOCK_HRC_1M  = 0x3U, /**< HRC divides to 1MHz */
    ALD_CMU_STOP1_CLOCK_HOSC    = 0x4U, /**< HOSC */
    ALD_CMU_STOP1_CLOCK_HOSC_1M = 0x5U, /**< HOSC divides to 1MHz */
    ALD_CMU_STOP1_CLOCK_LOSCM   = 0x6U, /**< LOSC security management */
} ald_cmu_stop1_clock_t;

/**
  * @}
  */

/**
  * @defgroup CMU_Private_Macros CMU Private Macros
  * @{
  */
#define IS_CMU_CLOCK(x)         (((x) == ALD_CMU_CLOCK_HRC)  || \
                                 ((x) == ALD_CMU_CLOCK_LRC)  || \
                                 ((x) == ALD_CMU_CLOCK_LOSC) || \
                                 ((x) == ALD_CMU_CLOCK_PLL)  || \
                                 ((x) == ALD_CMU_CLOCK_HOSC))
#define IS_CMU_PLL_OUTPUT(x)    (((x) == ALD_CMU_PLL_OUTPUT_48M) || \
                                 ((x) == ALD_CMU_PLL_OUTPUT_56M) || \
                                 ((x) == ALD_CMU_PLL_OUTPUT_64M) || \
                                 ((x) == ALD_CMU_PLL_OUTPUT_72M))
#define IS_CMU_PLL_INPUT(x)     (((x) == ALD_CMU_PLL_INPUT_HRC)  || \
                                 ((x) == ALD_CMU_PLL_INPUT_HOSC)   || \
                                 ((x) == ALD_CMU_PLL_INPUT_HOSC_2) || \
                                 ((x) == ALD_CMU_PLL_INPUT_HOSC_3) || \
                                 ((x) == ALD_CMU_PLL_INPUT_HOSC_4) || \
                                 ((x) == ALD_CMU_PLL_INPUT_HOSC_5) || \
                                 ((x) == ALD_CMU_PLL_INPUT_HOSC_6))
#define IS_CMU_HOSC_RANGE(x)    (((x) == ALD_CMU_HOSC_2M)  || \
                                 ((x) == ALD_CMU_HOSC_4M)  || \
                                 ((x) == ALD_CMU_HOSC_8M)  || \
                                 ((x) == ALD_CMU_HOSC_16M) || \
                                 ((x) == ALD_CMU_HOSC_24M))
#define IS_CMU_DIV(x)           (((x) == ALD_CMU_DIV_1)    || \
                                 ((x) == ALD_CMU_DIV_2)    || \
                                 ((x) == ALD_CMU_DIV_4)    || \
                                 ((x) == ALD_CMU_DIV_8)    || \
                                 ((x) == ALD_CMU_DIV_16)   || \
                                 ((x) == ALD_CMU_DIV_32)   || \
                                 ((x) == ALD_CMU_DIV_64)   || \
                                 ((x) == ALD_CMU_DIV_128)  || \
                                 ((x) == ALD_CMU_DIV_256)  || \
                                 ((x) == ALD_CMU_DIV_512)  || \
                                 ((x) == ALD_CMU_DIV_1024) || \
                                 ((x) == ALD_CMU_DIV_2048) || \
                                 ((x) == ALD_CMU_DIV_4096))
#define IS_CMU_BUS(x)           (((x) == ALD_CMU_SYS)      || \
                                 ((x) == ALD_CMU_PCLK_1)   || \
                                 ((x) == ALD_CMU_PCLK_2))
#define IS_CMU_OUTPUT_HIGH_SEL(x)   (((x) == ALD_CMU_OUTPUT_HIGH_SEL_HOSC) || \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_LOSC) || \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_HRC)  || \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_LRC)  || \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_HOSM) || \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_LOSM) || \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_PCLK1)|| \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_PCLK2)|| \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_PLL)  || \
                                     ((x) == ALD_CMU_OUTPUT_HIGH_SEL_SYSCLK))
#define IS_CMU_OUTPUT_HIGH_DIV(x)   (((x) == ALD_CMU_OUTPUT_DIV_1)  || \
                                     ((x) == ALD_CMU_OUTPUT_DIV_2)  || \
                                     ((x) == ALD_CMU_OUTPUT_DIV_4)  || \
                                     ((x) == ALD_CMU_OUTPUT_DIV_8)  || \
                                     ((x) == ALD_CMU_OUTPUT_DIV_16) || \
                                     ((x) == ALD_CMU_OUTPUT_DIV_32) || \
                                     ((x) == ALD_CMU_OUTPUT_DIV_64) || \
                                     ((x) == ALD_CMU_OUTPUT_DIV_128))
#define IS_CMU_OUTPUT_LOW_SEL(x)    (((x) == ALD_CMU_OUTPUT_LOW_SEL_LOSC) || \
                                     ((x) == ALD_CMU_OUTPUT_LOW_SEL_LRC ) || \
                                     ((x) == ALD_CMU_OUTPUT_LOW_SEL_LOSM) || \
                                     ((x) == ALD_CMU_OUTPUT_LOW_SEL_BUZZ) || \
                                     ((x) == ALD_CMU_OUTPUT_LOW_SEL_ULRC))
#define IS_CMU_SAFE_CLOCK_TYPE(x)   (((x) == ALD_CMU_SAFE_CLK_HOSC) || \
                                     ((x) == ALD_CMU_SAFE_CLK_LOSC) || \
                                     ((x) == ALD_CMU_SAFE_CLK_PLL))
#define IS_CMU_BUZZ_DIV(x)  (((x) == ALD_CMU_BUZZ_DIV_2)   || \
                             ((x) == ALD_CMU_BUZZ_DIV_4)   || \
                             ((x) == ALD_CMU_BUZZ_DIV_8)   || \
                             ((x) == ALD_CMU_BUZZ_DIV_16)  || \
                             ((x) == ALD_CMU_BUZZ_DIV_32)  || \
                             ((x) == ALD_CMU_BUZZ_DIV_64)  || \
                             ((x) == ALD_CMU_BUZZ_DIV_128) || \
                             ((x) == ALD_CMU_BUZZ_DIV_256))
#define IS_CMU_LP_PERH_CLOCK_SEL(x) (((x) == ALD_CMU_LP_PERH_CLOCK_SEL_PCLK2)   || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_PLL)     || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_HRC)     || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_HOSC)    || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_LRC)     || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_LOSC)    || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_ULRC)    || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_HRC_1M)  || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_HOSC_1M) || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_LOSM)    || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_PCLK2)   || \
                                     ((x) == ALD_CMU_LP_PERH_CLOCK_SEL_HOSM))
#define IS_CMU_RTC_CLOCK_SEL(x)  (((x) == ALD_CMU_RTC_SEL_LOSM)        || \
                                  ((x) == ALD_CMU_RTC_SEL_LRC)         || \
                                  ((x) == ALD_CMU_RTC_SEL_HRC_DIV_1M ) || \
                                  ((x) == ALD_CMU_RTC_SEL_HOSC_DIV_1M))
#define IS_CMU_PERH(x)      (((x) == ALD_CMU_PERH_GPIO)    || \
                             ((x) == ALD_CMU_PERH_CRC)     || \
                             ((x) == ALD_CMU_PERH_CALC)    || \
                             ((x) == ALD_CMU_PERH_PIS)     || \
                             ((x) == ALD_CMU_PERH_DMA0)    || \
                             ((x) == ALD_CMU_PERH_DMA1)    || \
                             ((x) == ALD_CUM_PERH_DMAMUX)  || \
                             ((x) == ALD_CMU_PERH_TIMER0)  || \
                             ((x) == ALD_CMU_PERH_TIMER2)  || \
                             ((x) == ALD_CMU_PERH_TIMER3)  || \
                             ((x) == ALD_CMU_PERH_TIMER4)  || \
                             ((x) == ALD_CMU_PERH_TIMER5)  || \
                             ((x) == ALD_CMU_PERH_TIMER6)  || \
                             ((x) == ALD_CMU_PERH_TIMER7)  || \
                             ((x) == ALD_CMU_PERH_UART0)   || \
                             ((x) == ALD_CMU_PERH_UART1)   || \
                             ((x) == ALD_CMU_PERH_UART2)   || \
                             ((x) == ALD_CMU_PERH_UART3)   || \
                             ((x) == ALD_CMU_PERH_UART4)   || \
                             ((x) == ALD_CMU_PERH_UART5)   || \
                             ((x) == ALD_CMU_PERH_SPI0)    || \
                             ((x) == ALD_CMU_PERH_SPI1)    || \
                             ((x) == ALD_CMU_PERH_I2C0)    || \
                             ((x) == ALD_CMU_PERH_I2C1)    || \
                             ((x) == ALD_CMU_PERH_CAN)     || \
                             ((x) == ALD_CMU_PERH_LPTIM0)  || \
                             ((x) == ALD_CMU_PERH_LPUART0) || \
                             ((x) == ALD_CMU_PERH_ADC0)    || \
                             ((x) == ALD_CMU_PERH_DAC0)    || \
                             ((x) == ALD_CMU_PERH_WWDT)    || \
                             ((x) == ALD_CMU_PERH_IWDT)    || \
                             ((x) == ALD_CMU_PERH_RTC)     || \
                             ((x) == ALD_CMU_PERH_DBGC)    || \
                             ((x) == ALD_CMU_PERH_ALL))
#define IS_CMU_CLOCK_STATE(x)   (((x) == ALD_CMU_CLOCK_STATE_HOSCACT) || \
                                 ((x) == ALD_CMU_CLOCK_STATE_LOSCACT) || \
                                 ((x) == ALD_CMU_CLOCK_STATE_HRCACT)  || \
                                 ((x) == ALD_CMU_CLOCK_STATE_LRCACT)  || \
                                 ((x) == ALD_CMU_CLOCK_STATE_ULRCACT) || \
                                 ((x) == ALD_CMU_CLOCK_STATE_PLLACT) || \
                                 ((x) == ALD_CMU_CLOCK_STATE_HOSCRDY) || \
                                 ((x) == ALD_CMU_CLOCK_STATE_LOSCRDY) || \
                                 ((x) == ALD_CMU_CLOCK_STATE_HRCRDY)  || \
                                 ((x) == ALD_CMU_CLOCK_STATE_LRCRDY)  || \
                                 ((x) == ALD_CMU_CLOCK_STATE_PLLRDY))
#define IS_CMU_STOP1_CLOCK(x)   (((x) == ALD_CMU_STOP1_CLOCK_LRC)     || \
                                 ((x) == ALD_CMU_STOP1_CLOCK_ULRC)     || \
                                 ((x) == ALD_CMU_STOP1_CLOCK_HRC)  || \
                                 ((x) == ALD_CMU_STOP1_CLOCK_HRC_1M)  || \
                                 ((x) == ALD_CMU_STOP1_CLOCK_HOSC)    || \
                                 ((x) == ALD_CMU_STOP1_CLOCK_HOSC_1M) || \
                                 ((x) == ALD_CMU_STOP1_CLOCK_LOSCM))
/**
  * @}
  */

/** @addtogroup CMU_Public_Functions
  * @{
  */
/** @addtogroup CMU_Public_Functions_Group1
  * @{
  */
/* System clock configure */
ald_status_t ald_cmu_clock_config_default(void);
ald_status_t ald_cmu_clock_config(ald_cmu_clock_t clk, uint32_t clock);
void ald_cmu_pll_config(ald_cmu_pll_input_t input, ald_cmu_pll_output_t output);
uint32_t ald_cmu_get_clock(void);
/**
  * @}
  */

/** @addtogroup CMU_Public_Functions_Group2
  * @{
  */
/* BUS division control */
void ald_cmu_div_config(ald_cmu_bus_t bus, ald_cmu_div_t div);
uint32_t ald_cmu_get_sys_clock(void);
uint32_t ald_cmu_get_pclk1_clock(void);
uint32_t ald_cmu_get_pclk2_clock(void);
/**
  * @}
  */

/** @addtogroup CMU_Public_Functions_Group3
  * @{
  */
/* Clock safe configure */
void ald_cmu_hosc_safe_config(ald_cmu_hosc_range_t clock, type_func_t status);
void ald_cmu_losc_safe_config(type_func_t status);
void ald_cmu_pll_safe_config(type_func_t status);
uint32_t ald_cmu_current_clock_source_get(ald_cmu_clock_safe_type_t type);
flag_status_t ald_cmu_get_clock_state(ald_cmu_clock_state_t sr);
void ald_cmu_irq_handler(void);
void ald_cmu_irq_cbk(ald_cmu_security_t se);
/**
  * @}
  */

/** @addtogroup CMU_Public_Functions_Group4
  * @{
  */
/* Clock output configure */
void ald_cmu_output_high_clock_config(ald_cmu_output_high_sel_t sel,
                                      ald_cmu_output_high_div_t div, type_func_t status);
void ald_cmu_output_low_clock_config(ald_cmu_output_low_sel_t sel, type_func_t status);
/**
  * @}
  */

/** @addtogroup CMU_Public_Functions_Group5
  * @{
  */
/* Peripheral Clock configure */
void ald_cmu_buzz_config(ald_cmu_buzz_div_t div, uint16_t dat, type_func_t status);
void ald_cmu_lptim0_clock_select(ald_cmu_lp_perh_clock_sel_t clock);
void ald_cmu_lpuart0_clock_select(ald_cmu_lp_perh_clock_sel_t clock);
void ald_cmu_rtc_clock_select(ald_cmu_rtc_clock_sel_t sel);
void ald_cmu_perh_clock_config(ald_cmu_perh_t perh, type_func_t status);
void ald_cmu_stop1_clock_sel(ald_cmu_stop1_clock_t clock);
/**
  * @}
  */
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_CMU_H__ */
