/**
 *********************************************************************************
 *
 * @file    ald_cache.h
 * @brief   Header file of cache module driver.
 *
 * @version V1.0
 * @date    12 Mar 2024
 * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
 */

#ifndef __ALD_CACHE_H__
#define __ALD_CACHE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"

/** @addtogroup ES32FXXX_ALD
 * @{
 */

/** @addtogroup CACHE
 * @{
 */

/** @defgroup CACHE_Public_Types CACHE Public Types
 * @{
 */

/**
  * @brief Cache statistic setting
  */
typedef enum
{
    ALD_AUTO_INV = 0x0U,    /**< Automatic cache invalidate when cache enabled */
    ALD_MAN_INV  = 0x1U,    /**< Manual cache invalidate mode */
} ald_cache_invalidate_t;

/**
  * @brief Cache Power control setting
  */
typedef enum
{
    ALD_AUTO_POWER = 0x0U,    /**< Automatic power when cache enabled */
    ALD_MAN_POWER  = 0x1U,    /**< Manual power mode */
} ald_cache_power_t;

/**
 * @}
 */

/**
  * @brief Cache configuration structure definition
  */
typedef struct
{
    type_func_t statistic;              /**< Cache statistics logic */
    type_func_t prefetch;               /**< Cache prefetch setting */
    ald_cache_invalidate_t invlidate;   /**< Cache invalidate setting */
    ald_cache_power_t power;            /**< Power control setting */
} ald_cache_config_t;

/**
  * @brief Cache power and invalidating status
  */
typedef enum
{
    ALD_CACHE_INV_STAT  = (1U << 2),     /**< Indicates if invalidation process is ongoing */
    ALD_CACHE_POW_STAT  = (1U << 4),     /**< SRAM power acknowledges */
} ald_inv_pow_stat_t;

/**
  * @brief Cache status
  */
typedef enum
{
    ALD_CACHE_DISABLED  = 0U,     /**< Cache disabled */
    ALD_CACHE_ENABLING  = 1U,     /**< Cache enabling */
    ALD_CACHE_ENABLED   = 2U,     /**< Cache enabled */
    ALD_CACHE_DISABLING = 3U,     /**< Cache disabling */
} ald_cache_stat_t;

/**
  * @brief Cache interrupt
  */
typedef enum
{
    ALD_CACHE_IT_POW_ERR      = (1U << 0),   /**< Power error interrupt */
    ALD_CACHE_IT_MAN_INV_ERR  = (1U << 1),   /**< Manual invalidation error interrupt */
} ald_cache_it_t;

/**
  * @brief Cache interrupt flag
  */
typedef enum
{
    ALD_CACHE_IF_POW_ERR      = (1U << 0),   /**< Power error indication */
    ALD_CACHE_IF_MAN_INV_ERR  = (1U << 1),   /**< Manual invalidation error indication */
} ald_cache_flag_t;


/** @defgroup Cache_Private_Macros Cache Private Macros
  * @{
  */
#define IS_CACHE_POWER_SEL(x)   (((x) == ALD_AUTO_POWER) || \
                                 ((x) == ALD_MAN_POWER))
#define IS_CACHE_INVALIDATE_SEL(x)  (((x) == ALD_AUTO_INV) || \
                                     ((x) == ALD_MAN_INV))
#define IS_CACHE_INV_POWER_FLAG(x)  (((x) == ALD_CACHE_INV_STAT) || \
                                     ((x) == ALD_CACHE_POW_STAT))

#define IS_CACHE_CACHE_STATUS_FLAG(x)   (((x) == ALD_CACHE_DISABLED) || \
        ((x) == ALD_CACHE_ENABLING) || \
        ((x) == ALD_CACHE_ENABLED)  || \
        ((x) == ALD_CACHE_DISABLING))
#define IS_CACHE_IT_SEL(x)  (((x) == ALD_CACHE_IT_POW_ERR) || \
                             ((x) == ALD_CACHE_IT_MAN_INV_ERR))
#define IS_CACHE_IF_SEL(x)  (((x) == ALD_CACHE_IF_POW_ERR) || \
                             ((x) == ALD_CACHE_IF_MAN_INV_ERR))




/** @addtogroup Cache_Public_Functions
 * @{
 */
void ald_cache_init(ald_cache_config_t *config);
void ald_cache_enable(void);
void ald_cache_disable(void);
flag_status_t ald_get_cache_inv_pow_status(ald_inv_pow_stat_t flag);
ald_cache_stat_t ald_get_cache_status(void);
void ald_cache_it_config(ald_cache_it_t it, type_func_t state);
flag_status_t ald_cache_get_flag_status(ald_cache_flag_t flag);
void ald_flag_clear_flag_status(ald_cache_flag_t flag);
uint32_t ald_cache_get_hits(void);
uint32_t ald_cache_get_miss_count(void);
void ald_cache_clear_hits(void);
void ald_cache_clear_miss_count(void);
void ald_cache_manual_pow(void);
void ald_cache_manual_inv(void);
/**
 * @}
 */
/**
 * @}
 */

/**
 * @}
 */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_CACHE_H__ */
