/**
  *********************************************************************************
  *
  * @file    ald_opamp.c
  * @brief   OPAMP module driver.
  *
  * @version V1.0
  * @date    27 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          27 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "ald_opamp.h"

/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup OPAMP OPAMP
  * @brief OPAMP module driver
  * @{
  */
#ifdef ALD_OPAMP

/** @defgroup OPAMP_Public_Functions OPAMP Public Functions
  * @{
  */
/**
  * @brief  Deinitialize the OPAMP peripheral registers to their default reset
  *         values.
  * @param  hperh: Pointer to a opamp_handle_t structure that contains
  *         the configuration information for the specified OPAMP module.
  * @retval None
  */
void ald_opamp_reset(opamp_handle_t *hperh)
{
	assert_param(IS_OPAMP_TYPE(hperh->perh));

	WRITE_REG(hperh->perh->CON, 0x0);
	WRITE_REG(hperh->perh->MUX0, 0x0);
	WRITE_REG(hperh->perh->MUX1, 0x0);
	WRITE_REG(hperh->perh->MUX2, 0x0);

	__UNLOCK(hperh);
	return;
}

/**
  * @brief  Initializes the OPAMP mode according to the specified parameters in
  *         the opamp_init_t and create the associated handle.
  * @param  hperh: Pointer to a opamp_handle_t structure that contains
  *         the configuration information for the specified OPAMP module.
  * @param  index: Specifies which opa to be initialized.
  * @retval None
  */
ald_status_t ald_opamp_init(opamp_handle_t *hperh, opamp_index_t index)
{
	uint32_t tmp;

	assert_param(IS_OPAMP_TYPE(hperh->perh));
	assert_param(IS_OPAMP_INDEX_TYPE(index));
	assert_param(IS_FUNC_STATE(hperh->init.enable));
	assert_param(IS_FUNC_STATE(hperh->init.hcmd));
	assert_param(IS_FUNC_STATE(hperh->init.p_lpfd));
	assert_param(IS_FUNC_STATE(hperh->init.n_lpfd));
	assert_param(IS_FUNC_STATE(hperh->init.npsen));
	assert_param(IS_OPAMP_POSSEL_TYPE(hperh->init.p_sel));
	assert_param(IS_OPAMP_NEGSEL_TYPE(hperh->init.n_sel));
	assert_param(IS_OPAMP_RINMUX_TYPE(hperh->init.rinmux));
	assert_param(IS_OPAMP_GAIN_TYPE(hperh->init.gain));
	assert_param(IS_FUNC_STATE(hperh->init.p_en));
	assert_param(IS_FUNC_STATE(hperh->init.n_en));
	assert_param(IS_OPAMP_OUT_SEL_TYPE(hperh->init.out_sel));
	assert_param(IS_OPAMP_OUT_MODE_TYPE(hperh->init.out_mode));
	assert_param(IS_OPAMP_POWER_TYPE(hperh->init.mode));
	assert_param(IS_OPAMP_NEXT_OUT_TYPE(hperh->init.o_next));
	assert_param(IS_OPAMP_RESSEL_TYPE(hperh->init.rsel));

	__LOCK(hperh);

	switch (index) {
	case OPAMP_INDEX_0:
		/* Clear index 0 bits */
		tmp = (OPAMP_CON_OPA0EN_MSK | OPAMP_CON_OPA0HCMD_MSK | OPAMP_CON_OPA0LPFD_MSK | OPAMP_CON_OPA0NPSEN_MSK);
		hperh->perh->CON &= ~tmp;
		/* Set index 0 bits */
		tmp = ((hperh->init.hcmd << OPAMP_CON_OPA0HCMD_POS) | (hperh->init.p_lpfd << OPAMP_CON_OPA0LPFD_POSS) |
		      (hperh->init.n_lpfd << OPAMP_CON_OPA0LPFD_POSE) | (hperh->init.npsen << OPAMP_CON_OPA0NPSEN_POS));
		hperh->perh->CON |= tmp;

		/* Set MUX0 */
		tmp = ((hperh->init.p_sel << OPAMP_MUX0_POSSEL_POSS) | (hperh->init.n_sel << OPAMP_MUX0_NEGSEL_POSS) |
		      (hperh->init.rinmux << OPAMP_MUX0_RINMUX_POSS) | (hperh->init.gain << OPAMP_MUX0_GAINSEL_POS) |
		      (hperh->init.p_en << OPAMP_MUX0_POSPEN_POS) | (hperh->init.n_en << OPAMP_MUX0_NEGPEN_POS) |
		      (hperh->init.out_sel << OPAMP_MUX0_OUTSEL_POSS) | (hperh->init.out_mode << OPAMP_MUX0_OUTMD_POSS) |
		      (hperh->init.mode << OPAMP_MUX0_OPMD_POSS) | (hperh->init.o_next << OPAMP_MUX0_NXTOUT_POS) |
		      (hperh->init.rsel << OPAMP_MUX0_RSEL_POSS));
		WRITE_REG(hperh->perh->MUX0, tmp);

		/* Enable/Disable index 0 */
		if (hperh->init.enable)
			SET_BIT(hperh->perh->CON, OPAMP_CON_OPA0EN_MSK);
		else
			CLEAR_BIT(hperh->perh->CON, OPAMP_CON_OPA0EN_MSK);

		break;

	case OPAMP_INDEX_1:
		/* Clear index 1 bits */
		tmp = (OPAMP_CON_OPA1EN_MSK | OPAMP_CON_OPA1HCMD_MSK | OPAMP_CON_OPA1LPFD_MSK | OPAMP_CON_OPA1NPSEN_MSK);
		hperh->perh->CON &= ~tmp;
		/* Set index 1 bits */
		tmp = ((hperh->init.hcmd << OPAMP_CON_OPA1HCMD_POS) | (hperh->init.p_lpfd << OPAMP_CON_OPA1LPFD_POSS) |
		      (hperh->init.n_lpfd << OPAMP_CON_OPA1LPFD_POSE) | (hperh->init.npsen << OPAMP_CON_OPA1NPSEN_POS));
		hperh->perh->CON |= tmp;

		/* Set MUX1 */
		tmp = ((hperh->init.p_sel << OPAMP_MUX1_POSSEL_POSS) | (hperh->init.n_sel << OPAMP_MUX1_NEGSEL_POSS) |
		      (hperh->init.rinmux << OPAMP_MUX1_RINMUX_POSS) | (hperh->init.gain << OPAMP_MUX1_GAINSEL_POS) |
		      (hperh->init.p_en << OPAMP_MUX1_POSPEN_POS) | (hperh->init.n_en << OPAMP_MUX1_NEGPEN_POS) |
		      (hperh->init.out_sel << OPAMP_MUX1_OUTSEL_POSS) | (hperh->init.out_mode << OPAMP_MUX1_OUTMD_POSS) |
		      (hperh->init.mode << OPAMP_MUX1_OPMD_POSS) | (hperh->init.o_next << OPAMP_MUX1_NXTOUT_POS) |
		      (hperh->init.rsel << OPAMP_MUX1_RSEL_POSS));
		WRITE_REG(hperh->perh->MUX1, tmp);

		/* Enable/Disable index 1 */
		if (hperh->init.enable)
			SET_BIT(hperh->perh->CON, OPAMP_CON_OPA1EN_MSK);
		else
			CLEAR_BIT(hperh->perh->CON, OPAMP_CON_OPA1EN_MSK);

		break;

	case OPAMP_INDEX_2:
		/* Clear index 2 bits */
		tmp = (OPAMP_CON_OPA2EN_MSK | OPAMP_CON_OPA2HCMD_MSK | OPAMP_CON_OPA2LPFD_MSK | OPAMP_CON_OPA2NPSEN_MSK);
		hperh->perh->CON &= ~tmp;
		/* Set index 2 bits */
		tmp = ((hperh->init.hcmd << OPAMP_CON_OPA2HCMD_POS) | (hperh->init.p_lpfd << OPAMP_CON_OPA2LPFD_POSS) |
		      (hperh->init.n_lpfd << OPAMP_CON_OPA2LPFD_POSE) | (hperh->init.npsen << OPAMP_CON_OPA2NPSEN_POS));
		hperh->perh->CON |= tmp;

		/* Set MUX2 */
		tmp = ((hperh->init.p_sel << OPAMP_MUX2_POSSEL_POSS) | (hperh->init.n_sel << OPAMP_MUX2_NEGSEL_POSS) |
		      (hperh->init.rinmux << OPAMP_MUX2_RINMUX_POSS) | (hperh->init.gain << OPAMP_MUX2_GAINSEL_POS) |
		      (hperh->init.p_en << OPAMP_MUX2_POSPEN_POS) | (hperh->init.n_en << OPAMP_MUX2_NEGPEN_POS) |
		      (hperh->init.out_sel << OPAMP_MUX2_OUTSEL_POS) | (hperh->init.out_mode << OPAMP_MUX2_OUTMD_POS) |
		      (hperh->init.mode << OPAMP_MUX2_OPMD_POSS) | (hperh->init.o_next << OPAMP_MUX2_NXTOUT_POS) |
		      (hperh->init.rsel << OPAMP_MUX2_RSEL_POSS));
		WRITE_REG(hperh->perh->MUX2, tmp);

		/* Enable/Disable index 2 */
		if (hperh->init.enable)
			SET_BIT(hperh->perh->CON, OPAMP_CON_OPA2EN_MSK);
		else
			CLEAR_BIT(hperh->perh->CON, OPAMP_CON_OPA2EN_MSK);

		break;

	default:
		break;
	}

	__UNLOCK(hperh);
	return OK;
}
/**
  * @}
  */
#endif /* ALD_OPAMP */
/**
  * @}
  */
/**
  * @}
  */
