/**
  *********************************************************************************
  *
  * @file    ald_opamp.h
  * @brief   Header file of OPAMP module driver.
  *
  * @version V1.0
  * @date    27 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          27 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __ALD_OPAMP_H__
#define __ALD_OPAMP_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"

/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @addtogroup OPAMP
  * @{
  */

/** @defgroup OPAMP_Public_Types OPAMP Public Types
  * @{
  */

/**
  * @brief Opa index
  */
typedef enum {
	OPAMP_INDEX_0 = 0x0U,	/**< Select opa0 */
	OPAMP_INDEX_1 = 0x1U,	/**< Select opa1 */
	OPAMP_INDEX_2 = 0x2U,	/**< Select opa2 */
} opamp_index_t;

/**
  * @brief Non-inverting input
  */
typedef enum {
	OPAMP_P_INPUT_OPA   = 0x0U,	/**< Opa x positive input */
	OPAMP_P_INPUT_NEXT  = 0x1U,	/**< Opa x next out input */
	OPAMP_P_INPUT_OPTAP = 0x2U,	/**< Opa x resistor ladder as input */
	OPAMP_P_INPUT_DAC   = 0x3U,	/**< DAC as input */
} opamp_possel_t;

/**
  * @brief Inverting input
  */
typedef enum {
	OPAMP_N_INPUT_OPA   = 0x0U,	/**< Opa x negative input */
	OPAMP_N_INPUT_OPTAP = 0x1U,	/**< Opa x resistor ladder as input */
	OPAMP_N_INPUT_UG    = 0x2U,	/**< Gain feedback path */
} opamp_negsel_t;

/**
  * @brief Resistor ladder input
  */
typedef enum {
	OPAMP_RINMUX_NEXT    = 0x0U,	/**< Set for next oupt */
	OPAMP_RINMUX_OPA_INP = 0x1U,	/**< Set for opa positive input */
	OPAMP_RINMUX_OPA_INN = 0x2U,	/**< Set for opa negative input */
	OPAMP_RINMUX_VSS     = 0x3U,	/**< VSS connect */
} opamp_rinmux_t;

/**
  * @brief Opamp gain function
  */
typedef enum {
	OPAMP_GAIN_UNIT   = 0x0U,	/**< gain function by unit */
	OPAMP_GAIN_MANUAL = 0x1U,	/**< gain function by manual */
} opamp_gaisel_t;

/**
  * @brief Select output
  */
typedef enum {
	OPAMP_OUTSEL_DIS_PIN = 0x0U,	/**< OPAMP3 output is disable */
	OPAMP_OUTSEL_PIN     = 0x1U,	/**< OPAMP3 output to pin enabled */
	OPAMP_OUTSEL_ALTOUT0 = 0x1U,	/**< OPAMP 1/2 altemative output0 */
	OPAMP_OUTSEL_ALTOUT1 = 0x2U,	/**< OPAMP 1/2 altemative output1 */
} opamp_out_sel_t;

/**
  * @brief Select output mode
  */
typedef enum {
	OPAMP_OUTMD_DISABLE = 0x0U,	/**< OPAMP 1/2/3 output is disable */
	OPAMP_OUTMD_MAIN    = 0x1U,	/**< OPAMP 1/2 output to main */
	OPAMP_OUTMD_ALT     = 0x2U,	/**< OPAMP 1/2 output to altemative */
	OPAMP_OUTMD_ALL     = 0x3U,	/**< OPAMP 1/2 output to main and altemative */
	OPAMP_OUTMD_PIN     = 0x1U,	/**< OPAMP3 output to pin */
} opamp_out_mode_t;

/**
  * @brief Select opamp power mode
  */
typedef enum {
	OPAMP_MODE_RSD    = 0x0U,	/**< OPAMP power reserved */
	OPAMP_MODE_LOW    = 0x1U,	/**< OPAMP work in low power mode */
	OPAMP_MODE_NORMAL = 0x2U,	/**< OPAMP work in normal power mode */
	OPAMP_MODE_HIGH   = 0x3U,	/**< OPAMP work in high power mode */
} opamp_mode_t;

/**
  * @brief Opamp next function
  */
typedef enum {
	OPAMP_NEXTOUT_DISABLE = 0x0U,	/**< Disable next output */
	OPAMP_NEXTOUT_ENABLE  = 0x1U,	/**< Makes output of OPA0/OPA1 available to OPA1/OPA2 */
} opamp_next_out_t;

/**
  * @brief Resistor ladder select
  */
typedef enum {
	OPAMP_RSEL_1  = 0x0U,	/**< R2/R1=1 */
	OPAMP_RSEL_3  = 0x1U,	/**< R2/R1=3 */
	OPAMP_RSEL_7  = 0x2U,	/**< R2/R1=7 */
	OPAMP_RSEL_15 = 0x3U,	/**< R2/R1=15 */
} opamp_rsel_t;

/**
  * @brief Opamp init structure definition
  */
typedef struct {
	type_func_t enable;		/**< ENABLE/DISABLE OPAMP */
	type_func_t hcmd;		/**< Disable high common mode */
	type_func_t p_lpfd;		/**< Disable positive low-pass filtering */
	type_func_t n_lpfd;		/**< Disable negative low-pass filtering */
	type_func_t npsen;		/**< Enable short circuit */
	opamp_possel_t p_sel;		/**< Positive input */
	opamp_negsel_t n_sel;		/**< Negative input */
	opamp_rinmux_t rinmux;		/**< Select divider resistor */
	opamp_gaisel_t gain;		/**< OPAMP gain mode */
	type_func_t p_en;		/**< Enable positive input */
	type_func_t n_en;		/**< Enable negative input */
	opamp_out_sel_t out_sel;	/**< Select output */
	opamp_out_mode_t out_mode;	/**< Output mode */
	opamp_mode_t mode;		/**< OPAMP work mode */
	opamp_next_out_t o_next;	/**< Enable output to next */
	opamp_rsel_t rsel;		/**< Resistor ladder select */
} opamp_init_t;

/**
  * @brief  OPAMP Handle Structure definition
  */
typedef struct opamp_handle_s {
	OPAMP_TypeDef *perh;	/**< Register base address */
	opamp_init_t init;	/**< Opamp required parameters */
	lock_state_t lock;	/**< Locking object */
} opamp_handle_t;
/**
  * @}
  */

/** @defgroup OPAMP_Public_Macros OPAMP Public Macros
  * @{
  */
#define OPAMP0_ENABLE(handle)		(SET_BIT((handle)->perh->CON, OPAMP_CON_OPA0EN_MSK))
#define OPAMP0_DISABLE(handle)		(CLEAR_BIT((handle)->perh->CON, OPAMP_CON_OPA0EN_MSK))
#define OPAMP1_ENABLE(handle)		(SET_BIT((handle)->perh->CON, OPAMP_CON_OPA1EN_MSK))
#define OPAMP1_DISABLE(handle)		(CLEAR_BIT((handle)->perh->CON, OPAMP_CON_OPA1EN_MSK))
#define OPAMP2_ENABLE(handle)		(SET_BIT((handle)->perh->CON, OPAMP_CON_OPA2EN_MSK))
#define OPAMP2_DISABLE(handle)		(CLEAR_BIT((handle)->perh->CON, OPAMP_CON_OPA2EN_MSK))
/**
  * @}
  */

/** @defgroup OPAMP_Private_Macros   OPAMP Private Macros
  * @{
  */
#define IS_OPAMP_TYPE(x)		((x) == OPAMP)
#define IS_OPAMP_INDEX_TYPE(x)		(((x) == OPAMP_INDEX_0) || \
					 ((x) == OPAMP_INDEX_1) || \
					 ((x) == OPAMP_INDEX_2))
#define IS_OPAMP_POSSEL_TYPE(x)		(((x) == OPAMP_P_INPUT_OPA)   || \
					 ((x) == OPAMP_P_INPUT_NEXT)  || \
					 ((x) == OPAMP_P_INPUT_OPTAP) || \
					 ((x) == OPAMP_P_INPUT_DAC))
#define IS_OPAMP_NEGSEL_TYPE(x)		(((x) == OPAMP_N_INPUT_OPA)   || \
					 ((x) == OPAMP_N_INPUT_OPTAP) || \
					 ((x) == OPAMP_N_INPUT_UG))
#define IS_OPAMP_RINMUX_TYPE(x) 	(((x) == OPAMP_RINMUX_NEXT)    || \
					 ((x) == OPAMP_RINMUX_OPA_INP) || \
					 ((x) == OPAMP_RINMUX_OPA_INN) || \
					 ((x) == OPAMP_RINMUX_VSS))
#define IS_OPAMP_GAIN_TYPE(x)		(((x) == OPAMP_GAIN_UNIT ) || \
					 ((x) == OPAMP_GAIN_MANUAL))
#define IS_OPAMP_OUT_SEL_TYPE(x) 	(((x) == OPAMP_OUTSEL_DIS_PIN) || \
					 ((x) == OPAMP_OUTSEL_PIN)     || \
					 ((x) == OPAMP_OUTSEL_ALTOUT0) || \
					 ((x) == OPAMP_OUTSEL_ALTOUT1))
#define IS_OPAMP_OUT_MODE_TYPE(x) 	(((x) == OPAMP_OUTMD_DISABLE) || \
					 ((x) == OPAMP_OUTMD_MAIN)    || \
					 ((x) == OPAMP_OUTMD_ALT)     || \
					 ((x) == OPAMP_OUTMD_ALL)     || \
					 ((x) == OPAMP_OUTMD_PIN))
#define IS_OPAMP_POWER_TYPE(x)		(((x) == OPAMP_MODE_RSD)    || \
					 ((x) == OPAMP_MODE_LOW)    || \
					 ((x) == OPAMP_MODE_NORMAL) || \
					 ((x) == OPAMP_MODE_HIGH))
#define IS_OPAMP_NEXT_OUT_TYPE(x)	(((x) == OPAMP_NEXTOUT_DISABLE) || \
					 ((x) == OPAMP_NEXTOUT_ENABLE))
#define IS_OPAMP_RESSEL_TYPE(x)		(((x) == OPAMP_RSEL_1) || \
					 ((x) == OPAMP_RSEL_3) || \
					 ((x) == OPAMP_RSEL_7) || \
					 ((x) == OPAMP_RSEL_15))
/**                                              
  * @}                                           
  */
/** @addtogroup OPAMP_Public_Functions
  * @{
  */
void ald_opamp_reset(opamp_handle_t *hperh);
ald_status_t ald_opamp_init(opamp_handle_t *hperh, opamp_index_t index);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
#ifdef __cplusplus
extern "C" }
#endif
#endif
