/**********************************************************************************
 *
 * @file    ald_dma.c
 * @brief   ald_dma C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov  2021    Ginger          the first version
 *          23 Dec  2021    Ginger          Modify macro define
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

#include <string.h>
#include "ald_conf.h"

/** @addtogroup ALD
  * @{
  */

/** @defgroup DMA DMA
  * @brief DMA module driver
  * @{
  */

#ifdef ALD_DMA
/** @defgroup DMA_Private_Variables DMA Private Variables
  * @{
  */
ald_dma_descriptor_t ald_dma_ctrl_base[12] __attribute__((aligned(512)));
ald_dma_call_back_t ald_dma_cbk[6];
/**
  * @}
  */

/** @defgroup DMA_Private_Functions DMA Private Functions
  * @{
  */

static void ald_dma_address_inject(volatile void *volatile *volatile reg, volatile void *value)
{
    while (*reg != value)
    {
        *reg = value;
    }
}

static void ald_dma_config_inject(volatile uint32_t *reg, volatile uint32_t value)
{
    while (*reg != value)
    {
        *reg = value;
    }
}


/**
  * @brief  Configure DMA channel using ald_dma_config_t structure
  * @param  DMAx: Pointer to DMA peripheral
  * @param  mode: DMA transfer mode. see @ref ald_dma_cycle_ctrl_t
  * @param  p: Pointer to ald_dma_cycle_ctrl_t which contains
  *            DMA channel parameter. see @ref ald_dma_config_t
  * @retval None
  */
static void ald_dma_config_base(DMA_TypeDef *DMAx, ald_dma_cycle_ctrl_t mode, ald_dma_config_t *p)
{
    ald_dma_descriptor_t *descr;
    uint32_t *ptr;

    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_CYCLECTRL_TYPE(mode));
    assert_param(p->src != NULL);
    assert_param(p->dst != NULL);
    assert_param(ALD_IS_DMA_DATA_SIZE(p->size));
    assert_param(ALD_IS_DMA_DATASIZE_TYPE(p->data_width));
    assert_param(ALD_IS_DMA_DATAINC_TYPE(p->src_inc));
    assert_param(ALD_IS_DMA_DATAINC_TYPE(p->dst_inc));
    assert_param(ALD_IS_DMA_ARBITERCONFIG_TYPE(p->R_power));
    assert_param(ALD_IS_DMA_CHANNEL(p->channel));

    if (p->primary)
        descr = (ald_dma_descriptor_t *)((DMA1_BASE + 0x100) + (p->channel * 0x10));
    else
        descr = (ald_dma_descriptor_t *)((DMA1_BASE + 0x200) + (p->channel * 0x10));

    if (p->src_inc == ALD_DMA_DATA_INC_NONE)
    {
        ald_dma_address_inject(&descr->src, p->src);
    }
    else
    {
        if (((p->size ) & 0x3FFUL) > 0)        
            ald_dma_address_inject(&descr->src, (void *)((uint32_t)p->src + (((p->size - 1) & 0x3FFUL) << p->src_inc)));
        else
            ald_dma_address_inject(&descr->src, (void *)((uint32_t)p->src));  
    }

    if (p->dst_inc == ALD_DMA_DATA_INC_NONE)
    {

        ald_dma_address_inject(&descr->dst, p->dst);
    }
    else
    {
        if (((p->size ) & 0x3FFUL) > 0)  
            ald_dma_address_inject(&descr->dst, (void *)((uint32_t)p->dst + (((p->size - 1) & 0x3FFUL) << p->dst_inc)));
        else
            ald_dma_address_inject(&descr->dst, (void *)((uint32_t)p->dst));      
    }

    if (((p->size ) & 0x3FFUL) > 0)
    {
        ald_dma_config_inject(&descr->ctrl.word, ((uint32_t)(mode) | (0UL << 3) | (((p->size - 1) & 0x3FFUL) << 4) | ((p->R_power & 0xFUL) << 14) | ((p->data_width & 0x3UL) << 24) | ((p->src_inc & 0x3UL) << 26) | ((p->data_width & 0x3UL) << 28) | ((p->dst_inc & 0x3UL) << 30)));
    }
    else
    {
        ald_dma_config_inject(&descr->ctrl.word, ((uint32_t)(mode) | (0UL << 3) | ((0UL) << 4) | ((p->R_power & 0xFUL) << 14) | ((p->data_width & 0x3UL) << 24) | ((p->src_inc & 0x3UL) << 26) | ((p->data_width & 0x3UL) << 28) | ((p->dst_inc & 0x3UL) << 30)));
    }

    if (p->primary)
        WRITE_REG(DMAx->CHPRIALTCLR, (1 << p->channel));
    else
        WRITE_REG(DMAx->CHPRIALTSET, (1 << p->channel));

    if (p->burst)
        WRITE_REG(DMAx->CHUSEBURSTSET, (1 << p->channel));
    else
        WRITE_REG(DMAx->CHUSEBURSTCLR, (1 << p->channel));

    if (p->high_prio)
        WRITE_REG(DMAx->CHPRSET, (1 << p->channel));
    else
        WRITE_REG(DMAx->CHPRCLR, (1 << p->channel));

    if (p->interrupt)
        SET_BIT(DMAx->IER, (1 << p->channel));
    else
        SET_BIT(DMAx->IDR, (1 << p->channel));

    ptr = (uint32_t *)(DMA1_BASE + 0x70 + (p->channel << 2));
    MODIFY_REG(*ptr, DMA_CH_SELCON_MSIGSEL_MSK, p->msigsel << DMA_CH_SELCON_MSIGSEL_POSS);
    return;
}

/**
  * @brief  Handle DMA interrupt
  * @retval None
  */
void ald_dma_irq_handler(void)
{
    uint32_t i, reg;
    ald_dma_descriptor_t *descr0, *descr1;
    uint32_t p_addr, a_addr;

    reg    = DMA1->IFM;
    p_addr = DMA1->PRI_CH00_SRC_DATA_END_PTR;
    a_addr = DMA1->ALT_CH00_SRC_DATA_END_PTR;

    for (i = 0; i < 6; ++i)
    {
        if (READ_BIT(reg, (1U << i)))
        {
            if (ald_dma_cbk[i].cplt_cbk != NULL)
                ald_dma_cbk[i].cplt_cbk(ald_dma_cbk[i].cplt_arg);

            DMA1->ICR = (1U << i);
            descr0     = (ald_dma_descriptor_t *)(p_addr) + i;
            descr1     = (ald_dma_descriptor_t *)(a_addr) + i;

            if ((descr0->ctrl.md_dma_config_typedef.cycle_ctrl == ALD_DMA_CYCLE_CTRL_NONE)
                    && (descr1->ctrl.md_dma_config_typedef.cycle_ctrl == ALD_DMA_CYCLE_CTRL_NONE))
                DMA1->CHENCLR = (1U << i);
        }
    }

    if (READ_BIT(reg, (1U << 31)))
    {
        DMA1->ICR = (1U << 31);

        for (i = 0; i < 6; ++i)
        {
            if (ald_dma_cbk[i].err_cbk != NULL)
                ald_dma_cbk[i].err_cbk(ald_dma_cbk[i].err_arg);
        }
    }

    return;
}
/**
  * @}
  */

/** @defgroup DMA_Public_Functions DMA Public Functions
  * @{
  */

/** @defgroup DMA_Public_Functions_Group1 Initialization functions
  * @brief Initialization functions
  *
  * @verbatim
    ===================================================================

                    #### Initialization functions ####

    ===================================================================
    [..]
    This subsection provides two functions to Initilizate DMA:
    (+) ald_dma_reset(): Reset the DMA register.

    (+) ald_dma_init(): Initializate the DMA module.
  this function do this:
  (++) Initializte private variable ald_dma_ctrl_base and ald_dma_cbk.
  (++) Reset DMA register.
  (++) Enable DMA module.

    (+) ald_dma_config_struct(): Configure ald_dma_config_t
        structure using default parameter.

    (+) ald_dma_config_sg_alt_desc(): Configure ald_dma_descriptor_t
        structure using specified parameter. This function used
  in scatter-gather mode(memory or peripheral).

    @endverbatim
  * @{
  */

/**
  * @brief  Reset the DMA register
  * @param  DMAx: Pointer to DMA peripheral
  * @retval None
  */
void ald_dma_reset(DMA_TypeDef *DMAx)
{
    assert_param(ALD_IS_DMA(DMAx));

    WRITE_REG(DMAx->CFG, 0x0);
    WRITE_REG(DMAx->CHUSEBURSTCLR, 0xFFF);
    WRITE_REG(DMAx->CHREQMASKCLR, 0xFFF);
    WRITE_REG(DMAx->CHENCLR, 0xFFF);
    WRITE_REG(DMAx->CHPRIALTCLR, 0xFFF);
    WRITE_REG(DMAx->CHPRCLR, 0xFFF);
    WRITE_REG(DMAx->ERRCLR, 0x1);
    WRITE_REG(DMAx->IER, 0x0);
    WRITE_REG(DMAx->ICR, 0x8000003F);

    WRITE_REG(DMAx->CH0_SELCON, 0x0);
    WRITE_REG(DMAx->CH1_SELCON, 0x0);
    WRITE_REG(DMAx->CH2_SELCON, 0x0);
    WRITE_REG(DMAx->CH3_SELCON, 0x0);
    WRITE_REG(DMAx->CH4_SELCON, 0x0);
    WRITE_REG(DMAx->CH5_SELCON, 0x0);
    return;
}

/**
  * @brief  DMA module initialization, this function
  * @param  DMAx: Pointer to DMA peripheral
  * @retval None
  */
void ald_dma_init(DMA_TypeDef *DMAx)
{
    assert_param(ALD_IS_DMA(DMAx));

    memset(ald_dma_ctrl_base, 0x0, sizeof(ald_dma_ctrl_base));
    memset(ald_dma_cbk, 0x0, sizeof(ald_dma_cbk));

    ald_dma_reset(DMAx);

    SET_BIT(RCU->AHBRST, RCU_AHBRST_DMA1EN);
    CLEAR_BIT(RCU->AHBRST, RCU_AHBRST_DMA1EN);
    SET_BIT(DMAx->CFG, DMA_CFG_MASTER_ENABLE);

    return;
}

/**
  * @brief  Configure ald_dma_config_t structure using default parameter.
  *         User can invoked this function, before configure ald_dma_config_t
  * @param  p: Pointer to ald_dma_config_t structure, see @ref ald_dma_config_t
  * @retval None
  */
void ald_dma_config_struct(ald_dma_config_t *p)
{
    p->data_width = ALD_DMA_DATA_SIZE_BYTE;
    p->src_inc    = ALD_DMA_DATA_INC_BYTE;
    p->dst_inc    = ALD_DMA_DATA_INC_BYTE;
    p->R_power    = ALD_DMA_R_POWER_1;
    p->primary    = ENABLE;
    p->burst      = DISABLE;
    p->high_prio  = DISABLE;
    p->interrupt  = ENABLE;

    return;
}

/**
  * @brief  Configure ald_dma_descriptor_t structure using specified parameter.
  * @note   This function used in scatter-gather mode(memory or peripheral).
  * @param  desc: Address of the alternate descriptor.
  * @param  config: Pointer to the ald_dma_config_t structure.
  * @param  memory: Memory or peripheral scatter-gather.
  * @retval None
  */
void ald_dma_config_sg_alt_desc(ald_dma_descriptor_t *desc, ald_dma_config_t *config, uint8_t memory)
{
    if ((desc == NULL) || (config == NULL))
        return;

    if (config->src_inc == ALD_DMA_DATA_INC_NONE)
    {
        ald_dma_address_inject(&desc->src, config->src);
    }
    else
    {
        ald_dma_address_inject(&desc->src, (void *)((uint32_t)config->src + ((((config->size - 1)) & 0x3FFUL) << config->data_width)));
    }

    if (config->dst_inc == ALD_DMA_DATA_INC_NONE)
        ald_dma_address_inject(&desc->dst, config->dst);
    else
        ald_dma_address_inject(&desc->dst, (void *)((uint32_t)config->dst + ((((config->size - 1)) & 0x3FFUL) << config->data_width)));



    ald_dma_config_inject(&desc->ctrl.word, ((memory ? ALD_DMA_CYCLE_CTRL_MEM_SG_ALTERNATE : ALD_DMA_CYCLE_CTRL_PER_SG_ALTERNATE) | ((memory ? 0 : 1) << 3) | (((config->size - 1) & 0x3FFUL) << 4) | ((config->R_power & 0xFUL) << 14) | ((config->data_width & 0x3UL) << 24) | ((config->src_inc & 0x3UL) << 26) | ((config->data_width & 0x3UL) << 28) | ((config->dst_inc & 0x3UL) << 30)));

    return;
}

/**
  * @}
  */

/** @defgroup DMA_Public_Functions_Group2 Configure DMA channel functions
  * @brief Configure DMA channel functions
  *
  * @verbatim
    ===================================================================

                    #### Configure DMA channel functions ####

    ===================================================================
    [..]
    This subsection provides some functions allowing to configure
    DMA channel. Include two type DMA transfer:
    (+) Carry data from memory to memory, this mode APIs are:
        (++) ald_dma_config_auto(): Configure DMA channel according to
       the specified parameter in the ald_dma_handle_t structure.
  (++) ald_dma_restart_auto(): Restart DMA transmitted.
        (++) ald_dma_config_auto_easy(): Configure DMA channel according
       to the specified parameter. If you want use the dma easily,
             you can invoke this function.
        (++) ald_dma_config_sg_mem(): Carry data used scatter-gather mode.
    (+) Carry data from peripheral to memory or from memory to peripheral,
        this mode APIs are:
  (++) ald_dma_config_basic(): Configure DMA channel according to
       the specified parameter in the ald_dma_handle_t structure.
  (++) ald_dma_restart_basic(): Restart DMA transmitted.
        (++) ald_dma_config_basic_easy(): Configure DMA channel according
       to the specified parameter. If you want use the dma easily,
             you can invoke this function.
        (++) ald_dma_ping_pong(): Carry data used ping-pong mode.
        (++) ald_dma_config_sg_per(): Carry data used scatter-gather mode.

    @endverbatim
  * @{
  */

/**
  * @brief  Configure DMA channel according to the specified parameter
  *         in the ald_dma_handle_t structure. The DMA mode is automatic.
  *         This mode is used to carry data from memory to memory.
  * @param  hperh: Pointer to ald_dma_handle_t structure that contains
  *               configuration information for specified DMA channel.
  * @retval None
  */
void ald_dma_config_auto(ald_dma_handle_t *hperh)
{
    ald_dma_cbk[hperh->config.channel].cplt_cbk = hperh->cplt_cbk;
    ald_dma_cbk[hperh->config.channel].cplt_arg = hperh->cplt_arg;
    ald_dma_cbk[hperh->config.channel].cplt_arg = hperh->cplt_arg;
    ald_dma_cbk[hperh->config.channel].err_arg  = hperh->err_arg;

    ald_dma_config_base(hperh->perh, ALD_DMA_CYCLE_CTRL_AUTO, &hperh->config);

    ald_dma_clear_flag_status(hperh->perh, hperh->config.channel);
    WRITE_REG(hperh->perh->CHENSET, (1 << hperh->config.channel));
    SET_BIT(hperh->perh->CHSWREQ, (1 << hperh->config.channel));

    return;
}

/**
  * @brief  Restart DMA transmitted. The DMA mode is automatic.
  *         The other parameters have not changed except 'size' and 'addr'.
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *               configuration information for specified DMA channel.
  * @param  src: Source data begin pointer
  * @param  dst: Destination data begin pointer
  * @param  size: Size.
  * @retval None
  */
void ald_dma_restart_auto(ald_dma_handle_t *hperh, void *src, void *dst, uint16_t size)
{
    ald_dma_descriptor_t *descr;

    if (hperh->config.primary)
        descr = (ald_dma_descriptor_t *)((DMA1_BASE + 0x100) + (hperh->config.channel * 0x10));
    else
        descr = (ald_dma_descriptor_t *)((DMA1_BASE + 0x200) + (hperh->config.channel * 0x10));

    if (src)
    {
        if (hperh->config.src_inc == ALD_DMA_DATA_INC_NONE)
            ald_dma_address_inject(&descr->src, src);
        else
            ald_dma_address_inject(&descr->src, (void *)((uint32_t)src + (((size - 1) & 0x3FFUL) << hperh->config.data_width)));
    }

    if (dst)
    {
        if (hperh->config.dst_inc == ALD_DMA_DATA_INC_NONE)
            ald_dma_address_inject(&descr->dst, dst);
        else
            ald_dma_address_inject(&descr->dst, (void *)((uint32_t)dst + (((size - 1) & 0x3FFUL) << hperh->config.data_width)));
    }

    ald_dma_clear_flag_status(hperh->perh, hperh->config.channel);
    ald_dma_config_inject(&descr->ctrl.word, ((ALD_DMA_CYCLE_CTRL_AUTO) | ((0UL) << 3) | (((size - 1) & 0x3FFUL) << 4) | ((0UL) << 14) | ((0UL) << 24) | ((0UL) << 26) | ((0UL) << 28) | ((0UL) << 30)));
    WRITE_REG(hperh->perh->CHENSET, (1 << hperh->config.channel));
    SET_BIT(hperh->perh->CHSWREQ, (1 << hperh->config.channel));
    return;
}



/**
  * @brief  Configure DMA channel according to the specified parameter.
  *         The DMA mode is automatic. This mode is used to carry data
  *         from memory to memory. If User want use the dma easily,
  *         they can invoke this function.
  * @param  DMAx: Pointer to DMA peripheral
  * @param  src: Source data begin pointer
  * @param  dst: Destination data begin pointer
  * @param  size: The total number of DMA transfers that DMA cycle contains
  * @param  channel: Channel index which will be used.
  * @param  cbk: DMA complete callback function
  *
  * @retval None
  */
void ald_dma_config_auto_easy(DMA_TypeDef *DMAx, void *src, void *dst,
                              uint16_t size, uint8_t channel, void (*cbk)(void *arg))
{
    ald_dma_handle_t hperh;

    assert_param(ALD_IS_DMA(DMAx));

    ald_dma_config_struct(&hperh.config);
    hperh.config.src     = src;
    hperh.config.dst     = dst;
    hperh.config.size    = size;
    hperh.config.msigsel = ALD_DMA_MSIGSEL_MEMORY;
    hperh.config.channel = channel;

    hperh.perh     = DMAx;
    hperh.cplt_cbk = cbk;
    hperh.cplt_arg = NULL;
    hperh.err_cbk  = NULL;

    ald_dma_clear_flag_status(DMAx, channel);
    ald_dma_config_auto(&hperh);

    return;
}

/**
  * @brief  Configure DMA channel according to the specified parameter
  *         in the dma_handle_t structure. The DMA mode is basic.
  *         This mode is used to carry data from peripheral to memory
  *         or from memory to peripheral.
  * @param  hperh: Pointer to ald_dma_handle_t structure that contains
  *         configuration information for specified DMA channel.
  * @retval None
  */
void ald_dma_config_basic(ald_dma_handle_t *hperh)
{
    ald_dma_cbk[hperh->config.channel].cplt_cbk = hperh->cplt_cbk;
    ald_dma_cbk[hperh->config.channel].cplt_arg = hperh->cplt_arg;
    ald_dma_cbk[hperh->config.channel].cplt_arg = hperh->cplt_arg;
    ald_dma_cbk[hperh->config.channel].err_arg  = hperh->err_arg;

    ald_dma_clear_flag_status(hperh->perh, hperh->config.channel);
    ald_dma_config_base(hperh->perh, ALD_DMA_CYCLE_CTRL_BASIC, &hperh->config);
    WRITE_REG(hperh->perh->CHENSET, (1 << hperh->config.channel));

    return;
}

/**
  * @brief  Restart DMA transmitted. The DMA mode is basic.
  *         The other parameters have not changed except 'size' and 'addr'.
  * @param  hperh: Pointer to ald_dma_handle_t structure that contains
  *               configuration information for specified DMA channel.
  * @param  src: Source data begin pointer
  * @param  dst: Destination data begin pointer
  * @param  size: Size.
  * @retval None
  */
void ald_dma_restart_basic(ald_dma_handle_t *hperh, void *src, void *dst, uint16_t size)
{
    ald_dma_descriptor_t *descr;

    if (hperh->config.primary)
        descr = (ald_dma_descriptor_t *)((DMA1_BASE + 0x100) + (hperh->config.channel * 0x10));
    else
        descr = (ald_dma_descriptor_t *)((DMA1_BASE + 0x200) + (hperh->config.channel * 0x10));

    if (src)
    {
        if (hperh->config.src_inc == ALD_DMA_DATA_INC_NONE)
            ald_dma_address_inject(&descr->src, src);
        else
            ald_dma_address_inject(&descr->src, (void *)((uint32_t)src + (((size - 1) & 0x3FFUL) << hperh->config.data_width)));
    }

    if (dst)
    {
        if (hperh->config.dst_inc == ALD_DMA_DATA_INC_NONE)
            ald_dma_address_inject(&descr->dst, dst);
        else
            ald_dma_address_inject(&descr->dst, (void *)((uint32_t)dst + (((size - 1) & 0x3FFUL) << hperh->config.data_width)));
    }

    ald_dma_clear_flag_status(hperh->perh, hperh->config.channel);
    ald_dma_config_inject(&descr->ctrl.word, ((ALD_DMA_CYCLE_CTRL_BASIC) | ((0UL) << 3) | (((size - 1) & 0x3FFUL) << 4) | ((0UL) << 14) | ((0UL) << 24) | ((0UL) << 26) | ((0UL) << 28) | ((0UL) << 30)));
    WRITE_REG(hperh->perh->CHENSET, (1 << hperh->config.channel));

    return;
}

/**
  * @brief  Configure DMA channel according to the specified parameter.
  *         The DMA mode is basic. This mode is used to carry data
  *         from peripheral to memory or negative direction. If user want
  *         use the dma easily, they can invoke this function.
  * @param  DMAx: Pointer to DMA peripheral
  * @param  src: Source data begin pointer
  * @param  dst: Destination data begin pointer
  * @param  size: The total number of DMA transfers that DMA cycle contains
  * @param  msigsel: Input signal to DMA channel @ref ald_dma_msigsel_t
  * @param  channel: Channel index which will be used
  * @param  cbk: DMA complete callback function
  *
  * @retval None
  *
 */
void ald_dma_config_basic_easy(DMA_TypeDef *DMAx, void *src, void *dst, uint16_t size,
                               ald_dma_msigsel_t msigsel, uint8_t channel, void (*cbk)(void *arg))
{
    ald_dma_handle_t hperh;

    assert_param(ALD_IS_DMA(DMAx));
    ald_dma_config_struct(&hperh.config);

    // if (((uint32_t)src) >= 0x40000000)
    // hperh.config.src_inc = DMA_DATA_INC_NONE;

    // if (((uint32_t)dst) >= 0x40000000)
    // hperh.config.dst_inc = DMA_DATA_INC_NONE;

    hperh.config.src     = src;
    hperh.config.dst     = dst;
    hperh.config.size    = size;
    hperh.config.msigsel = msigsel;
    hperh.config.channel = channel;

    hperh.perh     = DMAx;
    hperh.cplt_cbk = cbk;
    hperh.cplt_arg = NULL;
    hperh.err_cbk  = NULL;

    ald_dma_clear_flag_status(DMAx, channel);
    ald_dma_config_basic(&hperh);

    return;
}

/**
  * @brief  Configure DMA channel according to the specified parameter.
  *         The DMA mode is ping-pong.
  * @note   The ping-pong mode does not support memory to memory.
  * @param  DMAx: Pointer to DMA peripheral.
  * @param  config: Pointer to the ald_dma_config_t structure which contains
  *         the specified parameters.
  * @param  first: Whether it is the first transmission. 1-first, 0-not first.
  * @param  cbk: DMA complete callback function.
  * @retval None
  */
void ald_dma_config_ping_pong(DMA_TypeDef *DMAx, ald_dma_config_t *config,
                              uint8_t first, void (*cbk)(void *arg))
{
    ald_dma_descriptor_t *desc;
    uint32_t *ptr;

    assert_param(ALD_IS_DMA(DMAx));
    assert_param(config->src != NULL);
    assert_param(config->dst != NULL);
    assert_param(ALD_IS_DMA_DATA_SIZE(config->size));
    assert_param(ALD_IS_DMA_DATASIZE_TYPE(config->data_width));
    assert_param(ALD_IS_DMA_DATAINC_TYPE(config->src_inc));
    assert_param(ALD_IS_DMA_DATAINC_TYPE(config->dst_inc));
    assert_param(ALD_IS_DMA_ARBITERCONFIG_TYPE(config->R_power));
    assert_param(ALD_IS_DMA_CHANNEL(config->channel));

    ald_dma_cbk[config->channel].cplt_cbk = cbk;
    ald_dma_cbk[config->channel].cplt_arg = NULL;
    ald_dma_cbk[config->channel].cplt_arg = NULL;
    ald_dma_cbk[config->channel].err_arg  = NULL;

    if (config->primary)
        desc = (ald_dma_descriptor_t *)((DMA1_BASE + 0x100) + (config->channel * 0x10));
    else
        desc = (ald_dma_descriptor_t *)((DMA1_BASE + 0x200) + (config->channel * 0x10));

    if (config->src_inc == ALD_DMA_DATA_INC_NONE)
        ald_dma_address_inject(&desc->src, config->src);
    else
        ald_dma_address_inject(&desc->src, (void *)((uint32_t)config->src + (((config->size - 1) & 0x3FFUL) << config->data_width)));

    if (config->dst_inc == ALD_DMA_DATA_INC_NONE)
        ald_dma_address_inject(&desc->dst, config->dst);
    else
        ald_dma_address_inject(&desc->dst, (void *)((uint32_t)config->dst + (((config->size - 1) & 0x3FFUL) << config->data_width)));

    ald_dma_config_inject(&desc->ctrl.word, ((ALD_DMA_CYCLE_CTRL_PINGPONG) | ((0UL) << 3) | (((config->size - 1) & 0x3FFUL) << 4) | ((config->R_power & 0xFUL) << 14) | ((config->data_width & 0x3UL) << 24) | ((config->src_inc & 0x3UL) << 26) | ((config->data_width & 0x3UL) << 28) | ((config->dst_inc & 0x3UL) << 30)));



    if (config->burst)
        WRITE_REG(DMAx->CHUSEBURSTSET, (1 << config->channel));
    else
        WRITE_REG(DMAx->CHUSEBURSTCLR, (1 << config->channel));

    if (config->high_prio)
        WRITE_REG(DMAx->CHPRSET, (1 << config->channel));
    else
        WRITE_REG(DMAx->CHPRCLR, (1 << config->channel));

    if (config->interrupt)
        SET_BIT(DMAx->IER, (1 << config->channel));
    else
        SET_BIT(DMAx->IDR, (1 << config->channel));

    ptr = (uint32_t *)(DMA1_BASE + 0x70 + (config->channel << 2));
    MODIFY_REG(*ptr, DMA_CH_SELCON_MSIGSEL_MSK, config->msigsel << DMA_CH_SELCON_MSIGSEL_POSS);

    WRITE_REG(DMAx->ICR, (1 << config->channel));
    WRITE_REG(DMAx->CHENSET, (1 << config->channel));

    if (!first)
    {
        WRITE_REG(DMAx->CHSWREQ, (1 << config->channel));
//        return;
    }

    return;
}

/**
  * @brief  Configure DMA channel according to the specified parameter.
  *         The DMA mode is memory scatter-gather.
  * @param  DMAx: Pointer to DMA peripheral.
  * @param  desc: Pointer to first alternate descriptor.
  * @param  nr: Number of the alternate descriptor.
  * @param  channel: Channel index which will be used.
  * @param  cbk: DMA complete callback function.
  * @retval None
 */
void ald_dma_config_sg_mem(DMA_TypeDef *DMAx, ald_dma_descriptor_t *desc, uint32_t nr,
                           uint8_t channel, void (*cbk)(void *arg))
{
    ald_dma_descriptor_t *tmp  = (ald_dma_descriptor_t *)((DMA1_BASE + 0x100) + (channel * 0x10));
    ald_dma_descriptor_t *_tmp = (ald_dma_descriptor_t *)((DMA1_BASE + 0x200) + (channel * 0x10));

    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_CHANNEL(channel));

    if ((desc == NULL) || (nr == 0))
        return;

    ald_dma_cbk[channel].cplt_cbk = cbk;
    ald_dma_cbk[channel].cplt_arg = NULL;
    ald_dma_cbk[channel].cplt_arg = NULL;
    ald_dma_cbk[channel].err_arg  = NULL;

    ald_dma_address_inject(&tmp->src, (void *)((uint32_t)desc + ((((nr << 2) - 1) & 0x3FFUL) << ALD_DMA_DATA_INC_WORD)));
    ald_dma_address_inject(&tmp->dst, (void *)((uint32_t)_tmp + ((4 - 1) << ALD_DMA_DATA_INC_WORD)));


    ald_dma_config_inject(&tmp->ctrl.word, ((uint32_t)(ALD_DMA_CYCLE_CTRL_MEM_SG_PRIMARY) | ((0UL) << 3) | ((((nr << 2) - 1) & 0x3FFUL) << 4) | ((uint32_t)(ALD_DMA_R_POWER_4) << 14) | ((uint32_t)(ALD_DMA_DATA_SIZE_WORD) << 24) | ((uint32_t)(ALD_DMA_DATA_INC_WORD) << 26) | ((uint32_t)(ALD_DMA_DATA_SIZE_WORD) << 28) | ((uint32_t)(ALD_DMA_DATA_INC_WORD) << 30)));

    ald_dma_config_inject(&desc[nr - 1].ctrl.word, ((ALD_DMA_CYCLE_CTRL_AUTO) | ((0UL) << 3) | ((0UL) << 4) | ((0UL) << 14) | ((0UL) << 24) | ((0UL) << 26) | ((0UL) << 28) | ((0UL) << 30)));

    WRITE_REG(DMAx->CHPRIALTCLR, (1 << channel));
    WRITE_REG(DMAx->CHUSEBURSTCLR, (1 << channel));
    WRITE_REG(DMAx->CHPRCLR, (1 << channel));

    WRITE_REG(DMAx->ICR, (1 << channel));
    SET_BIT(DMAx->IER, (1 << channel));
    WRITE_REG(DMAx->CHENSET, (1 << channel));
    SET_BIT(DMAx->CHSWREQ, (1 << channel));

    return;
}

/**
  * @brief  Configure DMA channel according to the specified parameter.
  *         The DMA mode is peripheral scatter-gather.
  * @note   The size of the first transmission must be 5.
  * @param  DMAx: Pointer to DMA peripheral.
  * @param  desc: Pointer to first alternate descriptor.
  * @param  nr: Number of the alternate descriptor.
  * @param  msel: Input source to DMA channel @ref ald_dma_msel_t
  * @param  msigsel: Input signal to DMA channel @ref ald_dma_msigsel_t
  * @param  channel: Channel index which will be used.
  * @param  cbk: DMA complete callback function.
  * @retval None
 */
void ald_dma_config_sg_per(DMA_TypeDef *DMAx, ald_dma_descriptor_t *desc, uint32_t nr,
                           ald_dma_msigsel_t msigsel, uint8_t channel, void (*cbk)(void *arg))
{
    ald_dma_descriptor_t *tmp  = (ald_dma_descriptor_t *)((DMA1_BASE + 0x100) + (channel * 0x10));
    ald_dma_descriptor_t *_tmp = (ald_dma_descriptor_t *)((DMA1_BASE + 0x200) + (channel * 0x10));
    uint32_t *ptr;

    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_CHANNEL(channel));

    if ((desc == NULL) || (nr == 0))
        return;

    ald_dma_cbk[channel].cplt_cbk = cbk;
    ald_dma_cbk[channel].cplt_arg = NULL;
    ald_dma_cbk[channel].cplt_arg = NULL;
    ald_dma_cbk[channel].err_arg  = NULL;

    ald_dma_address_inject(&tmp->src, (void *)((uint32_t)desc + ((((nr << 2) - 1) & 0x3FFUL) << ALD_DMA_DATA_INC_WORD)));
    ald_dma_address_inject(&tmp->dst, (void *)((uint32_t)_tmp + ((4 - 1) << ALD_DMA_DATA_INC_WORD)));

    ald_dma_config_inject(&tmp->ctrl.word, ((uint32_t)(ALD_DMA_CYCLE_CTRL_PER_SG_PRIMARY) | ((0UL) << 3) | ((((nr << 2) - 1) & 0x3FFUL) << 4) | ((uint32_t)(ALD_DMA_R_POWER_4) << 14) | ((uint32_t)(ALD_DMA_DATA_SIZE_WORD) << 24) | ((uint32_t)(ALD_DMA_DATA_INC_WORD) << 26) | ((uint32_t)(ALD_DMA_DATA_SIZE_WORD) << 28) | ((uint32_t)(ALD_DMA_DATA_INC_WORD) << 30)));

    ald_dma_config_inject(&desc[nr - 1].ctrl.word, ((ALD_DMA_CYCLE_CTRL_BASIC) | ((0UL) << 3) | ((0UL) << 4) | ((0UL) << 14) | ((0UL) << 24) | ((0UL) << 26) | ((0UL) << 28) | ((0UL) << 30)));


    WRITE_REG(DMAx->CHPRIALTCLR, (1 << channel));
    WRITE_REG(DMAx->CHUSEBURSTCLR, (1 << channel));
    WRITE_REG(DMAx->CHPRCLR, (1 << channel));

    ptr = (uint32_t *)(DMA1_BASE + 0x70 + (channel << 2));
    MODIFY_REG(*ptr, DMA_CH_SELCON_MSIGSEL_MSK, msigsel << DMA_CH_SELCON_MSIGSEL_POSS);

    WRITE_REG(DMAx->ICR, (1 << channel));
    SET_BIT(DMAx->IER, (1 << channel));
    WRITE_REG(DMAx->CHENSET, (1 << channel));

    return;
}
/**
  * @}
  */

/** @defgroup DMA_Public_Functions_Group3 DMA Control functions
  * @brief DMA control functions
  *
  * @verbatim
    ===================================================================

                    #### DMA control functions ####

    ===================================================================
    [..]
    This subsection provides some functions allowing to control DMA:
    (+) ald_dma_channel_config(): Control DMA channel ENABLE/DISABLE.
    (+) ald_dma_interrupt_config(): Control DMA channel interrupt ENABLE or
        DISABLE.
    (+) ald_dma_get_it_status(): Check whether the specified channel
        interrupt is SET or RESET.
    (+) ald_dma_get_flag_status(): Check whether the specified channel
        flag is SET or RESET.
    (+) ald_dma_clear_flag_status(): Clear the specified channel
        pending flag

    @endverbatim
  * @{
  */

/**
  * @brief  Configure channel enable or disable. It will unbind descriptor with
  *         channel, when channel has been disable.
  * @param  DMAx: Pointer to DMA peripheral
  * @param  channel: channel index
  * @param  state: status of channel:
  *           @arg ENABLE: Enable the channel
  *           @arg DISABLE: Disable the channel
  * @retval None
  */
void ald_dma_channel_config(DMA_TypeDef *DMAx, uint8_t channel, TypeFunc state)
{
    ald_dma_descriptor_t *descr, *alt_descr;
    uint32_t *ptr;

    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_CHANNEL(channel));

    descr     = (ald_dma_descriptor_t *)((DMA1_BASE + 0x100) + (channel * 0x10));
    alt_descr = (ald_dma_descriptor_t *)((DMA1_BASE + 0x200) + (channel * 0x10));

    if (state)
    {
        WRITE_REG(DMAx->CHENSET, (1 << channel));
    }
    else
    {
        memset(descr, 0x00, sizeof(ald_dma_descriptor_t));
        memset(alt_descr, 0x00, sizeof(ald_dma_descriptor_t));
        ptr = (uint32_t *)(DMA1_BASE + 0x70 + (channel << 2));
        MODIFY_REG(*ptr, DMA_CH_SELCON_MSIGSEL_MSK, 0 << DMA_CH_SELCON_MSIGSEL_POSS);
        // WRITE_REG(DMAx->CH_SELCON[channel], 0x0);
        WRITE_REG(DMAx->CHENCLR, (1 << channel));
    }

    return;
}

/**
  * @brief  Configure the interrupt enable or disable
  * @param  DMAx: Pointer to DMA peripheral
  * @param  channel: Channel index.
  *           @arg 0~5: Channel index
  * @param  state: status of channel:
  *           @arg ENABLE: Enable the channel
  *           @arg DISABLE: Disable the channel
  *
  * @retval None
  */
void ald_dma_interrupt_config(DMA_TypeDef *DMAx, uint8_t channel, TypeFunc state)
{
    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_IT_TYPE(channel));

    if (state)
        SET_BIT(DMAx->IER, (1 << channel));
    else
        SET_BIT(DMAx->IDR, (1 << channel));

    return;
}

/**
  * @brief  Check whether the specified channel interrupt
  *         is set or reset
  * @param  DMAx: Pointer to DMA peripheral
  * @param  channel: Channel index
  *           @arg 0~5: Channel index
  * @retval Status:
  *           - SET: Channel interrupt is set
  *           - RESET: Channel interrupt is reset
  */
FlagStatus ald_dma_get_it_status(DMA_TypeDef *DMAx, uint8_t channel)
{
    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_IT_TYPE(channel));

    if (READ_BIT(DMAx->IFM, (1 << channel)))
        return SET;

    return RESET;
}

/**
  * @brief  Check whether the specified channel flag
  *         is set or reset
  * @param  DMAx: Pointer to DMA peripheral
  * @param  channel: Channel index
  *           @arg 0~5: Channel index
  * @retval Status:
  *           - SET: Channel flag is set
  *           - RESET: Channel flag is reset
  */
FlagStatus ald_dma_get_flag_status(DMA_TypeDef *DMAx, uint8_t channel)
{
    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_IT_TYPE(channel));

    if (READ_BIT(DMAx->RIF, (1 << channel)))
        return SET;

    return RESET;
}

/**
  * @brief  Clear the specified channel pending flag
  * @param  DMAx: Pointer to DMA peripheral
  * @param  channel: Channel index
  *           @arg 0~5: Channel index
  * @retval None
  */
void ald_dma_clear_flag_status(DMA_TypeDef *DMAx, uint8_t channel)
{
    assert_param(ALD_IS_DMA(DMAx));
    assert_param(ALD_IS_DMA_IT_TYPE(channel));

    WRITE_REG(DMAx->ICR, (1 << channel));
    return;
}

/**
  * @}
  */

/**
  * @}
  */

#endif
/**
  * @} DMA
  */

/**
  * @} ALD
  */
/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
