/**********************************************************************************
 *
 * @file    ald_dma.h
 * @brief   header file of ald_dma.c
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 June 2021    Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

#ifndef __ALD_DMA_H__
#define __ALD_DMA_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "utils.h"


/** @addtogroup ALD
  * @{
  */

/** @addtogroup DMA DMA
  * @brief DMA module driver
  * @{
  */

/**
  * @defgroup DMA_Public_Types DMA Public Types
  * @{
  */

/* DMA trigger select */
typedef enum
{
    ALD_DMA_MSIGSEL_UART1_TX        = 0,    /**< UART1_TX */
    ALD_DMA_MSIGSEL_UART2_TX        = 1,    /**< UART2_TX */
    ALD_DMA_MSIGSEL_UART3_TX        = 2,    /**< UART3_TX */
    ALD_DMA_MSIGSEL_UART4_TX        = 3,    /**< UART4_TX */
    ALD_DMA_MSIGSEL_SPI1_TX         = 5,    /**< SPI1_TX */
    ALD_DMA_MSIGSEL_I2S1_TX         = 5,    /**< I2S1_TX */
    ALD_DMA_MSIGSEL_SPI2_TX         = 6,    /**< SPI2_TX */
    ALD_DMA_MSIGSEL_I2C1_TX         = 7,    /**< I2C1_TX */
    ALD_DMA_MSIGSEL_I2C2_TX         = 8,    /**< I2C2_TX */
    ALD_DMA_MSIGSEL_AES_IN          = 9,    /**< AES_IN */
    ALD_DMA_MSIGSEL_CRC             = 11,   /**< CRC */
    ALD_DMA_MSIGSEL_SPI3_TX         = 12,   /**< SPI3_TX */
    ALD_DMA_MSIGSEL_UART1_RX        = 15,   /**< UART1_RX */
    ALD_DMA_MSIGSEL_UART2_RX        = 16,   /**< UART2_RX */
    ALD_DMA_MSIGSEL_UART3_RX        = 17,   /**< UART3_RX */
    ALD_DMA_MSIGSEL_UART4_RX        = 18,   /**< UART4_RX */
    ALD_DMA_MSIGSEL_SPI1_RX         = 20,   /**< SPI1_RX */
    ALD_DMA_MSIGSEL_I2S1_RX         = 20,   /**< I2S1_RX */
    ALD_DMA_MSIGSEL_SPI2_RX         = 21,   /**< SPI2_RX */
    ALD_DMA_MSIGSEL_I2C1_RX         = 22,   /**< I2C1_RX */
    ALD_DMA_MSIGSEL_I2C2_RX         = 23,   /**< I2C2_RX */
    ALD_DMA_MSIGSEL_AES_OUT         = 24,   /**< AES_OUT */
    ALD_DMA_MSIGSEL_ADC             = 25,   /**< ADC */
    ALD_DMA_MSIGSEL_SPI3_RX         = 29,   /**< SPI3_RX */
    ALD_DMA_MSIGSEL_KBCU            = 31,   /**< KBCU */
    ALD_DMA_MSIGSEL_BS16T1_UP       = 32,   /**< BS16T1_UP */
    ALD_DMA_MSIGSEL_AD16C4T1_CH1    = 33,   /**< AD16C4T1_CH1 */
    ALD_DMA_MSIGSEL_AD16C4T1_CH2    = 34,   /**< AD16C4T1_CH2 */
    ALD_DMA_MSIGSEL_AD16C4T1_CH3    = 35,   /**< AD16C4T1_CH3 */
    ALD_DMA_MSIGSEL_AD16C4T1_CH4    = 36,   /**< AD16C4T1_CH4 */
    ALD_DMA_MSIGSEL_AD16C4T1_UP     = 37,   /**< AD16C4T1_UP */
    ALD_DMA_MSIGSEL_AD16C4T1_TRGI   = 38,   /**< AD16C4T1_TRGI */
    ALD_DMA_MSIGSEL_AD16C4T1_COM    = 39,   /**< GP32C4T1_COM */
    ALD_DMA_MSIGSEL_GP32C4T1_CH1    = 40,   /**< GP32C4T1_CH1 */
    ALD_DMA_MSIGSEL_GP32C4T1_CH2    = 41,   /**< GP32C4T1_CH2 */
    ALD_DMA_MSIGSEL_GP32C4T1_CH3    = 42,   /**< GP32C4T1_CH3 */
    ALD_DMA_MSIGSEL_GP32C4T1_CH4    = 43,   /**< GP32C4T1_CH4 */
    ALD_DMA_MSIGSEL_GP32C4T1_UP     = 44,   /**< GP32C4T1_UP */
    ALD_DMA_MSIGSEL_GP32C4T1_TRGI   = 45,   /**< GP32C4T1_TRGI */
    ALD_DMA_MSIGSEL_GP16C4T1_CH1    = 46,   /**< GP16C4T1_CH1 */
    ALD_DMA_MSIGSEL_GP16C4T1_CH2    = 47,   /**< GP16C4T1_CH2 */
    ALD_DMA_MSIGSEL_GP16C4T1_CH3    = 48,   /**< GP16C4T1_CH3 */
    ALD_DMA_MSIGSEL_GP16C4T1_CH4    = 49,   /**< GP16C4T1_CH4 */
    ALD_DMA_MSIGSEL_GP16C4T1_UP     = 50,   /**< GP16C4T1_UP */
    ALD_DMA_MSIGSEL_GP16C4T1_TRGI   = 51,   /**< GP16C4T1_TRGI */
    ALD_DMA_MSIGSEL_GP16C4T2_CH1    = 52,   /**< GP16C4T2_CH1 */
    ALD_DMA_MSIGSEL_GP16C4T2_CH2    = 53,   /**< GP16C4T2_CH2 */
    ALD_DMA_MSIGSEL_GP16C4T2_CH3    = 54,   /**< GP16C4T2_CH3 */
    ALD_DMA_MSIGSEL_GP16C4T2_CH4    = 55,   /**< GP16C4T2_CH4 */
    ALD_DMA_MSIGSEL_GP16C4T2_UP     = 56,   /**< GP16C4T2_UP */
    ALD_DMA_MSIGSEL_GP16C4T2_TRGI   = 57,   /**< GP16C4T2_TRGI */
    ALD_DMA_MSIGSEL_GP16C4T3_CH1    = 58,   /**< GP16C4T3_CH1 */
    ALD_DMA_MSIGSEL_GP16C4T3_CH2    = 59,   /**< GP16C4T3_CH2 */
    ALD_DMA_MSIGSEL_GP16C4T3_CH3    = 60,   /**< GP16C4T3_CH3 */
    ALD_DMA_MSIGSEL_GP16C4T3_CH4    = 61,   /**< GP16C4T3_CH4 */
    ALD_DMA_MSIGSEL_GP16C4T3_UP     = 62,   /**< GP16C4T3_UP */
    ALD_DMA_MSIGSEL_GP16C4T3_TRGI   = 63,   /**< GP16C4T3_TRGI */
    ALD_DMA_MSIGSEL_GP16C2T1_CH1    = 64,   /**< GP16C2T1_CH1 */
    ALD_DMA_MSIGSEL_GP16C2T1_CH2    = 65,   /**< GP16C2T1_CH2 */
    ALD_DMA_MSIGSEL_GP16C2T1_UP     = 66,   /**< GP16C2T1_UP */
    ALD_DMA_MSIGSEL_GP16C2T1_TRGI   = 67,   /**< GP16C2T1_TRGI */
    ALD_DMA_MSIGSEL_GP16C2T1_COM    = 68,   /**< GP16C2T1_COM */
    ALD_DMA_MSIGSEL_GP16C2T2_CH1    = 70,   /**< GP16C2T2_CH1 */
    ALD_DMA_MSIGSEL_GP16C2T2_CH2    = 71,   /**< GP16C2T2_CH2 */
    ALD_DMA_MSIGSEL_GP16C2T2_UP     = 72,   /**< GP16C2T2_UP */
    ALD_DMA_MSIGSEL_GP16C2T2_TRGI   = 73,   /**< GP16C2T2_TRGI */
    ALD_DMA_MSIGSEL_GP16C2T2_COM    = 74,   /**< GP16C2T2_COM */
    ALD_DMA_MSIGSEL_GP16C2T3_CH1    = 76,   /**< GP16C2T3_CH1 */
    ALD_DMA_MSIGSEL_GP16C2T3_CH2    = 77,   /**< GP16C2T3_CH2 */
    ALD_DMA_MSIGSEL_GP16C2T3_UP     = 78,   /**< GP16C2T3_UP */
    ALD_DMA_MSIGSEL_GP16C2T3_TRGI   = 79,   /**< GP16C2T3_TRGI */
    ALD_DMA_MSIGSEL_GP16C2T3_COM    = 80,   /**< GP16C2T3_COM */
    ALD_DMA_MSIGSEL_GP16C2T4_CH1    = 82,   /**< GP16C2T4_CH1 */
    ALD_DMA_MSIGSEL_GP16C2T4_CH2    = 83,   /**< GP16C2T4_CH2 */
    ALD_DMA_MSIGSEL_GP16C2T4_UP     = 84,   /**< GP16C2T4_UP */
    ALD_DMA_MSIGSEL_GP16C2T4_TRGI   = 85,   /**< GP16C2T4_TRGI */
    ALD_DMA_MSIGSEL_GP16C2T4_COM    = 86,   /**< GP16C2T4_COM */
    ALD_DMA_MSIGSEL_MEMORY          = 127,  /**< MEMORY */
} ald_dma_msigsel_t;


/**
  * @brief DMA Descriptor control type
  */
typedef union
{
    struct
    {
        uint32_t cycle_ctrl    : 3; /**< DMA operating mode @ref ald_dma_cycle_ctrl_t */
        uint32_t next_useburst : 1; /**< Uses the alternate data structure when complete a DMA cycle */
        uint32_t n_minus_1     : 10; /**< Represent the total number of DMA transfers that DMA cycle contains. */
        uint32_t R_power       : 4; /**< Control how many DMA transfers can occur before re-arbitrates. @ref ald_dma_arbiter_config_t */
        uint32_t RESERVED1     : 3; /**< Reserved */
        uint32_t RESERVED2     : 3; /**< Reserved */
        uint32_t src_size      : 2; /**< Source data size @ref ald_dma_data_size_t */
        uint32_t src_inc       : 2; /**< Source address increment. @ref ald_dma_data_inc_t */
        uint32_t dst_size      : 2; /**< Destination data size. @ref ald_dma_data_size_t */
        uint32_t dst_inc       : 2; /**< Destination address increment. @ref ald_dma_data_inc_t */
    } md_dma_config_typedef;
    uint32_t word;
} ald_dma_ctrl_t;

/**
  * @brief Channel control data structure
  */
typedef struct
{
    volatile void *src;                /**< Source data end pointer */
    volatile void *dst;                /**< Destination data end pointer */
    ald_dma_ctrl_t ctrl;      /**< Control data configuration @ref ald_dma_ctrl_t */
    uint32_t use;             /**< Reserve for user */
} ald_dma_descriptor_t;

/**
  * @brief data increment
  */
typedef enum
{
    ALD_DMA_DATA_INC_BYTE     = 0x0,  /**< Address increment by byte */
    ALD_DMA_DATA_INC_HALFWORD = 0x1,  /**< Address increment by halfword */
    ALD_DMA_DATA_INC_WORD     = 0x2,  /**< Address increment by word */
    ALD_DMA_DATA_INC_NONE     = 0x3,  /**< No increment */
} ald_dma_data_inc_t;

/**
  * @brief Data size
  */
typedef enum
{
    ALD_DMA_DATA_SIZE_BYTE     = 0x0, /**< Byte */
    ALD_DMA_DATA_SIZE_HALFWORD = 0x1, /**< Halfword */
    ALD_DMA_DATA_SIZE_WORD     = 0x2, /**< Word */
} ald_dma_data_size_t;

/**
  * @brief The operating mode of the DMA cycle
  */
typedef enum
{
    ALD_DMA_CYCLE_CTRL_NONE             = 0x0,  /**< Stop */
    ALD_DMA_CYCLE_CTRL_BASIC            = 0x1,  /**< Basic */
    ALD_DMA_CYCLE_CTRL_AUTO             = 0x2,  /**< Auto-request */
    ALD_DMA_CYCLE_CTRL_PINGPONG         = 0x3,  /**< Ping-pong */
    ALD_DMA_CYCLE_CTRL_MEM_SG_PRIMARY   = 0x4,  /**< Memory scatter-gather using the primary structure */
    ALD_DMA_CYCLE_CTRL_MEM_SG_ALTERNATE = 0x5,  /**< Memory scatter-gather using the alternate structure */
    ALD_DMA_CYCLE_CTRL_PER_SG_PRIMARY   = 0x6,  /**< Peripheral scatter-gather using the primary structure */
    ALD_DMA_CYCLE_CTRL_PER_SG_ALTERNATE = 0x7,  /**< Peripheral scatter-gather using the alternate structure */
} ald_dma_cycle_ctrl_t;

/**
  * @brief Control how many DMA transfers can occur
  *        before the controller re-arbitrates
  */
typedef enum
{
    ALD_DMA_R_POWER_1    = 0x0,   /**< Arbitrates after each DMA transfer */
    ALD_DMA_R_POWER_2    = 0x1,   /**< Arbitrates after 2 DMA transfer */
    ALD_DMA_R_POWER_4    = 0x2,   /**< Arbitrates after 4 DMA transfer */
    ALD_DMA_R_POWER_8    = 0x3,   /**< Arbitrates after 8 DMA transfer */
    ALD_DMA_R_POWER_16   = 0x4,   /**< Arbitrates after 16 DMA transfer */
    ALD_DMA_R_POWER_32   = 0x5,   /**< Arbitrates after 32 DMA transfer */
    ALD_DMA_R_POWER_64   = 0x6,   /**< Arbitrates after 64 DMA transfer */
    ALD_DMA_R_POWER_128  = 0x7,   /**< Arbitrates after 128 DMA transfer */
    ALD_DMA_R_POWER_256  = 0x8,   /**< Arbitrates after 256 DMA transfer */
    ALD_DMA_R_POWER_512  = 0x9,   /**< Arbitrates after 512 DMA transfer */
    ALD_DMA_R_POWER_1024 = 0xA,   /**< Arbitrates after 1024 DMA transfer */
} ald_dma_arbiter_config_t;

/**
  * @brief Callback function pointer and param
  */
typedef struct
{
    void (*cplt_cbk)(void *arg);  /**< DMA transfers complete callback */
    void (*err_cbk)(void *arg);   /**< DMA occurs error callback */
    void *cplt_arg;               /**< The parameter of cplt_cbk() */
    void *err_arg;                /**< The parameter of err_cbk() */
} ald_dma_call_back_t;

/**
  * @brief DMA channal configure structure
  */
typedef struct
{
    void *src;                            /**< Source data begin pointer */
    void *dst;                            /**< Destination data begin pointer */
    uint16_t size;                        /**< The total number of DMA transfers that DMA cycle contains */
    ald_dma_data_size_t data_width;       /**< Data width, @ref ald_dma_data_size_t */
    ald_dma_data_inc_t src_inc;           /**< Source increment type. @ref ald_dma_data_inc_t */
    ald_dma_data_inc_t dst_inc;           /**< Destination increment type. @ref ald_dma_data_inc_t */
    ald_dma_arbiter_config_t R_power;     /**< Control how many DMA transfers can occur before re-arbitrates. @ref ald_dma_arbiter_config_t */
    TypeFunc primary;                     /**< Use primary descriptor or alternate descriptor */
    TypeFunc burst;                       /**< Uses the alternate data structure when complete a DMA cycle */
    TypeFunc high_prio;                   /**< High priority or default priority */
    TypeFunc interrupt;                   /**< Enable/disable interrupt */
    ald_dma_msigsel_t msigsel;            /**< Input signal to DMA channel @ref ald_dma_msigsel_t */
    uint8_t channel;                      /**< Channel index */
} ald_dma_config_t;

/**
  * @brief DMA handle structure definition
  */
typedef struct
{
    DMA_TypeDef *perh;              /**< DMA registers base address */
    ald_dma_config_t config;        /**< Channel configure structure. @ref ald_dma_config_t */
    void (*cplt_cbk)(void *arg);    /**< DMA transfers complete callback */
    void (*err_cbk)(void *arg);     /**< DMA bus occurs error callback */
    void *cplt_arg;                 /**< The parameter of cplt_cbk() */
    void *err_arg;                  /**< The parameter of err_cbk() */
} ald_dma_handle_t;

/**
  * @brief Descriptor complete state
  */
typedef enum
{
    ALD_DMA_DESCP_CPLT_PRI = 0x0U,    /**< Primary descriptor has been completed */
    ALD_DMA_DESCP_CPLT_ALT = 0x1U,    /**< Alternate descriptor has been completed */
    ALD_DMA_DESCP_CPLT_ALL = 0x2U,    /**< Both primary and alternate descriptors have been completed */
} ald_dma_descrp_cplt_t;

/**
  * @}
  */

/**
  * @defgroup DMA_Private_Macros DMA Private Macros
  * @{
  */
#define ALD_IS_DMA_DATAINC_TYPE(x)  (((x) == ALD_DMA_DATA_INC_BYTE)    || \
                                     ((x) == ALD_DMA_DATA_INC_HALFWORD) || \
                                     ((x) == ALD_DMA_DATA_INC_WORD)     || \
                                     ((x) == ALD_DMA_DATA_INC_NONE))
#define ALD_IS_DMA_DATASIZE_TYPE(x)  (((x) == ALD_DMA_DATA_SIZE_BYTE)     || \
                                      ((x) == ALD_DMA_DATA_SIZE_HALFWORD) || \
                                      ((x) == ALD_DMA_DATA_SIZE_WORD))
#define ALD_IS_DMA_CYCLECTRL_TYPE(x)  (((x) == ALD_DMA_CYCLE_CTRL_NONE)             || \
                                       ((x) == ALD_DMA_CYCLE_CTRL_BASIC)            || \
                                       ((x) == ALD_DMA_CYCLE_CTRL_AUTO)             || \
                                       ((x) == ALD_DMA_CYCLE_CTRL_PINGPONG)         || \
                                       ((x) == ALD_DMA_CYCLE_CTRL_MEM_SG_PRIMARY)   || \
                                       ((x) == ALD_DMA_CYCLE_CTRL_MEM_SG_ALTERNATE) || \
                                       ((x) == ALD_DMA_CYCLE_CTRL_PER_SG_PRIMARY)   || \
                                       ((x) == ALD_DMA_CYCLE_CTRL_PER_SG_ALTERNATE))
#define ALD_IS_DMA_ARBITERCONFIG_TYPE(x)  (((x) == ALD_DMA_R_POWER_1)   || \
        ((x) == ALD_DMA_R_POWER_2)   || \
        ((x) == ALD_DMA_R_POWER_4)   || \
        ((x) == ALD_DMA_R_POWER_8)   || \
        ((x) == ALD_DMA_R_POWER_16)  || \
        ((x) == ALD_DMA_R_POWER_32)  || \
        ((x) == ALD_DMA_R_POWER_64)  || \
        ((x) == ALD_DMA_R_POWER_128) || \
        ((x) == ALD_DMA_R_POWER_256) || \
        ((x) == ALD_DMA_R_POWER_512) || \
        ((x) == ALD_DMA_R_POWER_1024))
#define ALD_IS_DMA(x)              ((x) == DMA1)
#define ALD_IS_DMA_CHANNEL(x)      ((x) <= 5)
#define ALD_IS_DMA_DATA_SIZE(x)    ((x) <= 1024)
#define ALD_IS_DMA_IT_TYPE(x)      (((x) <= 5))
/**
  * @}
  */

/**
  * @addtogroup DMA_Public_Functions
  * @{
  */

/** @addtogroup DMA_Public_Functions_Group1
  * @{
  */
/* Initialization functions */
extern void ald_dma_reset(DMA_TypeDef *DMAx);
extern void ald_dma_init(DMA_TypeDef *DMAx);
extern void ald_dma_config_struct(ald_dma_config_t *p);
extern void ald_dma_config_sg_alt_desc(ald_dma_descriptor_t *desc, ald_dma_config_t *config, uint8_t memory);
/**
  * @}
  */


/** @addtogroup DMA_Public_Functions_Group2
  * @{
  */
/* Configure DMA channel functions */
extern void ald_dma_config_auto(ald_dma_handle_t *hperh);
extern void ald_dma_restart_auto(ald_dma_handle_t *hperh, void *src, void *dst, uint16_t size);
extern void ald_dma_config_auto_easy(DMA_TypeDef *DMAx, void *src, void *dst,
                                     uint16_t size, uint8_t channel, void (*cbk)(void *arg));
extern void ald_dma_config_basic(ald_dma_handle_t *hperh);
extern void ald_dma_restart_basic(ald_dma_handle_t *hperh, void *src, void *dst, uint16_t size);
extern void ald_dma_config_basic_easy(DMA_TypeDef *DMAx, void *src, void *dst, uint16_t size,
                                      ald_dma_msigsel_t msigsel, uint8_t channel, void (*cbk)(void *arg));
extern void ald_dma_config_ping_pong(DMA_TypeDef *DMAx, ald_dma_config_t *config,
                                     uint8_t first, void (*cbk)(void *arg));
extern void ald_dma_config_sg_mem(DMA_TypeDef *DMAx, ald_dma_descriptor_t *desc,
                                  uint32_t nr, uint8_t channel, void (*cbk)(void *arg));
extern void ald_dma_config_sg_per(DMA_TypeDef *DMAx, ald_dma_descriptor_t *desc, uint32_t nr,
                                  ald_dma_msigsel_t msigsel, uint8_t channel, void (*cbk)(void *arg));
/**
  * @}
  */

/** @addtogroup DMA_Public_Functions_Group3
  * @{
  */
/* DMA control functions */
extern void ald_dma_channel_config(DMA_TypeDef *DMAx, uint8_t channel, TypeFunc state);
extern void ald_dma_interrupt_config(DMA_TypeDef *DMAx, uint8_t channel, TypeFunc state);
extern FlagStatus ald_dma_get_it_status(DMA_TypeDef *DMAx, uint8_t channel);
extern FlagStatus ald_dma_get_flag_status(DMA_TypeDef *DMAx, uint8_t channel);
extern void ald_dma_clear_flag_status(DMA_TypeDef *DMAx, uint8_t channel);
extern void ald_dma_irq_handler(void);
/**
  * @}
  */

/**
  * @}
  */


/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif
/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
