/*********************************************************
*Copyright (C), 2021, Shanghai Eastsoft Microelectronics Co., Ltd.
*文件名:  main.c
*作  者:  AE Team
*版  本:  V1.00
*日  期:  2022/1/28
*描  述:  FreeRTOS例程，printf打印，TX:PA8
          可调用标准库使用，需要以下三个步骤
          1、keil勾选Use MicroLIB ，2、main.c #include "stdio.h"，3、 #define __PRINTF_USE_UART3__
          2、IAR的Library选择Full
*备  注:  适用于ES8H018x芯片
          本软件仅供学习和演示使用，对用户直接引用代码所带来的风险或后果不承担任何法律责任。
**********************************************************/
#include <stdio.h>
#include <FreeRTOS.h>
#include <task.h>
#include <queue.h>
#include <event_groups.h>
#include <stream_buffer.h>
#include <semphr.h>
#include "main.h"

/*任务句柄*/
static TaskHandle_t g_task_app_handle = NULL;
static TaskHandle_t g_task_1_handle = NULL;
static TaskHandle_t g_task_2_handle = NULL;
static TaskHandle_t g_task_3_handle = NULL;

/* 任务 */
static void task_app(void *arg);
static void task_1(void *arg);
static void task_2(void *arg);
static void task_3(void *arg);

/* 测试数据 */
static SemaphoreHandle_t s_mutex_test;
static uint32_t s_high_pri_cycle = 0U;
static uint32_t s_medium_pri_cycle = 0U;
static uint32_t s_low_pri_cycle = 0U;

int main(void)
{
    int retval;
    SystemHRCSelect(SCU_HRC_16M);       //HRC选择16MHz
    SystemClockConfig();                //配置时钟HRC
    DeviceClockAllEnable();             //打开所有外设时钟

    UARTInit();                         //UART初始化(115200,8,NO,1)
    printf("Startup\r\n");

    retval = xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
    if (pdPASS == retval)
        vTaskStartScheduler();
    else
        return -1;

    /* Should nerver run to here */
    while (1)
    {
    }
}

static void task_app(void *arg)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();
    /* Create mutex */
    s_mutex_test = xSemaphoreCreateMutex();

    if (s_mutex_test != NULL)
        printf("Create mutex OK\r\n");

    /* Create high priority task */
    retval = xTaskCreate(task_1, "HighPriTask", 128, NULL, 5, &g_task_1_handle);

    if (pdPASS == retval)
        printf("Task high priority Created\r\n");

    /* Create medium priority task*/
    retval = xTaskCreate(task_2, "MediumPriTask", 128, NULL, 4, &g_task_2_handle);

    if (pdPASS == retval)
        printf("Task medium priority Created\r\n");

    /* Create low priority task*/
    retval = xTaskCreate(task_3, "LowPriTask", 128, NULL, 3, &g_task_3_handle);

    if (pdPASS == retval)
        printf("Task low priority Created\r\n");

    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}

static void task_1(void *arg)
{
    while (1)
    {
        /* Get mutex */
        if (xSemaphoreTake(s_mutex_test, portMAX_DELAY) != pdTRUE)
        {
            printf("ERR:task_high_pri take fai\r\n");
        }

        printf("Mutex Take:  1\r\n");
        vTaskDelay(1000);

        if (xSemaphoreGive(s_mutex_test) != pdTRUE)
        {
            /* Error */
            printf("ERR:task_high_pri give fai\r\n");
        }

        /* Keep count of the number of cycles this thread has performed */
        s_high_pri_cycle++;

        vTaskSuspend(NULL);
    }
}

static void task_2(void *arg)
{
    while (1)
    {
        if (xSemaphoreTake(s_mutex_test, portMAX_DELAY) == pdTRUE)
        {

            printf("Mutex Take:    2\r\n");
            vTaskDelay(1000);

            if (eTaskGetState(g_task_1_handle) != eSuspended)
            {
                /* Did not expect to execute until the high priority thread was
                suspended. */
                printf("ERR:task_high_pri should be suspended!\r\n");
            }
            else
            {
                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(s_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_medium_pri give fai");
                }

                vTaskSuspend(NULL);
            }
        }
        else
        {
            /* We should not leave the osMutexWait() function
            until the mutex was obtained.*/
            printf("ERR:task_medium_pri cant take fai\r\n");
        }

        /* The High and Medium priority threads should be in lock step */
        if (s_high_pri_cycle != (s_medium_pri_cycle + 1))
        {
            printf("Lock step fail\r\n");
        }

        /* Print a message to indicate medium task is running */
        printf("Task2 Run :    2\r\n");

        /* Keep count of the number of cycles this task has performed so a
        stall can be detected */
        s_medium_pri_cycle++;
        vTaskDelay(500);
    }
}

static void task_3(void *arg)
{
    while (1)
    {
        /* Keep attempting to obtain the mutex.  We should only obtain it when
        the medium-priority thread has suspended itself, which in turn should only
        happen when the high-priority thread is also suspended */
        if (xSemaphoreTake(s_mutex_test, portMAX_DELAY) == pdTRUE)
        {
            printf("Mutex Take:      3\r\n");
            vTaskDelay(1000);

            /* Is the haigh and medium-priority threads suspended? */
            if ((eTaskGetState(g_task_1_handle) != eSuspended) || (eTaskGetState(g_task_2_handle) != eSuspended))
            {
                printf("ERR:task_high_pri and task_medium_pri should be suspended!\r\n");
            }
            else
            {
                /* Keep count of the number of cycles this task has performed
                so a stall can be detected */
                s_low_pri_cycle++;

                /* We can resume the other tasks here even though they have a
                higher priority than the this thread. When they execute they
                will attempt to obtain the mutex but fail because the low-priority
                thread is still the mutex holder.  this thread will then inherit
                the higher priority.  The medium-priority thread will block indefinitely
                when it attempts to obtain the mutex, the high-priority thread will only
                block for a fixed period and an error will be latched if the
                high-priority thread has not returned the mutex by the time this
                fixed period has expired */
                /* If binary semaphore is used, you will see "Task 2 run" before we give
                mutex to high priority task. Otherwise, Task 2 would not run before high
                priority task run*/
                vTaskResume(g_task_1_handle);
                vTaskResume(g_task_2_handle);

                /* The other two tasks should now have executed and no longer
                be suspended */
                if ((eTaskGetState(g_task_1_handle) == eSuspended) || (eTaskGetState(g_task_2_handle) == eSuspended))
                {
                    printf("ERR:task_high_pri and task_medium_pri should NOT be suspended!\r\n");
                }

                /* Give the mutex back before suspending ourselves to allow
                the low priority thread to obtain the mutex */
                if (xSemaphoreGive(s_mutex_test) != pdTRUE)
                {
                    printf("ERR:task_low_pri give fai");
                }
            }
        }
    }
}
