/*********************************************************

*文件名:  uart.c
*作  者:  AE Team
*版  本:  V1.00
*日  期:  2021/11/11
*描  述:  UART模块程序
*备  注:
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
**********************************************************/
#include "uart.h"
#include "string.h"
/*********************************************************
函数名: void UARTInit(void)
描  述: UART初始化子程序
输入值: 无
输出值: 无
返回值: 无
**********************************************************/
void UARTInit(void)
{
	volatile uint8_t BaudConfigStatus;
	
    GPIO_InitStruType y;
    UART_InitStruType uart;
    memset(&y, 0, sizeof(y));
    memset(&uart, 0, sizeof(uart));

    y.GPIO_Signal = GPIO_Pin_Signal_Digital;
    y.GPIO_Func = GPIO_Func_6;
    y.GPIO_Direction = GPIO_Dir_Out;
    y.GPIO_PUEN = GPIO_PUE_Input_Disable;
    y.GPIO_PDEN = GPIO_PDE_Input_Disable;
    y.GPIO_OD = GPIO_ODE_Output_Disable;
    y.GPIO_DS = GPIO_DS_Output_Strong;
    GPIO_Init(GPIOA, GPIO_Pin_8, &y);               //PA8---TxD

    y.GPIO_Signal = GPIO_Pin_Signal_Digital;
    y.GPIO_Func = GPIO_Func_6;
    y.GPIO_Direction = GPIO_Dir_In;
    y.GPIO_PUEN = GPIO_PUE_Input_Enable;
    y.GPIO_PDEN = GPIO_PDE_Input_Disable;
    y.GPIO_OD = GPIO_ODE_Output_Disable;
    y.GPIO_DS = GPIO_DS_Output_Strong;
    GPIO_Init(GPIOA, GPIO_Pin_7, &y);               //PA7---RxD

    uart.UART_StopBits = UART_StopBits_1;          //停止位：1
    uart.UART_TxMode = UART_DataMode_8;            //发送数据格式：8位数据
    uart.UART_TxPolar = UART_Polar_Normal;         //发送端口极性：正常
    uart.UART_RxMode = UART_DataMode_8;            //接收数据格式：8位数据
    uart.UART_RxPolar = UART_Polar_Normal;         //接收端口极性：正常
    uart.UART_BaudRate = 115200;                     //波特率
    uart.UART_ClockSet = UART_Clock_1;             //时钟选择：Pclk
	BaudConfigStatus = UART_Init(UART3, &uart);    //若BaudConfigStatus=ERROR，表示UART波特率配置错误；若BaudConfigStatus=SUCCESS，表示UART波特率配置成功。

    UART_TBIMConfig(UART3, UART_TRBIM_Byte);
    UART_RBIMConfig(UART3, UART_TRBIM_Byte);
    UART_ITConfig(UART3, UART_IT_RB, ENABLE);
    NVIC_Init(NVIC_UART3_IRQn, NVIC_Priority_1, ENABLE);
    UART3_TxEnable();                               //UART3发送使能
    UART3_RxEnable();
}
/*********************************************************
函数名: void UARTTxData(uint8_t *buf, uint8_t n)
描  述: UART发送数据整帧打包
输入值: buf—发送数据缓存
        n—发送数据个数
输出值: 无
返回值: 无
**********************************************************/
void UARTTxData(uint8_t *buf, uint8_t n)
{
    uint8_t i, check_sum;

    buf[0] = HEAD1;             //帧头0x55
    buf[1] = HEAD2;             //帧头0xAA
    buf[2] = SLAVE_ADDR;        //设备地址

    check_sum = 0;

    for (i = 0; i < n - 3; i++)
        check_sum ^= buf[i];    //计算所有字节（包括帧头）的异或和

    buf[n - 3] = check_sum;     //赋值校验和
    buf[n - 2] = TAIL1;         //帧尾0x5A
    buf[n - 1] = TAIL2;         //帧尾0xA5
    UART_ITConfig(UART3, UART_IT_TB, ENABLE);    //使能发送中断，在中断判断并结束发送数据帧
}

/*********************************************************
函数名: uint8_t UARTRxData(uint8_t *buf, uint8_t n)
描  述: UART接收数据整帧解包
输入值: 无
输出值: buf—接收数据缓存
        n—接收数据个数
返回值: 无
**********************************************************/
uint8_t UARTRxData(uint8_t *buf, uint8_t n)
{
    uint8_t i, check_sum;

    if ((buf[0] != HEAD1)
            || (buf[1] != HEAD2)
            || (buf[n - 2] != TAIL1)
            || (buf[n - 1] != TAIL2))    //判断帧头及帧尾有误
        return 1;

    if (buf[2] != SLAVE_ADDR)       //判断从机地址不匹配
        return 1;

    check_sum = 0;

    for (i = 0; i < n - 3; i++)
        check_sum ^= buf[i];        //计算所有字节（包括帧头）的异或和

    if (buf[n - 3] != check_sum)    //判断校验和错误
        return 1;

    return 0;                       //正确解包本帧数据
}

