/*********************************************************

*文件名:  iic.c
*作  者:  AE Team
*版  本:  V1.00
*日  期:  2021/11/4
*描  述:  IIC主机模块程序
          访问EEPROM
*备  注:
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
**********************************************************/
#include "iic.h"
#include "systick.h"
#include "string.h"
uint8_t g_txbuf[255];
uint8_t g_rxbuf[255] ;
uint8_t g_tx_length;
uint8_t g_rx_length;
volatile uint8_t g_tx_count;
volatile uint8_t g_rx_count;

/*********************************************************
函数名: void IIC1MasterInit(void)
描  述: IIC主机初始化子程序
输入值: 无
输出值: 无
返回值: 无
**********************************************************/
void IICMasterInit(void)
{
    GPIO_InitStruType x;
    I2C_InitStruType y;
    memset(&x, 0, sizeof(x));
    memset(&y, 0, sizeof(y));
    x.GPIO_Signal = GPIO_Pin_Signal_Digital;
    x.GPIO_Func = GPIO_Func_3;
    x.GPIO_Direction = GPIO_Dir_Out;
    x.GPIO_PUEN = GPIO_PUE_Input_Enable;
    x.GPIO_PDEN = GPIO_PDE_Input_Disable;
    x.GPIO_OD = GPIO_ODE_Output_Enable;
    x.GPIO_DS = GPIO_DS_Output_Normal;
    GPIO_Init(GPIOA, GPIO_Pin_30, &x);  //SDA
    GPIO_Init(GPIOA, GPIO_Pin_31, &x);  //SCL

    y.I2C_SclOd = I2C_PinMode_OD;
    y.I2C_SdaOd = I2C_PinMode_OD;
    y.I2C_16XSamp = DISABLE;
    y.I2C_Clk = 100000;        //100KHz
    y.I2C_Mode = I2C_Mode_Master;
    y.I2C_AutoStop = DISABLE;
    y.I2C_AutoCall = DISABLE;
    I2C_Init(&y);

    I2C_TBIMConfig(I2C_TRBIM_Word);
    I2C_RBIMConfig(I2C_TRBIM_Byte);
    I2C_RecModeConfig(I2C_RecMode_0);

    I2C_Enable();
}

/*********************************************************
函数名: void IICReadBuf(uint8_t slave_addr , char * buf, char size )
描  述: IIC读数据，连续字节读出(查询模式)
输入值: slave_addr—地址
        size-接收数据大小
输出值: 无
返回值: buf—接收数据缓存
**********************************************************/
void IICReadBuf(uint8_t slave_addr, uint8_t *buf, uint8_t size)
{
    uint32_t time_count = 0x0U;

    I2C_SendAddress(slave_addr, I2C_Mode_Read);
    I2C_SRTrigger();

    while (I2C_GetFlagStatus(I2C_Flag_SR) == RESET && (++time_count < 0xFFF));

    I2C_ClearITPendingBit(I2C_Clr_SR);

    I2C_RecModeConfig(I2C_RecMode_0);

    while (size-- > 1)
    {
        I2C_RDTrigger();
        time_count = 0x0U;

        while (I2C_GetFlagStatus(I2C_Flag_RB) == RESET && (++time_count < 0xFFF));

        *buf++ = I2C_RecByte();
    }

    I2C_RecModeConfig(I2C_RecMode_1);
    I2C_RDTrigger();
    time_count = 0x0U;

    while (I2C_GetFlagStatus(I2C_Flag_RB) == RESET && (++time_count < 0xFFF));

    *buf = I2C_RecByte();
    time_count = 0x0U;

    while (I2C_GetFlagStatus(I2C_Flag_RB) == SET && (++time_count < 0xFFF));

    time_count = 0x0U;

    I2C_SPTrigger();

    while (I2C_GetFlagStatus(I2C_Flag_SP) == RESET && (++time_count < 0xFFF));

    I2C_ClearITPendingBit(I2C_Clr_SP);

    return;
}

/*********************************************************
函数名: IICWriteBuf
描  述: IIC连续写数据
输入值: addr—地址
        buf—发送数据缓存
        size-发送数据大小
输出值: 无
返回值: ERROR/SUCCESS
**********************************************************/
ErrorStatus IICWriteBuf(uint8_t slave_addr, uint8_t *buf, uint8_t size)
{
    uint32_t time_count = 0x0U;

    I2C_ClearITPendingBit(I2C_Clr_NA);

    I2C_SendAddress(slave_addr, I2C_Mode_Write);
    I2C_SRTrigger();

    while (I2C_GetFlagStatus(I2C_Flag_SR) == RESET && (++time_count < 0xFFF));

    I2C_ClearITPendingBit(I2C_Clr_SR);

    while (size-- > 0)
    {
        time_count = 0x0U;

        I2C_ClearITPendingBit(I2C_Clr_TIDLE);  /* 清除发送空闲中断标志 */
        I2C_SendByte(*buf++);

        time_count = 0x0U;

        while ((I2C_GetFlagStatus(I2C_Flag_TIDLE) == RESET) && (++time_count < 0xFFF));

        if (I2C_GetFlagStatus(I2C_Flag_NA) == SET)
        {
            break;
        }
    }

    time_count = 0x0U;
    I2C_SPTrigger();

    while (I2C_GetFlagStatus(I2C_Flag_SP) == RESET && (++time_count < 0xFFF));

    I2C_ClearITPendingBit(I2C_Clr_SP);

    if (I2C_GetFlagStatus(I2C_Flag_NA) == SET)
        return ERROR;

    return SUCCESS;
}
