/***************************************************************
 *文件名：  lib_adc.c
 *作  者：  AE Team
 *版  本：  V1.00
 *日  期：  2021/11/2
 *描  述：  ADC模块库函数
 *备  注：
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************/
#include "lib_adc.h"

/***************************************************************
  函数名：ADC_Init
  描  述：初始化ADC模块
  输入值：初始化配置结构体地址
  输出值：无
  返回值：无
 ***************************************************************/
void ADC_Init(ADC_InitStruType *ADC_InitStruct)
{
#if defined(ES8H0384) || defined(ES8H0364) || defined(ES8H0394) || defined(ES8H0354) ||defined(ES8H0161) || defined(ES8H0163) || defined(ES8H0173) || defined(ES8H0181) || defined(ES8H0183) || defined(ES8H0193) || defined(ES8H0184)
    ADC->VREFCON.VREF_EN = ENABLE;                 //内部VREFP使能
    ADC->VREFCON.IREF_EN = ENABLE;                 //IREF使能位
#elif defined(ES8H0133)

#else
	#error The chip model is not defined
#endif
    ADC->CON0.BIT_SEL = ADC_InitStruct->ADC_BITSEL;//ADC分辨率选择
    ADC->CON1.HSEN = ADC_SPEED_HIGH;               //ADC转换速度：高速
    ADC->CON1.CLKS = ADC_InitStruct->ADC_ClkS;     //ADCCON1:bit3 ADC时钟源选择
    ADC->CON1.CLKDIV = ADC_InitStruct->ADC_ClkDiv; //ADCCON1:bit2-0 ADC时钟源预分频
#if defined(ES8H0384) || defined(ES8H0364) || defined(ES8H0394) || defined(ES8H0354) ||defined(ES8H0161) || defined(ES8H0163) || defined(ES8H0173) || defined(ES8H0181) || defined(ES8H0183) || defined(ES8H0193) || defined(ES8H0184)
    ADC->CON1.VREFP = ADC_InitStruct->ADC_VrefP;   //ADCCON1:bit9-8 ADC正向参考电压选择
#elif defined(ES8H0133)

#else
	#error The chip model is not defined
#endif
    ADC->CON1.SMPS = ADC_InitStruct->ADC_SampS;    //ADCCON1:bit12 ADC采样模式选择
    ADC->CHS.CHS = ADC_InitStruct->ADC_ChS;        //ADCCHS:bit0-3 ADC模拟通道选择
    ADC->CON1.ST = ADC_InitStruct->ST;             //ADCCON1:bit20-16 ADC采样时间选择 ST*2+1个Tadclk
#if defined(ES8H0384) || defined(ES8H0364) || defined(ES8H0394) || defined(ES8H0354) ||defined(ES8H0161) || defined(ES8H0163) || defined(ES8H0173) || defined(ES8H0181) || defined(ES8H0183) || defined(ES8H0193) || defined(ES8H0184)
    ADC->CON1.VREFN = ADC_InitStruct->ADC_VREFN;   //负向参考电压选择
#elif defined(ES8H0133)

#else
	#error The chip model is not defined
#endif
}

/***************************************************************
  函数名：ADC_Set_CH
  描  述：选择ADC模拟通道
  输入值：通道
  输出值：无
  返回值：无
 ***************************************************************/
void ADC_Set_CH(ADC_TYPE_CHS AdcCH)
{
    ADC->CHS.CHS = AdcCH;
}

/***************************************************************
  函数名：ADC_GetConvValue
  描  述：获取ADC转换结果
  输入值：无
  输出值：无
  返回值：转换值
 ***************************************************************/
uint16_t ADC_GetConvValue(void)
{
    if (ADC->CON0.BIT_SEL == ADC_BITSEL_12)
        return ((uint16_t)ADC->DR.DR);
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_10)
        return ((uint16_t)(ADC->DR.DR >> 2));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_8)
        return ((uint16_t)(ADC->DR.DR >> 4));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_6)
        return ((uint16_t)(ADC->DR.DR >> 6));
    else
        return 0;
}

/***************************************************************
  函数名：ADC_GetConvStatus
  描  述：获取ADC转换状态
  输入值：无
  输出值：无
  返回值：RESET(完成)/SET(正在转换)
 ***************************************************************/
FlagStatus ADC_GetConvStatus(void)
{
    FlagStatus bitstatus = RESET;

    /* 检测转换状态寄存器 */
    if ((ADC->CON0.TRIG != (uint32_t)RESET))
        bitstatus = SET;                    //正在转换
    else
        bitstatus = RESET;                  //转换完成

    return  bitstatus;
}

/***************************************************************
  函数名：ADC_ACPConfig
  描  述：ADC 自动比较功能初始化
  输入值：ADC_ACP_InitStruct 自动比较功能初始化结构体
  输出值：无
  返回值：SUCCESS 成功
  ERROR 失败
 ***************************************************************/
ErrorStatus ADC_ACPConfig(ADC_ACP_TypeDef *ADC_ACP_InitStruct)
{
    if (ADC_ACP_InitStruct->ACPC_OVER_TIME > ADC_ACPC_OVFL_TIME_MAX
            || ADC_ACP_InitStruct->ACPC_TIMES > ADC_ACPC_TIMES_MAX
            || ADC_ACP_InitStruct->ACPC_MIN_TARGET > ADC_ACP_MIN_MAX
            || ADC_ACP_InitStruct->ACPC_MAX_TARGET > ADC_ACP_MAX_MAX)
    {
        return ERROR;
    }

    if (ADC_ACP_InitStruct->ACP_EN == ENABLE)
    {
        if (ADC->CON1.CLKS == ADC_ClkS_PCLK)
        {
            ADC->ACPC.CLKS = ADC_ACPC_CLKS_PCLK;
        }
        else
        {
            ADC->ACPC.CLKS = ADC_ACPC_CLKS_LRC;
        }

        ADC->ACPC.OVFL_TIME = ADC_ACP_InitStruct->ACPC_OVER_TIME;
        ADC->ACPC.TIMES = ADC_ACP_InitStruct->ACPC_TIMES;
        ADC->IE.ACPOVIE = ENABLE;

        /* 假设用户将高阈值设置成0(最小值)，我们可以认为其想关闭该功能 */
        if (ADC_ACP_InitStruct->ACPC_MAX_TARGET == 0x0)
        {
            ADC->IE.ACPMAXIE = DISABLE;
        }
        else
        {
            ADC->ACPCMP.CMP_MAX = ADC_ACP_InitStruct->ACPC_MAX_TARGET;
            ADC->IE.ACPMAXIE = ENABLE;
        }

        /* 假设用户将低阈值设置成0xfff(最大值)，我们可以认为其想关闭该功能 */
        if (ADC_ACP_InitStruct->ACPC_MIN_TARGET == 0xFFF)
        {
            ADC->IE.ACPMINIE = DISABLE;
        }
        else
        {
            ADC->ACPCMP.CMP_MIN = ADC_ACP_InitStruct->ACPC_MIN_TARGET;
            ADC->IE.ACPMINIE = ENABLE;
        }

        ADC->CON0.ACP_EN = ENABLE;

    }
    else
    {
        ADC->CON0.ACP_EN = DISABLE;
    }

    return SUCCESS;
}

/***************************************************************
  函数名：ADC_SampStart
  描  述：ADC 采样软件控制-启动函数
  输入值：无
  输出值：无
  返回值：SUCCESS 成功
  ERROR 失败
 ***************************************************************/
ErrorStatus ADC_SoftStart(void)
{
    if (ADC->CON1.SMPS == ADC_SMPS_HARD)
        return ERROR;

    ADC->CON1.SMPON = SET;
    return SUCCESS;
}

/***************************************************************
  函数名：ADC_SampStop
  描  述：ADC 采样软件控制-停止函数
  输入值：无
  输出值：无
  返回值：SUCCESS 成功
  ERROR 失败
 ***************************************************************/
ErrorStatus ADC_SoftStop(void)
{
    if (ADC->CON1.SMPS == ADC_SMPS_HARD)
        return ERROR;

    ADC->CON1.SMPON = RESET;
    return SUCCESS;
}

/***************************************************************
  函数名：ADC_GetACPMeanValue
  描  述：ADC 获得单次自动比较平均值
  输入值：无
  输出值：无
  返回值：采样数据
 ***************************************************************/
uint16_t ADC_GetACPMeanValue(void)
{
    if (ADC->CON0.BIT_SEL == ADC_BITSEL_12)
        return ((uint16_t)ADC->ACPMEAN.MEAN_DATA);
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_10)
        return ((uint16_t)(ADC->ACPMEAN.MEAN_DATA >> 2));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_8)
        return ((uint16_t)(ADC->ACPMEAN.MEAN_DATA >> 4));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_6)
        return ((uint16_t)(ADC->ACPMEAN.MEAN_DATA >> 6));
    else
        return 0;
}

/***************************************************************
  函数名：ADC_GetACPMINValue
  描  述：ADC 获得单次自动比较低阈值
  输入值：无
  输出值：无
  返回值：采样数据
 ***************************************************************/
uint16_t ADC_GetACPMinValue(void)
{
    if (ADC->CON0.BIT_SEL == ADC_BITSEL_12)
        return ((uint16_t)ADC->ACPCMP.CMP_MIN);
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_10)
        return ((uint16_t)(ADC->ACPCMP.CMP_MIN >> 2));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_8)
        return ((uint16_t)(ADC->ACPCMP.CMP_MIN >> 4));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_6)
        return ((uint16_t)(ADC->ACPCMP.CMP_MIN >> 6));
    else
        return 0;
}

/***************************************************************
  函数名：ADC_GetACPMAXValue
  描  述：ADC 获得单次自动比较高阈值
  输入值：无
  输出值：无
  返回值：采样数据
 ***************************************************************/
uint16_t ADC_GetACPMaxValue(void)
{
    if (ADC->CON0.BIT_SEL == ADC_BITSEL_12)
        return ((uint16_t)ADC->ACPCMP.CMP_MAX);
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_10)
        return ((uint16_t)(ADC->ACPCMP.CMP_MAX >> 2));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_8)
        return ((uint16_t)(ADC->ACPCMP.CMP_MAX >> 4));
    else if (ADC->CON0.BIT_SEL == ADC_BITSEL_6)
        return ((uint16_t)(ADC->ACPCMP.CMP_MAX >> 6));
    else
        return 0;
}

/***************************************************************
  函数名：ADC_GetFlagStatus
  描  述：读取ADC标志位状态
  输入值：IFName：ADC中断标志
  输出值：无
  返回值：SET/RESET
 ***************************************************************/
FlagStatus ADC_GetFlagStatus(ADC_TYPE_IF IFName)
{
    FlagStatus bitstatus = RESET;

    /* 检查中断标志位 */
    if (((ADC->IF.Word & IFName) != (uint32_t)RESET))
        bitstatus = SET;                    //转换完成
    else
        bitstatus = RESET;                  //无中断

    return  bitstatus;
}

/***************************************************************
  函数名：ADC_GetITStatus
  描  述：读取ADC中断状态，未使能相应中断时不会返回SET
  输入值：无
  输出值：无
  返回值：SET（中断）/RESET（无中断）
 ***************************************************************/
ITStatus ADC_GetITStatus(ADC_TYPE_IE IEName)
{
    ITStatus bitstatus = RESET;

    /* 检查中断标志位 */
    if (((ADC->IE.Word & IEName) != (uint32_t)RESET))
        bitstatus = SET;                    //转换完成，进中断
    else
        bitstatus = RESET;                  //无中断

    return  bitstatus;
}

/***************************************************************
  函数名：ADC_ClearIFStatus
  描  述：ADC 清除特定类型中断
  输入值：IFName 中断类型选择
  ADC_IF            ADC中断
  ADC_IF_ACPMIN 自动转换低阈值超出中断
  ADC_IF_ACPMAX 自动转换高阈值超出中断
  ADC_IF_ACPOVER    自动转换溢出中断
  输出值：无
  返回值：SUCCESS 成功
  ERROR 失败
 ***************************************************************/
ErrorStatus ADC_ClearIFStatus(ADC_TYPE_IF IFName)
{
    ADC->IF.Word = (uint32_t)IFName;

    return SUCCESS;
}
/***************************************************************
  函数名：ADC_Reset
  描  述：ADC复位
  输入值：无
  输出值：无
  返回值：无
 ***************************************************************/
void ADC_Reset(void)
{
    ADC->CON0.Word = 0x00000030;
    ADC->CON1.Word = 0x00041000;
    ADC->CHS.Word = 0x00000100;
    ADC->IE.Word = 0x00000000;
    ADC->IF.Word = 0x00000000;
    ADC->ACPC.Word = 0x00000000;
    ADC->ACPCMP.Word = 0x0FFF0000;
#if defined(ES8H0384) || defined(ES8H0364) || defined(ES8H0394) || defined(ES8H0354) ||defined(ES8H0161) || defined(ES8H0163) || defined(ES8H0173) || defined(ES8H0181) || defined(ES8H0183) || defined(ES8H0193) || defined(ES8H0184)
    ADC->VREFCON.Word = 0x00000000;
#elif defined(ES8H0133)

#else
	#error The chip model is not defined
#endif
}

/*************************END OF FILE**********************/


