/*********************************************************
*Copyright (C), 2023, Shanghai Eastsoft Microelectronics Co., Ltd.
 * @文件名:  main.c
 * @作  者:  AE Team
 * @版  本:  V1.00
 * @日  期:  2023/10/17
 * @描  述:  MCU自带Flash模拟EEPROM
 *           开辟两个16kB Flash区域：0x8000~0xBFFF, 0xC000~0xFFFF，实现按halfword(16bit)写数据和读数据功能。
 *           测试步骤：
 *           1) 通过ESBurner使能芯片DATAFLASH空间：Page64~Page127；
 *           2) 在线调试，将地址0x8000和0xC000加入Memory Window，将g_var_data_tab加入Watch Window；
 *           3) 观察0x8000~0xBFFF和0xC000~0xFFFF的区域以及数组g_var_data_tab的数据变化。
 *           注意：
 *           1) 32bit为一组数据，高16bit为模拟地址（禁止使用0xFFFF），低16bit为模拟地址对应的数据。
 *           2) 电压低于设置的LVD阈值（本例程为4.1V）时，将停止后续的IAP操作。
 *           3) 按照本例程开辟两个16kB空间计算，IAP最长操作时间为166.4ms（极端情况），
 *              出现在一个空间满转移数据时，且有效数据有(16 * 1024 / 4 - 1)个，
 *              计算方法：
 *              转移数据时间：(16 * 1024 / 4 - 1)word * 0.025ms/word = 102.4ms
 *              擦数据时间：16 * 1024byte / 512byte * 2ms = 64ms
 *              总时间：166.4ms
 *           4) 实际应用时，应考虑电路板掉电后，VDD从LVD阈值降低到BOR电压点的时间，该时间应大于IAP最长操作时间。
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Oct. 2023    AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "main.h"

/* Virtual address defined by the user: 0xFFFFFFFF value is prohibited */
uint32_t g_virt_add_var_tab[NB_OF_VAR] = {0x5555U, 0x6666U, 0x7777U};
uint32_t g_var_data_tab[NB_OF_VAR] = {0U, 0U, 0U};
uint32_t g_var_value = 0U;

int main(void)
{
    HRC_Config(Enable, SCU_HRC_48M, Enable);
    SystemInit();
    DeviceClockAllEnable();
    User_SysTickInit();

    SCU_RegUnLock();
    SCU_LVDVS_4V1();        /* LVD阈值4.1V */
    SCU_LVDFLT_Enable();
    SCU_LVDIFS_High();      /* LVDO高电平产生中断（VDD低于阈值产生中断） */
    SCU_LVD_Enable();
    SCU_RegLock();

    __disable_irq();

    /* EEPROM Init */
    while (SCU_GetLVDFlagStatus(SCU_LVDFlag_IF) != RESET);  /* 等到VDD电压高于LVD阈值 */
    ee_init();

    /* --- Store successively many values of the three variables in the EEPROM ---*/
    /* Store 0x1000 values of Variable1 in EEPROM */
    for (g_var_value = 1; g_var_value <= 0x1000; g_var_value++)
        ee_write_variable(g_virt_add_var_tab[0], g_var_value);

    /* read the last stored variables data */
    ee_read_variable(g_virt_add_var_tab[0], &g_var_data_tab[0]);

    /* Store 0x2000 values of Variable2 in EEPROM */
    for (g_var_value = 1; g_var_value <= 0x2000; g_var_value++)
        ee_write_variable(g_virt_add_var_tab[1], g_var_value);

    /* read the last stored variables data*/
    ee_read_variable(g_virt_add_var_tab[0], &g_var_data_tab[0]);
    ee_read_variable(g_virt_add_var_tab[1], &g_var_data_tab[1]);

    /* Store 0x3000 values of Variable3 in EEPROM */
    for (g_var_value = 1; g_var_value <= 0x3000; g_var_value++)
        ee_write_variable(g_virt_add_var_tab[2], g_var_value);

    /* read the last stored variables data*/
    ee_read_variable(g_virt_add_var_tab[0], &g_var_data_tab[0]);
    ee_read_variable(g_virt_add_var_tab[1], &g_var_data_tab[1]);
    ee_read_variable(g_virt_add_var_tab[2], &g_var_data_tab[2]);

    ee_write_variable(g_virt_add_var_tab[0], 0x1234);
    ee_write_variable(g_virt_add_var_tab[1], 0x8765);
    ee_write_variable(g_virt_add_var_tab[2], 0x55AA);

    ee_read_variable(g_virt_add_var_tab[0], &g_var_data_tab[0]);
    ee_read_variable(g_virt_add_var_tab[1], &g_var_data_tab[1]);
    ee_read_variable(g_virt_add_var_tab[2], &g_var_data_tab[2]);

    __enable_irq();

    while (1);
}

/*************************END OF FILE**********************/
