/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    3 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          3 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
uint16_t g_duty_buffer[BUFFER_SIZE];
uint16_t g_cycle_buffer[BUFFER_SIZE];

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize CH1IN IO as input */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_INPUT;  /* input */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_FLOATING;    /* no pull up and pull down */
	gpio_init.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
	gpio_init.flt = MD_GPIO_FILTER_DISABLE;
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_3;      /* GPIO function 3 */
    md_gpio_init(CAPTURE_CH1_PORT, CAPTURE_CH1_PIN, &gpio_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;
    md_timer_ic_init_t ic_init;

    md_timer_base_struct_init(&ad16c4t_init);   /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 48 - 1;            /* clk_count: 1MHz */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1; /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_UP;   /* count up */
    ad16c4t_init.period = 0xFFFF;               /* set max period */
    ad16c4t_init.re_cnt = 1 - 1;                /* 0 repeat count */
    md_timer_base_set_config(AD16C4T, &ad16c4t_init);

    md_timer_ic_struct_init(&ic_init);                 /* initialize the ic_init  */
    ic_init.filter      = 0;                           /* 0 filter */
    ic_init.polarity    = MD_TIMER_IC_POLARITY_RISE;   /* capture rising edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_DIRECT;      /* capture this channel */
    md_timer_ic_init(AD16C4T, MD_TIMER_CHANNEL_1, &ic_init);
    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;
    ic_init.polarity    = MD_TIMER_IC_POLARITY_FALL;   /* capture falling edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_INDIRECT;    /* capture adjacent channel */
    md_timer_ic_init(AD16C4T, MD_TIMER_CHANNEL_2, &ic_init);

    md_timer_set_slave_mode_smods(AD16C4T, 4);        /* 0x4: Reset Mode */
    md_timer_set_slave_trigger_tssel(AD16C4T, 5);     /* 0x5: Filtered Timer Input 1 */

    md_timer_set_cc_dma_select_ccdmasel(AD16C4T, 0);  /* CCn DMA request sent when CCx event occurs */
    md_timer_enable_cc1dma(AD16C4T);                  /* enable CC1 DMA request */
    md_timer_enable_cc2dma(AD16C4T);                  /* enable CC2 DMA request */

    md_timer_enable_cc1it_interrupt(AD16C4T);         /* enable CC1 interrupt request */
    md_timer_enable_cc2it_interrupt(AD16C4T);         /* enable CC1 interrupt request */
    md_timer_enable_uit_interrupt(AD16C4T);           /* enable update interrupt request */
    csi_vic_enable_sirq(AD16C4T_UP_TRIG_COM_IRQn);
    csi_vic_enable_sirq(AD16C4T_CC_IRQn);
    __enable_irq();

    /* enable AD16C4T */
    md_timer_enable_counter_cnten(AD16C4T);
}

/**
  * @brief:  Use MD library to config DMA.
  * @param:  None
  * @retval: None
  */
static void init_dma(void)
{
    md_dma_config_t dma_config_cycle;
    md_dma_config_t dma_config_duty;

    memset(&dma_config_cycle, 0x0, sizeof(md_dma_config_t));
    dma_config_cycle.src            = (void *)(&AD16C4T->CCVAL1);
    dma_config_cycle.dst            = (void *)g_cycle_buffer;
    dma_config_cycle.size           = BUFFER_SIZE;
    dma_config_cycle.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_cycle.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_cycle.src_inc        = DISABLE;
    dma_config_cycle.dst_inc        = ENABLE;
    dma_config_cycle.R_power        = MD_DMA_R_POWER_1;
    dma_config_cycle.priority       = MD_DMA_LOW_PRIORITY;
    dma_config_cycle.mem_to_mem     = DISABLE;
    dma_config_cycle.circle_mode    = ENABLE;
    dma_config_cycle.msel           = MD_DMA_MSEL_AD16C4T;
    dma_config_cycle.msigsel        = MD_DMA_MSIGSEL_TIMER_CH1;
    md_dma_init(MD_DMA_CH_0, &dma_config_cycle);

    memset(&dma_config_duty, 0x0, sizeof(md_dma_config_t));
    dma_config_duty.src            = (void *)(&AD16C4T->CCVAL2);
    dma_config_duty.dst            = (void *)g_duty_buffer;
    dma_config_duty.size           = BUFFER_SIZE;
    dma_config_duty.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_duty.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_duty.src_inc        = DISABLE;
    dma_config_duty.dst_inc        = ENABLE;
    dma_config_duty.R_power        = MD_DMA_R_POWER_1;
    dma_config_duty.priority       = MD_DMA_LOW_PRIORITY;
    dma_config_duty.mem_to_mem     = DISABLE;
    dma_config_duty.circle_mode    = ENABLE;
    dma_config_duty.msel           = MD_DMA_MSEL_AD16C4T;
    dma_config_duty.msigsel        = MD_DMA_MSIGSEL_TIMER_CH2;
    md_dma_init(MD_DMA_CH_1, &dma_config_duty);

    csi_vic_enable_sirq(DMA_IRQn);
    md_dma_enable_channel(MD_DMA_CH_0);        /* enable DMA channel 0 */
    md_dma_enable_channel(MD_DMA_CH_1);        /* enable DMA channel 1 */
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Initialize the gpio */
    init_gpio();

    /* Initialize dma */
    init_dma();

    /* Initialize timer_AD16C4T */
    init_timer();

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
