/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    23 Sep. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Sep. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"
#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

static md_spi_init_t s_spi;
static md_spi_init_t s_spi1;

/* Public Variables ---------------------------------------------------------- */

md_dma_config_t g_dma_rx_config;

uint8_t g_complete;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

static void delay(int i);
static void spi_pin_init(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval Status.
  */
md_status_t flash_sector_erase(uint32_t addr)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;           /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != MD_OK)  /*First write send enabling instruction*/
    {
        FLASH_CS_SET();
        return MD_ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_write(uint32_t addr, char *buf, uint16_t size)
{
    uint8_t cmd_buf[4];
    uint16_t i = 0U;

    if (buf == NULL)
        return MD_ERROR;

    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != MD_OK)
    {
        FLASH_CS_SET();
        return MD_ERROR;
    }

    FLASH_CS_SET();    /*Pick up and release of Flash*/

    delay(100);
    FLASH_CS_CLR();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        if (md_spi_send_byte_fast(&s_spi, buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  dma_start_rx init.
  * @param  None
  * @retval None
  */
void dma_recv_rx(uint16_t size)
{
    md_dma_enable_it_tc(MD_DMA_CH_0);
    memset(&g_dma_rx_config, 0x0, sizeof(md_dma_config_t));
    g_dma_rx_config.src            = (void *)&SPI0->DATA;
    g_dma_rx_config.dst            = (void *)&SPI1->DATA;
    g_dma_rx_config.size           = size;
    g_dma_rx_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.src_inc        = DISABLE;
    g_dma_rx_config.dst_inc        = DISABLE;
    g_dma_rx_config.R_power        = MD_DMA_R_POWER_1;
    g_dma_rx_config.priority       = MD_DMA_LOW_PRIORITY;
    g_dma_rx_config.mem_to_mem     = DISABLE;
    g_dma_rx_config.circle_mode    = DISABLE;
    g_dma_rx_config.msel           = MD_DMA_MSEL_SPI0;
    g_dma_rx_config.msigsel        = MD_DMA_MSIGSEL_SPI_RNR;
    md_dma_init(MD_DMA_CH_0, &g_dma_rx_config);
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  size: Amount of data to be received
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_read(uint32_t addr, uint16_t size)
{
    uint32_t cnt = 40000U;
    uint8_t cmd_buf[4];
    uint16_t i = 0U;

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    s_spi.SPIx      = SPI0;                 /* Using SPI0 */
    s_spi.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi.baud      = MD_SPI_BAUD_32;      /* clock */
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;
    md_spi_init(&s_spi);                    /* According to the parameter initialization SPI peripherals */

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return MD_ERROR;
        }
    }

    g_complete = 0U;

    md_gpio_write_pin(GPIOB, MD_GPIO_PIN_12, 0);

    dma_recv_rx(size);
    md_dma_enable_channel(MD_DMA_CH_0);
    md_spi_enable_rxdma(SPI0);

	md_spi_enable_recv_only(s_spi.SPIx);

    cnt = 40000U;
    while (!g_complete && --cnt);

    md_spi_disable_recv_only(s_spi.SPIx);

    md_gpio_write_pin(GPIOB, MD_GPIO_PIN_12, 1);

	FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_wait_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    if (md_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != MD_OK)   /*Send to read status command*/
    {
        FLASH_CS_SET();     /* Pick up and release of Flash */
        return MD_ERROR;
    }

    do
    {
        status = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (md_spi_send_byte_fast(&s_spi, flash_id[i]) != MD_OK)
        {
            FLASH_CS_SET();     /* Pick up and release of Flash */
            return MD_ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief  Initialize PIS.
  * @param  None
  * @retval None
  */
void pis_init(void)
{
    md_pis_init_t pis_init;
    memset(&pis_init, 0, sizeof(md_pis_init_t));

    MD_SYSCFG_UNLOCK();
    md_syscfg_pisiocfg_io13sel(SYSCFG, MD_SYSCFG_PISIOCFG_PB);  /* PIS producer IO13 selects PB */
	md_syscfg_pisiocfg_io15sel(SYSCFG, MD_SYSCFG_PISIOCFG_PC);   /* PIS producer IO15 selects PC */
    MD_SYSCFG_LOCK();

    /* Enable PIS CH3 output */
    md_pis_enable_ch3_output(PIS);
    /* Enable PIS CH4 output */
    md_pis_enable_ch4_output(PIS);

    pis_init.p_src     = MD_PIS_GPIO_PIN13;     /* PIS SRC: PB13(SPI1_SCK) */
    pis_init.p_clk     = MD_PIS_CLK_HCLK;
    pis_init.p_edge    = MD_PIS_EDGE_NONE;
    pis_init.p_output  = MD_PIS_OUT_LEVEL;
    pis_init.c_trig    = 0x3;                   /* PIS CH3 output PB13 signal */
    pis_init.c_clk     = MD_PIS_CLK_HCLK;
    md_pis_init(&pis_init);

    pis_init.p_src     = MD_PIS_GPIO_PIN15;     /* PIS SRC: PC15(SPI1_MOSI) */
    pis_init.p_clk     = MD_PIS_CLK_HCLK;
    pis_init.p_edge    = MD_PIS_EDGE_NONE;
    pis_init.p_output  = MD_PIS_OUT_LEVEL;
    pis_init.c_trig    = 0x4;                   /* PIS CH4 output PC15 signal */
    pis_init.c_clk     = MD_PIS_CLK_HCLK;
    md_pis_init(&pis_init);
}

/**
  * @brief spi function
  * @retval None.
  */
void mcu_spi_init(void)
{
	pis_init();
    spi_pin_init();

    md_spi_struct_init(&s_spi);
    s_spi.SPIx      = SPI0;                 /* Using SPI0 */
    s_spi.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi.baud      = MD_SPI_BAUD_32;       /* clock */
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;
    md_spi_init(&s_spi);                    /* According to the parameter initialization SPI peripherals */

    md_spi_struct_init(&s_spi1);
    s_spi1.SPIx      = SPI1;                 /* Using SPI1 */
    s_spi1.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi1.baud      = MD_SPI_BAUD_32;       /* clock */
    s_spi1.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi1.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi1.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi1.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi1.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi1.ss_en     = DISABLE;
    s_spi1.crc_calc  = DISABLE;
    s_spi1.crc_poly  = 0;
    md_spi_init(&s_spi1);                    /* According to the parameter initialization SPI peripherals */
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void spi_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_PUSH_PULL;
    l_gpio.pupd  = MD_GPIO_PUSH_UP;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_STRONG;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;

    l_gpio.func  = MD_GPIO_FUNC_1;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &l_gpio);    /* Initialize PA4 for selected pin */
    FLASH_CS_SET();   /* Choose the output high, release of Flash */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &l_gpio);    /* Initialize PA5 for clock output pin */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_7, &l_gpio);    /* Initialize PA7 for MOSI pin */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_6, &l_gpio);    /* Initialization for PA6 MISO pins */

    l_gpio.func  = MD_GPIO_FUNC_1;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOB, MD_GPIO_PIN_12, &l_gpio);    /* Initialize PB12 for selected pin */
    md_gpio_write_pin(GPIOB, MD_GPIO_PIN_12, 1);   /* Choose the output high, release of Flash */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOB, MD_GPIO_PIN_13, &l_gpio);    /* Initialize PB13 for clock output pin */

    l_gpio.func = MD_GPIO_FUNC_6;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &l_gpio);    /* PIS_CH3 output PB13 */

    l_gpio.func  = MD_GPIO_FUNC_5;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOC, MD_GPIO_PIN_15, &l_gpio);    /* Initialize PC15 for MOSI pin */

    l_gpio.func = MD_GPIO_FUNC_6;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOB, MD_GPIO_PIN_4, &l_gpio);     /* PIS_CH4 output PC15 */
}

/**
  * @brief  delay some time.
  * @retval None.
  */
static void delay(int i)
{
    while (i--) ;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
