/**********************************************************************************
 *
 * @file    wm_i2s.c
 * @brief   I2S driver for WM8978
 *
 * @date    04 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          31 Dec 2021     shiwa           the first version
 *          04 Aug 2022     shiwa           for md lib
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */
#include <md_conf.h>
#include <md_spi.h>
#include <md_i2s.h>
#include <md_gpio.h>
#include <md_dma.h>
#include "main.h"
#include "wm8978.h"
#include "wm_i2s.h"

/* Private Macros ----------------------------------------------------------- */
#define LOG_E
#define LOG_I
#define DMA_MSEL_SPIX (s_I2SX==SPI0?MD_DMA_MSEL_SPI0:MD_DMA_MSEL_SPI1)
/* Private Variables--------------------------------------------------------- */
static md_i2s_init_t s_h_i2s;
static SPI_I2S_TypeDef *s_I2SX;
static void *s_i2s_buffers[2];
static uint16_t s_buffer_length;
static func_I2S_callback s_feed_data;
static func_I2S_callback s_recv_data;
static uint32_t s_i2s_send_dma_ch, s_i2s_recv_dma_ch;
static uint32_t s_status = 0;

/* Public Variables -------------------------------------------------------- */
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */
void i2s_dma_config(uint32_t is_recv, uint32_t ch);
/* Private Function---------------------------------------------------------- */
void wm_set_vol(uint32_t type, uint8_t vol)
{
    if (type & 0x1)
        WM8978_HPvol_Set(vol, vol);   /* Headphone volume */

    if (type & 0x2)
        WM8978_SPKvol_Set(vol);      /* Speaker volume */

    if (type & 0x4)
        WM8978_MIC_Gain(vol); /* Microphone gain */

    if (type & 0x8)
        WM8978_LINEIN_Gain(vol); /* Linein gain (not used) */
}

void wm_set_mode(uint32_t mode)
{
    if (mode == WM_SEND)
    {
        WM8978_ADDA_Cfg(1, 0);                              /*Open the DAC*/
        WM8978_Input_Cfg(0, 0, 0);                          /*Close the input channel*/
        WM8978_Output_Cfg(1, 0);                            /*Open the DAC output*/

    }
    else if (mode == WM_RECV)
    {
        WM8978_ADDA_Cfg(0, 1);
        WM8978_Input_Cfg(1, 1, 0);
        WM8978_Output_Cfg(0, 1);
    }
    else if (mode == WM_SENDRECV)
    {
        WM8978_ADDA_Cfg(1, 1);
        WM8978_Input_Cfg(1, 1, 0);
        WM8978_Output_Cfg(1, 0);
    }
	else
    {

	}
}

void wm_set_callback(uint32_t is_recv, func_I2S_callback f)
{
    if (is_recv)
        s_recv_data = f;
    else
        s_feed_data = f;
}

void wm_start(void)
{
    md_i2s_enable(s_I2SX);
    s_status = 1;
}

void wm_stop(void)
{
    md_i2s_disable(s_I2SX);
    s_status = 0;
}

void wm_pause(void)
{
    if (s_status)
        md_i2s_disable(s_I2SX);
}

void wm_resume(void)
{
    if (s_status)
        md_i2s_enable(s_I2SX);
}

uint32_t wm_status(void)
{
    return s_status;
}

int wm_set_sample_rate(uint32_t rate)
{
    if (rate != s_h_i2s.sampling)
    {
        md_i2s_disable(s_I2SX);
        s_h_i2s.sampling = rate;

        if (md_i2s_init(s_I2SX, &s_h_i2s))
            return 1;
    }

    return 0;
}

/**
  * @brief  Init I2S for WM8978
  * @retval None
  */
int wm_i2s_init(struct wm_i2s_info *wm_info)
{
    s_I2SX = (SPI_I2S_TypeDef *)wm_info->spi_x;
    md_i2s_struct_init(&s_h_i2s);
    s_h_i2s.ch_len      = MD_I2S_WIDE_16;
    s_h_i2s.data_len    = MD_I2S_LEN_16;
    s_h_i2s.polarity    = MD_I2S_INACTIVE_LOW;
    s_h_i2s.standard    = MD_I2S_STD_PHI;
    s_h_i2s.ext_clk_en  = DISABLE;
    s_h_i2s.ext_clk     = 0;
    s_h_i2s.mck_en      = ENABLE;
    s_h_i2s.sampling    = wm_info->sample_rate;

    if (md_i2s_init(s_I2SX, &s_h_i2s))
        return 1;

    s_i2s_buffers[0] = wm_info->buffers[0];
    s_i2s_buffers[1] = wm_info->buffers[1];
    s_buffer_length = wm_info->buffer_length > 1024 ? 1024 : wm_info->buffer_length;
    s_feed_data = wm_info->feed_data;
    s_recv_data = wm_info->recv_data;

    s_i2s_send_dma_ch = wm_info->dma_tx_ch;
    s_i2s_recv_dma_ch = wm_info->dma_rx_ch;

    md_i2s_disable(s_I2SX);
    md_i2s_set_mode(s_I2SX, MD_I2S_MASTER_DUPLEX);
    md_i2s_enable_mclock(s_I2SX);

    i2s_dma_config(0, s_i2s_send_dma_ch);
    md_dma_enable_it_tc(s_i2s_send_dma_ch);
    md_dma_enable_it_ht(s_i2s_send_dma_ch);
    md_i2s_enable_txdma(s_I2SX);
    md_dma_enable_channel(s_i2s_send_dma_ch);

    i2s_dma_config(1, s_i2s_recv_dma_ch);
    md_dma_enable_it_tc(s_i2s_recv_dma_ch);
    md_dma_enable_it_ht(s_i2s_recv_dma_ch);
    md_dma_enable_channel(s_i2s_recv_dma_ch);
    md_i2s_enable_rxdma(s_I2SX);
    return 0;
}

/**
  * @brief  callback to set I2Ssend data
  * @retval None
  */
void i2s_dma_send_callback(void *arg)
{
    s_feed_data(arg, s_buffer_length / 2);
}

/**
  * @brief  callback to receive i2s data
  * @retval None
  */
void i2s_dma_recv_callback(void *arg)
{
    s_recv_data(arg, s_buffer_length / 2);
}

/**
  * @brief  setup dma for i2s
  * @retval None
  */
void i2s_dma_config(uint32_t is_recv, uint32_t ch)
{
    md_dma_config_t dma_config;

    if (is_recv)
    {
        g_dma_tc_callbacks[ch] = i2s_dma_recv_callback;
        g_dma_ht_callbacks[ch] = i2s_dma_recv_callback;
        g_dma_tc_callback_datas[ch] = s_i2s_buffers[1] + s_buffer_length / 2 * 2;
        g_dma_ht_callback_datas[ch] = s_i2s_buffers[1];
        dma_config.src            = (void *)&s_I2SX->DATA;
        dma_config.dst            = s_i2s_buffers[1];
        dma_config.msigsel        = MD_DMA_MSIGSEL_SPI_RNR;
        dma_config.src_inc        = DISABLE;
        dma_config.dst_inc        = ENABLE;
    }
    else
    {
        g_dma_tc_callbacks[ch] = i2s_dma_send_callback;
        g_dma_ht_callbacks[ch] = i2s_dma_send_callback;
        g_dma_tc_callback_datas[ch] = s_i2s_buffers[0] + s_buffer_length / 2 * 2;
        g_dma_ht_callback_datas[ch] = s_i2s_buffers[0];
        dma_config.src            = s_i2s_buffers[0];
        dma_config.dst            = (void *)&s_I2SX->DATA;
        dma_config.msigsel        = MD_DMA_MSIGSEL_SPI_TXEMPTY;
        dma_config.src_inc        = ENABLE;
        dma_config.dst_inc        = DISABLE;
    }

    dma_config.size           = s_buffer_length;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_HIGHEST_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = ENABLE;
    dma_config.msel           = DMA_MSEL_SPIX;

    md_dma_clear_flag_tc(ch);
    md_dma_init(ch, &dma_config);
}
