/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    28 Dec. 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          28 Dec. 2022    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
md_adc_init_t g_adc_init;
uint32_t g_adc_result;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  Configure the ADC Pins.
  * @param  None
  * @retval None
  */
void gpio_pin_config(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize adc gpio pin */
    gpio_init.mode  = MD_GPIO_MODE_CLOSE;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;

    md_gpio_init(ADC_CHANNEL6_GPIO_PORT, ADC_CHANNEL6_PIN, &gpio_init);

    return;
}

/**
  * @brief  Configure the ADC.
  * @param  None
  * @retval None
  */
void adc_base_config(void)
{
    md_adc_struct_init(&g_adc_init);

    /* Set ADC Base Configuration:  */
    g_adc_init.align    = MD_ADC_DATAALIGN_RIGHT;     /* Specifies ADC data alignment */
    g_adc_init.data_bit = MD_ADC_CONV_BIT_12;
    g_adc_init.div      = MD_ADC_POSDIV_16;           /* ADCCLK divider */
    g_adc_init.n_ref    = MD_ADC_NEG_REF_VSS;
    g_adc_init.p_ref    = MD_ADC_POS_REF_VDD;         /* The positive reference voltage*/
    md_adc_init(ADC, &g_adc_init);

    MD_ADC_SPEED_HIGH_ENABLE(ADC);
    /* Disable ADC sequencers scan mode */
    md_adc_scan_mode_disable_scanen(ADC);
    /* Set ADC group REG sequencer length and scan direction, ADC conversion 1 channel */
    md_adc_set_insert_channel_length_isl(ADC, MD_ADC_ICH_NR_1);
    /* Set ADC continuous conversion mode on ADC group REG.*/
    md_adc_continuous_conversion_disable_cm(ADC);
    /* Set ADC group REG sequence: channel on the selected scan sequence rank. */
    md_adc_set_insert_1st_conv_is1(ADC, MD_ADC_CHANNEL_6);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_6);

    /* ADC insert trig polarity */
    md_adc_set_insert_conv_extern_polarity(ADC, MD_ADC_ETS_RISE);

    /* Enable interruption ADC group insert end of sequence conversions. */
    md_adc_inserted_channel_interrupt_enable_icheie(ADC);
}

/**
  * @brief  Initialize TIMER.
  * @param  None
  * @retval None
  */
void init_timer(void)
{
    md_timer_base_init_t init;
    md_timer_base_struct_init(&init);
    init.prescaler = 480;
    init.period = 10000;
    md_timer_base_set_config(AD16C4T, &init);
}

/**
  * @brief  Initialize PIS.
  * @param  None
  * @retval None
  */
void init_pis(void)
{
    md_pis_init_t pis_init;

    /* Initialize PIS */
    memset(&pis_init, 0, sizeof(md_pis_init_t));

    pis_init.p_src     = MD_PIS_TIMER0_UPDATA;
    pis_init.p_clk     = MD_PIS_CLK_PCLK;
    pis_init.p_edge    = MD_PIS_EDGE_NONE;
    pis_init.p_output  = MD_PIS_OUT_PULSE;
    pis_init.c_trig    = MD_PIS_CH5_ADC0_INSERT;
    pis_init.c_clk     = MD_PIS_CLK_PCLK;
    md_pis_init(&pis_init);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Enable the selected ADC instance */
    md_adc_converter_enable_adcen(ADC);
    /* Disable the selected ADC instance */
    md_adc_converter_disable_adcen(ADC);

    /* Initialize PIS */
    init_pis();

    /* Initialize TIMER */
    init_timer();

    /* Pin Configuration */
    gpio_pin_config();

    md_mcu_irq_config(ADC_IRQn, 0, ENABLE);
    __enable_irq();

    /* ADC Base Feature Configuration */
    adc_base_config();

    /* Enable the selected ADC instance.*/
    md_adc_converter_enable_adcen(ADC);

    /* Start TIMER */
    md_timer_enable_counter_cnten(AD16C4T);

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
