/**********************************************************************************
 *
 * @file    crc32.c
 * @brief   crc32 C file
 *
 * @date    28 Apr. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          28 Apr. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "crc32.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */
/* Public Variables --------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Reset the CRC peripheral.
  * @retval None
  */
void set_crc_reset_reg(void)__attribute((section(".crc32_section")));
void set_crc_reset_reg(void)
{
    CRC->DATA = 0x0;
    CRC->CR   = 0x2;
    CRC->SEED = 0xFFFFFFFF;

    return;
}

/**
  * @brief  Init CRC module.
  * @retval None.
  */
void set_crc_init(md_crc_init_t *init)__attribute((section(".crc32_section")));
void set_crc_init(md_crc_init_t *init)
{
    uint32_t tmp = 0U;

    set_crc_reset_reg();

    tmp |= ((init->chs_rev << CRC_CR_CHSREV_POS) | (init->data_inv << CRC_CR_DATINV_POS) |
            (init->chs_inv << CRC_CR_CHSINV_POS) | (init->mode << CRC_CR_MODE_POSS) |
            (init->len << CRC_CR_DATLEN_POSS) | (init->data_rev << CRC_CR_DATREV_POS) |
            (init->order << CRC_CR_BYTORD_POS));
    WRITE_REG(CRC->CR, tmp);
    WRITE_REG(CRC->SEED, init->seed);
    MD_CRC_RESET();
    MD_CRC_ENABLE();

    return;
}

/**
  * @brief  Init structure which describe CRC configuration.
  * @retval None.
  */
void set_crc_init_struct(md_crc_init_t *init)__attribute((section(".crc32_section")));
void set_crc_init_struct(md_crc_init_t *init)
{
    init->mode     = MD_CRC_MODE_CCITT;
    init->len      = MD_CRC_DATASIZE_8;
    init->order    = MD_CRC_BYTORD_LOW;
    init->data_rev = DISABLE;
    init->data_inv = DISABLE;
    init->chs_rev  = DISABLE;
    init->chs_inv  = DISABLE;
    init->seed     = 0x0;

    return;
}

/**
  * @brief  Init CRC module as standard CRC32.
  * @retval None.
  */
void init_crc32_module(void)__attribute((section(".crc32_section")));
void init_crc32_module(void)
{
    md_crc_init_t g_init;
    set_crc_init_struct(&g_init);

    g_init.mode = MD_CRC_MODE_32;
    g_init.seed = 0xFFFFFFFF;
    g_init.chs_inv = ENABLE;
    g_init.chs_rev = ENABLE;
    g_init.data_inv = DISABLE;
    g_init.data_rev = ENABLE;

    set_crc_init(&g_init);

    return;
}

/**
  * @brief  Calculate result accoding to crc algorithm.
  * @retval CRC32 calculation result.
  */
uint32_t get_crc32_calculation_result(uint32_t *buf, uint32_t size)__attribute((section(".crc32_section")));
uint32_t get_crc32_calculation_result(uint32_t *buf, uint32_t size)
{
    uint32_t i;

    for (i = 0; i < size; i++)
        CRC->DATA = buf[i];

    return (CRC->CHECKSUM);
}

/**
  * @}
  */
/**
  * @}
  */
