/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    10 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          10 Jan. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
md_adc_init_t g_adc_init;
uint16_t g_adc_result[4];
uint8_t g_index = 0;
uint8_t g_tx_buf[20];

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  格式化用户数据，使之符合传输协议
  * @param  srcdata-保存用户数据的首地址；
  *         desdata-保存处理好的数据的内存首地址(其内存大小至少大于前者5字节)；
  *         len-用户数据长度(单位:字节)
  * @retval 返回数据帧的总长度
  */
uint32_t data_initialize(uint8_t *desdata, uint8_t *srcdata, uint32_t len)
{
    uint32_t i;

    /* 帧头 */
    desdata[0] = 0xAA;
    desdata[1] = 0xAA;

    /* 功能码 */
    desdata[2] = 0xF1;

    /* 数据长度 */
    desdata[3] = len;

    /* 用户数据 */
    memcpy(desdata + 4, srcdata, len);

    /* 累加校验和 */
    desdata[len + 4] = 0U;

    for (i = 0; i < len + 4; i++)
    {
        /* 累加校验和, 高位舍去, 只保留低8位 */
        desdata[len + 4] += desdata[i];
    }

    return (len + 5);
}

/**
  * @brief  Configure the ADC Pins.
  * @param  None
  * @retval None
  */
void gpio_pin_config(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize adc gpio pin */
    gpio_init.mode  = MD_GPIO_MODE_CLOSE;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;

    md_gpio_init(ADC_CHANNEL4_GPIO_PORT, ADC_CHANNEL4_PIN, &gpio_init);
    md_gpio_init(ADC_CHANNEL5_GPIO_PORT, ADC_CHANNEL5_PIN, &gpio_init);
    md_gpio_init(ADC_CHANNEL6_GPIO_PORT, ADC_CHANNEL6_PIN, &gpio_init);
    md_gpio_init(ADC_CHANNEL7_GPIO_PORT, ADC_CHANNEL7_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200U;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);
}

/**
  * @brief  Configure the ADC.
  * @param  None
  * @retval None
  */
void adc_base_config(void)
{
    md_adc_struct_init(&g_adc_init);

    /* Set ADC Base Configuration:  */
    g_adc_init.align    = MD_ADC_DATAALIGN_RIGHT;     /* Specifies ADC data alignment */
    g_adc_init.data_bit = MD_ADC_CONV_BIT_12;
    g_adc_init.div      = MD_ADC_POSDIV_16;           /* ADCCLK divider */
    g_adc_init.n_ref    = MD_ADC_NEG_REF_VSS;
    g_adc_init.p_ref    = MD_ADC_POS_REF_VDD;         /* The positive reference voltage*/
    md_adc_init(ADC, &g_adc_init);

    MD_ADC_SPEED_HIGH_ENABLE(ADC);
    /* Trigger the NCHE FALG mode */
    md_adc_set_eoc_selection_nchesel(ADC, MD_ADC_NCHESEL_MODE_ONE);
    /*Enable ADC sequencers scan mode*/
    md_adc_scan_mode_enable_scanen(ADC);
    /* Set ADC group REG sequencer length and scan direction, ADC conversion 4 channel */
    md_adc_set_normal_channel_length_nsl(ADC, MD_ADC_NCH_NR_4);
    /* Set ADC continuous conversion mode on ADC group REG.*/
    md_adc_continuous_conversion_disable_cm(ADC);
    /*Set ADC group REG sequencer discontinuous mode */
    md_adc_normal_channel_discon_mode_enable_nchdcen(ADC);
    /*Set ADC group REG sequencer discontinuous channels number */
    md_adc_set_ex_trigger_discon_etrgn(ADC, MD_ADC_DISC_NR_2);

    /*Set ADC group REG sequence: channel on the selected scan sequence rank.*/
    md_adc_set_normal_1st_conv_ns1(ADC, MD_ADC_CHANNEL_4);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_4);
    md_adc_set_normal_2st_conv_ns2(ADC, MD_ADC_CHANNEL_5);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_5);
    md_adc_set_normal_3st_conv_ns3(ADC, MD_ADC_CHANNEL_6);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_6);
    md_adc_set_normal_4st_conv_ns4(ADC, MD_ADC_CHANNEL_7);
    md_adc_set_smpt1_cht(ADC, MD_ADC_SAMPLETIME_4, MD_ADC_CHANNEL_7);

    /*Enable interruption ADC group REG end of sequence conversions.*/
    md_adc_conv_end_interrupt_enable_ncheie(ADC);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    uint16_t cnt;
	uint32_t length, i;

    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Enable the selected ADC instance */
    md_adc_converter_enable_adcen(ADC);
    /* Disable the selected ADC instance */
    md_adc_converter_disable_adcen(ADC);

    /* Pin Configuration */
    gpio_pin_config();
	/* UART Configuration */
	uart_init();
    /* ADC Base Feature Configuration */
    adc_base_config();

    md_mcu_irq_config(ADC_IRQn, 0, ENABLE);
    __enable_irq();

    /* Enable the selected ADC instance.*/
    md_adc_converter_enable_adcen(ADC);

    while (1)
    {
		if(g_index >= 4)
		{
			g_index = 0;

            length = data_initialize(g_tx_buf, (uint8_t *)g_adc_result, 2 * 4);

            for (i = 0; i < length; i++)
            {
			    md_uart_set_send_data8(UARTX, g_tx_buf[i]);

                cnt = 4000U;
                while (!md_uart_is_active_it_tbc(UARTX) && (--cnt));

                md_uart_clear_it_tbc(UARTX);
            }
		}

		/* Enable ADC group REG conversion trigger source internal (SW start)*/
	    md_adc_set_normal_channel_conv_start_nchtrg(ADC);

        md_delay_1ms(10);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
